/*  TilEm, TI-Linux Emulator
 *  Copyright (C) 2001 Solignac Julien <x1cygnus@xcalc.org>
 *  Portions copyright (C) 2004 Benjamin Moody <benjamin@ecg.mit.edu>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

extern byte lcdactive, lcdmode, lcdinc;

void handlelink(byte line);

int x4_protectport(byte portaddr);
byte x4_gettime(int shift);

extern byte xs_calc_md5(byte mode, unsigned long *regs, byte shift, byte oshift);

byte x4_Z80_In(byte Port)
{
	byte v, i;

	switch(Port) {
		case 0x00:
			handlelink((~tilnk & pclnk) & 0x03);
			return((~tilnk & pclnk) & 0x03);

		case 0x01:
			updatekb();

			v = 0xFF;
			for (i = 0; i < 7; i++)
				v &= ((keymask >> i) & 0x01)?0xFF:scode[i];

			return(v);

		case 0x02:
		case 0x03:
			return(0xe3 | (flashlock << 2));

		case 0x04:
			if (port4) {
				port4--;
				return(0x01);
			}

			if (onkey) {
				onkey--;
				return(0x01);
			}

			return(0x8A);

		case 0x05:
			return(pager[3-(2*memmode)] & 0x07);

		case 0x06:
			return(pager[1+memmode]&0x3f)|
			  ((pager[1+memmode]&0x40)<<1);

		case 0x07:
			return(pager[2+memmode]&0x3f)|
			  ((pager[2+memmode]&0x40)<<1);

		case 0x08:
			return(dbusstat & DBUS_DISABLED);

		case 0x09:
			handlelink((~tilnk & pclnk) & 0x03);
			return(dbusstat & 0x78);

		case 0x0A:
			handlelink((~tilnk & pclnk) & 0x03);
			if (dbusstat & DBUS_DATA) {
				pclnk = 3;
				tilnk = 0;
				dbusstat = DBUS_READY;
				/* printf("read %02X\n",dbusread); */
			}
			return(dbusread);

		case 0x10:
		  	return(0x40 | lcdmode<<6 | lcdactive<<5 | (lcdinc&3));

		case 0x11:
			return(LCDIn());

		case 0x1C:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 0));

		case 0x1D:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 8));

		case 0x1E:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 16));

		case 0x1F:
			return(xs_calc_md5(md5mode, md5regs, md5shift, 24));

		case 0x20:
			return(port20);

		case 0x21:
			return(port21);

		case 0x22:
			return(port22);

		case 0x23:
			return(port23);

		case 0x27:
			return(port27);

		case 0x28:
			return(port28);

		case 0x30:
		case 0x31:
		case 0x32:
		case 0x33:
		case 0x34:
		case 0x35:
		case 0x36:
		case 0x37:
		case 0x38:
			return 0;

		case 0x39:
			return(0xf0);

		case 0x40:
			return clockmode;

		case 0x41:	
			return clockset&0xff;

		case 0x42:
			return (clockset>>8)&0xff;

		case 0x43:
			return (clockset>>16)&0xff;

		case 0x44:
			return (clockset>>24)&0xff;

		case 0x45:
			return x4_gettime(0);

		case 0x46:
			return x4_gettime(8);

		case 0x47:
			return x4_gettime(16);

		case 0x48:
			return x4_gettime(24);

		case 0x4C:
			return(0x22);

		case 0x4D:
			/* USB port - not emulated, calculator should
			   recognize that the USB cable is
			   disconnected.

			   Thanks go to Dan Englender for these
			   values. */

			return(0xA5);

		case 0x55:
			return(0x1F);

		case 0x56:
			return(0x00);

		case 0x57:
			return(0x50);
	}
	printf("** unimplemented port ** %d\n", Port);
	return(0x00);
}


void x4_Z80_Out(byte Port, byte Value)
{
	extern byte lcdshow;

	time_t curtime;
	byte x;

	switch(Port) {
		case 0x00:
			tilnk = Value;
			handlelink((~tilnk & pclnk) & 0x03);
			break;

		case 0x01:
			keymask = Value;
			break;

		case 0x04:
		  	if (memmode != (Value & 1)) {
			  if (memmode == 0) {
			    x = pager[3];
			    pager[3] = pager[2]; /* old port 7 value */
			    pager[2] = pager[1]; /* old port 6 value */
			    pager[1] = x; /* old port 5 value */
			  }
			  else {
			    x = pager[1];
			    pager[1] = pager[2]; /* old port 6 value */
			    pager[2] = pager[3]; /* old port 7 value */
			    pager[3] = x; /* old port 5 value */
			  }
			}
			memmode = Value & 1;
			break;
	
		case 0x05:
		  	pager[3-(2*memmode)] = (Value & 0x07) | 0x40;
			break;

		case 0x06:
			pager[1+memmode] = Value&0x80 ? 0x40 | (Value & 7) :
				(Value & 0x3f);
			break;

		case 0x07:
		  	pager[2+memmode] = Value&0x80 ? 0x40 | (Value & 7) :
				(Value & 0x3f);
			break;

		case 0x08:
			if (Value & 0x80)
				dbusstat |= DBUS_DISABLED;
			else
				dbusstat = DBUS_READY;
			break;

		case 0x0D:
		  /* printf("write %02X\n",Value); */
			if (!(dbusstat & DBUS_READY)) {
			  printf("** writing data too fast to DBUS **\n");
			  dbusstat |= DBUS_ERROR;
			}
			else {
			  dbusstat &= ~DBUS_READY;
			  dbusstat |= DBUS_WRITE;
			  dbuswrite = Value;
			}
			handlelink((~tilnk & pclnk) & 0x03);
			break;

		case 0x10:
			LCDDriver(Value);
			break;

		case 0x11:
			lcdshow = 1;
			LCDOut(Value);
			break;

		case 0x14:
			if (x4_protectport(0x14))
			  flashlock = Value&1;
			break;

		case 0x18:
		case 0x19:
		case 0x1A:
		case 0x1B:
		case 0x1C:
		case 0x1D:
			md5regs[Port-0x18] = (md5regs[Port-0x18] >> 8) |
				(Value << 24);
			break;

		case 0x1E:
			md5shift = Value & 0x1f;
			break;

		case 0x1F:
			md5mode = Value & 3;
			break;

		case 0x20:
			/* Speed control - NYI */
			port20 = Value;
			break;

		case 0x21:
			if (x4_protectport(0x21))
			  port21 = Value;
			break;

		case 0x22:
			if (x4_protectport(0x22))
			  port22 = Value;
			break;

		case 0x23:
		  	if (x4_protectport(0x23))
			  port23 = Value;
			break;

		case 0x40:
			time(&curtime);
			if ((clockmode&1) != (Value&1)) {
				/* turning off: TITime = SysTime - offset;
				   turning on: offset = SysTime - TITime */
				clockoff = curtime - clockoff;
			}
			if (!(Value&2)) {
				clockoff = curtime - clockset;
			}
			clockmode = Value&3;
			break;

		case 0x41:
			clockset = (clockset&0xffffff00)|Value;
			break;

		case 0x42:
			clockset = (clockset&0xffff00ff)|(Value<<8);
			break;

		case 0x43:
			clockset = (clockset&0xff00ffff)|(Value<<16);
			break;

		case 0x44:
			clockset = (clockset&0x00ffffff)|(Value<<24);
			break;
	}

	return;
}


/* void x4_emkeyin(unsigned short keyv, byte i) */
/* { */
/* 	if (i == 0xFF) { */

/* 		i = 0; */
/* 		while ((keyv != keycodes[i]) && (i < sizeof keycodes)) */
/* 			i++; */

/* /\* Support for 2nd Enter Key *\/ */
/* 		if (keyv == GDK_KP_Enter) */
/* 			i = 0x08; */
/* 	} */

/* 	if (i == 40) { */
/* 		onkey = -1; */
/* 		return; */
/* 	} */

/* 	if (i != sizeof keycodes) */
/* 		scode[i >> 3] &= ~(1 << (i & 0x07)); */

/* 	return; */
/* } */


/* void x4_emkeyup(unsigned short keyv, byte i) */
/* { */
/* 	if (i == 0xFF) { */
/* 		i = 0; */
/* 		while ((keyv != keycodes[i]) && (i < sizeof keycodes)) */
/* 			i++; */

/* /\* Support for 2nd Enter Key *\/ */
/* 		if (keyv == GDK_KP_Enter) */
/* 			i = 0x08; */
/* 	} */

/* 	if (i == 40) { */
/* 	  onkey = 0; */
/* 	  return; */
/* 	} */

/* 	if (i != sizeof keycodes) */
/* 		scode[i >> 3] |= 1 << (i & 0x07); */

/* 	return; */
/* } */


int x4_protectport(byte portaddr)
{
  /* Note this is not as strong as TI's protection (which monitors the
   * data bus and requires you to actually execute the instructions
   * NOP, NOP, IM 1, DI.)  This only requires the appropriate sequence
   * of instructions to be present in memory.
   */

  unsigned long pa;

  byte unprotect_opstring[] = {0x00,0x00,0xed,0x56,0xf3,0xd3};

  pa = (R.PC.W.l & 0x3FFF) + 0x4000*pager[(R.PC.W.l)>>14];

  if ((pa < (0x3c*0x4000) || pa >= (0x40*0x4000)) &&
      (pa < (0x2c*0x4000) || pa >= (0x30*0x4000))) {
	  printf("** writing to port %02x from illegal address %06lx **\n",
		 portaddr,pa);
	  return 0;
  }

  if (memcmp(mpages+pa-7,unprotect_opstring,6) || mpages[pa-1]!=portaddr) {
	  printf("** writing to port %02x without proper unlock sequence **\n",
		 portaddr);
	  return 0;
  }

  return 1;
}


byte x4_gettime(int shift)
{
  static time_t curtime;

  if (clockmode&1) {
    time(&curtime);
    return ((curtime-clockoff)>>shift)&0xff;
  }
  else
    return (clockoff>>shift)&0xff;
}
