/*  TilEm, TI-Linux Emulator
 *  Copyright (C) 2001 Solignac Julien <x1cygnus@xcalc.org>
 *  Portions copyright (C) 2004 Benjamin Moody <benjamin@ecg.mit.edu>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* >>> Imported */
#include "../../../tools.h"
/* Imported <<< */

extern char *romfile;
extern char *savfile;

#ifdef extlink
void ti68k_close_linkport();
void ti68k_open_linkport();
#endif

void xp_fix_cert(byte *cert);

void xp_Rezet(byte i)
{
	extern Z80_Regs R;
	extern byte lcdshow;
	extern byte lcdactive;

	FILE *rfile;
	char *fname;

	lcdmode = xlcd = ylcd = 0;
	lcdinc = 4;
	port2 = 0;
	port4 = 100000;
	onkey = 0;
	tilnk = 0;
	pclnk = 0x03;
	dbusstat = DBUS_DISABLED | DBUS_READY;

	pager[0] = 0x00;
	pager[1] = 0x1F; 
	pager[2] = 0x1F; 
	pager[3] = 0x20;
	memmode = 0;

	R.PC.D = 0x8000;

	flashmode = 0;
	flashlock = 1;
	port5 = 0;
	noexec[0] = noexec[1] = noexec[2] = noexec[3] = noexec[4] = 0;

	memset(mpages + 0x20 * 0x4000, 0x00, 0x02 * 0x4000);
	if (i == 0) return;

	fname = savfile ? savfile : ahome("xp.sav");
	if ((rfile = fopen(fname, "rb"))) {
		lcdshow = lcdactive = 1;
		fread(pager, sizeof(byte), 4, rfile);
		fread(&R, sizeof(byte), sizeof(Z80_Regs), rfile);
		fread(mpages + 0x20 * 0x4000, sizeof(byte), 0x02 * 0x4000 + 0x300, rfile);
		lcdmode = fgetc(rfile);
		xlcd = fgetc(rfile);
		ylcd = fgetc(rfile);
		lcdinc = fgetc(rfile);
		memmode = fgetc(rfile);
		flashmode = fgetc(rfile);
		flashlock = fgetc(rfile);
		fclose(rfile);
	}
	if (!savfile)
		free(fname);

	fname = romfile ? romfile : ahome("xp.rom");
	if ((rfile = fopen(fname, "rb"))) {
		fread(mpages, sizeof(byte), 0x20 * 0x4000, rfile);
		xp_fix_cert(mpages + (0x1E * 0x4000L));
		fclose(rfile);
	}
	if (!romfile)
		free(fname);
}


byte xp_initmem(void)
{
	FILE *rfile;
	char *fname;

	if (romfile)
		printf(">>> TI-83 Plus (%s)\n", romfile);
	else
		puts(">>> TI-83 Plus");

	if (!(mpages = malloc((0x20 + 2) * 0x4000 + 0x300)))  {
		fprintf(stderr, "Error Allocating Memory\n");
		return(0);
	}

	lcdbuf = mpages + (0x20 + 2) * 0x4000;
	memset(lcdbuf, 0xFF, 0x300);

	fname = romfile ? romfile : ahome("xp.rom");
	if (!(rfile = fopen(fname, "rb"))) {
		fprintf(stderr, "Rom File IO Error\n");
		free(mpages);
		return(0);
	}
	if (!romfile)
		free(fname);

	if (fread(mpages, sizeof(byte), 0x20 * 0x4000, rfile) != 0x20 * 0x4000) {
		fprintf(stderr, "Rom File Corrupted\n");
		free(mpages);
		return(0);
	}

	fclose(rfile);

	xp_fix_cert(mpages + (0x1E * 0x4000L));

#ifdef extlink
	ti68k_open_linkport();
#endif

	return(1);
}


void xp_cleanmem()
{
	extern byte savecode;
	extern Z80_Regs R;

	FILE *rfile;
	char *fname;

#ifdef extlink
	ti68k_close_linkport();
#endif

	if (savecode) {
		fname = romfile ? romfile : ahome("xp.rom");
		if (!(rfile = fopen(fname, "wb"))) return;
		if (!romfile)
			free(fname);

		fwrite(mpages, sizeof(byte), 0x20 * 0x4000, rfile);
		fclose(rfile);

		fname = savfile ? savfile : ahome("xp.sav");
		if (!(rfile = fopen(fname, "wb"))) return;
		if (!savfile)
			free(fname);

		fwrite(pager, sizeof(byte), 4, rfile);
		fwrite(&R, sizeof(byte), sizeof(Z80_Regs), rfile);
		fwrite(mpages + 0x20 * 0x4000, sizeof(byte), 0x02 * 0x4000 + 0x300, rfile);
		fputc(lcdmode, rfile);
		fputc(xlcd, rfile);
		fputc(ylcd, rfile);
		fputc(lcdinc, rfile);
		fputc(memmode, rfile);
		fputc(flashmode, rfile);
		fputc(flashlock, rfile);
		fclose(rfile);
	}

	free(mpages);
}

void xp_fix_cert(byte *cert)
{
	int i;
	int base;

	/* If the ROM was dumped from an unpatched OS, the certificate
	   needs to be patched for some calculator functions to
	   work. */

	/* First, check if the certificate is already valid */

	if (cert[0x2000] == 0)
		base = 0x2000;
	else
		base = 0;

	if (cert[base] == 0) {
		/* Unfortunately, older versions of TilEm patched the
		   certificate incompletely.  Check for app-enabled
		   bits to be set in the correct sequence, which they
		   most likely won't be in a dumped ROM. */

		for (i = (base + 0x1FE0); i < (base + 0x1FED); i++)
			if ((((cert[i] ^ 0xff) >> 1) & cert[i])
			    || (i > (base + 0x1FE0)
				&& (cert[i - 1] & 0x80)
				&& (cert[i] != 0xff)))
				i = 0x4000;

		if (i < 0x4000)
			return;
	}

	fprintf(stderr, "Invalid certificate found; patching...\n");

	cert[0] = 0;
	for (i = 1; i < 0x1FE0; i++)
		cert[i] = 0xff;
	for (; i < 0x1fed; i++)
		cert[i] = 0;
	for (; i < 0x4000; i++)
		cert[i] = 0xff;
}

int xp_checkrom(char *fname)
{
	FILE *rfile;

	rfile = fopen(fname, "rb");
	if (!rfile)
		return 0;

	if (find_string("TI-83 Plus", rfile)) {
		fclose(rfile);
		return 1;
	}

	fclose(rfile);
	return 0;
}

