/***************************************************************************
  common.c  Generic functions used in different emulators.
***************************************************************************/

#include "driver.h"

int readroms(const struct RomModule *rommodule,const char *basename)
{
	int region;
	const struct RomModule *romp;
	romp = rommodule;
	for (region = 0;region < MAX_MEMORY_REGIONS;region++)
		Machine->memory_region[region] = 0;
	region = 0;

	while (romp->name || romp->offset || romp->length)
	{
		int region_size;
		if (romp->name || romp->length)
		{
			printf("Error in RomModule definition: expecting ROM_REGION\n");
			goto getout;
		}

		region_size = romp->offset;
		if ((Machine->memory_region[region] = malloc(region_size)) == 0)
		{
			printf("Unable to allocate %d bytes of RAM\n",region_size);
			goto getout;
		}

		memset(Machine->memory_region[region],0,region_size);
		romp++;

		while (romp->length)
		{
			FILE *f;
			char buf[100];
			char name[100];
			if (romp->name == 0)
			{
				printf("Error in RomModule definition: ROM_CONTINUE not preceded by ROM_LOAD\n");
				goto getout;
			}

			sprintf(buf,romp->name,basename);
			sprintf(name,"%s",buf);

			if ((f = fopen(name,"rb")) == 0)
			{
				printf("Unable to open ROM %s\n",name);
				goto printromlist;
			}

			do
			{
				if (romp->offset + romp->length > region_size)
				{
					printf("Error in RomModule definition: %s out of memory region space\n",name);
					fclose(f);
					goto getout;
				}

				if (fread(Machine->memory_region[region] + romp->offset,1,romp->length,f) != romp->length)
				{
					printf("Unable to read ROM %s\n",name);
					fclose(f);
					goto printromlist;
				}

				romp++;
			} while (romp->length && romp->name == 0);

			fclose(f);
		}

		region++;
	}

	return 0;


printromlist:
	romp = rommodule;
	printf("These ROMs are required on this disk.\n"
			"Name             Size\n");
	while (romp->name || romp->offset || romp->length)
	{
		romp++;	/* skip memory region definition */

		while (romp->length)
		{
			char name[100];
			int length;
			sprintf(name,romp->name,basename);
			length = 0;

			do
			{
				length += romp->length;
				romp++;
			} while (romp->length && romp->name == 0);

			printf("%-12s %5d bytes\n",name,length);
		}
	}

getout:
	for (region = 0;region < MAX_MEMORY_REGIONS;region++)
	{
		free(Machine->memory_region[region]);
		Machine->memory_region[region] = 0;
	}
	return 1;
}


int readbit(const unsigned char *src,int bitnum)
{
	int bit;
	bit = src[bitnum / 8] << (bitnum % 8);
	if (bit & 0x80) return 1;
	else return 0;
}


void decodechar(struct GfxElement *gfx,int num,const unsigned char *src,const struct GfxLayout *gl)
{
	int plane;
	for (plane = 0;plane < gl->planes;plane++)
	{
		int offs,y;
		offs = num * gl->charincrement + gl->planeoffset[plane];
		for (y = 0;y < gl->height;y++)
		{
			int x;
			for (x = 0;x < gl->width;x++)
			{
				unsigned char *dp;
				dp = gfx->gfxdata->line[num * gl->height + y];
				if (plane == 0) dp[x] = 0;
				else dp[x] <<= 1;
				dp[x] += readbit(src,offs + gl->yoffset[y] + gl->xoffset[x]);
			}
		}
	}
}


struct GfxElement *decodegfx(const unsigned char *src,const struct GfxLayout *gl)
{
	int c;
	struct osd_bitmap *bm;
	struct GfxElement *gfx;
	if ((bm = osd_create_bitmap(gl->width,gl->total * gl->height)) == 0)
		return 0;
	if ((gfx = malloc(sizeof(struct GfxElement))) == 0)
	{
		osd_free_bitmap(bm);
		return 0;
	}
	gfx->width = gl->width;
	gfx->height = gl->height;
	gfx->total_elements = gl->total;
	gfx->color_granularity = 1 << gl->planes;
	gfx->gfxdata = bm;

	for (c = 0;c < gl->total;c++)
		decodechar(gfx,c,src,gl);

	return gfx;
}


void freegfx(struct GfxElement *gfx)
{
	if (gfx)
	{
		osd_free_bitmap(gfx->gfxdata);
		free(gfx);
	}
}

void drawgfx(struct osd_bitmap *dest,const struct GfxElement *gfx,
		unsigned int code,unsigned int color,int flipx,int flipy,int sx,int sy,
		const struct rectangle *clip,int transparency,int transparent_color)
{
	int ox,oy,ex,ey,x,y,start;
	const unsigned char *sd;
	unsigned char *bm;
	int col;

	if (!gfx) return;

	/* check bounds */
	ox = sx;
	oy = sy;
	ex = sx + gfx->width-1;
	if (sx < 0) sx = 0;
	if (clip && sx < clip->min_x) sx = clip->min_x;
	if (ex >= dest->width) ex = dest->width-1;
	if (clip && ex > clip->max_x) ex = clip->max_x;
	if (sx > ex) return;
	ey = sy + gfx->height-1;
	if (sy < 0) sy = 0;
	if (clip && sy < clip->min_y) sy = clip->min_y;
	if (ey >= dest->height) ey = dest->height-1;
	if (clip && ey > clip->max_y) ey = clip->max_y;
	if (sy > ey) return;

	start = (code % gfx->total_elements) * gfx->height;
	if (gfx->colortable)	/* remap colors */
	{
		const unsigned char *paldata;
		paldata = &gfx->colortable[gfx->color_granularity * (color % gfx->total_colors)];
		switch (transparency)
		{
			case TRANSPARENCY_NONE:
				if (flipx)
				{
					if (flipy)	/* XY flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
								*(bm++) = paldata[*(sd--)];
						}
					}
					else 	/* X flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
								*(bm++) = paldata[*(sd--)];
						}
					}
				}
				else
				{
					if (flipy)	/* Y flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + (sx-ox);
							for (x = sx;x <= ex;x++)
								*(bm++) = paldata[*(sd++)];
						}
					}
					else		/* normal */
					{
						/* unrolled loop for the most common case */
						if (ex-sx+1 == 8)
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd = gfx->gfxdata->line[start + (y-oy)] + (sx-ox);
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*(bm++) = paldata[*(sd++)];
								*bm = paldata[*sd];
							}
						}
						else
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd = gfx->gfxdata->line[start + (y-oy)] + (sx-ox);
								for (x = sx;x <= ex;x++)
									*(bm++) = paldata[*(sd++)];
							}
						}
					}
				}
				break;

			case TRANSPARENCY_PEN:
				if (flipx)
				{
					if (flipy)	/* XY flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = *(sd--);
								if (col != transparent_color) *bm = paldata[col];
								bm++;
							}
						}
					}
					else 	/* X flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = *(sd--);
								if (col != transparent_color) *bm = paldata[col];
								bm++;
							}
						}
					}
				}
				else
				{
					if (flipy)	/* Y flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = *(sd++);
								if (col != transparent_color) *bm = paldata[col];
								bm++;
							}
						}
					}
					else		/* normal */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = *(sd++);
								if (col != transparent_color) *bm = paldata[col];
								bm++;
							}
						}
					}
				}
				break;

			case TRANSPARENCY_COLOR:
				if (flipx)
				{
					if (flipy)	/* XY flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = paldata[*(sd--)];
								if (col != transparent_color) *bm = col;
								bm++;
							}
						}
					}
					else 	/* X flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = paldata[*(sd--)];
								if (col != transparent_color) *bm = col;
								bm++;
							}
						}
					}
				}
				else
				{
					if (flipy)	/* Y flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = paldata[*(sd++)];
								if (col != transparent_color) *bm = col;
								bm++;
							}
						}
					}
					else		/* normal */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + (sx-ox);
							for (x = sx;x <= ex;x++)
							{
								col = paldata[*(sd++)];
								if (col != transparent_color) *bm = col;
								bm++;
							}
						}
					}
				}
				break;
		}
	}
	else
	{
		switch (transparency)
		{
			case TRANSPARENCY_NONE:		/* do a verbatim copy (faster) */
				if (flipx)
				{
					if (flipy)	/* XY flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
								*(bm++) = *(sd--);
						}
					}
					else 	/* X flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + gfx->width-1 - (sx-ox);
							for (x = sx;x <= ex;x++)
								*(bm++) = *(sd--);
						}
					}
				}
				else
				{
					if (flipy)	/* Y flip */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + (sx-ox);
							memcpy(bm,sd,ex-sx+1);
						}
					}
					else		/* normal */
					{
						for (y = sy;y <= ey;y++)
						{
							bm = dest->line[y] + sx;
							sd = gfx->gfxdata->line[start + (y-oy)] + (sx-ox);
							memcpy(bm,sd,ex-sx+1);
						}
					}
				}
				break;

			case TRANSPARENCY_PEN:
			case TRANSPARENCY_COLOR:
				{
					int *sd4,x1;
					int trans4;


					trans4 = transparent_color * 0x01010101;

					if (flipx)
					{
						if (flipy)	/* XY flip */
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd4 = (int *)(gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + gfx->width-1 - (sx-ox) - 3);
								for (x = sx;x <= ex;x+=4)
								{
									if (*sd4 == trans4)
									{
										bm += 4;
									}
									else
									{
										sd = ((unsigned char *)sd4) + 3;
										x1 = ex - x;
										if (x1 > 3) x1 = 3;
										while (x1 >= 0)
										{
											col = *(sd--);
											if (col != transparent_color) *bm = col;
											bm++;
											x1--;
										}
									}
									sd4--;
								}
							}
						}
						else 	/* X flip */
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd4 = (int *)(gfx->gfxdata->line[start + (y-oy)] + gfx->width-1 - (sx-ox) - 3);
								for (x = sx;x <= ex;x+=4)
								{
									if (*sd4 == trans4)
									{
										bm += 4;
									}
									else
									{
										sd = ((unsigned char *)sd4) + 3;
										x1 = ex - x;
										if (x1 > 3) x1 = 3;
										while (x1 >= 0)
										{
											col = *(sd--);
											if (col != transparent_color) *bm = col;
											bm++;
											x1--;
										}
									}
									sd4--;
								}
							}
						}
					}
					else
					{
						if (flipy)	/* Y flip */
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd4 = (int *)(gfx->gfxdata->line[start + gfx->height-1 - (y-oy)] + (sx-ox));
								for (x = sx;x <= ex;x+=4)
								{
									if (*sd4 == trans4)
									{
										bm += 4;
									}
									else
									{
										sd = (unsigned char *)sd4;
										x1 = ex - x;
										if (x1 > 3) x1 = 3;
										while (x1 >= 0)
										{
											col = *(sd++);
											if (col != transparent_color) *bm = col;
											bm++;
											x1--;
										}
									}
									sd4++;
								}
							}
						}
						else		/* normal */
						{
							for (y = sy;y <= ey;y++)
							{
								bm = dest->line[y] + sx;
								sd4 = (int *)(gfx->gfxdata->line[start + (y-oy)] + (sx-ox));
								for (x = sx;x <= ex;x+=4)
								{
									if (*sd4 == trans4)
									{
										bm += 4;
									}
									else
									{
										sd = (unsigned char *)sd4;
										x1 = ex - x;
										if (x1 > 3) x1 = 3;
										while (x1 >= 0)
										{
											col = *(sd++);
											if (col != transparent_color) *bm = col;
											bm++;
											x1--;
										}
									}
									sd4++;
								}
							}
						}
					}
				}
				break;
		}
	}
}

void copybitmap(struct osd_bitmap *dest,struct osd_bitmap *src,int flipx,int flipy,int sx,int sy,
		const struct rectangle *clip,int transparency,int transparent_color)
{
	static struct GfxElement mygfx =
	{
		0,0,0,	/* filled in later */
		1,1,0,1
	};

	mygfx.width = src->width;
	mygfx.height = src->height;
	mygfx.gfxdata = src;
	drawgfx(dest,&mygfx,0,0,flipx,flipy,sx,sy,clip,transparency,transparent_color);
}

void copyscrollbitmap(struct osd_bitmap *dest,struct osd_bitmap *src,
		int rows,int *rowscroll,int cols,int *colscroll,
		const struct rectangle *clip,int transparency,int transparent_color)
{
	if (rows == 0)
	{
		/* scrolling columns */
		int col,colwidth;
		struct rectangle myclip;
		colwidth = src->width / cols;
		myclip.min_y = clip->min_y;
		myclip.max_y = clip->max_y;

		col = 0;
		while (col < cols)
		{
			int cons,scroll;

			scroll = colscroll[col];
			cons = 1;
			while (col + cons < cols &&	colscroll[col + cons] == scroll)
				cons++;

			myclip.min_x = col * colwidth;
			if (myclip.min_x < clip->min_x) myclip.min_x = clip->min_x;
			myclip.max_x = (col + cons) * colwidth - 1;
			if (myclip.max_x > clip->max_x) myclip.max_x = clip->max_x;

			if (scroll < 0) scroll = src->height - (-scroll) % src->height;
			else scroll %= src->height;

			copybitmap(dest,src,0,0,0,scroll,&myclip,transparency,transparent_color);
			copybitmap(dest,src,0,0,0,scroll - src->height,&myclip,transparency,transparent_color);

			col += cons;
		}
	}
	else if (cols == 0)
	{
		int row,rowheight;
		struct rectangle myclip;

		rowheight = src->height / rows;

		myclip.min_x = clip->min_x;
		myclip.max_x = clip->max_x;

		row = 0;
		while (row < rows)
		{
			int cons,scroll;

			scroll = rowscroll[row];
			cons = 1;
			while (row + cons < rows &&	rowscroll[row + cons] == scroll)
				cons++;

			myclip.min_y = row * rowheight;
			if (myclip.min_y < clip->min_y) myclip.min_y = clip->min_y;
			myclip.max_y = (row + cons) * rowheight - 1;
			if (myclip.max_y > clip->max_y) myclip.max_y = clip->max_y;

			if (scroll < 0) scroll = src->width - (-scroll) % src->width;
			else scroll %= src->width;

			copybitmap(dest,src,0,0,scroll,0,&myclip,transparency,transparent_color);
			copybitmap(dest,src,0,0,scroll - src->width,0,&myclip,transparency,transparent_color);

			row += cons;
		}
	}
	else if (rows == 1 && cols == 1)
	{
		int scrollx,scrolly;

		if (rowscroll[0] < 0) scrollx = src->width - (-rowscroll[0]) % src->width;
		else scrollx = rowscroll[0] % src->width;

		if (colscroll[0] < 0) scrolly = src->height - (-colscroll[0]) % src->height;
		else scrolly = colscroll[0] % src->height;

		copybitmap(dest,src,0,0,scrollx,scrolly,clip,transparency,transparent_color);
		copybitmap(dest,src,0,0,scrollx,scrolly - src->height,clip,transparency,transparent_color);
		copybitmap(dest,src,0,0,scrollx - src->width,scrolly,clip,transparency,transparent_color);
		copybitmap(dest,src,0,0,scrollx - src->width,scrolly - src->height,clip,transparency,transparent_color);
	}
}



void clearbitmap(struct osd_bitmap *bitmap)
{
	int i;
	for (i = 0;i < bitmap->height;i++)
		memset(bitmap->line[i],Machine->background_pen,bitmap->width);
}



int setdipswitches(void)
{
	struct DisplayText dt[40];
	int settings[20];
	int i,s,key;
	int total;
	const struct DSW *dswsettings;

	dswsettings = Machine->gamedrv->dswsettings;

	total = 0;
	while (dswsettings[total].num != -1)
	{
		int msk,val;

		msk = dswsettings[total].mask;
		if (msk == 0) return 0;	/* error in DSW definition, quit */
		val = Machine->gamedrv->input_ports[dswsettings[total].num].default_value;
		while ((msk & 1) == 0)
		{
			val >>= 1;
			msk >>= 1;
		}
		settings[total] = val & msk;

		total++;
	}

	s = 0;
	do
	{
		for (i = 0;i < total;i++)
		{
			dt[2 * i].color = (i == s) ? Machine->gamedrv->yellow_text : Machine->gamedrv->white_text;
			dt[2 * i].text = dswsettings[i].name;
			dt[2 * i].x = 2*Machine->gfx[0]->width;
			dt[2 * i].y = 2*Machine->gfx[0]->height * i + (Machine->drv->screen_height - 2*Machine->gfx[0]->height * (total - 1)) / 2;
			dt[2 * i + 1].color = (i == s) ? Machine->gamedrv->yellow_text : Machine->gamedrv->white_text;
			dt[2 * i + 1].text = dswsettings[i].values[settings[i]];
			dt[2 * i + 1].x = Machine->drv->screen_width - 2*Machine->gfx[0]->width - Machine->gfx[0]->width*strlen(dt[2 * i + 1].text);
			dt[2 * i + 1].y = dt[2 * i].y;
		}
		dt[2 * i].text = 0;	/* terminate array */

		displaytext(dt,1);
		key = osd_read_key();

		switch (key)
		{
			case OSD_KEY_DOWN:
				if (s < total - 1) s++;
				break;

			case OSD_KEY_UP:
				if (s > 0) s--;
				break;

			case OSD_KEY_RIGHT:
				if (dswsettings[s].reverse == 0)
				{
					if (dswsettings[s].values[settings[s] + 1] != 0) settings[s]++;
				}
				else
				{
					if (settings[s] > 0) settings[s]--;
				}
				break;

			case OSD_KEY_LEFT:
				if (dswsettings[s].reverse == 0)
				{
					if (settings[s] > 0) settings[s]--;
				}
				else
				{
					if (dswsettings[s].values[settings[s] + 1] != 0) settings[s]++;
				}
				break;
		}
	} while (key != OSD_KEY_TAB && key != OSD_KEY_ESC);

	while (osd_key_pressed(key));	/* wait for key release */

	while (--total >= 0)
	{
		int msk;

		msk = dswsettings[total].mask;
		while ((msk & 1) == 0)
		{
			settings[total] <<= 1;
			msk >>= 1;
		}

		Machine->gamedrv->input_ports[dswsettings[total].num].default_value =
				(Machine->gamedrv->input_ports[dswsettings[total].num].default_value
				& ~dswsettings[total].mask) | settings[total];
	}

	clearbitmap(Machine->scrbitmap);
	if (key == OSD_KEY_ESC) return 1;
	else return 0;
}


void displaytext(const struct DisplayText *dt,int erase)
{
	if (erase) clearbitmap(Machine->scrbitmap);
	while (dt->text)
	{
		int x,y;
		const char *c;

		x = dt->x;
		y = dt->y;
		c = dt->text;

		while (*c)
		{
			if (*c == '\n')
			{
				x = dt->x;
				y += Machine->gfx[0]->height + 1;
			}
			else if (*c == ' ')
			{
				if (x == dt->x)
					x += Machine->gfx[0]->width;
				else
				{
					int nextlen;
					const char *nc;

					x += Machine->gfx[0]->width;
					nc = c+1;
					while (*nc && *nc != ' ' && *nc != '\n')
						nc++;

					nextlen = nc - c - 1;

					if (x + nextlen * Machine->gfx[0]->width >= Machine->drv->screen_width)
					{
						x = dt->x;
						y += Machine->gfx[0]->height + 1;
					}
				}
			}
			else
			{
				if (*c >= '0' && *c <= '9')
					drawgfx(Machine->scrbitmap,Machine->gfx[0],Machine->gamedrv->charset[*c - '0'],dt->color,0,0,x,y,0,TRANSPARENCY_NONE,0);
				else if (*c >= 'A' && *c <= 'Z')
					drawgfx(Machine->scrbitmap,Machine->gfx[0],Machine->gamedrv->charset[*c - 'A' + 10],dt->color,0,0,x,y,0,TRANSPARENCY_NONE,0);

				x += Machine->gfx[0]->width;
			}
			c++;
		}
		dt++;
	}
	osd_update_display();
}

