/***************************************************************************
Pengo driver - Sil's version
***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"

extern unsigned char *pengo_soundregs;
extern void pengo_gfxbank_w(int offset,int data);
extern void pengo_vh_convert_color_prom(unsigned char *palette, unsigned char *colortable,const unsigned char *color_prom);
extern int pengo_vh_start(void);
extern void pengo_vh_screenrefresh(struct osd_bitmap *bitmap);

extern void pengo_sound_enable_w(int offset,int data);
extern void pengo_sound_w(int offset,int data);
extern void pengo_sh_update(void);

static struct MemoryReadAddress readmem[] =
{
	{ 0x8000, 0x8fff, MRA_RAM },	/* video and color RAM, scratchpad RAM, sprite codes */
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x90c0, 0x90ff, input_port_0_r },	/* IN0 */
	{ 0x9080, 0x90bf, input_port_1_r },	/* IN1 */
	{ 0x9040, 0x907f, input_port_2_r },	/* DSW1 */
	{ 0x9000, 0x903f, input_port_3_r },	/* DSW2 */
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress writemem[] =
{
	{ 0x8800, 0x8fef, MWA_RAM },
	{ 0x8000, 0x83ff, videoram_w, &videoram },
	{ 0x8400, 0x87ff, colorram_w, &colorram },
	{ 0x9000, 0x901f, pengo_sound_w, &pengo_soundregs },
	{ 0x8ff0, 0x8fff, MWA_RAM, &spriteram},
	{ 0x9020, 0x902f, MWA_RAM, &spriteram_2 },
	{ 0x9040, 0x9040, interrupt_enable_w },
	{ 0x9070, 0x9070, MWA_NOP },
	{ 0x9041, 0x9041, pengo_sound_enable_w },
	{ 0x9047, 0x9047, pengo_gfxbank_w },
	{ 0x9042, 0x9046, MWA_NOP },
	{ 0x0000, 0x7fff, MWA_ROM },
	{ -1 }	/* end of table */
};

static struct InputPort input_ports[] =
{
	{	/* IN0 */
		0xff,
		{ OSD_KEY_UP, OSD_KEY_DOWN, OSD_KEY_LEFT, OSD_KEY_RIGHT,
				0, 0, OSD_KEY_3, OSD_KEY_CONTROL },
		{ OSD_JOY_UP, OSD_JOY_DOWN, OSD_JOY_LEFT, OSD_JOY_RIGHT,
				0, 0, 0, OSD_JOY_FIRE }
	},
	{	/* IN1 */
		0xff,
		{ 0, 0, 0, 0, OSD_KEY_F2, OSD_KEY_1, OSD_KEY_2, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSW1 */
		0xb0,
		{ 0, 0, 0, 0, 0, OSD_KEY_F1, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSW2 */
		0xff,
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{ -1 }	/* end of table */
};

static struct DSW dsw[] =
{
	{ 2, 0x18, "LIVES", { "5", "4", "3", "2" }, 1 },
	{ 2, 0x01, "BONUS", { "30000", "50000" } },
	{ 2, 0xc0, "DIFFICULTY", { "HARDEST", "HARD", "MEDIUM", "EASY" }, 1 },
	{ 2, 0x02, "DEMO SOUNDS", { "ON", "OFF" }, 1 },
	{ -1 }
};

static struct GfxLayout charlayout =
{
	8,8,	/* 8*8 characters */
	256,	/* 256 characters */
	2,	/* 2 bits per pixel */
	{ 0, 4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 7*8, 6*8, 5*8, 4*8, 3*8, 2*8, 1*8, 0*8 }, /* characters are rotated 90 degrees */
	{ 8*8+0, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 },	/* bits are packed in groups of four */
	16*8	/* every char takes 16 bytes */
};
static struct GfxLayout spritelayout =
{
	16,16,	/* 16*16 sprites */
	64,	/* 64 sprites */
	2,	/* 2 bits per pixel */
	{ 0, 4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3, 0, 1, 2, 3 },
	64*8	/* every sprite takes 64 bytes */
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,      0, 32 },	/* first bank */
	{ 1, 0x1000, &spritelayout,    0, 32 },
	{ 1, 0x2000, &charlayout,   4*32, 32 },	/* second bank */
	{ 1, 0x3000, &spritelayout, 4*32, 32 },
	{ -1 } /* end of array */
};

static unsigned char color_prom[] =
{
	/* palette */
	0x00,0xF6,0x07,0x38,0xC9,0xF8,0x3F,0xEF,0x6F,0x16,0x2F,0x7F,0xF0,0x36,0xDB,0xC6,
	0x00,0xF6,0xD8,0xF0,0xF8,0x16,0x07,0x2F,0x36,0x3F,0x7F,0x28,0x32,0x38,0xEF,0xC6,
	/* color lookup table */
	0x00,0x00,0x00,0x00,0x00,0x05,0x03,0x01,0x00,0x05,0x02,0x01,0x00,0x05,0x06,0x01,
	0x00,0x05,0x07,0x01,0x00,0x05,0x0A,0x01,0x00,0x05,0x0B,0x01,0x00,0x05,0x0C,0x01,
	0x00,0x05,0x0D,0x01,0x00,0x05,0x04,0x01,0x00,0x03,0x06,0x01,0x00,0x03,0x02,0x01,
	0x00,0x03,0x07,0x01,0x00,0x03,0x05,0x01,0x00,0x02,0x03,0x01,0x00,0x00,0x00,0x00,
	0x00,0x08,0x03,0x01,0x00,0x09,0x02,0x05,0x00,0x08,0x05,0x0D,0x04,0x04,0x04,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x02,0x00,0x03,0x03,0x03,
	0x00,0x06,0x06,0x06,0x00,0x07,0x07,0x07,0x00,0x0A,0x0A,0x0A,0x00,0x0B,0x0B,0x0B,
	0x00,0x01,0x01,0x01,0x00,0x05,0x05,0x05,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
	0x00,0x00,0x00,0x00,0x00,0x03,0x07,0x0D,0x00,0x0C,0x0F,0x0B,0x00,0x0C,0x0E,0x0B,
	0x00,0x0C,0x06,0x0B,0x00,0x0C,0x07,0x0B,0x00,0x0C,0x03,0x0B,0x00,0x0C,0x08,0x0B,
	0x00,0x0C,0x0D,0x0B,0x00,0x0C,0x04,0x0B,0x00,0x0C,0x09,0x0B,0x00,0x0C,0x05,0x0B,
	0x00,0x0C,0x02,0x0B,0x00,0x0C,0x0B,0x02,0x00,0x08,0x0C,0x02,0x00,0x08,0x0F,0x02,
	0x00,0x03,0x02,0x01,0x00,0x02,0x0F,0x03,0x00,0x0F,0x0E,0x02,0x00,0x0E,0x07,0x0F,
	0x00,0x07,0x06,0x0E,0x00,0x06,0x05,0x07,0x00,0x05,0x00,0x06,0x00,0x00,0x0B,0x05,
	0x00,0x0B,0x0C,0x00,0x00,0x0C,0x0D,0x0B,0x00,0x0D,0x08,0x0C,0x00,0x08,0x09,0x0D,
	0x00,0x09,0x0A,0x08,0x00,0x0A,0x01,0x09,0x00,0x01,0x04,0x0A,0x00,0x04,0x03,0x01
};

/* waveforms for the audio hardware */
static unsigned char samples[8*32] =
{
	0x00,0x00,0x00,0x00,0x00,0x77,0x77,0x00,0x00,0x88,0x88,0x88,0x00,0x00,0x00,0x00,
	0x77,0x77,0x77,0x00,0x88,0x88,0x00,0x00,0x00,0x00,0x77,0x77,0x00,0x00,0x88,0x88,

	0xff,0x11,0x22,0x33,0xff,0x55,0x55,0xff,0x66,0xff,0x55,0x55,0xff,0x33,0x22,0x11,
	0xff,0xdd,0xff,0xbb,0xff,0x99,0xff,0x88,0xff,0x88,0xff,0x99,0xff,0xbb,0xff,0xdd,

	0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,0x66,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,

	0x00,0x00,0x00,0x88,0x00,0x00,0x77,0x77,0x88,0x88,0x00,0x00,0x00,0x77,0x77,0x77,
	0x88,0x00,0x00,0x88,0x77,0x77,0x00,0x00,0x00,0x00,0x77,0x00,0x88,0x88,0x88,0x00,

	0xff,0x22,0x44,0x55,0x66,0x55,0x44,0x22,0xff,0xcc,0xaa,0x99,0x88,0x99,0xaa,0xcc,
	0xff,0x33,0x55,0x66,0x55,0x33,0xff,0xbb,0x99,0x88,0x99,0xbb,0xff,0x66,0xff,0x88,

	0xff,0x66,0x44,0x11,0x44,0x66,0x22,0xff,0x44,0x77,0x55,0x00,0x22,0x33,0xff,0xaa,
	0x00,0x55,0x11,0xcc,0xdd,0xff,0xaa,0x88,0xbb,0x00,0xdd,0x99,0xbb,0xee,0xbb,0x99,

	0xff,0x00,0x22,0x44,0x66,0x55,0x44,0x44,0x33,0x22,0x00,0xff,0xdd,0xee,0xff,0x00,
	0x00,0x11,0x22,0x33,0x11,0x00,0xee,0xdd,0xcc,0xcc,0xbb,0xaa,0xcc,0xee,0x00,0x11,

	0x22,0x44,0x44,0x22,0xff,0xff,0x00,0x33,0x55,0x66,0x55,0x22,0xee,0xdd,0xdd,0xff,
	0x11,0x11,0x00,0xcc,0x99,0x88,0x99,0xbb,0xee,0xff,0xff,0xcc,0xaa,0xaa,0xcc,0xff
};

static struct MachineDriver machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			3072000,	/* 3.072 Mhz */
			0,
			readmem,writemem,0,0,
			interrupt,1
		}
	},
	60,
	0,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	gfxdecodeinfo,
	32,4*64,
	pengo_vh_convert_color_prom,

	0,
	pengo_vh_start,
	generic_vh_stop,
	pengo_vh_screenrefresh,

	/* sound hardware */
	samples,
	0,
	0,
	0,
	pengo_sh_update
};

/***************************************************************************
  Game driver(s)
***************************************************************************/

ROM_START( pengo_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "pengo.u8",  0x0000, 0x1000 )
	ROM_LOAD( "pengo.u7",  0x1000, 0x1000 )
	ROM_LOAD( "pengo.u15", 0x2000, 0x1000 )
	ROM_LOAD( "pengo.u14", 0x3000, 0x1000 )
	ROM_LOAD( "pengo.u21", 0x4000, 0x1000 )
	ROM_LOAD( "pengo.u20", 0x5000, 0x1000 )
	ROM_LOAD( "pengo.u32", 0x6000, 0x1000 )
	ROM_LOAD( "pengo.u31", 0x7000, 0x1000 )

	ROM_REGION(0x4000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "pengo.u92", 0x0000, 0x2000 )
	ROM_LOAD( "pengo.105", 0x2000, 0x2000 )
ROM_END

static int hiload(const char *name)
{
	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x8840],"\xd0\x07",2) == 0 &&
			memcmp(&RAM[0x8858],"\xd0\x07",2) == 0)
	{
		FILE *f;
		if ((f = fopen(name,"rb")) != 0)
		{
			fread(&RAM[0x8840],1,6*5,f);
			RAM[0x880c] = RAM[0x8858];
			RAM[0x880d] = RAM[0x8859];
			fclose(f);
		}
		return 1;
	}
	else return 0;
}

static void hisave(const char *name)
{
	FILE *f;
	if ((f = fopen(name,"wb")) != 0)
	{
		fwrite(&RAM[0x8840],1,6*5,f);
		fclose(f);
	}
}

struct GameDriver pengo_driver =
{
	"pengo",
	&machine_driver,
	pengo_rom,
	0, 0,
	0,
	input_ports, dsw,
	color_prom, 0, 0,
	{ 0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,	/* numbers */
		0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,	/* letters */
		0x4e,0x4f,0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a },
	0x01, 0x18,
	8*11, 8*20, 0x16,

	hiload, hisave
};

