#pragma once
#include "M2x.h"
#include "memory.h"
#include "debugman.h"
#include "fileman.h"
#include "video.h"
#include "timer.h"

// Seem necessary
class CPUManager;
class FileManager;
struct Timer;


#define Core		EmulatorCore::instance()

enum state
{
	EMULATOR_UNLOADED,		// Waiting for user to select game
	EMULATOR_INITIALISE,
	EMULATOR_RESET,
	EMULATOR_RUNNING,
	EMULATOR_PAUSED,
	EMULATOR_SHUTDOWN,
};

class EmulatorCore
{
private:

	EmulatorCore(const EmulatorCore &);				// Prevent copy construction
	EmulatorCore&operator=(const EmulatorCore &);	// Prevent assignment

	static EmulatorCore *_singleton;				// The global instance

	static EmulatorCore *MakeInstance()
	{
		return new EmulatorCore();
	}

	static void DestroyInstance()
	{
		delete _singleton;
		_singleton = NULL;
	}

	enum state eState;
	bool bFirstRun;

	UINT64 real_base;
	UINT64 real_time;

	VideoUpdateCB ActiveVideoUpdate;
	const char *active_set_name;
	const char *active_game_name;

public:
	CPUManager *CPUMan;
	DebugManager *DebugMan;
	FileManager *FileMan;
	Video *VideoMan;
	Timer *timer_list;

	emu_time emulated_time;
	UINT64 RealTicksPerSec;

	emu_time GetEmulationTime();
	void SetState(enum state State);
	enum state GetState() { return eState; }
	void RunEmulator();
	bool InitialiseEmulator();
	void ShutdownEmulator();
	void (*system_video_update)(EmulatorCore *Core);

	void SetActiveVideoUpdate(VideoUpdateCB Callback)
	{
		ActiveVideoUpdate = Callback;
	}

	void SetActiveGameName(const char *name, const char *set)
	{
		active_set_name = set;
		active_game_name = name;
	}

	const char *GetActiveSetName()
	{
		return active_set_name;
	}
	void DoActiveVideoUpdate();

	void CreateTimer(emu_time period, UINT32 flags, bool active, void (*callback)());
	void ResetRealTime();
	UINT64 GetSystemTicks();

	void SystemReset();

	// TODO
	//EmulatorCore() : CPUMan(NULL), timer_list(NULL), emulated_time(0), real_time(0), real_base(0), RealTicksPerSec(0){}

	static EmulatorCore& instance()
	{
		if (_singleton == NULL)
			_singleton = EmulatorCore::MakeInstance();
		return *_singleton;
	}

	static void destroy()
	{
		EmulatorCore::DestroyInstance();
	}

protected:
	EmulatorCore() :
		bFirstRun(true),
		eState(EMULATOR_UNLOADED),
		timer_list(NULL),
		emulated_time(0),
		real_time(0),
		real_base(0),
		RealTicksPerSec(0),
		CPUMan(NULL),
		DebugMan(NULL),
		FileMan(NULL),
		VideoMan(NULL)
		{

		}
	virtual ~EmulatorCore() { }
};

typedef struct
{
	//CPUTag *CPUTags;	// CPU Type, Tag, Frequency, Memory Map

	UINT32 VideoHeight;
	UINT32 VideoWidth;
	double VideoFreq;
	void (*video_update)(BufferInfo *BInfo);
} GameSystemInfo;

typedef struct
{
	UINT8	tag;
	UINT32	start;
	UINT32	end;
	UINT32	(*read_handler)(UINT32 addr, UINT32 mask);
	void	(*write_handler)(UINT32 addr, UINT32 data, UINT32 mask);
} memory_map;

UINT32 read_mem32(const memory_map *map, UINT32 addr, UINT32 mask);
void write_mem32(const memory_map *map, UINT32 addr, UINT32 data, UINT32 mask);

UINT16 read_mem16(const memory_map *map, UINT32 addr, UINT16 mask);
void write_mem16(const memory_map *map, UINT32 addr, UINT16 data, UINT16 mask);

UINT8 read_mem8(const memory_map *map, UINT32 addr);
void write_mem8(const memory_map *map, UINT32 addr, UINT8 data);
