#include "ppc602.h"
#include "log.h"

/*
 *   What are the WIMG default bits in HID0??? HACKED for now.
 *   Konami BIOS uses CMPDI, which is for 64-bit quantities (L bit is set). Guess we ignore it.
 *   Instruction timings (see page 316) - MOSTLLY OK
 */

#define CYCLE_CHECK						1
#define DISABLE_ADDRESS_TRANSLATION		1

#define UNHANDLED()						LOGERROR("Unhandled/Illegal Instruction: 0x%0.8x (PC:0x%0.8x)\n", inst, pc);


UINT32 PPC602::Execute(UINT32 cycles)
{
	cycles_remaining = cycles;
	cycles_to_run = cycles;
	static int trace = 0;

	do
	{
		if (exceptions_pending)
			HandleExceptions();

		// 2 cpu cycles = 1 bus cycle
		// 3 cpu cycles = 1 bus cycle

		// TODO: Assumes 2:1 clock ratio
		// These registers tick every four bus cycles
		if ((GetTotalCycles() & 7) == 0)
		{
			// Time base
			++tbr.dw;

			// Decrementer
			if (dec-- == 0LL)
				SIGNAL_EXCEPTION(DECREMENTER);
		}

		next_pc = pc + 4;
		inst = ReadWordMMU(ADDRESS_READ | ADDRESS_INSTRUCTION, pc, 0xffffffff);

		// Disable shell console - POLYSTARS/TOTAL VICE
#if 0
		if (pc == 0x40035958)
			gpr[11] = 1;
#else
		// Disable shell console - everything else
		if (pc == 0x400385c8)
			gpr[11] = 0;
#endif

#if 0
	// This is for Total Vice 10K check
	if (pc == 0x400b5354)
		gpr[3] = 1;

	if (pc == 0x40089acc || pc == 0x40089c30)
		gpr[12] = gpr[3];
#endif


// *************************** TESTEND

//#ifdef _DEBUG
	//	Core.DebugMan->DebugStep(this);
//#endif

		switch ((inst >> 26) & 0x3f)
		{
			case  7: mulli();	break;
			case  8: subfic();	break;
			case 10: cmpli();	break;
			case 11: cmpi();	break;
			case 12: addic();	break;
			case 13: addicr();	break;
			case 14: addi();	break;
			case 15: addis();	break;
			case 16: bcx();		break;
			case 17: sc();		break;
			case 18: bx();		break;
			case 19:
			{
				switch ((inst >> 1) & 0x3ff)
				{
					case 16:  bclrx();	break;
					case 33:  crnor();	break;
					case 50:  rfi();	break;
					case 129: crandc();	break;
					case 150: isync();	break;
					case 193: crxor();	break;
					case 225: crnand();	break;
					case 257: crand();	break;
					case 289: creqv();	break;
					case 417: crorc();	break;
					case 449: cror();	break;
					case 528: bcctrx();	break;
					default:
						UNHANDLED();
						break;
				}
				break;
			}
			case 20: rlwimix();	break;
			case 21: rlwinmx();	break;
			case 23: rlwnmx();	break;
			case 24: ori();		break;
			case 25: oris();	break;
			case 26: xori();	break;
			case 27: xoris();	break;
			case 31:
			{
				switch ((inst >> 1) & 0x3ff)
				{
					case   0: cmp();	break;

					case   8:
					case 520: subfc();	break;

					case  10:
					case 522: addcx();	break;

					case  11: mulhwux();break;
					case  19: mfcr();	break;
					case  23: lwzx();	break;
					case  24: slwx();	break;
					case  26: cntlzw();	break;
					case  28: andx();	break;
					case  32: cmpl();	break;

					case  40:
					case 552: subfx();	break;

					case  54: dcbtst();	break;
					case  55: lwzux();	break;
					case  60: andcx();	break;
					case  75: mulhwx();	break;
					case  83: mfmsr();	break;
					case  86: dcbf();	break;
					case  87: lbzx();	break;

					case 104:
					case 616: negx();	break;

					case 124: norx();	break;

					case 136:
					case 648: subfex();	break;

					case 138:
					case 650: addex();	break;

					case 144: mtcrf();	break;
					case 146: mtmsr();	break;
					case 150: stwcx();	break;
					case 151: stwx();	break;

					case 200:
					case 712: subfze(); break;

					case 202:
					case 714: addzex();	break;

					case 210: mtsr();	break;
					case 215: stbx();	break;

					case 232:
					case 744: subfmex();break;

					case 234:
					case 746: addmex();	break;

					case 235:
					case 747: mullwx();	break;

					case 242: mtsrin();	break;
					case 246: dcbtst();	break;

					case 266:
					case 778: addx();	break;

					case 278: dcbt();	break;
					case 279: lhzx();	break;
					case 284: eqvx();	break;
					case 306: tlbie();	break;
					case 316: xorx();	break;
					case 339: mfspr();	break;
					case 343: lhax();	break;
					case 371: mftb();	break;
					case 375: lhaux();	break;
					case 407: sthx();	break;
					case 412: orcx();	break;
					case 444: orx();	break;
					case 467: mtspr();	break;
					case 476: nandx();	break;

					case 459:
					case 971: divwux();	break;

					case 470: dcbi();	break;

					case 491:
					case 1003:divwx();	break;

					case 535: lfsx();	break;
					case 536: srwx();	break;
					case 595: mfsr();	break;
					case 596: esa();	break;
					case 598: sync();	break;
					case 628: dsa();	break;
					case 663: stfsx();	break;
					case 792: srawx();	break;
					case 824: srawix();	break;
					case 978: tlbld();	break;
					case 922: extshx();	break;
					case 954: extsbx();	break;
					case 983: stfiwx();	break;
					case 1010:tlbli();	break;
					default:
						UNHANDLED();
						break;
				}
				break;
			}
			case 28: andi();	break;
			case 29: andis();	break;
			case 32: lwz();		break;
			case 33: lwzu();	break;
			case 34: lbz();		break;
			case 35: lbzu();	break;
			case 36: stw();		break;
			case 37: stwu();	break;
			case 38: stb();		break;
			case 39: stbu();	break;
			case 40: lhz();		break;
			case 41: lhzu();	break;
			case 42: lha();		break;
			case 43: lhau();	break;
			case 44: sth();		break;
			case 45: sthu();	break;
			case 46: lmw();		break;
			case 47: stmw();	break;
			case 48: lfs();		break;
			case 49: lfsu();	break;
			case 52: stfs();	break;
			case 53: stfsu();	break;
			case 59:
			{
				switch ((inst >> 1) & 0x1f)
				{
					case 18: fdivsx();		break;
					case 20: fsubsx();		break;
					case 21: faddsx();		break;
					case 28: fmsubsx();		break;
					case 29: fmaddsx();		break;
					case 30: fnmsubsx();	break;
					case 31: fnmaddsx();	break;
					case 25: fmulsx();		break;
					default: UNHANDLED();
				}
				break;
			}
			case 63:
			{
				switch ((inst >> 1) & 0x3ff)
				{
					case   0: fcmpu();		break;
					case  14: fctiwx();		break;
					case  15: fctiwz();		break;
					case  26: frsqrtex();	break;
					case  38: mtfsb1x();	break;
					case  40: fnegx();		break;
					case  70: mtfsb0x();	break;
					case  72: fmrx();		break;
					case 264: fabsx();		break;
					case 583: mffsx();		break;
					case 711: mtfsfx();		break;
					default:
					{
						switch ((inst >> 1) & 0x1f)
						{
							case  23: fselx();		break;
							default: UNHANDLED();
						}
					}
				}
				break;
			}
			default:
				UNHANDLED();
				break;
		}

	pc = next_pc;

//		LOGERROR("...........R11=%x\n", gpr[11]);

	// POLYSTARS
//	if (pc == 0x40137380)
//		pc = 0x40137384;

	} while(cycles_remaining > 0);

	return cycles_to_run - cycles_remaining - stolen_cycles;
}


#define MemoryProtectionViolation    \
	if (flags & ADDRESS_INSTRUCTION) \
	{                                \
		srr1 |= (1 << (31 - 4));     \
		SIGNAL_EXCEPTION(ISI);       \
	}                                \
	else                             \
	{                                \
		dsisr |= (1 << (31 - 4));    \
		SIGNAL_EXCEPTION(DSI);       \
	}

bool PPC602::TranslateAddress(UINT32 flags, UINT32 *addr)
{
#if DISABLE_ADDRESS_TRANSLATION
	return true;
#endif

	// Compare address with Instruction BAT array
	BAT *bat_array = (flags & ADDRESS_INSTRUCTION) ? ibat : dbat;

	for (int i = 0; i < 4; ++i)
	{
		const BAT bat = bat_array[i];
		bool valid = msr.pr ? bat.vp : bat.vs;

		// CHECK
		if (valid && ((((*addr >> 17) & 0x7fff) & ~bat.bl) == bat.bepi))
		{
			if (bat.pp == 0) // No access
			{
				MemoryProtectionViolation(flags);
				return FALSE;
			}
			if ((flags & ADDRESS_WRITE) && (bat.pp & 1))
			{
				MemoryProtectionViolation(flags);
				return FALSE;
			}
			if (((bat.wimg & 1) || bat.ne) && (flags & ADDRESS_INSTRUCTION))
			{
				MemoryProtectionViolation(flags);
				return FALSE;
			}

			UINT32 bits0_3 = (bat.brpn << 17) & 0xf0000000;
			UINT32 bits4_14 = (bat.brpn & 0x7ff) | (*addr >> 17) & bat.bl;
			UINT32 bits15_31 = *addr & 0x1ffff;

			*addr = bits0_3 | (bits4_14 << 17) | bits15_31;
			return TRUE;
		}
	}

	if (hid0.po == 1)
	{
		// TODO: Other translation
		DEBUG_BREAK;
	}
	else
	{
		// Use EA0-EA3 to select one of 16 SRs
		//DEBUG_BREAK;
		UINT32 sd = sr[(*addr >> 28) & 0xf];
	}
	return TRUE;
}

UINT32 PPC602::ReadWordMMU(UINT32 flags, UINT32 addr, UINT32 mask)
{
	if (((flags & ADDRESS_INSTRUCTION) && msr.ir) || ((flags & ADDRESS_DATA) && msr.dr))
	{
		if (TranslateAddress(flags, &addr))
		{
			return read_mem32(cpuinfo.map, addr, mask);
		}
		else
		{
			DEBUG_BREAK;
			return 0;
		}
	}
	return read_mem32(cpuinfo.map, addr, mask);
}

void PPC602::WriteWordMMU(UINT32 flags, UINT32 addr, UINT32 data, UINT32 mask)
{
	if (addr == 0x40a88ede)
		LOGERROR("%x %x %x\n",pc, addr, data);

	if (((flags & ADDRESS_INSTRUCTION) && msr.ir) || ((flags & ADDRESS_DATA) && msr.dr))
	{
		if (TranslateAddress(flags, &addr))
			write_mem32(cpuinfo.map, addr, data, mask);
		else
			DEBUG_BREAK;
	}
	else
		write_mem32(cpuinfo.map, addr, data, mask);
}

UINT8 PPC602::ReadData8(UINT32 addr)
{
	UINT32 shift = 8 * (3 - (addr & 3));
	return (ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, addr, 0xff << shift) >> shift) & 0xff;
}

UINT16 PPC602::ReadData16(UINT32 addr)
{
	UINT32 shift = 8 * (2 - (addr & 2));
	if ((addr & 1) == 0)
		return (ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, addr, 0xffff << shift) >> shift) & 0xffff;
	else
	{
		DEBUG_BREAK;
		return 0;
	}
}

UINT32 PPC602::ReadData32(UINT32 addr)
{
	if ((addr & 3) == 0)
		return ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, addr, 0xffffffff);
	else
	{
		UINT32 shift = 8 * (addr & 3);
		UINT32 w0 = ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, addr & ~3, 0xffffffff);
		UINT32 w1 = ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, (addr & ~3) + 4, 0xffffffff);

		w0 = (w0 & (0xffffffff >> 8*((addr & 3)))) << shift;
		w1 = (w1 & (0xffffffff << 8*(3-(addr & 3)))) >> 8*((4-(addr & 3)));
		return	w0 | w1;
	}
}

void PPC602::WriteData8(UINT32 addr, UINT8 data)
{
	UINT32 shift = 8 * (3 - (addr & 3));
	WriteWordMMU(ADDRESS_WRITE | ADDRESS_DATA, addr, data << shift, 0xff << shift);
}

void PPC602::WriteData16(UINT32 addr, UINT16 data)
{
	UINT32 shift = 8 * (2 - (addr & 2));
	if ((addr & 1) == 0)
		WriteWordMMU(ADDRESS_WRITE | ADDRESS_DATA, addr, data << shift, 0xffff << shift);
	else
	{
		// Unaligned case
		DEBUG_BREAK;
	}
}

void PPC602::WriteData32(UINT32 addr, UINT32 data)
{
	if ((addr & 3) == 0)
		WriteWordMMU(ADDRESS_WRITE | ADDRESS_DATA, addr, data, 0xffffffff);
	else
	{
		UINT32 w0 = ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, addr & ~3, 0xffffffff);
		UINT32 w1 = ReadWordMMU(ADDRESS_READ | ADDRESS_DATA, (addr & ~3) + 4, 0xffffffff);

		UINT32 shift = 8 * (addr & 3);
		w0 = (w0 & ~(0xffffffff >> 8*((addr & 3))));
		w0 |= data >> 8*(addr & 3);
		WriteWordMMU(ADDRESS_WRITE | ADDRESS_DATA, addr & ~3, w0, 0xffffffff);

		w1 = (w1 & ~(0xffffffff << 8*(4-(addr & 3))));
		w1 |= data << 8*(4-(addr & 3));
		WriteWordMMU(ADDRESS_WRITE | ADDRESS_DATA, (addr & ~3) + 4, w1, 0xffffffff);
	}
}

void PPC602::HandleExceptions(void)
{
	for (int i = 0; i < 32; ++i)
	{
		UINT32 vector = 0;

		if ((exceptions_pending & (1 << i)) == 0)
			continue;

		switch (i)
		{
			case 4:
			{
				if (msr.ee == 0)
					continue;

				srr0 = pc;
				srr1 &= ~0x783FFF73;
				srr1 |= msr.w & 0xFF73;
				msr.pow = 0;
				msr.fp = 0;
				msr.be = 0;
				msr.dr = 0;
				msr.fe1 = 0;
				msr.ri = 0;
				msr.ee = 0;
				msr.fe0 = 0;
				msr.pr = 0;
				msr.se = 0;
				msr.ir = 0;
				msr.le = msr.ile;
				vector = 0x500;
				break;
			}
			case 6:
			{
				srr0 = pc;
				srr1 &= ~0x783FFF73;
				srr1 |= msr.w & 0xFF73;
				msr.pow = 0;
				msr.fp = 0;
				msr.be = 0;
				msr.dr = 0;
				msr.fe1 = 0;
				msr.ri = 0;
				msr.ee = 0;
				msr.fe0 = 0;
				msr.pr = 0;
				msr.se = 0;
				msr.ir = 0;
				msr.le = msr.ile;
				vector = 0x800;
				break;
			}
			case 5:
			{
				if (msr.ee == 0)
					continue;

				srr0 = pc;
				srr1 &= ~0x783FFF73;
				srr1 |= msr.w & 0xFF73;
				msr.pow = 0;
				msr.fp = 0;
				msr.be = 0;
				msr.dr = 0;
				msr.fe1 = 0;
				msr.ri = 0;
				msr.ee = 0;
				msr.fe0 = 0;
				msr.pr = 0;
				msr.se = 0;
				msr.ir = 0;
				msr.le = msr.ile;
				vector = 0x900;
				break;
			}
			default:
				continue;
				//DEBUG_BREAK;
		}

		// Determine the new PC
		pc = (msr.ip ? 0xfff00000 : ibr) + vector;

		// Clear the exception
		exceptions_pending &= ~(1 << i);
	}
}


// TEST CODE
void PPC602::Interrupt(void)
{
	SIGNAL_EXCEPTION(EXT_INTERRUPT);
}

void PPC602::Reset(void)
{
	pc = next_pc = 0xfff00100;

	exceptions_pending = 0;

	ZeroMemory(gpr, sizeof(gpr));
	ZeroMemory(fpr, sizeof(fpr));
	fpscr.w = 0x00000000;
	cr.w = 0x00000000;
	msr.w = 0x00000040;
	xer.w = 0x00000000;
	tbr.u = 0x00000000;
	tbr.l = 0x00000000;
	lr = 0x00000000;
	ctr = 0x00000000;
	dsisr = 0x00000000;
	dar = 0x00000000;
	dec = 0xffffffff;
	sdr1 = 0x00000000;
	srr0 = 0x00000000;
	srr1 = 0x00000000;
	ZeroMemory(sprg, sizeof(sprg));
	// TAG DIRECTORY
	pvr = 0x00050100 | PPC602_REVISION_LEVEL;
	hid0.w = 0x00000000 | 1;	// HACK?
	hid1.w = 0x00000000;
	dmiss = 0x00000000;
	imiss = 0x00000000;
	dcmp = 0x00000000;
	icmp = 0x00000000;
	rpa = 0x00000000;
	iabr = 0x00000000;
	esasrr.w = 0x00000000;
	ser = 0x00000000;
	sebr = 0x00000000;
	ibr = 0x00000000;
	hash1 = 0x00000000;
	hash2 = 0x00000000;
	// INVALIDATE CACHE
}

void PPC602::SoftReset(void)
{
	srr0 = next_pc;
	srr1 = msr.w & 0xffffffff;

	pc = msr.ip ? 0xfff00100 : 0x00000100;
	next_pc = pc;

	exceptions_pending = 0;

	msr.ap = 0;
	msr.sa = 0;
	msr.pow = 0;
	msr.ee = 0;
	msr.pr = 0;
	msr.fp = 0;
	msr.fe0 = 0;
	msr.se = 0;
	msr.be = 0;
	msr.fe1 = 0;
	msr.ir = 0;
	msr.dr = 0;
	msr.ri = 0;
	msr.le = msr.ile;
}

UINT32 PPC602::GetReg(UINT32 id)
{
	return gpr[id];
}

UINT32 PPC602::GetLR()
{
	return lr;
}
