#pragma once
#include "cpuexec.h"
#include "log.h"
#include <math.h>

// Note: Use and its flagsa _fpclass
#define isnan _isnan

static const UINT32 ExceptionVectors[] =
{
	0x100,
	0x100,
	0x300,
	0x400,
	0x500,
	0x600,
	0x700,
	0x800,
	0x900,
	0xc00,
	0xe00,
};

#define FP32_SGN(f)		(((*(UINT32 *)&(f)) >> 31) & 0x1)
#define FP32_EXP(f)		(((*(UINT32 *)&(f)) >> 23) & 0xff)
#define FP32_MAN(f)		((*(UINT32 *)&(f)) & 0x7ffff)

#define FP32_QNAN		0x7FC00000

#define FP32_EXP_MIN	1
#define FP32_EXP_MAX	254

static bool FloatIsZero(float f)
{
	if (FP32_EXP(f) == 0 && FP32_MAN(f) == 0)
		return true;
	else
		return false;
}

static bool FloatIsNorm(float f)
{
	if (FP32_EXP(f) <= FP32_EXP_MAX && FP32_EXP(f) >= FP32_EXP_MIN)
		return true;
	else
		return false;
}

static bool FloatIsDenorm(float f)
{
	if (FP32_EXP(f) == 0 && FP32_MAN(f) != 0)
		return true;
	else
		return false;
}

static bool FloatIsPosInf(float f)
{
	if (FP32_SGN(f) == 0 && FP32_EXP(f) == 255 && FP32_MAN(f) == 0)
		return true;
	else
		return false;
}

static bool FloatIsNegInf(float f)
{
	if (FP32_SGN(f) == 1 && FP32_EXP(f) == 255 && FP32_MAN(f) == 0)
		return true;
	else
		return false;
}

static bool FloatIsNaN(float f)
{
	if (FP32_EXP(f) == 255 && FP32_EXP(f) != 0)
		return true;
	else
		return false;
}

static bool FloatIsQNaN(float f)
{
	if (FP32_EXP(f) == 255)
		if (FP32_EXP(f) & 0x400000)
			return true;

	return false;
}

static bool FloatIsSNaN(float f)
{
	if (FP32_EXP(f) == 255)
		if (!(FP32_EXP(f) & 0x400000))
			return true;

	return false;
}

inline INT32 FloatRoundToNearest(float f)
{
	if (f >= 0.0f)
		return (INT32)(f + 0.5);
	else
		return (INT32)(f - 0.5);
}

inline INT32 FloatRoundToZero(float f)
{
	return (INT32)f;
}

inline INT32 FloatRoundToPosInf(float f)
{
	return (INT32)ceil(f);
}

inline INT32 FloatRoundToNegInf(float f)
{
	return (INT32)floor(f);
}



class PPC602 : public CPU
{
private:

#define PPC602_REVISION_LEVEL	1
#define CYCLES(x)				cycles_remaining -= x

#define BIT_TEST(r, b)		((r) & (1 << (b)))
#define BIT(r, b)			(((r) >> (b)) & 1)
#define ROTL32(v, s)		(((UINT32)(v) << (s)) | ((UINT32)(v) >> (32 - (s))))
#define ROTR32(v, s)		(((UINT32)(v) >> (s)) | ((UINT32)(v) << (32 - (s))))

// MMU Translation
#define ADDRESS_READ		(1 << 0)
#define ADDRESS_WRITE		(1 << 1)
#define ADDRESS_INSTRUCTION	(1 << 2)
#define ADDRESS_DATA		(1 << 3)

#define CHECK_FP_AVAILABLE()	\
	do {						\
		if (msr.fp == 0)		\
		{						\
			SIGNAL_EXCEPTION(FP_UNAVAILABLE);	\
			return;				\
		}						\
	}while(0);
void SIGNAL_EXCEPTION(int x)
{
	exceptions_pending |= x;
}

// Ordered by priority
enum
{
	SYSTEM_RESET	= (1 << 0),
	MACHINE_CHECK	= (1 << 1),
	INSTRUCTION		= (1 << 2),
	FP_IMPRECISE	= (1 << 3),
	EXT_INTERRUPT	= (1 << 4),
	DECREMENTER		= (1 << 5),
	FP_UNAVAILABLE	= (1 << 6),
}Exceptions;

//
// Instruction sub-exception group
//

enum
{
	INTEGER_ALIGNMENT	= (1 << 0),
	INTEGER_DSI			= (1 << 1),
	INTEGER_TRACE		= (1 << 2),
}SubExceptionsInteger;

enum
{
//	FP_UNAVAILABLE		= (1 << 0),
	FP_ALIGNMENT		= (1 << 1),
	FP_DSI				= (1 << 2),
	FP_TRACE			= (1 << 3),
}SubExceptionsFPLoadStore;

enum
{
	FPM_UNAVAILABLE		= (1 << 0),
	FPM_PROGRAM			= (1 << 1),
	FPM_ASSIST			= (1 << 2),
	FPM_TRACE			= (1 << 3),
}SubExceptionsFPMisc;

enum
{
	MTMSR_PROGRAM_PRIVILEGED	= (1 << 0),
	MTMSR_PROGRAM_PRECISEFP		= (1 << 1),
	MTMSR_TRACE					= (1 << 2),
}SubExceptionsMTMSR;

enum
{
	OTHER_TRAP			= (1 << 0),
	OTHER_SC			= (1 << 0),
	OTHER_PRIVILEGED	= (1 << 0),
	OTHER_ILLEGAL		= (1 << 0),
	OTHER_TRACE			= (1 << 1),
}SubExceptionsOther;
// WRONG
#define ISI (1 << 5)
#define DSI (1 << 4)


UINT32 MAKE_MASK(INT32 b, INT32 e)
{
	UINT32 m = 0;
	// TODO: CRAP!!!!!!
	if (!(b <= e))
	{
		for (int i=b; i<32; ++i)
		{
			m |= (1 << (31 - i));
		}

		for (int i=0; i<=e; ++i)
		{
			m |= (1 << (31 - i));
		}
	}
	else
	{
		for (int i=0; i<32; ++i)
		{
			if ((i >= (int)b) && (i <= (int)e))
				m |= (1 << (31 - i));
		}
	}
	return m;
}


	//
	//  Types
	//
	typedef union
	{
		INT32	l;
		float	f;
	} FP32;

	typedef union
	{
		struct
		{
			UINT32 l;
			UINT32 u;
		};
		UINT64 dw;
	} PAIR64;

	typedef union
	{
		struct
		{
			union
			{
				struct
				{
					unsigned pp : 2;
					unsigned reserved1 : 1;
					unsigned wimg : 4;
					unsigned reserved2 : 2;
					unsigned se : 1;
					unsigned ne : 1;
					unsigned reserved: 6;
					unsigned brpn : 15;
				};
				UINT32 l;
			};
			union
			{
				struct
				{
					unsigned vp : 1;
					unsigned vs : 1;
					unsigned bl : 11;
					unsigned reserved: 4;
					unsigned bepi : 15;
				};
				UINT32 u;
			};
		};
		UINT64 dw;
	} BAT;


	//
	//  Virtual Registers
	//
	UINT32		pc;
	UINT32		next_pc;
	UINT32		BREAKPC;
	UINT32		inst;
	UINT32		exceptions_pending;

	// PLL Configuration Register
	union
	{
		struct
		{
			unsigned reserved : 28;
			unsigned pc3 : 1;
			unsigned pc2 : 1;
			unsigned pc1 : 1;
			unsigned pc0 : 1;
		};
		UINT32 w;
	} hid1;

	// Machine State Register
	union
	{
		struct
		{
			unsigned le : 1;
			unsigned ri : 1;
			unsigned reserved1 : 2;
			unsigned dr : 1;
			unsigned ir : 1;
			unsigned ip : 1;
			unsigned reserved2 : 1;
			unsigned fe1 : 1;
			unsigned be : 1;
			unsigned se : 1;
			unsigned fe0 : 1;
			unsigned me : 1;
			unsigned fp : 1;
			unsigned pr : 1;
			unsigned ee : 1;
			unsigned ile : 1;
			unsigned tgpr : 1;
			unsigned pow : 1;
			unsigned reserved4 : 3;
			unsigned sa : 1;
			unsigned ap : 1;
			unsigned reserved5 : 8;
		};
		UINT32 w;
	} msr;

	// Machine State Register
	union
	{
		struct
		{
			unsigned wimg : 4;
			unsigned unused7 : 1;
			unsigned sl : 1;
			unsigned unused6 : 2;
			unsigned po : 1;
			unsigned dcfi : 1;
			unsigned icfi : 1;
			unsigned dlock : 1;
			unsigned ilock : 1;
			unsigned dce : 1;
			unsigned unused5 : 1;
			unsigned nhr : 1;
			unsigned unused4 : 2;
			unsigned riseg : 1;
			unsigned dpm : 1;
			unsigned sleep : 1;
			unsigned nap : 1;
			unsigned doze : 1;
			unsigned unused3 : 1;
			unsigned eclk : 1;
			unsigned unused2 : 1;
			unsigned sbclk : 1;
			unsigned unused : 3;
			unsigned emcp : 1;
		};
		UINT32 w;
	} hid0;

	UINT32		pvr;		// Processor Version Register

	BAT			ibat[4];	// Instruction BAT
	BAT			dbat[4];	// Data BAT

	UINT32		dmiss;		// Software Table Search Registers
	UINT32		dcmp;
	UINT32		hash1;
	UINT32		hash2;
	UINT32		imiss;
	UINT32		icmp;
	UINT32		rpa;

	UINT32		reserve;	// TODO: USED BY STWCX


	UINT32		sdr1;
	UINT32		sr[16];		// Segment Registers

	UINT32		ser;
	UINT32		dar;
	UINT32		dsisr;
	UINT32		sebr;

	union
	{
		struct
		{
			unsigned ee : 1;
			unsigned sa : 1;
			unsigned ap : 1;
			unsigned pr : 1;
			unsigned reserved: 28;
		};
		UINT32 w;
	} esasrr;

	UINT32		sprg[4];	//
	UINT32		srr0;		//
	UINT32		srr1;		//
	UINT32		sp;			//
	UINT32		lt;			//
	UINT32		tcr;		// Timer control
	UINT32		ibr;		// Interrupt base
	UINT32		iabr;
	UINT32		dec;		// Decrementer

	UINT32		gpr[32];	// General-Purpose Registers
	FP32		fpr[32];	// Single-Precision Floating-Point Registers

	union
	{
		struct
		{
			unsigned cr7_so : 1; unsigned cr7_z : 1; unsigned cr7_p : 1; unsigned cr7_n : 1;
			unsigned cr6_so : 1; unsigned cr6_z : 1; unsigned cr6_p : 1; unsigned cr6_n : 1;
			unsigned cr5_so : 1; unsigned cr5_z : 1; unsigned cr5_p : 1; unsigned cr5_n : 1;
			unsigned cr4_so : 1; unsigned cr4_z : 1; unsigned cr4_p : 1; unsigned cr4_n : 1;
			unsigned cr3_so : 1; unsigned cr3_z : 1; unsigned cr3_p : 1; unsigned cr3_n : 1;
			unsigned cr2_so : 1; unsigned cr2_z : 1; unsigned cr2_p : 1; unsigned cr2_n : 1;
			unsigned cr1_so : 1; unsigned cr1_z : 1; unsigned cr1_p : 1; unsigned cr1_n : 1;
			unsigned cr0_so : 1; unsigned cr0_z : 1; unsigned cr0_p : 1; unsigned cr0_n : 1;
		};
		UINT32 w;
	} cr;

	union
	{
		struct
		{
			unsigned rn : 2;
			unsigned ni : 1;
			unsigned xe : 1;
			unsigned ze : 1;
			unsigned ue : 1;
			unsigned oe : 1;
			unsigned ve : 1;
			unsigned vxcvi : 1;
			unsigned vxsqrt : 1;
			unsigned vxsoft : 1;
			unsigned reserved : 1;
			unsigned fprf : 5;
			unsigned fi : 1;
			unsigned fr : 1;
			unsigned vxvc : 1;
			unsigned vximz : 1;
			unsigned vxzdz : 1;
			unsigned vxidi : 1;
			unsigned vxisi : 1;
			unsigned vxsnan : 1;
			unsigned xx : 1;
			unsigned zx : 1;
			unsigned ux : 1;
			unsigned ox : 1;
			unsigned vx : 1;
			unsigned fex : 1;
			unsigned fx : 1;
		};
		UINT32 w;
	} fpscr;

	union
	{
		struct
		{
			unsigned bytecount : 7;
			unsigned reserved : 22;
			unsigned ca : 1;
			unsigned ov : 1;
			unsigned so : 1;
		};
		UINT32 w;
	} xer;

	UINT32		lr;
	UINT32		ctr;
	PAIR64		tbr;

#define SUPERVISOR_MODE		(msr.pr == 0)

#define CALC_CR0_N(r)		(cr.cr0_n = (r & (1 << 31)) > 0)
#define CALC_CR0_P(r)		(cr.cr0_p = r && !(r & (1 << 31)))
#define CALC_CR0_Z(r)		(cr.cr0_z = !r)
#define UPDATE_CR0_SO()		(cr.cr0_so = xer.so)

void UPDATE_CR0(UINT32 r)
{
	CALC_CR0_N(r);
	CALC_CR0_P(r);
	CALC_CR0_Z(r);
	UPDATE_CR0_SO();
}

void UPDATE_CR1()
{
	cr.cr1_n = fpscr.fx;
	cr.cr1_p = fpscr.fex;
	cr.cr1_z = fpscr.vx;
	cr.cr1_so = fpscr.ox;
}

void UpdateFPRF(float r)
{
	if (r < 0.0f) fpscr.fprf |= 8;
	if (r > 0.0f) fpscr.fprf |= 4;
	if (r == 0.0f) fpscr.fprf |= 2;
	// TODO
}

#define ADD_CA(r,a)					((UINT32)(r) < (UINT32)(a))
#define SUB_CA(a,b)					(!((UINT32)(a) < (UINT32)(b)))
#define CALC_XER_CA_ADD(r, a)		(xer.ca = (UINT32)(r) < (UINT32)(a))
#define CALC_XER_CA_SUB(a, b)		(xer.ca = !((UINT32)(a) < (UINT32)(b)))

#define CALC_XER_OV_SUB(r, a, b)	(xer.ov = ((INT32)(((a) ^ (b)) & ((a) ^ (r))) < 0))
#define CALC_XER_OV_ADD(r, a, b)	(xer.ov = ((INT32)(~((a) ^ (b)) & ((a) ^ (r))) < 0))
#define CALC_XER_SO()				(xer.so |= xer.ov)

	//
	// Instructions
	//

#define A			((inst >> 16) & 0x1f)
#define B			((inst >> 11) & 0x1f)
#define C			((inst >> 6) & 0x1f)
#define S			((inst >> 21) & 0x1f)
#define D			((inst >> 21) & 0x1f)
#define MB			((inst >> 6) & 0x1f)
#define ME			((inst >> 1) & 0x1f)
#define SH			((inst >> 11) & 0x1f)
#define UIMM		((UINT16)(inst & 0xffff))
#define SIMM		((INT16)(inst & 0xffff))
#define SPR			((inst >> 11) & 0x3ff)
#define rC			(inst & 1)

#define SEX16(x)	((INT32)(INT16)(x))

#define crfD		((inst >> 23) & 7)
#define crbD		((inst >> 21) & 0x1f)
#define crbA		((inst >> 16) & 0x1f)
#define crbB		((inst >> 11) & 0x1f)

#define R(x)		gpr[x]
#define FR(x)		fpr[x].f
#define FRx(x)		fpr[x].l

#define SR			((inst >> 16) & 0xf)

#define BO			((inst >> 21) & 0x1f)
#define BI			((inst >> 16) & 0x1f)
#define BD			((inst >> 2) & 0x3fff)
#define LK			(inst & 1)
#define LI			((inst >> 2) & 0xffffff)
#define AA			(inst & 2)

#define L			(inst & (1 << 21))
#define OE			(inst & (1 << 10))

	//
	//   Integer Arithmetic
	//
	void addi()
	{
		R(D) = (A ? R(A) : 0) + (INT32)(SIMM);
		CYCLES(1);
	}

	void addic()
	{
		UINT32 a = R(A);
		UINT32 b = (INT32)(SIMM);
		UINT32 r = a + b;
		R(D) = r;

		CALC_XER_CA_ADD(r, a);
		CYCLES(1);
	}

	void addicr()
	{
		UINT32 a = R(A);
		UINT32 b = (INT32)(SIMM);
		UINT32 r = a + b;
		R(D) = r;

		UPDATE_CR0(r);
		CALC_XER_CA_ADD(R(D), a);
		CYCLES(1);
	}

	void addis()
	{
		R(D) = (A ? R(A) : 0) + (SIMM << 16);
		CYCLES(1);
	}

	void addmex()
	{
		UINT32 tmp = R(A) + xer.ca;
		UINT32 r = R(A) + xer.ca + -1;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, R(A), xer.ca + -1);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		xer.ca = ADD_CA(tmp, R(A)) || SUB_CA(tmp, 1);

		CYCLES(1);
	}

	void addcx()
	{
		UINT32 a = R(A);
		UINT32 b = R(B);
		UINT32 r = a + b;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, a, b);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		CALC_XER_CA_ADD(r, a);
		CYCLES(1);
	}

	void addex()
	{
		UINT32 a = R(A);
		UINT32 b = R(B);
		UINT32 r = a + b + xer.ca;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, a, b + xer.ca);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		UINT32 tmp = a + b;
		xer.ca = ADD_CA(tmp, a) || ADD_CA(tmp + xer.ca, xer.ca);
		CYCLES(1);
	}

	void addzex()
	{
		UINT32 r = R(A) + xer.ca;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, R(A), xer.ca);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		CALC_XER_CA_ADD(r, R(A));
		CYCLES(1);
	}

	void addx()
	{
		UINT32 a = R(A);
		UINT32 b = R(B);
		UINT32 r = a + b;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, a, b);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);
		CYCLES(1);
	}

	void divwx()
	{
		INT32 dend = (INT32)R(A);
		INT32 dsor = (INT32)R(B);
		INT32 r;

		if (dsor == 0 && dend < 0x80000000)
		{
			r = 0;
			if (OE)
			{
				xer.ov = 1;
				CALC_XER_SO();
			}
		}
		else if(dsor == 0 || dsor == 0xffffffff && dend == 0x80000000)
		{
			r = 0xffffffff;
			if (OE)
			{
				xer.ov = 1;
				CALC_XER_SO();
			}
		}
		else
		{
			r = dend / dsor;
			R(D) = r;

			if (OE)
				xer.ov = 0;
		}

		R(D) = r;
		if (rC)
			UPDATE_CR0(r);
		CYCLES(37);
	}

	void divwux()
	{
		UINT32 dend = (UINT32)R(A);
		UINT32 dsor = (UINT32)R(B);

		if (dsor == 0)
		{
			if (OE)
			{
				xer.ov = 1;
				CALC_XER_SO();
			}
			if (rC)
			{
				cr.cr0_z = 1;
				UPDATE_CR0_SO();
			}
		}
		else
		{
			INT32 r = dend / dsor;
			R(D) = r;

			if (OE)
				xer.ov = 0;
			if (rC)
				UPDATE_CR0(r);
		}
		CYCLES(37);
	}

	void mulli()
	{
		UINT64 prod = (INT64)(INT32)R(A) * (INT64)(INT32)(SIMM);
		R(D) = prod & 0xffffffff;
		CYCLES(1);	// TODO Naaaah
	}

	void mullwx()
	{
		INT64 prod = (INT64)(INT32)R(A) * (INT64)(INT32)R(B);
		INT32 r = prod & 0xffffffff;
		R(D) = r;

		if (OE)
		{
			xer.ov = prod != (INT64)(INT32)r;
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(prod & 0xffffffff);

		CYCLES(1);	// TODO Naaaah
	}

	void mulhwx()
	{
		UINT64 prod = (INT64)(INT32)R(A) * (INT64)(INT32)R(B);
		UINT32 r = prod >> 32;
		R(D) = r;

		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);	// TODO Naaaah
	}

	void mulhwux()
	{
		UINT64 prod = (UINT64)(UINT32)R(A) * (UINT64)(UINT32)R(B);
		UINT32 r = prod >> 32;
		R(D) = r;

		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);	// TODO Naaaah
	}

	void subfex()
	{
		UINT32 a = R(A);
		UINT32 b = R(B);
		UINT32 tmp = ~a + xer.ca;
		UINT32 r = tmp + b;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, ~a, b + xer.ca);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		xer.ca = ADD_CA(tmp, ~a) || (r < tmp);
		CYCLES(1);
	}

	void subfx()
	{
		UINT32 a = R(B);
		UINT32 b = R(A);
		UINT32 r = a - b;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_SUB(r, a, b);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);
	}

	void subfc()
	{
		UINT32 a = R(B);
		UINT32 b = R(A);
		UINT32 r = a - b;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_SUB(r, a, b);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		CALC_XER_CA_SUB(a, b);
		CYCLES(1);
	}

	void subfmex()
	{
		UINT32 tmp = ~R(A) + xer.ca;
		UINT32 r = ~R(A) + xer.ca - 1;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_ADD(r, ~R(A), xer.ca + -1);
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		xer.ca = ADD_CA(tmp, ~R(A)) || (r < tmp);

		CYCLES(1);
	}

	void subfic()
	{
		UINT32 a = R(A);
		UINT32 b = (INT32)SIMM;
		R(D) = b - a;
		CALC_XER_CA_SUB(b, a);
		CYCLES(1);
	}

	void subfze()
	{
		UINT32 r = ~R(A) + xer.ca;
		R(D) = r;

		if (OE)
		{
			CALC_XER_OV_SUB(r, xer.ca, R(A));
			CALC_XER_SO();
		}
		if (rC)
			UPDATE_CR0(r);

		CALC_XER_CA_ADD(r, ~R(A));

		CYCLES(1);
	}

	//
	//   Integer Compare
	//
	void cmp()
	{

//		if (L)
//		// Instruction form is invalid. What to do?
//			DEBUG_BREAK;

		UINT32 c;

		if ((INT32)R(A) < (INT32)R(B))
			c = 4;
		else if ((INT32)R(A) > (INT32)R(B))
			c = 2;
		else
			c = 1;

		cr.w &= ~(0xf << (4 * (7 - crfD)));
		cr.w |= ((c << 1) | xer.so) << (4 * (7 - crfD));
		CYCLES(1);
	}

	void cmpi()
	{
//		if (L)
//		// Instruction form is invalid. What to do?
//			DEBUG_BREAK;

		UINT32 a = R(A);
		UINT32 c;

		if ((INT32)a < (INT32)(SIMM))
			c = 4;
		else if ((INT32)a > (INT32)(SIMM))
			c = 2;
		else
			c = 1;

		cr.w &= ~(0xf << (4 * (7 - crfD)));
		cr.w |= ((c << 1) | xer.so) << (4 * (7 - crfD));
		CYCLES(1);
	}

	void cmpl()
	{
//		if (L)
//		// Instruction form is invalid. What to do?
//			DEBUG_BREAK;
		UINT32 c;

		if ((UINT32)R(A) < (UINT32)R(B))
			c = 4;
		else if ((UINT32)R(A) > (UINT32)R(B))
			c = 2;
		else
			c = 1;

		cr.w &= ~(0xf << (4 * (7 - crfD)));
		cr.w |= ((c << 1) | xer.so) << (4 * (7 - crfD));
		CYCLES(1);
	}

	void cmpli()
	{
//		if (L)
//		// Instruction form is invalid. What to do?
//			DEBUG_BREAK;
		UINT32 c;

		if ((UINT32)R(A) < (UINT32)UIMM)
			c = 4;
		else if ((UINT32)R(A) > UIMM)
			c = 2;
		else
			c = 1;

		cr.w &= ~(0xf << (4 * (7 - crfD)));
		cr.w |= ((c << 1) | xer.so) << (4 * (7 - crfD));
		CYCLES(1);
	}

	//
	//   Integer Logical
	//
	void extsbx()
	{
		UINT32 s = R(S) & 0xff;
		INT32 r = (INT32)(INT8)(s);
		R(A) = r;
		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);
	}

	void extshx()
	{
		UINT32 s = R(S) & 0xffff;
		INT32 r = (INT32)(INT16)(s);
		R(A) = r;
		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);
	}

	void negx()
	{
		UINT32 r;

		if (R(A) == 0x80000000)
		{
			r = 0x80000000;

			if (OE)
			{
				xer.ov = 1;
				CALC_XER_SO();
			}
		}
		else
		{
			r = ~R(A) + 1;

			if (OE)
			{
				xer.ov = 0;
				CALC_XER_SO();
			}
		}
		R(D) = r;

		if (rC)
			UPDATE_CR0(r);

		CYCLES(1);
	}

	void andx()
	{
		R(A) = R(S) & R(B);
		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void andcx()
	{
		R(A) = R(S) & ~R(B);
		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void andi()
	{
		R(A) = R(S) & UIMM;
		UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void andis()
	{
		R(A) = R(S) & (UIMM << 16);
		UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void cntlzw()
	{
		UINT32 r = R(S);

		__asm
		{
			bsr   eax,r
			jnz   skip
			mov   eax,63
		skip:
			xor   eax,31
			mov   r,eax
		}
		R(A) = r;

		if (rC)
		{
			cr.cr0_n = 0;
			CALC_CR0_P(r);
			CALC_CR0_Z(r);
			UPDATE_CR0_SO();
		}
		CYCLES(1);
	}

	void nandx()
	{
		R(A) = ~(R(S) & R(B));

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void norx()
	{
		R(A) = ~(R(S) | R(B));

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void orcx()
	{
		R(A) = R(S) | ~R(B);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void orx()
	{
		R(A) = R(S) | R(B);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void ori()
	{
		R(A) = R(S) | UIMM;
		CYCLES(1);
	}

	void oris()
	{
		R(A) = R(S) | (UIMM << 16);
		CYCLES(1);
	}

	void xori()
	{
		R(A) = R(S) ^ UIMM;
		CYCLES(1);
	}

	void xoris()
	{
		R(A) = R(S) ^ (UIMM << 16);
		CYCLES(1);
	}

	void xorx()
	{
		R(A) = R(S) ^ R(B);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	//
	//   Integer Rotate
	//
	void rlwimix()
	{
		UINT32 r = ROTL32(R(S), SH);
		UINT32 m = MAKE_MASK(MB, ME);
		R(A) = (r & m) | (R(A) & ~m);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void rlwinmx()
	{
		UINT32 r = ROTL32(R(S), SH);
		UINT32 m = MAKE_MASK(MB, ME);
		R(A) = r & m;

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void rlwnmx()
	{
		UINT32 n = R(B) & 0x1f;
		UINT32 r = ROTL32(R(S), n);
		UINT32 m = MAKE_MASK(MB, ME);
		R(A) = r & m;

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	//
	//   Integer Shift
	//
	void srawx()
	{
		UINT32 n = R(B) & 0x1f;
		UINT32 r = ROTL32(R(S), 32 - n);
		UINT32 m = 0;
		if (BIT_TEST(R(B), 31-26) == 0)
			m = MAKE_MASK(n, 31);
		UINT32 s = BIT(R(S), 31);

		R(A) = (r & m) | ((s ? 0xffffffff : 0) & ~m);
		xer.ca = s && ((r & ~m) != 0);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void srawix()
	{
		UINT32 n = SH;
		UINT32 r = ROTL32(R(S), 32 - n);
		UINT32 m = MAKE_MASK(n, 31);
		UINT32 s = BIT(R(S), 31);

		R(A) = (r & m) | ((s ? 0xffffffff : 0) & ~m);
		xer.ca = s && ((r & ~m) != 0);

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void slwx()
	{
		UINT32 n = R(B) & 0x1f;
		UINT32 r = ROTL32(R(S), n);
		UINT32 m = 0;
		if (BIT_TEST(R(B), 31-26) == 0)
			m = MAKE_MASK(0, 31 - n);
		R(A) = r & m;

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	void srwx()
	{
		UINT32 n = R(B) & 0x1f;
		UINT32 r = ROTL32(R(S), 32 - n);
		UINT32 m = 0;
		if (BIT(R(B), 31-26) == 0)
			m = MAKE_MASK(n, 31);
		R(A) = r & m;

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	//
	//   Floating-Point Arithmetic
	//
	void mffsx()
	{
		CHECK_FP_AVAILABLE();

		FRx(D) = fpscr.w;

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
	}

	void fabsx()
	{
		CHECK_FP_AVAILABLE();

		FRx(D) = FRx(B) & ~(1 << 31);

		if (rC)
			UPDATE_CR1();

		CYCLES(1);
	}

	void fmulsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = FR(A) * FR(C);

		// TODO - WE MUST SET STATUS BITS
		UpdateFPRF(FR(D));

	//	if (FloatIsNaN(FR(D))
	//		LOGERROR("************ NAN");

//		if (rC)
//			UPDATE_CR1();

		CYCLES(1);	// ??
	}

	void fmaddsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = (FR(A) * FR(C)) + FR(B);

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void fsubsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = FR(A) - FR(B);

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void faddsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = FR(A) + FR(B);

		UpdateFPRF(FR(D));

		if (rC)
			UPDATE_CR1();

		CYCLES(1);	// ??
	}

	void fmsubsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = (FR(A) * FR(C)) - FR(B);

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void fnmaddsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = -((FR(A) * FR(C)) + FR(B));

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void fnmsubsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = -((FR(A) * FR(C)) - FR(B));

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void fdivsx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = FR(A) / FR(B);

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);	// ??
		UpdateFPRF(FR(D));
	}

	void fnegx()
	{
		CHECK_FP_AVAILABLE();

		FRx(D) = FRx(B) ^ 0x80000000;

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);
	}

	void frsqrtex()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = 1.0f / sqrtf(FR(B));

		if (rC)
			DEBUG_BREAK;

		CYCLES(1);
		UpdateFPRF(FR(D));
	}
	//
	//   Floating-Point Multiply-Add
	//

	//
	//   Floating-Point Rounding and Conversion
	//

	//
	//   Floating-Point Compare
	//
	void fcmpu()
	{
		UINT32 c;

		CHECK_FP_AVAILABLE();

		if (FloatIsNaN(FR(A)) || FloatIsNaN(FR(B)))
			c = 1;
		else if (FR(A) < FR(B))
			c = 8;
		else if (FR(A) > FR(B))
			c = 4;
		else
			c = 2;

		fpscr.w &= ~(0xf << 12);
		fpscr.w |= c << 12;

		cr.w &= ~(0xf << (4 * (7 - crfD)));
		cr.w |= (c << (4 * (7 - crfD)));
#if 1
		if (FloatIsSNaN(FR(A)) || FloatIsSNaN(FR(B)))
			fpscr.vxsnan = 1;

//		fpscr.fx = ?
#endif
		CYCLES(1);
	}

	void fctiwx()
	{
		CHECK_FP_AVAILABLE();

		DEBUG_BREAK;
		if (FR(B) > (INT32)0x7fffffff)
			FRx(D) = 0x7fffffff;
		else if (FR(B) < (INT32)0x80000000)
			FRx(D) = 0x80000000;
		else
		{
			if (fpscr.rn == 0)
				FRx(D) = FloatRoundToNearest(FR(B));
			else if (fpscr.rn == 1)
				FRx(D) = FloatRoundToZero(FR(B));
			else if (fpscr.rn == 2)
				FRx(D) = FloatRoundToPosInf(FR(B));
			else
				FRx(D) = FloatRoundToNegInf(FR(B));
		}

		if (rC)
			DEBUG_BREAK;

		// More shit
		// Update CR float bits

		CYCLES(1);
	}

	void fctiwz()
	{
		CHECK_FP_AVAILABLE();

		if (FR(B) > (INT32)0x7fffffff)
			FRx(D) = 0x7fffffff;
		else if (FR(B) < (INT32)0x80000000)
			FRx(D) = 0x80000000;
		else
			FRx(D) = FloatRoundToZero(FR(B));

		if (rC)
			DEBUG_BREAK;

		// More shit
		// Update CR float bits

		CYCLES(1);
	}

	//
	//   Floating-Point Status and Control Register
	//
	void mtfsfx()
	{
		CHECK_FP_AVAILABLE();

#pragma message ("MTFSFX might not be correct")
		UINT32 fm = (inst >> 17) & 0xff;
		UINT32 mask = 0;

		for (int i = 0; i < 8; ++i)
		{
			UINT32 sub = (fm & (1 << (7 - i))) ? 0xf : 0;
			mask |= sub << (4*(7 - i));
		}

		fpscr.w &= ~mask;
		fpscr.w |= FRx(B) & mask;

		if (rC)
			DEBUG_BREAK;
		CYCLES(1);
	}

	void mtfsb0x()
	{
		CHECK_FP_AVAILABLE();

		// Bits 1 and 2 (FEX and VEX) cannot be explicitly set
		if (crbD != 0 && crbD > 2)
		{
			fpscr.w &= ~(1 << (31 - crbD));
			// Update CR1
			if (rC && crbD < 4)
			{
				DEBUG_BREAK; // THIS LOOKS WRONG
				
			}
		}
		CYCLES(1);
	}

	void mtfsb1x()
	{
		CHECK_FP_AVAILABLE();

		// Bits 1 and 2 (FEX and VEX) cannot be explicitly set
		if (crbD != 0 && crbD > 2)
		{
			fpscr.w |= (1 << (31 - crbD));
			// Update CR1
			if (rC && crbD < 4)
			{
				DEBUG_BREAK; // Check me
				
			}
		}
		CYCLES(1);
	}

	//
	//   Integer Load
	//
	void lhzx()
	{
		UINT32 b = A ? R(A) : 0;
		UINT32 ea = b + R(B);
		R(D) = ReadData16(ea);
		CYCLES(1); // TODO
	}

	void lha()
	{
		UINT32 b = A ? R(A) : 0;
		UINT32 ea = b + (INT32)(SIMM);
		R(D) = (INT32)(INT16)ReadData16(ea);
		CYCLES(1);
	}

	void lhax()
	{
		UINT32 b = A ? R(A) : 0;
		UINT32 ea = b + R(B);
		R(D) = (INT32)(INT16)ReadData16(ea);
		CYCLES(1);
	}

	void lhau()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		R(D) = (INT32)(INT16)ReadData16(ea);
		R(A) = ea;
		CYCLES(1);
	}

	void lhaux()
	{
		UINT32 ea = R(A) + R(B);
		R(D) = (INT32)(INT16)ReadData16(ea);
		R(A) = ea;
		CYCLES(1);
	}

	void lbzx()
	{
		UINT32 b = A ? R(A) : 0;
		UINT32 ea = b + R(B);
		R(D) = ReadData8(ea);
		CYCLES(1);
	}

	void lbz()
	{
		UINT32 b = A ? R(A) : 0;
		UINT32 ea = b + (INT32)(SIMM);
		R(D) = ReadData8(ea);
		CYCLES(1);
	}

	void lbzu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		R(D) = ReadData8(ea);
		R(A) = ea;
		CYCLES(1); // TODO
	}

	void lhz()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		R(D) = ReadData16(ea);
		CYCLES(1);
	}

	void lhzu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		R(D) = ReadData16(ea);
		R(A) = ea;
		CYCLES(1);
	}

	void lwz()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		R(D) = ReadData32(ea);
		CYCLES(1);
	}

	void lwzx()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		R(D) = ReadData32(ea);
		CYCLES(1);
	}

	void lwzu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		R(D) = ReadData32(ea);
		R(A) = ea;
		CYCLES(1);
	}

	void lwzux()
	{
		UINT32 ea = R(A) + R(B);
		R(D) = ReadData32(ea);
		R(A) = ea;
		CYCLES(1);
	}

	//
	//   Integer Store
	//
	void stb()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		WriteData8(ea, R(S) & 0xff);
		CYCLES(1); // TODO: One cycle?!
	}

	void stbu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		WriteData8(ea, R(S) & 0xff);
		R(A) = ea;
		CYCLES(1); // TODO: One cycle?!
	}

	void stbx()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		WriteData8(ea, R(S));
		CYCLES(1); // TODO: One cycle?!
	}

	void sth()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		WriteData16(ea, R(S));
		CYCLES(1); // TODO: One cycle?!
	}

	void sthu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		WriteData16(ea, R(S));
		R(A) = ea;
		CYCLES(1); // TODO: One cycle?!
	}

	void sthx()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		WriteData16(ea, R(S));
		CYCLES(1); // TODO: One cycle?!
	}

	void lmw()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		UINT32 r = D;

		if (ea & 3)
		{
			DEBUG_BREAK;
			// ALIGNMENT EXCEPTION!!!!
			// ALSO, WHAT ABOUT CROSSING PAGES OR MMU VIOLATIONS
		}

		do
		{
			R(r) = ReadData32(ea);
			ea += 4;
		} while(++r <= 31);
		CYCLES(1 + (32 - r));
	}

	void stmw()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		UINT32 r = S;

		if (ea & 3)
		{
			DEBUG_BREAK;
			// ALIGNMENT EXCEPTION!!!!
			// ALSO, WHAT ABOUT CROSSING PAGES OR MMU VIOLATIONS
		}

		do
		{
			WriteData32(ea, R(r));
			ea += 4;
		} while(++r <= 31);
		CYCLES(1 + (32 - r));
	}

	void stw()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		WriteData32(ea, R(S));
		CYCLES(1); // TODO: One cycle?!
	}

	void stwx()
	{
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		WriteData32(ea, R(S));
		CYCLES(1); // TODO: One cycle?!
	}

	void stwcx()
	{
		// TODO
		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);

		cr.w &= ~(0xf << (4 * (7 - 0)));

		//if (reserve)
		if (1)
		{
			WriteData32(ea, R(S));
			cr.w |= ((1 << 1) | xer.so) << (4 * (7 - 0));
			reserve = 0;
		}
		else
			cr.w |= ((0 << 1) | xer.so) << (4 * (7 - 0));

		CYCLES(1); // TODO: One cycle?!
	}

	void stwu()
	{
		UINT32 ea = R(A) + (INT32)(SIMM);
		WriteData32(ea, R(S));
		R(A) = ea;
		CYCLES(1); // TODO: One cycle?!
	}

	//
	//   Integer Load and Store with Byte-Reverse
	//


	//
	//   Integer Load and Store Multiple
	//

	//
	//   Integer Load and Store String
	//

	//
	//   Memory Synchronization
	//
	void isync()
	{
		// Nothing to do?
		CYCLES(1);
	}

	void sync()
	{
		// Nothing to do?
		CYCLES(1); // ????
	}

	//
	//   Floating-Point Store
	//
	void fmrx()
	{
		CHECK_FP_AVAILABLE();

		FRx(D) = FRx(B);

		if (rC)
			DEBUG_BREAK;
		CYCLES(1);
	}

	void fselx()
	{
		CHECK_FP_AVAILABLE();

		FR(D) = (FR(A) >= 0.0) ? FR(C) : FR(B);

		if (rC)
			DEBUG_BREAK;
		CYCLES(1);
	}

	void lfs()
	{
		CHECK_FP_AVAILABLE();

		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		FRx(D) = ReadData32(ea);
		CYCLES(1);
	}

	void lfsu()
	{
		CHECK_FP_AVAILABLE();

		UINT32 ea = R(A) + (INT32)(SIMM);
		FRx(D) = ReadData32(ea);
		R(A) = ea;
		CYCLES(1);
	}

	void lfsx()
	{
		CHECK_FP_AVAILABLE();

		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		FRx(D) = ReadData32(ea);
		CYCLES(1);
	}

	void stfs()
	{
		CHECK_FP_AVAILABLE();

		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + (INT32)(SIMM);
		WriteData32(ea, FRx(S));
		CYCLES(1);
	}

	void stfsu()
	{
		CHECK_FP_AVAILABLE();

		UINT32 ea = R(A) + (INT32)(SIMM);
		WriteData32(ea, FRx(S));
		R(A) = ea;
		CYCLES(1);
	}

	void stfsx()
	{
		CHECK_FP_AVAILABLE();

		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		WriteData32(ea, FRx(S));
		CYCLES(1);
	}

	void stfiwx()
	{
		CHECK_FP_AVAILABLE();

		UINT32 b = (A ? R(A) : 0);
		UINT32 ea = b + R(B);
		WriteData32(ea, FRx(S));
		CYCLES(1);
	}

	//
	//   Branch
	//
	void bcctrx()
	{
		int cnd_ok = BIT_TEST(BO, 4) || (BIT(cr.w, 31 - BI) == BIT(BO, 3));

		if (cnd_ok)
			next_pc = ctr & ~3;
		if (LK)
			lr = pc + 4;
		CYCLES(1);
	}

	void bx()
	{
		UINT32 sexli = (LI << 2) | (((LI << 2) & (1 << 25)) ? 0xfc000000 : 0);
		next_pc = AA ? sexli : pc + sexli;

		if (LK)
			lr = pc + 4;

		CYCLES(1);
	}

	void bcx()
	{
		if (!BIT_TEST(BO, 2))
			ctr = ctr - 1;

		int ctr_ok = BIT_TEST(BO, 2) || ((ctr != 0) ^ BIT(BO, 1));
		int cnd_ok = BIT_TEST(BO, 4) || (BIT(cr.w, 31 - BI) == BIT(BO, 3));

		int tmp = BIT_TEST(cr.w, 31 - BI);
		int tmp2 = BIT_TEST(BO, 3);

		if (ctr_ok && cnd_ok)
		{
			UINT32 sexbd = SEX16(BD << 2);
			next_pc = AA ? sexbd : pc + sexbd;
		}
		if (LK)
			lr = pc + 4;

		CYCLES(1);
	}

	// bcctrx

	void bclrx()
	{
		if (!BIT_TEST(BO, 2))
			ctr = ctr - 1;

		int ctr_ok = BIT_TEST(BO, 2) || ((ctr != 0) ^ BIT(BO, 1));
		int cnd_ok = BIT_TEST(BO, 4) || (BIT(cr.w, 31 - BI) == BIT(BO, 3));

		if (ctr_ok && cnd_ok)
			next_pc = lr & ~3;
		if (LK)
			lr = pc + 4;

		CYCLES(1);
	}

	//
	//   Condition Register Logical
	//
	void crand()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) & BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void crandc()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) & !BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void crnand()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = !(BIT(cr.w, 31 - crbA) & BIT(cr.w, 31 - crbB));
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void creqv()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) == BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void cror()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) | BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void crorc()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) | !BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void crnor()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = !(BIT(cr.w, 31 - crbA) | BIT(cr.w, 31 - crbB));
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	void crxor()
	{
		UINT32 d = 31 - crbD;
		UINT32 res = BIT(cr.w, 31 - crbA) ^ BIT(cr.w, 31 - crbB);
		cr.w &= ~(1 << d);
		cr.w |= res << d;
		CYCLES(1);
	}

	//
	//   System Linkage
	//

	void esa()
	{
		if (msr.sa == 0)
		{
			esasrr.pr = msr.pr;
			esasrr.ap = msr.ap;
			esasrr.sa = msr.sa;
			esasrr.ee = msr.ee;

			msr.pr = 0;
			msr.ap = 0;
			msr.sa = 1;
			msr.ee = 0;
		}
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED); // TODO

		// TODO: Memory region checking

		CYCLES(1);
	}

	void dsa()
	{
		// TODO: CHECK ME
		if (msr.sa == 1)
		{
			msr.pr = esasrr.pr;
			msr.ap = esasrr.ap;
			msr.sa = esasrr.sa;
			msr.ee = esasrr.ee;
		}
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED); // TODO
		CYCLES(1);
	}


	// mfrom
	void rfi()
	{
		msr.w &= ~0x87c0ff73;
		msr.w |= srr1 & 0x87c0ff73;
		msr.w &= ~(1 << (31 - 13));
		// TODO: Check for pending intterupts!!!
		next_pc = srr0 & ~0x3;
		CYCLES(3);
	}

	// sc
	void sc()
	{
		// TODO: Check reserved bits
		srr0 = pc + 4;

		srr1 &= ~0x783f0000;
		srr1 &= ~0x87C0FF73; // Could combine the above
		srr1 |= (msr.w & 0x87C0FF73);

		// TODO: Make a table of MSR mask bits for other exceptions
		msr.pow = 0;
		msr.fp = 0;
		msr.be = 0;
		msr.dr = 0;
		msr.fe1 = 0;
		msr.ri = 0;
		msr.ee = 0;
		msr.fe0 = 0;
		msr.le = msr.ile;
		msr.pr = 0;
		msr.se = 0;
		msr.ir = 0;

		next_pc = (msr.ip ? 0xfff00000 : ibr) + 0xc00;
		CYCLES(1);
	}


	//
	//   Trap
	//

	//
	//   Processor Control
	//

	void eqvx()
	{
		R(A) = ~(R(S) ^ R(B));

		if (rC)
			UPDATE_CR0(R(A));
		CYCLES(1);
	}

	// mcrxr
	void mtcrf()
	{
		UINT32 crm = (inst >> 12) & 0xff;
		UINT32 mask = 0;

		for (UINT32 i = 0; i < 8; ++i)
		{
			UINT32 sub = (crm & (1 << (7 - i))) ? 0xf : 0;
			mask |= sub << (4 * (7 - i));
		}

		cr.w &= ~mask;
		cr.w |= R(S) & mask;
		CYCLES(1);
	}

	void mfcr()
	{
		R(D) = cr.w;
		CYCLES(1);
	}

	void mfmsr()
	{
		if (SUPERVISOR_MODE)
			R(D) = msr.w;
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);

		CYCLES(1);
	}

	void mfspr()
	{
		UINT32 n = ((SPR & 0x1f) << 5) | ((SPR >> 5) & 0x1f);

		if ((n & 0x10) && !SUPERVISOR_MODE)
		{
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);
		}
		else
		{
			switch (n)
			{
				case 1:		R(D) = xer.w;		break;
				case 8:		R(D) = lr;			break;
				case 9:		R(D) = ctr;			break;
				case 18:	R(D) = dsisr;		break;
				case 19:	R(D) = dar;			break;
				case 22:	R(D) = dec;			break;
				case 25:	R(D) = sdr1;		break;
				case 26:	R(D) = srr0;		break;
				case 27:	R(D) = srr1;		break;
				case 272:	R(D) = sprg[0];		break;
				case 273:	R(D) = sprg[1];		break;
				case 274:	R(D) = sprg[2];		break;
				case 275:	R(D) = sprg[3];		break;
//				case 282:	R(D) = ear;			break;
				case 284:	R(D) = tbr.l;		break;
				case 285:	R(D) = tbr.u;		break;
				case 287:	R(D) = pvr;			break;
				case 528:	R(D) = ibat[0].u;	break;
				case 529:	R(D) = ibat[0].l;	break;
				case 530:	R(D) = ibat[1].u;	break;
				case 531:	R(D) = ibat[1].l;	break;
				case 532:	R(D) = ibat[2].u;	break;
				case 533:	R(D) = ibat[2].l;	break;
				case 534:	R(D) = ibat[3].u;	break;
				case 535:	R(D) = ibat[3].l;	break;
				case 536:	R(D) = dbat[0].u;	break;
				case 537:	R(D) = dbat[0].l;	break;
				case 538:	R(D) = dbat[1].u;	break;
				case 539:	R(D) = dbat[1].l;	break;
				case 540:	R(D) = dbat[2].u;	break;
				case 541:	R(D) = dbat[2].l;	break;
				case 542:	R(D) = dbat[3].u;	break;
				case 543:	R(D) = dbat[3].l;	break;
//				case 1013:	R(D) = dabr;		break;
				// PPC602 specific
				case 1008:	R(D) = hid0.w;	break;
				case 1009:	R(D) = hid1.w;	break;
				case 1010:	R(D) = iabr;	break;
				case 1021:	R(D) = sp;		break;
				case 1022:	R(D) = lt;		break;
				case 976:	R(D) = dmiss;	break;
				case 977:	R(D) = dcmp;	break;
				case 978:	R(D) = hash1;	break;
				case 979:	R(D) = hash2;	break;
				case 980:	R(D) = imiss;	break;
				case 981:	R(D) = icmp;	break;
				case 982:	R(D) = rpa;		break;
				case 984:	R(D) = tcr;		break;
				case 986:	R(D) = ibr;		break;
				case 987:	R(D) = esasrr.w;break;
				case 991:	R(D) = ser;		break;
				case 990:	R(D) = sebr;	break;
				default:
				{
					DEBUG_BREAK;
					//if (SUPERVISOR_MODE)
					//	SIGNAL_EXCEPTION(PROGRAM);
					//else
					//SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);
				}
			}
		}
		if ((n >= 528) && (n <= 543))
			CYCLES(3);
		else
			CYCLES(1);
	}

	void mftb()
	{
		UINT32 n = ((inst >> 6) & 0x1e0) | ((inst >> 16) & 0x1f);

		if (n == 268)
			R(D) = tbr.l;
		else if (n == 269)
			R(D) = tbr.u;
		else
		{
			// EXCEPTIONS and shit
			DEBUG_BREAK;
		}
		CYCLES(1);
	}

	void mtmsr()
	{
		// TODO: Side effects!
		if (SUPERVISOR_MODE)
		{
			msr.w = R(D);
		}
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);

		CYCLES(2);
	}

	void mtspr()
	{
		UINT32 n = ((SPR & 0x1f) << 5) | ((SPR >> 5) & 0x1f);

		// NOTE: Some of these will require seperate write functions
		// to handle side-effects
		if ((n & 0x10) && !SUPERVISOR_MODE)
		{
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);
		}
		else
		{
			switch (n)
			{
				case 1:		xer.w = R(S);		break;
				case 8:		lr = R(S);			break;
				case 9:		ctr = R(S);			break;
				case 18:	dsisr = R(S);		break;
				case 19:	dar = R(S);			break;
				case 22:	dec = R(S);			break;	// TODO: Cause exception if fiddling bit 0
				case 25:	sdr1 = R(S);		break;
				case 26:	srr0 = R(S);		break;
				case 27:	srr1 = R(S);		break;
				case 272:	sprg[0] = R(S);		break;
				case 273:	sprg[1] = R(S);		break;
				case 274:	sprg[2] = R(S);		break;
				case 275:	sprg[3] = R(S);		break;
//				case 282:	ear = R(S);			break;
				case 284:	tbr.l = R(S);		break;
				case 285:	tbr.u = R(S);		break;
				case 528:	ibat[0].u = R(S);	break;
				case 529:	ibat[0].l = R(S);	break;
				case 530:	ibat[1].u = R(S);	break;
				case 531:	ibat[1].l = R(S);	break;
				case 532:	ibat[2].u = R(S);	break;
				case 533:	ibat[2].l = R(S);	break;
				case 534:	ibat[3].u = R(S);	break;
				case 535:	ibat[3].l = R(S);	break;
				case 536:	dbat[0].u = R(S);	break;
				case 537:	dbat[0].l = R(S);	break;
				case 538:	dbat[1].u = R(S);	break;
				case 539:	dbat[1].l = R(S);	break;
				case 540:	dbat[2].u = R(S);	break;
				case 541:	dbat[2].l = R(S);	break;
				case 542:	dbat[3].u = R(S);	break;
				case 543:	dbat[3].l = R(S);	break;
//				case 1013:	dabr = R(S);		break;
				// PPC602 specific
				case 1008:	hid0.w = R(S);		break;
				case 1009:	DEBUG_BREAK;		break;	// hid1 is read-only
				case 1010:	iabr = R(S);		break;
				case 1021:	sp = R(S);			break;
				case 1022:	lt = R(S);			break;
				case 976:	dmiss = R(S);		break;
				case 977:	dcmp = R(S);		break;
				case 978:	DEBUG_BREAK;		break;	// hash1 is read-only
				case 979:	DEBUG_BREAK;		break;	// hash2 is read-only
				case 980:	imiss = R(S);		break;
				case 981:	icmp = R(S);		break;
				case 982:	rpa = R(S);			break;
				case 984:	tcr = R(S);			break;
				case 986:	ibr = R(S);			break;
				case 987:	esasrr.w = R(S);	break;
				case 991:	ser = R(S);			break;
				case 990:	sebr = R(S);		break;
#pragma message ("Need to handle SP and LT SPRs")
				default:
				{
					DEBUG_BREAK;
#if 0
					if (SUPERVISOR_MODE)
						SIGNAL_EXCEPTION(SUPER_PROGRAMR);
					else
						SIGNAL_EXCEPTION(PROGRAMR);
#endif
				}
			}
		}

		if ((n >= 528) && (n <= 543))
			CYCLES(2);
		else
			CYCLES(1);
	}


	//
	//   Cache Management
	//
	void dcbf()
	{
		// TODO?
		CYCLES(1);
	}

	void dcbi()
	{
		// TODO?
		CYCLES(1);
	}

	void dcbt()
	{
		// TODO?
		UINT32 ea = (A ? R(A) : 0) + R(B);
		ReadData8(ea);
		CYCLES(1);
	}

	void dcbst()
	{
		// TODO?
		UINT32 ea = (A ? R(A) : 0) + R(B);
		ReadData8(ea);
		CYCLES(1);
	}

	void dcbtst()
	{
		// TODO?
		UINT32 ea = (A ? R(A) : 0) + R(B);
		ReadData8(ea);
		CYCLES(1);
	}

	//
	//   Segment Register Manipulation
	//
	void mfsr()
	{
		if (SUPERVISOR_MODE)
			R(S) = sr[SR];
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);

		CYCLES(3);
	}

	void mtsr()
	{
		if (SUPERVISOR_MODE)
			sr[SR] = R(S);
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);

		CYCLES(3);
	}

	void mtsrin()
	{
		if (SUPERVISOR_MODE)
			sr[(R(B) >> 28) & 0xf] = R(S);
		else
			SIGNAL_EXCEPTION(MTMSR_PROGRAM_PRIVILEGED);

		CYCLES(3);
	}

	//
	//   Segment Register Manipulation
	//

	//
	//   Lookaside Buffer Management
	//
	void tlbli()
	{
		// TODO
		CYCLES(1);
	}

	void tlbie()
	{
		// TODO
		CYCLES(1);
	}

	void tlbld()
	{
		// TODO
		CYCLES(1);
	}

	//
	//   External Control
	//
	// eciwx - not supported
	// ecowx - not supported

	//
	//  Functions
	//
	UINT32 Execute(UINT32 cycles);
	void Reset(void);
	UINT32 GetPC() { return pc; }

	UINT32 GetReg(UINT32 id); // SHIT
	UINT32 GetLR(); // SHIT

	UINT8 ReadData8(UINT32 addr);
	UINT16 ReadData16(UINT32 addr);
	UINT32 ReadData32(UINT32 addr);
	void WriteData8(UINT32 addr, UINT8 data);
	void WriteData16(UINT32 addr, UINT16 data);
	void WriteData32(UINT32 addr, UINT32 data);

	UINT32 ReadWordMMU(UINT32 flags, UINT32 addr, UINT32 mask);
	void WriteWordMMU(UINT32 flags, UINT32 addr, UINT32 data, UINT32 mask);
	bool TranslateAddress(UINT32 flags, UINT32 *addr);
	void HandleExceptions();

	
public:
	void SoftReset(void);
	void Interrupt(void);
};
