#include "M2x.h"
#include "core.h"
#include "log.h"

#include "eeprom.h"


static char *ins_codes_8[] =
{
	"110",
	"10011xxxxx",
	"111",
	"101",
	"10010xxxxx",
	"10001xxxxx",
	"10000xxxxx",
};

static char *ins_codes_16[] =
{
	"110",
	"10011xxxx",
	"111",
	"101",
	"10010xxxx",
	"10001xxxx",
	"10000xxxx",
};

EEPROM::EEPROM(UINT32 size) :
				org(1),
				cs(0),
				di(0),
				ck(0),
				cmd_ptr(0),
				inst_valid(false),
				cmd_matched(false),
				write_enable(false),
				size(size),
				state(INSTRUCTION)
{
	data = new UINT8[size];
}

bool EEPROM::InitialiseEEPROM(const char *eeprom_file)
{
	char eeprom_path[256];
	sprintf(eeprom_path, "nvram\\%s\\%s", Core.GetActiveSetName(), eeprom_file);

	/* Use the new EEPROM file if it exists */
	if (Core.FileMan->FileExists(eeprom_path) == FILE_OK)
	{
		Core.FileMan->Open(0, eeprom_path, FILE_READ | FILE_WRITE, file_handle);

		if (Core.FileMan->Read(file_handle, data, size) != FILE_OK)
			LOGERROR("Warning: Error reading EEPROM file!\n");
		else
			LOGERROR("Loaded existing EEPROM file\n");

#pragma message ("Add size check to EEPROM file?")

		Core.FileMan->SetPos(file_handle, 0, FILE_POS_BEGIN);
	}
	else
	{
		FileHandle tmp;
		/* We have to load the original NVRAM file provided with the ROM set */
		sprintf(eeprom_path, "roms\\%s\\%s", Core.GetActiveSetName(), eeprom_file);

		Core.FileMan->Open(0, eeprom_file, FILE_READ | FILE_OPEN_ALWAYS, tmp);

		if (Core.FileMan->Read(tmp, data, size) != FILE_OK)
			LOGERROR("Warning: Error reading EEPROM file!\n");

		Core.FileMan->Close(tmp);

		/* Now create a new file in the NVRAM folder */
		sprintf(eeprom_path, "nvram\\%s", Core.GetActiveSetName());
		if (Core.FileMan->Open(eeprom_path, eeprom_file, FILE_READ | FILE_WRITE | FILE_OPEN_ALWAYS, file_handle) != FILE_OK)
			LOGERROR("Warning: Couldn't create new EEPROM file!\n");

		LOGERROR("Created new EEPROM file\n");
	}
	return true;
}

EEPROM::~EEPROM()
{
	if (file_handle)
	{
		Core.FileMan->Write(file_handle, data, size);
		Core.FileMan->Close(file_handle);
	}

	delete[] data;
}

void EEPROM::SetState(enum eState newstate)
{
	if (newstate == INSTRUCTION)
		state = INSTRUCTION;
	else if (newstate == INSTRUCTION_MATCHED)
	{
		switch (matching_cmd)
		{
			case READ:
			case ERASE:
			case WRITE:
			{
				bits_to_shift = org ? 6 : 7;
				state = ADDRESS;
				address_in = 0;
				break;
			}
			case WRAL:
			{
				bits_to_shift = org ? 16 : 8;
				state = DATA;
				data_in = 0;
				break;
			}
			case EWEN:
			{
				write_enable = true;
				state = INSTRUCTION;
				break;
			}
			case EWDS:
			{
				write_enable = false;
				state = INSTRUCTION;
				break;
			}
			case ERAL:
			{
				if (write_enable == true)
				{
					memset(data, 0, size);
				}

				LOGERROR("ERASE ALL\n");
				state = STATUS;
				status_delay = 2;
				break;
			}
		}
	}
	else if (newstate == GOT_ADDRESS)
	{
		switch (matching_cmd)
		{
			case READ:
			{
				// First bit is a dummy '0'
				bits_to_shift = (org ? 16 : 8);
				state = READ_DATA;
				break;
			}
			case ERASE:
			{
				if (write_enable)
				{
					if (org == 0)
					{
						data[address_in] = 0;
					}
					else
					{
						data[address_in*2 + 0] = 0;
						data[address_in*2 + 1] = 0;
					}
				}

				LOGERROR("ERASE: %x\n", address_in);
				state = STATUS;
				status_delay = 2;
				break;
			}
			case WRITE:
			{
				bits_to_shift = org ? 16 : 8;
				state = DATA;
				data_in = 0;
				break;
			}
			default:
				assert(0);
		}
	}
	else if (newstate == GOT_DATA)
	{
		if (matching_cmd == WRITE)
		{
			if (org == 0)
			{
				LOGERROR("WRITE %x TO ADDRESS : %x\n", data_in, address_in);
				data[address_in] = data_in;
			}
			else
			{
				LOGERROR("READ FROM ADDRESS : %x\n", address_in);
				data[address_in*2 + 0] = data_in >> 8;
				data[address_in*2 + 1] = data_in & 0xff;
			}
		}
		else if (matching_cmd == WRAL)
		{
			LOGERROR("WRAL Not implemented\n");
		}
		else
			assert(0);

		state = STATUS;
		status_delay = 2;
	}
}

void EEPROM::SetCS(UINT32 val)
{
	val = val && 1;

	// A valid instruction starts with a rising edge of CS
	if (!cs && val)
		inst_valid = true;

	if (!val)
	{
		//state = INSTRUCTION;
		inst_valid = false;
		cmd_matched = false;
		cmd_ptr = 0;
		ck = 0;
	}

	cs = val;
}

void EEPROM::SetDI(UINT32 val)
{
	di = val && 1;
}

void EEPROM::ClockDataOut()
{
	UINT32 val = 0;

	if (cs && inst_valid == true)
	{
		/*
		if (state == STATUS)
		{
			// TODO: Should not immediately be high
			val = 1;
			SetState(INSTRUCTION);
		}*/
		if (state == READ_DATA)
		{
			if (org == 0)
			{
				if (bits_to_shift == 9)
					val = 0;
				else
					val = data[address_in] >> (bits_to_shift -1);
			}
			else
			{
				if (bits_to_shift == 17)
					val = 0;
				else if (bits_to_shift > 8)
					val = data[address_in*2 + 0] >> (bits_to_shift - 8 - 1);
				else
					val = data[address_in*2 + 1] >> (bits_to_shift - 1);
			}

			if (--bits_to_shift == 0)
				SetState(INSTRUCTION);
		}
		else
			assert(0);
	}
	dO = val & 1;
}

UINT32 EEPROM::GetDO()
{
	if (state == STATUS)
	{
		if (--status_delay == 0)
		{
			SetState(INSTRUCTION);
			return 1;
		}
		else
			return 0;
	}

	return dO & 1;
}

void EEPROM::SetCK(UINT32 val)
{
	if (!cs || inst_valid == false)
		return;

	val = val && 1;

	// Clock in command/data on rising edge
	if (!ck && val)
	{
		if (state == READ_DATA)
			ClockDataOut();
		else
			ProcessData(di);
	}

	ck = val;
}

void EEPROM::ProcessData(UINT32 di)
{
	cmd_buf[cmd_ptr] = di;
	UpdateCmdPtr();

	if (state == INSTRUCTION)
	{
		int matches = 0;
		int cmd;

		// Compare the incoming bit against the list of instructions
		for (cmd = 0; cmd < sizeof(ins_codes_16) / sizeof(char*); ++cmd)
		{
			UINT32 len = strlen(ins_codes_16[cmd]);
			bool matched = true;

			if (cmd_ptr > len)
			{
				matched = false;
				continue;
			}

			for (UINT32 i = 0; i < cmd_ptr; ++i)
			{
				char a = *(ins_codes_16[cmd] + i);
				int b = cmd_buf[i];

				if (((a == '1' && b == 1)
					|| (a == '0' && b == 0)
					|| (a == 'X' || a == 'x')) == 0)
				{
					matched = false;
					break;
				}
			}

			if (matched == true)
			{
				if (++matches > 1)
					break;

				if (len == cmd_ptr)
				{
					cmd_matched = true;
					matching_cmd = cmd;
					SetState(INSTRUCTION_MATCHED);
				}
			}
		}

		if (matches == 0)
		{
			LOGERROR("%s: Bad EEPROM Command\n", __FILE__);
			assert(0);
		}
	}
	else if (state == ADDRESS)
	{
		address_in |= di << (bits_to_shift - 1);

		if (--bits_to_shift == 0)
		{
			SetState(GOT_ADDRESS);
		}
	}
	else if (state == DATA)
	{
		data_in |= di << (bits_to_shift - 1);

		if (--bits_to_shift == 0)
			SetState(GOT_DATA);
	}
	else
	{
		assert(0);
	}
}

