/******************************************************************************

	M48T58

	Emulation of M48T58 TIMEKEEPER SRAM

******************************************************************************/

#include "M2x.h"
#include "core.h"
#include "cpuexec.h"
#include "memory.h"
#include "log.h"

#include "m48t58.h"

M48T58::M48T58() :
	file_handle(0)
{
	SRAM = new UINT8[SRAM_SIZE];
}

bool M48T58::InitialiseSRAM(const char *nvram_file)
{
	char nvram_path[256];
	sprintf(nvram_path, "nvram\\%s\\%s", Core.GetActiveSetName(), nvram_file);

	/* Use the new SRAM file if it exists */
	if (Core.FileMan->FileExists(nvram_path) == FILE_OK)
	{
		Core.FileMan->Open(0, nvram_path, FILE_READ | FILE_WRITE, file_handle);

		if (Core.FileMan->Read(file_handle, SRAM, SRAM_SIZE) != FILE_OK)
			LOGERROR("Warning: Error reading SRAM file!\n");
		else
			LOGERROR("Loaded existing SRAM file\n");

		Core.FileMan->SetPos(file_handle, 0, FILE_POS_BEGIN);
	}
	else
	{
		FileHandle tmp;
		/* We have to load the original NVRAM file provided with the ROM set */
		sprintf(nvram_path, "roms\\%s\\%s", Core.GetActiveSetName(), nvram_file);

		Core.FileMan->Open(0, nvram_path, FILE_READ | FILE_OPEN_ALWAYS, tmp);

		if (Core.FileMan->Read(tmp, SRAM, SRAM_SIZE) != FILE_OK)
			LOGERROR("Warning: Error reading SRAM file!\n");

		Core.FileMan->Close(tmp);

		/* Now create a new file in the NVRAM folder */
		sprintf(nvram_path, "nvram\\%s", Core.GetActiveSetName());
		if (Core.FileMan->Open(nvram_path, nvram_file, FILE_READ | FILE_WRITE | FILE_OPEN_ALWAYS, file_handle) != FILE_OK)
			LOGERROR("Warning: Couldn't create new SRAM file!\n");

		LOGERROR("Created new SRAM file\n");
	}

	return true;
}

M48T58::~M48T58()
{
	if (file_handle)
	{
		Core.FileMan->Write(file_handle, SRAM, SRAM_SIZE);
		Core.FileMan->Close(file_handle);
	}

	if (SRAM)
		delete[] SRAM;
}

UINT8 M48T58::Read(UINT32 offset)
{
#pragma message("M48T58 needs proper register support")
	if (offset < 0x1ff8)
		return SRAM[offset];
	else
	{
		switch (offset)
		{
			case 0x1ff8: return 0x00;
			case 0x1ff9: return 0x00;
			case 0x1ffa: return 0x00;
			case 0x1ffb: return 0x14;
			case 0x1ffc: return 0x05;
			case 0x1ffd: return 0x31;
			case 0x1ffe: return 0x05;
			case 0x1fff: return 0x09;
		}
		LOGERROR("M48T58 REGISTER READ: %x\n", offset);
		return 0;
	}

}

void M48T58::Write(UINT32 offset, UINT8 data)
{
	if (offset < 0x1ff8)
		SRAM[offset] = data;
	else
	{
		LOGERROR("M48T58 REGISTER WRITE: %x with %x\n", offset, data);
	}
}
