/******************************************************************************

    fileman

    Class for loading and manipulation of files

******************************************************************************/

#include "M2x.h"
#include "log.h"
#include "fileman.h"

FileInfo::~FileInfo()
{
	//CloseHandle(sys_handle);
}

FileManager::FileManager()
{
	
}

FileManager::~FileManager()
{
	/* Close any open files */
	std::vector<FileInfo>::iterator it;
	for (it = file_list.begin(); it < file_list.end(); ++it)
		CloseHandle((*it).sys_handle);
}

bool FileManager::Validate(FileHandle handle, int &idx)
{
	std::vector<FileInfo>::iterator it;
	idx = 0;

	for (it = file_list.begin() ; it < file_list.end(); ++it, ++idx)
	{
		if (handle == (*it).handle)
			return true;
	}
	return false;
}

FileStatus FileManager::Open(const char *path, const char *file, int flags, FileHandle &handle)
{
	HANDLE hFile = 0;
	int file_flags = 0;
	int file_flags_ex = 0;
	char buffer[256];
	FileInfo *fileinfo;

	if (flags == 0)
		return FILE_ERROR;

	if (flags & FILE_READ)
		file_flags |= GENERIC_READ;
	if (flags & FILE_WRITE)
		file_flags |= GENERIC_WRITE;

	if (flags & FILE_OPEN_ALWAYS)
		file_flags_ex |= OPEN_ALWAYS;
	else
		file_flags_ex |= OPEN_EXISTING;
	

	/* Append the path, if provided */
	if (path != NULL)
		sprintf(buffer, "%s\\%s", path, file);
	else
		sprintf(buffer, "%s", file);

	/* Make sure we haven't already opened this file */
	std::vector<FileInfo>::iterator it;
	int idx = 0;

	for (it = file_list.begin() ; it < file_list.end(); ++it, ++idx)
	{
		if (strcmp(buffer, (*it).name) == 0)
		{
			LOGERROR("File %s already open!\n", buffer);
			return FILE_ERROR;
		}
	}

	/* If creating a new file, make sure the path exists */
	if (path != NULL && flags & FILE_OPEN_ALWAYS)
		CreateFolder(path);

	/* Convert to UTF-16 format */
	WCHAR *tmp = CharToWString(buffer);

	/* Call the OS function */
	hFile = CreateFile( tmp,
						file_flags,
						FILE_SHARE_READ,
						0,
						file_flags_ex,
						0,
						NULL );

	delete[] tmp;

	if (hFile == INVALID_HANDLE_VALUE)
		return FILE_ERROR;

	/* Create a file record for the new file */
	fileinfo = new FileInfo();

	/* TODO: Decide on a suitable hash */
	fileinfo->handle = (int)hFile * (int)hFile;
	fileinfo->sys_handle = hFile;
	fileinfo->flags = flags & (FILE_READ | FILE_WRITE);
	strcpy(fileinfo->name, buffer);

	/* Add it to the file list */
	file_list.push_back(*fileinfo);

	/* Return its handle */
	handle = fileinfo->handle;

	LOGERROR("File %s opened (SYS:%x HDL:%x)\n", buffer, hFile, handle);
	return FILE_OK;
}

FileStatus FileManager::Close(FileHandle &handle)
{
	int idx;

	if (Validate(handle, idx) == false)
		return FILE_ERROR;

	LOGERROR("CLOSING FILE %x\n", handle);

	CloseHandle(file_list.at(idx).sys_handle);

	file_list.erase(file_list.begin() + idx);
	handle = 0;

	return FILE_OK;
}

FileStatus FileManager::Read(FileHandle handle, void *buffer, int bytes)
{
	int idx;
	DWORD bytes_read;

	if (Validate(handle, idx) == false)
		return FILE_ERROR;

	/* Read flag must be set */
	if ((file_list.at(idx).flags & FILE_READ) == 0)
		return FILE_ERROR;

	ReadFile(file_list.at(idx).sys_handle, buffer, bytes, &bytes_read, NULL);

	if (bytes_read == 0)
		return FILE_ERROR;

	return FILE_OK;
}

FileStatus FileManager::Write(FileHandle handle, void *buffer, int bytes)
{
	int idx;
	DWORD bytes_written;

	if (Validate(handle, idx) == false)
		return FILE_ERROR;

	/* Write flag must be set */
	if ((file_list.at(idx).flags & FILE_WRITE) == 0)
	{
		LOGERROR("Trying to write to a file without flag set\n");
		return FILE_ERROR;
	}

	WriteFile(file_list.at(idx).sys_handle, buffer, bytes, &bytes_written, NULL);
	
	if (bytes_written == 0)
	{
		LOGERROR("Failed to write file (SYS:%x HDL:%x)\n", (int)file_list.at(idx).sys_handle, handle);
		return FILE_ERROR;
	}

	return FILE_OK;
}

FileStatus FileManager::Delete(const char *path, const char *file)
{
	char buffer[256];

	/* Append the path, if provided */
	if (path != NULL)
		sprintf(buffer, "%s\\%s", path, file);
	else
		sprintf(buffer, "%s", file);

	/* Convert to UTF-16 format */
	WCHAR *tmp = CharToWString(buffer);

	int res = DeleteFile(tmp);
	delete []tmp;

	if (res == TRUE)
		return FILE_OK;
	else
		return FILE_ERROR;
}

FileStatus FileManager::SetPos(FileHandle handle, int offset, FilePos pos)
{
#pragma message ("Need to finish " __FUNCTION__)
	int fpos;
	int idx;

	if (Validate(handle, idx) == false)
		return FILE_ERROR;

	if (pos == FILE_POS_BEGIN)
		fpos = FILE_BEGIN;
	else if (pos == FILE_POS_CURRENT)
		fpos = FILE_CURRENT;
	else if (pos == FILE_END)
		fpos = FILE_END;
	else
		return FILE_ERROR;

	int res = SetFilePointer(file_list.at(idx).sys_handle, offset, NULL, fpos);

	if (res == INVALID_SET_FILE_POINTER)
		return FILE_ERROR;
	else
		return FILE_OK;
}


FileStatus FileManager::GetFileSize(FileHandle handle, UINT64 &size)
{
	int pos;

	if (Validate(handle, pos) == false)
		return FILE_ERROR;

	LARGE_INTEGER fsize;
	GetFileSizeEx(file_list.at(pos).sys_handle, &fsize);
	size = fsize.QuadPart;
	return FILE_OK;
}

FileStatus FileManager::FileExists(const char *name)
{
	HANDLE result;
	WIN32_FIND_DATA tmp;

	/* Convert to wide char format */
	WCHAR *path = CharToWString(name);

	result = FindFirstFile(path, &tmp);

	delete[] path;
	return result == INVALID_HANDLE_VALUE ? FILE_ERROR : FILE_OK;
}

FileStatus FileManager::CreateFolder(const char *name)
{
	int result;

	/* Convert to wide char format */
	WCHAR *dir = CharToWString(name);

	result = CreateDirectory(dir, 0);
	
	delete[] dir;
	return result ? FILE_OK : FILE_ERROR;
}
