/******************************************************************************

    memory

    Emulated system memory access handling

******************************************************************************/

#include "memory.h"
#include "core.h"
#include "cpuexec.h"
#include "log.h"


static UINT32 read_mem(const memory_map *map, UINT32 addr, UINT32 mask)
{
	for (int i = 0; map[i].tag != 0xff; ++i)
	{
		const memory_map &entry = map[i];

		if ((addr >= entry.start) && (addr <= entry.end))
		{
			if (entry.read_handler)
				return (UINT32)entry.read_handler((addr - entry.start) >> 2, mask);
			else
				break;
		}
	}

	CLOGERROR(CYAN, "CPU[%d]: Unmapped read from 0x%0.8x (PC:0x%0.8x)\n", Core.CPUMan->GetActiveCPU(), addr, Core.CPUMan->ActiveCPUGetPC());
	return 0;
}

static void write_mem(const memory_map *map, UINT32 addr, UINT32 data, UINT32 mask)
{
	for (int i = 0; map[i].tag != 0xff; ++i)
	{
		const memory_map &entry = map[i];

		if ((addr >= entry.start) && (addr <= entry.end))
		{
			if (entry.write_handler)
			{
				entry.write_handler((addr - entry.start) >> 2, data, mask);
				return;
			}
			else
				break;
		}
	}

	CLOGERROR(GREEN, "CPU[%d]: Unmapped write to 0x%0.8x with 0x%0.8x (PC:0x%0.8x)\n", Core.CPUMan->GetActiveCPU(), addr, data, Core.CPUMan->ActiveCPUGetPC());
}


UINT32 read_mem32(const memory_map *map, UINT32 addr, UINT32 mask)
{
	return read_mem(map, addr, mask);
}

void write_mem32(const memory_map *map, UINT32 addr, UINT32 data, UINT32 mask)
{
	write_mem(map, addr, data, mask);
}

UINT16 read_mem16(const memory_map *map, UINT32 addr, UINT16 mask)
{
	UINT32 shift = 16*(1 - (addr & 1));
	return read_mem(map, addr & ~1, 0xffff << shift) >> shift;
}

void write_mem16(const memory_map *map, UINT32 addr, UINT16 data, UINT16 mask)
{
	UINT32 shift = 8*(2 - (addr & 2));
	write_mem(map, addr & ~1, data << shift, 0xffff << shift);
}

UINT8 read_mem8(const memory_map *map, UINT32 addr)
{
	UINT32 shift = 8* (3 - (addr & 3));
	return read_mem(map, addr & ~3, 0xff << shift) >> shift;
}

void write_mem8(const memory_map *map, UINT32 addr, UINT8 data)
{
	UINT32 shift = 8*(3 - (addr & 3));
	write_mem(map, addr & ~3, data << shift, 0xff << shift);
}

READ32_HANDLER( nop_r )
{
	return 0;
}

WRITE32_HANDLER( nop_w )
{
}
