/******************************************************************************

    M2 Hardware Driver

******************************************************************************/

#include "M2x.h"
#include "core.h"
#include "romload.h"
#include "cpuexec.h"
#include "memory.h"
#include "log.h"
#include "video.h"

#include "konamim2.h"
#include "trengine.h"
#include "m2defs.h"
#include "atapi.h"
#include "eeprom.h"
#include "m48t58.h"
#include "ymz280b.h"


/**************************************
 *
 *  Defines
 *
 *************************************/

#define POLYSTARS		0
#define TOTALVICE		1
#define BATTLETRYST		2
#define EVILNIGHT		3
#define HELLNIGHT		4
#define HEATOF11		5
#define FZ35S			6

#define SELECTED_GAME	HEATOF11


/**************************************
 *
 *  Game File Definitions
 *
 *************************************/

static const m2_game_info gamelist[] =
{
	{
		"polystar",
		"Tobe! Polystars",
		"JAA",
		"1997",
		{ "623b01.8q", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ "polystar.img", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_8MB
	},

	{
		"totlvice",
		"Total Vice",
		"JAD",
		"1997",
		{ "623b01.8q", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "639jaa02.bin", 0x100000, 0xc6163818, "b6f8f2d808b98610becc0a5be5443ece3908df0b" },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ "vice.raw", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_YMZ280B
	},

	{
		"btltryst",
		"Battle Tryst",
		"JAC",
		"1998",
		{ "636a01.8q", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "m48t58.bin", 0x002000, 0x8611ff09, "6410236947d99c552c4a1f7dd5fd8c7a5ae4cba1" },
		{ "btl.raw", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_TIMEKEEPER
	},

	{
		"evilngt",
		"Evil Night",
		"EAA",
		"1998",
		{ "636a01.8q",  0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "810a03.16h", 0x400000, 0x4cd79d98, "12fea41cfc5c1b883ffbeda7e428dd1d1bf54d7f" },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "m48t58.bin", 0x002000, 0x8611ff09, "6410236947d99c552c4a1f7dd5fd8c7a5ae4cba1" },
		{ "810uba02.toc", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_YMZ280B | GAME_HAS_TIMEKEEPER
	},

	{
		"hellngt",
		"Hell Night",
		"EAA",
		"1998",
		{ "636a01.8q",  0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "810a03.16h", 0x400000, 0x4cd79d98, "12fea41cfc5c1b883ffbeda7e428dd1d1bf54d7f" },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "m48t58.bin", 0x002000, 0x8611ff09, "6410236947d99c552c4a1f7dd5fd8c7a5ae4cba1" },
		{ "hellngt.toc", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_YMZ280B | GAME_HAS_TIMEKEEPER
	},

	{
		"heatof11",
		"Heat of Eleven '98",
		"EAA",
		"1998",
		{ "636a01.8q",  0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ "93c46.7k", 0x200000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ "dallas.5e",  0x002000, 0x8611ff09, "6410236947d99c552c4a1f7dd5fd8c7a5ae4cba1" },
		{ "heatof11.raw", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		GAME_HAS_TIMEKEEPER
	},

	{
		"fz35s",
		"FZ-35S",
		"???",
		"1998",
		{ "fz35s.bin",  0x100000, 0xbd879f93, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		{ 0 },
		{ 0 },
		{ 0 },
		{ "cd.img", 0, 0, "e2d63bfbd2b15260a2664082652442eadea3eab6" },
		0
	},
};


/**************************************
 *
 *  Statics
 *
 *************************************/

UINT8 *RAM;
UINT32 MEMCFG;
UINT32 RAM_SIZE_BYTES;
UINT32 RAM_SIZE_MASK;

static UINT8 *bios;
static UINT8 *ymz_rom;

// Devices
static YMZ280B *ymz;
static EEPROM *eeprom;
static M48T58 *m48t58;

static int _2004val;
static int _208val;
static int _284val;

// Video
static struct
{
	UINT32 vloc;
	UINT32 vint;
	UINT32 vdc0;
	UINT32 vdc1;
	UINT32 fv0a;
	UINT32 fv1a;
	UINT32 avdi;
	UINT32 vdli;
	UINT32 vcfg;
	UINT32 dmt0;
	UINT32 dmt1;
	UINT32 lfsr;

	UINT32 field;
} vdu;


// Interrupt controller
static UINT32 int_enable;
static UINT32 int_status;

static UINT32 cde_int_enable;
static UINT32 cde_status;

static UINT32 hw_timer;


/**************************************
 *
 *  Forward Definitions
 *
 *************************************/

static READ32_HANDLER( cpu_id_r );
static WRITE32_HANDLER( cpu_id_w );

static READ32_HANDLER( ram_r );
static WRITE32_HANDLER( ram_w );
static READ32_HANDLER( tmp_r );
static READ32_HANDLER( rom_r );
static READ32_HANDLER( unk_r );
static WRITE32_HANDLER( unk_w );
static READ32_HANDLER( memcfg_r );

static READ32_HANDLER( cde_r );
static WRITE32_HANDLER( cde_w );


static WRITE32_HANDLER( video_w );

static READ32_HANDLER( slot2_r );
static READ32_HANDLER( slot3_r );
static WRITE32_HANDLER( slot3_w );
static READ32_HANDLER( slot5_r );
static READ32_HANDLER( slot6_r );
static READ32_HANDLER( slot7_r );

static READ32_HANDLER( random_r );


static READ32_HANDLER( reg_r );
static WRITE32_HANDLER( reg_w );

static READ32_HANDLER( x30000_r );

static bool LoadCDImage();



#define SET		0
#define CLEAR	1
#define WRITE	2

void WriteM2Reg(UINT32 &reg, UINT32 data, UINT32 access)
{
	switch (access)
	{
		case SET:	reg |= data;	break;
		case CLEAR:	reg &= ~data;	break;
		case WRITE: reg = data;		break;
	}
}

/**************************************
 *
 *  Interrupt Control
 *
 *************************************/

void SetBDAInterrupt(int id)
{
	int_status |= id;

	if (int_enable & id)
	{
		Core.CPUMan->SignalInterrupt(0);
	}
}

void ClearBDAInterrupt(int id)
{
	int_status &= ~id;
}

void SetCDEInterrupt(int id)
{
	cde_status |= id;

	if (cde_int_enable & id)
	{
		SetBDAInterrupt(INT_CDE);
	}
}





// When masked with 0x07e6000, it MUST be 0x364. Don't mess with that.
#if (SELECTED_GAME == CONSOLE)
#define SYSCONFIG	0x00078000 | SYSCFG_AUDIO_ASASHI | SYSCFG_REGION_JAPAN | SYSCFG_VIDEO_ENCODER_MEIENC
#else
#define SYSCONFIG	0x03640000 | SYSCFG_AUDIO_ASASHI | SYSCFG_REGION_JAPAN | SYSCFG_VIDEO_ENCODER_MEIENC
#endif

// 00xx

static READ32_HANDLER( cpu_r )
{
	return _2004val;
}

static WRITE32_HANDLER( cpu_w )
{
	int cpuid = strcmp(Core.CPUMan->ActiveCPUGetName(), "PPC1");

//	CLOGERROR(CYAN, "[%x] CPU COMM %x (PC:%x)\n", cpuid, data, Core.CPUMan->ActiveCPUGetPC());

	_2004val = data;

	// Master writes 0xc00000 to trigger INT
	// Slave writes  0x200000 as an int clear?

//	if ((data & 0xc00000) == 0xc00000)
	{
		if (cpuid == 0)
			Core.CPUMan->SignalInterrupt(1);
	}

//	if ((data & 0x200000))
//	{
//		// LIKELY INCORRECT
//		if (cpuid == 1)
//			SetBDAInterrupt(INT_CPUCOMM);
//	}
}


static WRITE32_HANDLER( vbltimer_w )
{
	// 0x8000 for VBLANK
	if (data & 0x8000)
		int_status &= ~INT_VINT1;

	// 0x80000000 for BDA
	if (data & 0x80000000)
		int_status &= ~INT_VINT0;

//	LOGERROR("VBLTIMER: %x %x\n", data, Core.CPUMan->ActiveCPUGetPC());
}


UINT32 uartbuf;

static READ32_HANDLER( uartbuf_r )
{
	return uartbuf;
}

static WRITE32_HANDLER( uartbuf_w )
{
	uartbuf = data;
}

//     1111222222222233
//     6789012345678901
// 6: |               x| Coin 1
//    |              x | Coin 2
//    |            x   | Service coin
//    |.......x........| EEPROM DO
//    |..x.............| ATAPI/CD status?
//    |.....x..........| ADC
//    |...xx...........| SIO related? (set to 3 NOTE: Total Vice doesn't like this!)



UINT16 GetCoinInputs()
{
	UINT16 val = 0xffff;

	if (GetAsyncKeyState('5') & 0x8000)
		val &= ~1;
	if (GetAsyncKeyState('6') & 0x8000)
		val &= ~2;
	if (GetAsyncKeyState('9') & 0x8000)
		val &= ~8;
	return val;
}

/*
	Polystars
	=========
	8000 = P2 START
	4000 = P2 BUTTON 3
	2000 = P2 BUTTON 2
	1000 = P2 BUTTON 1
	0800 = P2 DOWN
	0400 = P2 UP
	0200 = P2 RIGHT
	0100 = P2 LEFT
	0080 = P1 START
	0040 = P1 BUTTON 3
	0020 = P1 BUTTON 2
	0010 = P1 BUTTON 1
	0008 = P1 DOWN
	0004 = P1 UP
	0002 = P1 RIGHT
	0001 = P1 LEFT
*/
UINT16 ReadPort6()
{
	UINT16 val = 0xffff;

	if (GetAsyncKeyState(VK_F2) & 0x8000)
		val &= ~4;

	return val;
}

UINT16 ReadPort4()
{
	UINT16 val = 0xffff;

	if (GetAsyncKeyState('2') & 0x8000)
		val &= ~0x8000;
	if (GetAsyncKeyState('H') & 0x8000)		// P2 BUTTON3
		val &= ~0x4000;
	if (GetAsyncKeyState('G') & 0x8000)		// P2 BUTTON2
		val &= ~0x2000;
	if (GetAsyncKeyState('F') & 0x8000)		// P2 BUTTON1
		val &= ~0x1000;
	if (GetAsyncKeyState('E') & 0x8000)		// P2 DOWN
		val &= ~0x800;
	if (GetAsyncKeyState('D') & 0x8000)		// P2 UP
		val &= ~0x400;
	if (GetAsyncKeyState('S') & 0x8000)		// P2 RIGHT
		val &= ~0x200;
	if (GetAsyncKeyState('A') & 0x8000)		// P2 LEFT
		val &= ~0x100;

	if (GetAsyncKeyState('1') & 0x8000)
		val &= ~0x80;
//	if (GetAsyncKeyState('Y') & 0x8000)			// BUTTON 3 (UNUSED)
//		val &= ~0x40;
	if (GetAsyncKeyState('X') & 0x8000)			// BUTTON 2
		val &= ~0x20;
	if (GetAsyncKeyState('Z') & 0x8000)			// BUTTON 1
		val &= ~0x10;
	if (GetAsyncKeyState(VK_DOWN) & 0x8000)		// DOWN
		val &= ~0x8;
	if (GetAsyncKeyState(VK_UP) & 0x8000)		// DOWN
		val &= ~0x4;
	if (GetAsyncKeyState(VK_RIGHT) & 0x8000)	// RIGHT
		val &= ~0x2;
	if (GetAsyncKeyState(VK_LEFT) & 0x8000)		// LEFT
		val &= ~0x1;

	return val;
}

static READ32_HANDLER( port_r )
{
	UINT32 val = 0;
	UINT32 shift = 0;
	offset <<= 1;

	static int ip0 = 0xffffffff;
	static int ip1 = 0xffffffff;
	static int ip2 = 0xffffffff;
	static int ip3 = 0xffffffff;

	if (mask == 0x0000ffff)
		offset += 1;
	else
		shift = 16;

	switch (offset)
	{
		case 0: val = 0xffffffff; break;
		case 1: val = 0xffffffff; break;
		case 2: val = 0xffffffff;		// DIP SWITCHES
				break;
		case 3: val = (rand() & (1 << 13))
					| (3 << 11)
					| (rand() & (1 << 10))
					| (eeprom->GetDO() << 8)
					| (GetCoinInputs() & 0xf);
				break;
		case 4: val = ReadPort4();
				break;
		case 5: val = 0xffffffff;
				break;
		case 6: val = ReadPort6(); // 4 = Service mode
				break;
		case 7: val = 0xffffffff;
				break;
				
	}
	return val << shift;
}

static WRITE32_HANDLER( port_w )
{
	// 0x00000200 = ADC?
	// 0x20000000 = CD-MUTE
	// 0xa0000000 = ?
	LOGERROR("PORT W:%x %x\n", data, Core.CPUMan->ActiveCPUGetPC());
}


static WRITE32_HANDLER( eeprom_w )
{
//	LOGERROR("NVRAM W:%x %x  ", data, Core.CPUMan->ActiveCPUGetPC());
	data >>= 20;
	// 3 = CS
	// 2 = CLK
	// 1 = DATA
	// 0 = ? (From Port)
	eeprom->SetCS(data & 8);
	eeprom->SetDI(data & 2);
	eeprom->SetCK(data & 4);
}

static READ32_HANDLER( controlport_r )
{
//	LOGERROR("CONTROL PORT: %x %x\n", offset, Core.CPUMan->ActiveCPUGetPC());
	return rand();
}

static WRITE32_HANDLER( controlport_w )
{
//	LOGERROR("CONTROL PORT W: %x %.8x %.8x\n", offset, data, Core.CPUMan->ActiveCPUGetPC());
	int_status &= ~INT_CONTROLPORT;
}


/**************************************
 *
 *  Konami ACIO
 *
 *************************************/

// Gun triggers are read here + other stuff
UINT32 GetGunInputs()
{
	UINT32 val = 0xffffffff;
#if 1
	if (GetAsyncKeyState('Q') & 0x8000)
		val &= ~0x80000000;
	if (GetAsyncKeyState('W') & 0x8000)
		val &= ~0x40000000;				// Player 3 gun sw
	if (GetAsyncKeyState('E') & 0x8000)
		val &= ~0x20000000;				// Player 2 gun sw
	if (GetAsyncKeyState('R') & 0x8000)
		val &= ~0x10000000;				// Player 1 gun sw

	if (GetAsyncKeyState('N') & 0x8000)
		val &= ~0x08000000;
	if (GetAsyncKeyState('B') & 0x8000)
		val &= ~0x04000000;				// Player 3 trig

	if (GetAsyncKeyState('L') & 0x8000)	// Player 2 trig
		val &= ~0x02000000;
	if (GetAsyncKeyState('K') & 0x8000)
		val &= ~0x01000000;
#endif
	return val;
}

static READ32_HANDLER( kacio_r )
{
/*
Evil Night
==========

Offset C
0x40000000 = 3P GUN SWITCH
0x20000000 = 2P GUN SWITCH
0x10000000 = 1P GUN SWITCH
0x04000000 = 3P GUN TRIGGER
0x02000000 = 2P GUN TRIGGER
0x01000000 = 1P GUN TRIGGER
*/
	int value = 0xffffffff;

/*
	static int p2_x[] = { 64, 34 };

	int gun_x;
	int gun_y;

	extern void GetMousePosition(int &x, int &y);

	GetMousePosition(gun_x, gun_y);

	if (gun_x == 0)
		gun_x = 1;
	else if (gun_x < 256)
		gun_x = (256 - gun_x) << 8;
	else if (gun_x < 512)
		gun_x = (((512 - gun_x) & 0xff) << 8) | 0xff;

	if (offset == 0x8)
		value = (gun_x << 16) | p2_x[0];
*/
	if (offset == 0xc)
		value = GetGunInputs();

//	LOGERROR("KACIO_R: %x\n", offset);
	return value;
}

static WRITE32_HANDLER( kacio_w )
{
//	LOGERROR("KACIO_W: %x\n", offset, data);
}


/**************************************
 *
 *  Konami 003461 SIO
 *
 *************************************/

/*
  37c00010-1 = In/Output
  37c00012-3 = ?
  37c00014-5 = ?
  37c00016-7 = ?
  37c00018-9 = ?
  37c0001a-b = Status
             - 
             - Bit 5 = Transmit Ready
  37c0001c-d = ?
  37c0001e-f = Register Read/Write Port
*/

static UINT16 sio_reg;

void SIOTransmit(UINT16 data)
{
	data >>= 8;
//	CLOGERROR(GREEN, "%c", data);
}

static READ32_HANDLER( sio_r )
{
	UINT32 val;
	UINT32 shift = 0;
	offset <<= 1;

	if (mask == 0x0000ffff)
		offset += 1;
	else
		shift = 16;

	switch (offset)
	{
		case 5: val = 0x2100;	break;
		case 7: val = sio_reg;	break;
		default:
			val = rand();
//			LOGERROR("SIO R: %x %x %x\n", offset, mask, Core.CPUMan->ActiveCPUGetPC());
	}
	
	return val << shift;
}

static WRITE32_HANDLER( sio_w )
{
	offset <<= 1;

	if (mask == 0x0000ffff)
		offset += 1;
	else
		data >>= 16;

	switch (offset)
	{
		case 0:
			SIOTransmit(data);
			break;
		case 7:
			sio_reg = data & 0xffff;
			break;
		default:
			LOGERROR("SIO W: %x %x %x %x\n", offset, data, mask, Core.CPUMan->ActiveCPUGetPC());
	}
}

static READ32_HANDLER( bda_r )
{
	return 0;
}







static READ32_HANDLER( vdu_r )
{
	UINT32 val = 0;

	switch (offset & 0xff)
	{
		case 0x00/4:
		{
			/* VPOS -THIS IS TOTALLY WRONG */
			static UINT32 vpos = 0x00000068;
			val = vpos--;
			break;
		}
		case 0x04/4:
		{
			val = vdu.vint;
			break;
		}
		default:
			LOGERROR("VDU R: [%x] (%x)\n", offset<<2, Core.CPUMan->ActiveCPUGetPC());
	}

	return val;
}


static WRITE32_HANDLER( vdu_w )
{
	int access = (offset & 0x100) ? CLEAR : WRITE;

	switch (offset & 0xff)
	{
		case 0x04/4:
		{
			if (access == CLEAR)
			{
				if (data & 0x8000)
					int_status &= ~INT_VINT1;

				if (data & 0x80000000)
					int_status &= ~INT_VINT0;
			}
			else
			{
				vdu.vint = data;
				//LOGERROR("VINT W: %x (%x)\n", data, Core.CPUMan->ActiveCPUGetPC());
			}
			break;
		}
		case 0x10/4:
		{
			vdu.fv0a = data;
			break;
		}
		case 0x14/4:
		{
			vdu.fv1a = data;
			break;
		}
		default:
			LOGERROR("VDU W: [%x] %x (%x)\n", offset<<2, data, Core.CPUMan->ActiveCPUGetPC());
	}
}


/***************************************
 *
 *    M48T58 RTC (Konami only)
 *
***************************************/

static READ32_HANDLER( m48t58_r )
{
	offset <<= 1;
	UINT32 shift = 0;

	if (mask == 0x000ffff)
		offset += 1;
	else
		shift = 16;

	return (m48t58->Read(offset) << 8) << shift;
}

static WRITE32_HANDLER( m48t58_w )
{
	offset <<= 1;

	if (mask == 0x000ffff)
		offset += 1;
	else
		data >>= 16;

	m48t58->Write(offset, data >> 8);
}


/***************************************
 *
 *    Sound DSP
 *
***************************************/

static UINT16 dsp_nram[1024];	// Instruction
static UINT16 dsp_iram[1024];	// Internal
static UINT32 dsp_ch_regs[32];

//FILE *dsp_log = NULL;

static READ32_HANDLER( dsp_r )
{

	return rand();
}

static WRITE32_HANDLER( dsp_w )
{

}

static READ32_HANDLER( random_r )
{
	return rand();
}



UINT8 ymz_rom_r(int offset)
{
	return ymz_rom[offset];
}

static READ32_HANDLER( ymz1_r )
{
	return 0;
}

static READ32_HANDLER( ymz0_r )
{
	if (mask == 0xffff0000)
		return ymz->Read(0) << 24;
	else if (mask == 0x0000ffff)
		return ymz->Read(1) << 8;

	return 0xffffffff;
}

static WRITE32_HANDLER( ymz0_w )
{
	if (mask == 0xffff0000)
		ymz->Write(0, data >> 24);
	else if (mask == 0x0000ffff)
		ymz->Write(1, data >> 8);
}

static WRITE32_HANDLER( ymz1_w )
{

}



static WRITE32_HANDLER( powerbus_w )
{
	int access = (offset & 0x100) ? CLEAR : SET;

	switch (offset & 0xff)
	{
		case 0x10/4:
		{
			LOGERROR("PowerBus Control: %.8x %.8x\n", data, Core.CPUMan->ActiveCPUGetPC());
			break;
		}
		case 0x40/4:
		{
			WriteM2Reg(int_enable, data, access);

			if (int_enable & int_status)
				Core.CPUMan->SignalInterrupt(0);
			break;
		}
		case 0x60/4:
		{
			/* TODO: I'm not sure about this lot */
			if (access == CLEAR)
				WriteM2Reg(int_status, data, access);

			if (access == SET)
			{
				if (data & 1)
					SetBDAInterrupt(INT_CPUCOMM);
			}
			break;
		}
		default:
		{
			LOGERROR("PowerBus W[%x] %.8x %.8x\n", offset*4, data, Core.CPUMan->ActiveCPUGetPC());
		}
	}
}

static READ32_HANDLER( powerbus_r )
{
	UINT32 val = 0;

	switch (offset & 0xff)
	{
		case 0x10/4:
		{
			break;
		}
		case 0x40/4:
		{
			val = int_enable;
			break;
		}
		case 0x50/4:
		{
			val = int_status;
			break;
		}
		default:
		{
			LOGERROR("PowerBus R[%x] %.8x\n", offset*4, Core.CPUMan->ActiveCPUGetPC());
			val = rand() & 0x40;
		}
	}
	return val;
}


/***************************************
 *
 *    CPU Memory Maps
 *
***************************************/

ADDRESS_MAP( m2_memory_map )
	MAP_RANGE( 0x20000000, 0x201fffff, rom_r, NULL )
	MAP_RANGE( 0x40000000, 0x40ffffff, ram_r, ram_w )

	MAP_RANGE( 0x00010000, 0x0001ffff, powerbus_r, powerbus_w )

	MAP_RANGE( 0x00020000, 0x00020003, memcfg_r, NULL )
	MAP_RANGE( 0x00020004, 0x00020007, cpu_r, cpu_w )

	MAP_RANGE( 0x00030000, 0x0003ffff, vdu_r, vdu_w )

	MAP_RANGE( 0x00040000, 0x0004ffff, te_r, te_w )

	MAP_RANGE( 0x00060000, 0x0006ffff, dsp_r, dsp_w )
	MAP_RANGE( 0x00061ffc, 0x00061fff, random_r, NULL )

	MAP_RANGE( 0x00070000, 0x0007ffff, controlport_r, controlport_w )
	MAP_RANGE( 0x01000000, 0x01ffffff, cde_r, cde_w )
	MAP_RANGE( 0x02000000, 0x02000003, slot2_r, NULL )
	MAP_RANGE( 0x03000000, 0x03000fff, slot3_r, slot3_w )
	MAP_RANGE( 0x04000000, 0x04ffffff, reg_r, reg_w )
	MAP_RANGE( 0x05000000, 0x05000003, slot5_r, NULL )
	MAP_RANGE( 0x06000000, 0x06000003, slot6_r, NULL )
	MAP_RANGE( 0x07000000, 0x07000003, slot7_r, NULL )
	MAP_RANGE( 0x10000000, 0x1000101f, cpu_id_r, cpu_id_w )
	MAP_RANGE( 0x36c00000, 0x36cfffff, m48t58_r, m48t58_w )
//	MAP_RANGE( 0x37200000, 0x37200003, NULL, led_w)
	MAP_RANGE( 0x37400000, 0x37400003, NULL, eeprom_w)
	MAP_RANGE( 0x37600000, 0x37600000, NULL, atapi_dma_w )
	MAP_RANGE( 0x37a00000, 0x37a0003f, kacio_r, kacio_w )
	MAP_RANGE( 0x37c00010, 0x37c0001f, sio_r, sio_w )			// Konami 11k
	MAP_RANGE( 0x37e00000, 0x37e0000f, port_r, port_w )			// Konami? - 37e00006 = Read
	MAP_RANGE( 0x3c000007, 0x3c00000f, uartbuf_r, uartbuf_w)	// FZ35S only?
	MAP_RANGE( 0x3e000000, 0x3effffff, ymz0_r, ymz0_w )			// Konami
//	MAP_RANGE( 0x3e900000, 0x3e9fffff, ymz1_r, ymz1_w )			// Konami
	MAP_RANGE( 0x3f000000, 0x3fffffff, atapi_r, atapi_w )		// PCMCIA/ATAPI
	MAP_RANGE( 0xff000000, 0xff000fff, tmp_r, NULL )
	MAP_RANGE( 0xfff00000, 0xffffffff, rom_r, NULL )
ADDRESS_MAP_END

static READ32_HANDLER( rom_r )
{
	return	READ32_BE(bios, offset*4);
}

static READ32_HANDLER( cpu_id_r )
{
	// 0x40000000 = VIDEO LOW_RES
	// 0x00000000 = VIDEO HIGH RES
	if (strcmp(Core.CPUMan->ActiveCPUGetName(), "PPC1") == 0)
		return 0x00000000;
	else
		return 0x80000000;
}

static WRITE32_HANDLER( cpu_id_w )
{
	// 0xc0000000 is written when asking for the video resolution
	LOGERROR("Writing to CPUID: %x (PC:%.08x)\n", data, Core.CPUMan->ActiveCPUGetPC());
}

static READ32_HANDLER( memcfg_r )
{
	return MEMCFG;
}

static READ32_HANDLER( unk_r )
{
	return 0x00000000;
}

static READ32_HANDLER( tmp_r )
{
	return rand() & 0x40;
}

// TODO: Timing related. Used to calculate the bus clock
static READ32_HANDLER( x30000_r )
{
	static UINT32 poo = 0x00000068;
	return poo--;
}

static READ32_HANDLER( reg_r )
{
	switch (offset)
	{
		case     0 / 4: return 0 << 24;
		case 0x00c / 4: return 0xffffffff;			/* Serial? Read and discarded */
		case 0x010 / 4: return 0;					/* Serial in */
		case 0x018 / 4: return 0x18000000;			/* Serial status */
		case 0x208 / 4: return _208val;
		case 0x280 / 4: return SYSCONFIG;
		case 0x284 / 4: return _284val;
		default:
			LOGERROR("SYSREG_R 0x%0.8x (PC:%0.8x)\n", offset, Core.CPUMan->ActiveCPUGetPC());
			return 0;
	}
}


static WRITE32_HANDLER( reg_w )
{
	assert(mask == 0xffffffff);

	switch (offset)
	{
		case 0x0c / 4:
		{
			// Serial
			break;
		}
		case 0x14 / 4:
		{
			// Debug print
			char outchar = data & 0xff;
			if (outchar == 0xd)
				LOGERROR("\n");
			else
				LOGERROR("%c", outchar);
			break;
		}
		case 0x418 / 4:
			// See above (written with 0x10000000)
			break;
		case 0x20 / 4:
		{
			if (data == 1)
			{
				// Soft Reset
				_208val = 0;
				Core.CPUMan->ResetCPU(0);
			}
			else
				DEBUG_BREAK;
			break;
		}
		case 0x284 / 4:
			_284val = data;
			break;
		default:
			LOGERROR("SYSREG_W 0x%0.8x 0x%0.8x (PC:%0.8x)\n", offset, data, Core.CPUMan->ActiveCPUGetPC());
	}
}

static READ32_HANDLER( ram_r )
{
	return READ32_BE(RAM, offset*4);
}

static WRITE32_HANDLER( unk_w )
{

}

static WRITE32_HANDLER( ram_w )
{
	UINT32 old = READ32_BE(RAM, offset * 4);
	data |= (old & ~mask);
	WRITE32_BE(RAM, offset * 4, data);
}

/*
 * Physical memory space has 7 'slot' ranges
 * Byte 1 stores the device ID
 * What are they?
 * Device 1 is CDE
 * 0x01000000 = 0x01ffffff
 * 0x02000000 = 0x02ffffff
 * 0x03000000 = 0x03ffffff
 * 0x04000000 = 0x04ffffff = System Regs
 * 0x05000000 = 0x05ffffff
 * 0x06000000 = 0x06ffffff
 * 0x07000000 = 0x07ffffff
 */


/************************************
 *
 *    CDE (CD and DMA controller)
 *
 ***********************************/
INT32 cde_status_ptr = 0;
UINT32 dma_countdown;
UINT32 dma_pending;
UINT8 cde_status_fifo[32]; // ?

// Aux DMA
UINT32 dma_src;
UINT32 dma_dst;
UINT32 dma_cnt;

UINT32 cde_data_fifo[16];
UINT32 cde_data_ptr = 0;
UINT32 cmd_bytes;
UINT32 report_enabled = 0;
UINT32 q_report_enabled = 0;
UINT32 seek_report_enabled = 0;

UINT32 last_byte = 0;
UINT32 test = 0;

UINT32 cd_block_pos;

struct
{
	UINT32 addr;
	UINT32 bytes;

	UINT32 next_addr;
	UINT32 next_bytes;

	UINT32 control;
	UINT32 pending;
}dma[2];

enum
{
	EJECTED = 0,
	UNKNOWN = 1,
	PAUSED = 2,
	PLAYING = 3,
} cd_states;

UINT8 drive_state = PAUSED;
static const char *cmd[] = { "eject", "????", "pause", "play" };

static const UINT32 first_track_num = 1;
static const UINT32 disc_type = 0;
static const UINT32 last_track_num = 1;
static const UINT32 lead_out_time = 0x00500000;	// TEST


void ClearCDEStatus(int id)
{
	cde_status &= ~id;
}

void StartCDEDMA(int chan, int which);



void UPDATE_STATUS_PTR(UINT32 value)
{
	cde_status_ptr += value;
}

void CreateReport(UINT32 report_type)
{
	switch (report_type)
	{
		case 0xb:
		{
			if (seek_report_enabled)
			{
				UPDATE_STATUS_PTR(2);
				cde_status_fifo[1] = 0x1b;
				cde_status_fifo[0] = 2;
			}
			else
			{
				UPDATE_STATUS_PTR(2);
				cde_status_fifo[1] = 0x1b;
				cde_status_fifo[0] = drive_state;
			}
			break;
		}
		case 0xc:
		{
			UPDATE_STATUS_PTR(11);
			cde_status_fifo[10] = 0x1c;

			if (test == 0)
			{
				cde_status_fifo[9] = 0x01; // session number?
				cde_status_fifo[8] = 0x00;
				cde_status_fifo[7] = 0xa0;
				cde_status_fifo[6] = 0x00;
				cde_status_fifo[5] = 0x00;
				cde_status_fifo[4] = 0x00;
				cde_status_fifo[3] = 0x00;
				cde_status_fifo[2] = first_track_num;
				cde_status_fifo[1] = disc_type;
				cde_status_fifo[0] = 0x00;
				++test;
			}
			else if (test == 1)
			{
				cde_status_fifo[9] = 0x01; // session number?
				cde_status_fifo[8] = 0x00;
				cde_status_fifo[7] = 0xa1;
				cde_status_fifo[6] = 0x00;
				cde_status_fifo[5] = 0x00;
				cde_status_fifo[4] = 0x00;
				cde_status_fifo[3] = 0x00;
				cde_status_fifo[2] = last_track_num;
				cde_status_fifo[1] = 0x00;
				cde_status_fifo[0] = 0x00;
				++test;
			}
			else if (test == 2)
			{
				cde_status_fifo[9] = 0x01; // session number?
				cde_status_fifo[8] = 0x00;
				cde_status_fifo[7] = 0xa2;
				cde_status_fifo[6] = 0x00;
				cde_status_fifo[5] = 0x00;
				cde_status_fifo[4] = 0x00;
				cde_status_fifo[3] = 0x00;
				cde_status_fifo[2] = (lead_out_time >> 16) & 0xff;
				cde_status_fifo[1] = (lead_out_time >> 8) & 0xff;
				cde_status_fifo[0] = (lead_out_time) & 0xff;
				++test;
			}
			else if (test == 3)
			{
				// TRACK INFO
				cde_status_fifo[9] = 0x41; // Something to do with track type (data/audio etc)
				cde_status_fifo[8] = 0x00; // should be 0
				cde_status_fifo[7] = 0x01; // TRACK 1
				cde_status_fifo[6] = 0x00;
				cde_status_fifo[5] = 0x00;
				cde_status_fifo[4] = 0x00;
				cde_status_fifo[3] = 0x00;
				cde_status_fifo[2] = 0x00; // Start Pmin
				cde_status_fifo[1] = 0x00; // Start Psec
				cde_status_fifo[0] = 0x00; // Start Pframe
				test = 0;
			}
			break;
		}
		case 0xd:
		{
			// Switch state report
			UPDATE_STATUS_PTR(4);
			cde_status_fifo[3] = 0x1d;
			cde_status_fifo[2] = 2;
			cde_status_fifo[1] = 0xa5;
			cde_status_fifo[0] = 0x5a;
			break;
		}
	}
}


void ProcessCDECommand()
{
	cde_status |= 0x18; //?
	switch (cde_data_fifo[0])
	{
		case 0x4:
		{
			UPDATE_STATUS_PTR(1);
			cde_status_fifo[0] = 0x4;
			report_enabled = 0;

			LOGERROR("CD speed set to %dX\n", cde_data_fifo[1]);
			break;
		}
		case 0x6:
		{
			UPDATE_STATUS_PTR(1);
			cde_status_fifo[0] = 0x6;
			report_enabled = 0;

			assert((cde_data_fifo[1] == 0) || (cde_data_fifo[1] == 0x78));

			LOGERROR("CD %s command\n", cde_data_fifo[1] == 0x78 ? "data" : "audio");
			break;
		}
		case 0x8:
		{
			UPDATE_STATUS_PTR(1);
			cde_status_fifo[0] = 0x8;
			report_enabled = 0;

			drive_state = cde_data_fifo[1];

			LOGERROR("CD %s command\n", cmd[cde_data_fifo[1]]);
			break;
		}
		case 0x9:
		{
			UPDATE_STATUS_PTR(1);
			cde_status_fifo[0] = 0x9;
			report_enabled = 0;

			cd_block_pos = (cde_data_fifo[1] << 16) | (cde_data_fifo[2] << 8) | cde_data_fifo[3];

			LOGERROR("CD seek to LBA %.6x command\n", cd_block_pos);
			break;
		}
		case 0xb:
		{
			UPDATE_STATUS_PTR(1);
			report_enabled = cde_data_fifo[1] & 1;
			seek_report_enabled = cde_data_fifo[1] & 2;
			cde_status_fifo[0] = 0xb;
			LOGERROR("CD get drive state command (%x)\n", cde_data_fifo[1]);
			break;
		}
		case 0xc:
		{
			UPDATE_STATUS_PTR(1);
			q_report_enabled = cde_data_fifo[1] & 2;
			cde_status_fifo[0] = 0xc;
			LOGERROR("CD get ????? state command (%x)\n", cde_data_fifo[1]);
			break;
		}
		case 0xd:
		{
			UPDATE_STATUS_PTR(1);
			report_enabled = cde_data_fifo[1];
			cde_status_fifo[0] = 0xd;
			LOGERROR("CD get switch state command\n");
			break;
		}
		case 0x21:
		{
#define MECH_REPORT_SPEED_2X		(0)
#define MECH_REPORT_SPEED_4X		(1 << 1)
#define MECH_REPORT_SPEED_6X		(1 << 2)
#define MECH_REPORT_SPEED_8X		(1 << 3)

			UPDATE_STATUS_PTR(8);
			report_enabled = 0;

			cde_status_fifo[7]  = 0x21;
			cde_status_fifo[6]  = 0xff;
			cde_status_fifo[5]  = MECH_REPORT_SPEED_8X;
			cde_status_fifo[4]  = 0xff;
			cde_status_fifo[3]  = 0xff;
			cde_status_fifo[2]  = 0xff;
			cde_status_fifo[1]  = 0xff;
			cde_status_fifo[0]  = 0xff;

			LOGERROR("CD mech report command\n");
			break;
		}
		case 0x83:
		{
			UPDATE_STATUS_PTR(0xc);
			report_enabled = 0;

			cde_status_fifo[11] = 0x83;
			cde_status_fifo[10] = 0x1;
			cde_status_fifo[9]  = 0x2;
			cde_status_fifo[8]  = 0x3;
			cde_status_fifo[7]  = 0x4;
			cde_status_fifo[6]  = 0x5;
			cde_status_fifo[5]  = 0x6;
			cde_status_fifo[4]  = 0x7;
			cde_status_fifo[3]  = 0x8;
			cde_status_fifo[2]  = 0x9;
			cde_status_fifo[1]  = 0x1;
			cde_status_fifo[0]  = 0x2;

			LOGERROR("CD ID report command\n");
			break;
		}
	}
	return;
}


static READ32_HANDLER( cde_r )
{
	switch (offset)
	{
		case 0x0: return 0x00010000;	// CDE ID
		case 0x6:
		{
			if (dma_pending)
			{
				if (--dma_countdown == 0)
				{
					StartCDEDMA(0, 0);
				}
			}

			return cde_status | 0x00100000 | 0x18;	// TODO
		}
		case 7:
		{
			return cde_int_enable;
		}
		case 0xb:
		{
			if (cde_status_ptr >= 0)
			{
				if (cde_status_ptr == 0)
					last_byte = 1;
				return cde_status_fifo[cde_status_ptr--] | 0x100;
			}
			else if (!(cde_status & 0x18) && (report_enabled || q_report_enabled || seek_report_enabled))
			{
				if (last_byte)
				{
					last_byte = 0;
					return 0;
				}
				else
				{
					if (report_enabled)
					{
						CreateReport(cde_data_fifo[0]);
						report_enabled = 0; // Necessary?
					}
					else if (q_report_enabled)
					{
						CreateReport(0xc);
					}
					else if (seek_report_enabled)
					{
						CreateReport(0xb);
					}
					return cde_status_fifo[cde_status_ptr--] | 0x100;
				}
			}
			else
			{
				last_byte = 0;
				return 0;
			}
		}
		case 0x0d:
		{
			UINT32 val = hw_timer;

			if (hw_timer)
			{
				hw_timer -=2;
			}

			return hw_timer;
		}
		case 0x81:
		{
			// LCCD??
			return 0x88;
		}
		case 0x89:
		case 0x8a:
		{
			return 0;
		}
		case 0x90:
		case 0x9e:
		{
			return 0; // Read very frequently by new BIOS
		}
		case 0xa4:
		case 0xa5:
		case 0xa6:
		case 0xa7:
		{
			return 0x20;
		}
		case 0xa8:
		{
			static int ret = 0x20;
			// Microcard?
			return ret;
		}
		case 0x400:
		{
			// DMA 1 (0x80 means active)
			return 0;
		}
		case 0x408:
		{
			// DMA 2 (0x80 means active)
			return 0;
		}
		default:
			LOGERROR("CDE_R 0x%0.8x (PC:%0.8x)\n", offset, Core.CPUMan->ActiveCPUGetPC());
			return 0;
	}
}


void StartCDEDMA(int chan, int which)
{
//	if (dma[chan].pending & (0x20 << which))
	{
		UINT32 addr  = dma[chan].addr;
		UINT32 bytes = dma[chan].bytes;

		CLOGERROR(RED, "DMA[%d] Start %.06x -> %.08x (%x)\n", chan, cd_block_pos, addr, bytes);

		// First word is the MSF
		UINT32 msf = lba_to_msf(cd_block_pos);

		// First and last words must have MSF????
		write_mem32(m2_memory_map, addr, msf << 8, 0xffffffff);
		write_mem32(m2_memory_map, addr + 0x924, msf << 8, 0xffffffff);

		for (UINT32 i = 0; i < bytes - 8; ++i)
		{
			// TODO
			UINT32 data = 0;//CD[cd_block_pos * 2048 + i];
			write_mem8(m2_memory_map, addr + 4 + i, data);
		}

		// Complete
		cde_status |= 0x400 << chan;
		dma_pending &= ~0x40;

		// Do we have another DMA pending?
		if (dma_pending & 0x20)
		{
			// If so, queue it up.
			dma[chan].addr = dma[chan].next_addr;
			dma[chan].bytes = dma[chan].next_bytes;
			dma_pending &= ~0x20;
			dma_countdown = 16384;
		}

		cd_block_pos++;
	}
}

/*
      0xa: CDE Command
    0x106: Status clear
*/
static WRITE32_HANDLER( cde_w )
{
	assert(mask == 0xffffffff); // TEST
	switch (offset)
	{
		case 0x7:
		{
			LOGERROR("CDE Interrupt Enable: %x (%x)\n", data, Core.CPUMan->ActiveCPUGetPC());
			cde_int_enable |= data;
			break;
		}
		case 0xd:
		{
			// TODO: Does this exist in Konami M2?
			hw_timer = data;
			break;
		}
		case 0xc0:
		case 0xc8:
		{
			int chan = (offset & 0xf) >> 3;
			dma[chan].pending |= data & 0x60;

//			if (data & 0x200)
//				LOGERROR("DMA[%d] Reset\n", chan);

			if (data & 0x40) // DMA
				dma_pending |= 0x40;
			else if (data & 0x20) // NEXT DMA
				dma_pending |= 0x20;

			if (data & 0x80)
			{
				dma_countdown = 16384;
//				StartCDEDMA(Core, chan, 0);
			}

			break;
		}

		case 0xc2:
		case 0xca:
		{
			int chan = (offset & 0xf) >> 3;
			dma[chan].addr = data;
			LOGERROR("DMA[%d]:A address: %x (%x)\n", chan, data, Core.CPUMan->ActiveCPUGetPC());
			break;
		}

		case 0xc6:
		case 0xce:
		{
			int chan = (offset & 0xf) >> 3;
			dma[chan].next_addr = data;
			LOGERROR("DMA[%d]:B address: %x (%x)\n", chan, data, Core.CPUMan->ActiveCPUGetPC());
			break;
		}

		case 0xc3:
		case 0xcb:
		{
			int chan = (offset & 0xf) >> 3;
			dma[chan].bytes = data;
			LOGERROR("DMA[%d]:A byte count: %x (%d) (%x)\n", chan, data, data, Core.CPUMan->ActiveCPUGetPC());
			break;
		}

		case 0xc7:
		case 0xcf:
		{
			int chan = (offset & 0xf) >> 3;
			dma[chan].next_bytes = data;
			LOGERROR("DMA[%d]:B byte count: %x (%d) (%x)\n", chan, data, data, Core.CPUMan->ActiveCPUGetPC());
			break;
		}

		case 0x1c0:
		case 0x1c8:
		{
			// DMA
			int chan = (offset & 0xf) >> 3;
			//LOGERROR("DMA[%d] %s (%x) (%x)\n", chan, data & 0x80 ? "disabled" : " ", data, Core->CPUMan->ActiveCPUGetPC());
			break;
		}
		case 0x106:
		{
			// Status clear
			//LOGERROR("CDE STATUS CLEAR %x\n", data);
			cde_status &= ~data;
			break;
		}
		case 0xa:
		{
			// Command data
			cde_status |= 0x20;

			// ???
			if (data == 0x180)
			{
				//cde_data_ptr = 0;
				//LOGERROR("**0x180 Command**\n");
				//report_enabled = 0;
				break;
			}

			cde_data_fifo[cde_data_ptr] = data;

			if (cde_data_ptr++ == 0)
			{
				switch (data)
				{
					case 0x04:
					{
						// Speed Command
						cmd_bytes = 3;
						break;
					}
					case 0x06:
					{
						// Audio command
						cmd_bytes = 3;
						break;
					}
					case 0x08:
					{
						// Play, pause, eject
						cmd_bytes = 2;
						break;
					}
					case 0x09:
					{
						// Seek
						cmd_bytes = 4;
						break;
					}
					case 0x0b:
					{
						// Get drive state?
						cmd_bytes = 2;
						break;
					}
					case 0x0c:
					{
						// Get drive state?
						cmd_bytes = 2;
						break;
					}
					case 0x0d:
					{
						// Enable reports
						// Get switch state?
						cmd_bytes = 2;
						break;
					}
					case 0x21:
					{
						// Mech type
						cmd_bytes = 4;
						break;
					}
					case 0x83:
					{
						// Read ID
						cmd_bytes = 7;
						break;
					}
					case 0x180:
						
						break;
					default:
					{
						LOGERROR("Unknown CDE Command: %x\n", data);
					}
				}
			}

			if (cde_data_ptr == cmd_bytes)
			{
				ProcessCDECommand();
				cde_data_ptr = 0; // ?
			}
			break;
		}
		case 0x88:
			break;

		case 0xa4:
		case 0xa5:
		case 0xa6:
		case 0xa7:
		{
			break;
		}
		case 0xa8:
		{
			// Write 4 then 0...
			break;
		}
		case 0x107:
		{
			cde_int_enable &= ~data;
			break;
		}
		// Some sort of DMAC?
		case 0x500:
		{
			break;	// 0xffffffff
		}
		case 0x401:
		{
			//LOGERROR("DMA Src: %x\n", data);
			dma_src = data;
			break;
		}
		case 0x402:
		{
			//LOGERROR("DMA Dst: %x\n", data);
			dma_dst = data;
			break;
		}
		case 0x403:
		{
			//LOGERROR("DMA Count: %x\n", data);
			dma_cnt = data;
			assert((dma_cnt & 0x3) == 0);
			break;
		}
		case 0x400:
		{
			// 0x80 = Start?
			//LOGERROR("DMA Ctrl: %x %x -> %x (%x bytes) (%.8x)\n", data, dma_src, dma_dst, dma_cnt, Core.CPUMan->ActiveCPUGetPC());

			if (dma_src == 0x3f000000)
			{
				for (UINT32 i = 0; i < dma_cnt; i += 2)
				{
					UINT16 src = read_mem16(m2_memory_map, 0x3f000000, 0xffff);
					write_mem16(m2_memory_map, dma_dst + i, src, 0xffff);
				}
			}
			else
			{
				for (UINT32 i = 0; i < dma_cnt; ++i)
				{
					UINT8 src = read_mem8(m2_memory_map, dma_src + i);
					write_mem8(m2_memory_map, dma_dst + i, src);
				}
			}

			SetCDEInterrupt(CDE_INT_DMA1);
			break;
		}
		default:
			LOGERROR("CDE W: 0x%0.8x %0.8x (PC:%0.8x)\n", offset, data, Core.CPUMan->ActiveCPUGetPC());
			break;
	}
	
	
}

static WRITE32_HANDLER( video_w )
{
	LOGERROR("VDL[%d] 0x%0.8x (PC:%0.8x)\n", offset, data, Core.CPUMan->ActiveCPUGetPC());

	if (offset == 0)
		vdu.vdc0 = data;
	else
		vdu.vdc1 = data;
}


static READ32_HANDLER( slot2_r ) { return 0x44444444; }
static READ32_HANDLER( slot5_r ) { return 0x01000000; }
static READ32_HANDLER( slot6_r ) { return 0x02020202; }
static READ32_HANDLER( slot7_r ) { return 0x33333333; }

static READ32_HANDLER( slot3_r )
{
	if (offset == 0)
		return 0x00050000;
	// Counter
	if (offset == 0xd)
	{
		UINT32 val = hw_timer;

		if (hw_timer)
		{
			hw_timer -=2;
		}

		return val;
	}
	return 0;
}

static WRITE32_HANDLER( slot3_w )
{
	if (offset == 0xd)
	{
		hw_timer = data;
	}
}

void bdavideo()
{
	SetBDAInterrupt(INT_VINT0);
}

void dspint()
{
	SetBDAInterrupt(INT_DSP);
}

VIDEO_UPDATE( konami_m2 )
{
	SetBDAInterrupt(INT_VINT1);

	/* TODO!!! */
	if (vdu.vint & VDU_VINT_VLINE0_MASK)
		Core.CreateTimer(TIME_IN_MSEC(3), TIMER_ONESHOT, true, bdavideo);

	/*
		Parse the VDL
	*/
	UINT32 ptr = vdu.field ? vdu.fv1a : vdu.fv0a;

	if (ptr == 0)
		goto field_swap;

	while (1)
	{
		/* Get the four mandatory words */
		UINT32 dma_ctl = READ32_BE(RAM, ptr & RAM_SIZE_MASK); ptr +=4;
		UINT32 lwr_ptr = READ32_BE(RAM, ptr & RAM_SIZE_MASK); ptr +=4;
		UINT32 upr_ptr = READ32_BE(RAM, ptr & RAM_SIZE_MASK); ptr +=4;
		UINT32 nxt_ptr = READ32_BE(RAM, ptr & RAM_SIZE_MASK); ptr +=4;

		int words = (dma_ctl & VDL_DMA_NWORDS_MASK) >> VDL_DMA_NWORDS_SHIFT;
		int lines = (dma_ctl & VDL_DMA_NLINES_MASK) >> VDL_DMA_NLINES_SHIFT;

		if (lines == 0)
			break;

		assert(words >= 4);
		words -=4;

		while (words--)
		{
			UINT32 data = READ32_BE(RAM, ptr & RAM_SIZE_MASK); ptr +=4;

			switch ((data >> 28) & 0xe)
			{
				case 0x8:
				{
				}
				case 0xa:
				{
					if (data & VDL_AV_LD_VDOUBLE)
					{
					}

					if (data & VDL_AV_LD_HDOUBLE)
					{
					}

					if (data & VDL_AV_LD_HWIDTH)
					{
						int width = (data & VDL_AV_HWIDTH_MASK) >> VDL_AV_HWIDTH_SHIFT;
						vdu.avdi &= ~VDU_AVDI_HWIDTH_MASK;
						vdu.avdi |= width << VDU_AVDI_HEND_SHIFT;
					}

					if (data & VDL_AV_LD_HSTART)
					{
						int width = (data & VDL_AV_HSTART_MASK) >> VDL_AV_HSTART_SHIFT;
						vdu.avdi &= ~VDU_AVDI_HSTART_MASK;
						vdu.avdi |= width << VDU_AVDI_HSTART_SHIFT;
					}
				}
				case 0xc:
				{
					/* TODO: There's more fields */
					if (data & VDL_LC_LD_BYPASSTYPE)
					{
					}

					if (data & VDL_LC_LD_FBFORMAT)
					{
						int format = (data & VDL_LC_FBFORMAT) >> VDL_LC_FBFORMAT_SHIFT;
						vdu.vdli &= ~VDU_VDLI_FBFORMAT;
						vdu.vdli |= format << VDU_VDLI_FBFORMAT_SHIFT;
					}
				}
			}
		}

		if (dma_ctl & VDL_DMA_ENABLE)
		{
			UINT32	src_addr;
			INT32	height;
			INT32	width;

			width = (vdu.avdi & VDU_AVDI_HWIDTH_MASK) >> VDU_AVDI_HEND_SHIFT;
			height = lines;
			src_addr = lwr_ptr;

			if (vdu.vdli & VDU_VDLI_FBFORMAT_32)
			{
				for (int y = 0; y < height; ++y)
					for (int x = 0; x < width; ++x)
						BITMAP_ADDR32(y, x) = READ32_BE(RAM, (src_addr + (width * y + x)*4) & RAM_SIZE_MASK);
			}
			else
			{
				for (int y = 0; y < height; ++y)
					for (int x = 0; x < width; ++x)
						BITMAP_ADDR32(y, x) = BGR1555(READ16_BE(RAM, (src_addr + (width * y + x)*2) & RAM_SIZE_MASK));
			}
		}

		/* Jump to the next entry */
		ptr = nxt_ptr;
	}

//	if (GetAsyncKeyState(VK_F6) & 0x8000)
//		WriteBMP(&RAM[vboffset], width, height, bytespp);

field_swap:
	vdu.field ^= 1;
}


/**************************************
 *
 *  System Initialisation
 *
 *************************************/

bool SystemInitM2(int game)
{
	// TODO!
	game = SELECTED_GAME;

	/* Use the setname for directory and file naming */
	Core.SetActiveGameName(gamelist[game].title, gamelist[game].setname);

	/* Load BIOS ROM */
	if (LoadROMFile(gamelist[game].bios, (void **)&bios) == false)
		return false;

	/* Load YMZ280B ROM if present */
	if (gamelist[game].flags & GAME_HAS_YMZ280B)
	{
		if (LoadROMFile(gamelist[game].ymz, (void **)&ymz_rom) == false)
			return false;

		ymz = new YMZ280B();

		ymz->SetExternalReadHandler(ymz_rom_r);
	}

	if (LoadCDImage(gamelist[game].cd) == FALSE)
		return false;

	/* Add our CPUs */
#pragma message("PPC602s are running at half speed")
	Core.CPUMan->AddCPU("PPC1", POWERPC_CLOCK / 2, m2_memory_map);
	Core.CPUMan->AddCPU("PPC2", POWERPC_CLOCK / 2, m2_memory_map);
//	Core.CPUMan->AddCPU("DSP", POWERPC_CLOCK);

	/* Hook the video update */
	Core.SetActiveVideoUpdate(konami_m2_video_update);

	/* Create the system RAM */
	RAM_SIZE_BYTES = (gamelist[game].flags & GAME_HAS_8MB ? 8 : 16) * 1024 * 1024;
	RAM_SIZE_MASK = RAM_SIZE_BYTES - 1;

	/* Set the MEMCFG register */
	if (RAM_SIZE_BYTES == 8 * 1024 * 1024)
		MEMCFG = (2 << 13) | (2 << 10);
	else
		MEMCFG = (5 << 13) | (5 << 10);

	/* Create the system RAM */
	RAM = new UINT8[RAM_SIZE_BYTES];

	/* Create the Timekeeper device if present */
	if (gamelist[game].flags & GAME_HAS_TIMEKEEPER)
	{
		m48t58 = new M48T58();

		if (m48t58->InitialiseSRAM(gamelist[game].nvram.name) == false)
			return false;
	}

	/* Create the EEPROM */
	eeprom = new EEPROM(128);

	if (eeprom->InitialiseEEPROM(gamelist[game].eeprom.name) == false)
		return false;


	/* Initialise the triangle engine */
	InitTriangleEngine();

	return true;
}

#include <stdio.h>
void SystemShutDownM2()
{
	// TEST: DUMP RAM

#if 1
	if (RAM)
	{
		FILE *f = fopen("tmp\\ramdump.bin","wb");
		if (f)
		{
			fwrite(RAM, 1, RAM_SIZE_BYTES, f);
			fclose(f);
		}
	}

	{
		FILE *f = fopen("tmp\\dsp_nram.bin","wb");
		if (f)
		{
			fwrite(dsp_nram, 1, 2048, f);
			fclose(f);
		}
	}
#endif

	delete eeprom;
	delete m48t58;
	if (RAM)
	{
		delete[] RAM;
		RAM = NULL;
	}
	if (bios)
	{
		delete[] bios;
		bios = NULL;
	}
}

void SystemResetM2()
{
	_208val = 1;
	_284val = 0;

	cde_status = 0;
	cde_status_ptr = 0;
	dma_pending = 0;

	cde_data_ptr = 0;
	cmd_bytes = 0;
	report_enabled = 0;
	q_report_enabled = 0;
	seek_report_enabled = 0;

	last_byte = 0;
	test = 0;

	vdu.vdc0 = 0;
	vdu.vdc1 = 0;
}
