/******************************************************************************

    Core

    Class for core emulator functions

******************************************************************************/

#include "M2x.h"
#include "core.h"
#include "cpuexec.h"
#include "fileman.h"
#include "timer.h"
#include "video.h"
#include "log.h"

#define ARCADE

#ifdef PAL
	#define WIDTH	768
	#define HEIGHT	576
#elif defined NTSC
	#define WIDTH	736
	#define HEIGHT	544
#elif defined VGA
	#define WIDTH	640
	#define HEIGHT	480
#elif defined ARCADE
	#define WIDTH	512
	#define HEIGHT	384
#else
	#error "No video mode defined"
#endif

EmulatorCore *EmulatorCore::_singleton = 0;

//
//   Prototypes
//
static TIMER_CALLBACK( core_video_update );

//
//   Functions
//

// TODO: Maybe change this to LoadGame
// and call all the initialisation at the start
// of the emulator.
bool EmulatorCore::InitialiseEmulator(void)
{
	//LoadROM(eGameID);

	// TODO: Maybe this 
	if (bFirstRun == true)
	{
		CPUMan = new CPUManager();
		FileMan = new FileManager();
		VideoMan = new Video();

		if (VideoMan->Initialise(WIDTH, HEIGHT) == FALSE)
			return false;

		FileMan->CreateFolder("nvram");

		// HACK - should also go below
		extern bool SystemInitM2(int game);
		if (SystemInitM2(0) == false)
			return false;

		bFirstRun = false;
	}

	ResizeAppWindow(WIDTH, HEIGHT);

	return true;
}

void EmulatorCore::ShutdownEmulator()
{
	extern void SystemShutDownM2();
	SystemShutDownM2();

	delete CPUMan;
	delete FileMan;
	delete VideoMan;
}

void EmulatorCore::RunEmulator()
{
	switch (eState)
	{
		case EMULATOR_UNLOADED:
		{
			EnableResetMenuItem(false);
			Sleep(20);
			break;
		}
		case EMULATOR_INITIALISE:
		{
			if (!InitialiseEmulator())
				eState = EMULATOR_SHUTDOWN;
			else
				eState = EMULATOR_RESET;
			break;
		}
		case EMULATOR_RESET:
		{
			SystemReset();
			CPUMan->ResetAllCPUs();

			EnableResetMenuItem(true);
			eState = EMULATOR_RUNNING;
			break;
		}
		case EMULATOR_RUNNING:
		{
			// Execute the CPU timeslices
			CPUMan->RunCPUs();

			// Check for input?
			// Update sound
			break;
		}
		case EMULATOR_SHUTDOWN:
		{
			ShutdownEmulator();
		#ifdef _DEBUG
			CloseErrorLog();
		#endif
			eState = EMULATOR_PAUSED;
			break;
		}
		default:
		case EMULATOR_PAUSED:
		{
			// Do nothing
			break;
		}
	}
}

emu_time EmulatorCore::GetEmulationTime()
{
	return emulated_time + CPUMan->GetExecutedTime();
}

void EmulatorCore::CreateTimer(emu_time period, UINT32 flags, bool active, void (*callback)())
{
	emu_time timenow = GetEmulationTime();
	emu_time time_slice = CPUMan->GetTimeslice();

	if (period == 0)
	{

	}
	else if (period < time_slice)
	{
		InsertTimer(&timer_list, period, flags, active, callback);
		CPUMan->Resync();
	}
	else
		InsertTimer(&timer_list, period, flags, active, callback);
}

void EmulatorCore::SetState(enum state State)
{
	eState = State;
}

void EmulatorCore::SystemReset()
{
	ResetRealTime();
	ClearTimerList(&timer_list);

	// Create a VBLANK timer
	CreateTimer(TIME_IN_HZ(60), TIMER_PERIODIC, true, core_video_update);

	// TEST
//	InsertTimer(&timer_list, TIME_IN_NSEC(100), TIMER_PERIODIC, TRUE, NULL);

	// TODO
	extern void SystemResetM2();
	SystemResetM2();
}

void EmulatorCore::ResetRealTime()
{
	if (RealTicksPerSec == 0)
	{
		LARGE_INTEGER TickFreq;
		QueryPerformanceFrequency(&TickFreq);
		RealTicksPerSec = TickFreq.QuadPart;
	}

	LARGE_INTEGER Count;
	QueryPerformanceCounter(&Count);
	real_base = Count.QuadPart;
	real_time = 0;
}

UINT64 EmulatorCore::GetSystemTicks()
{
	LARGE_INTEGER Count;
	QueryPerformanceCounter(&Count);
	return Count.QuadPart;
}


//
//   Local Functions
//
static TIMER_CALLBACK( core_video_update )
{
	bool bDrawFrame = true;
	UINT64 ticks = Core.GetSystemTicks();

	// VBLANK Interrupt?

	// TODO: Synchronise emu time and real time!

	// HACK - should call the core->VideoUpdate!!
	if (bDrawFrame)
	{
		Core.DoActiveVideoUpdate();
	}

//	core->last_ticks = ticks;
}

void EmulatorCore::DoActiveVideoUpdate()
{
	BufferInfo *BInfo = VideoMan->GetBufferInfo();
	ActiveVideoUpdate(BInfo);
	VideoMan->ReleaseBuffer();
	VideoMan->D3DRender();
}
