/******************************************************************************

    cpuexec

    Handling of emulated CPU execution

******************************************************************************/

#include "M2x.h"
#include "cpuexec.h"
#include "core.h"
#include "cpu\powerpc\ppc602.h"

#define DEBUG_TIMING		0

void CPUManager::AddCPU(const char *name, UINT32 clock, memory_map *map)
{
	CPU *newcpu;

	assert(cpu_count <= MAX_CPU);
	assert(clock > 0);

	// TODO: Take in an enum, scan through a list and declare
	newcpu = new PPC602();
	newcpu->cpuinfo.clock = clock;
	newcpu->cpuinfo.name = name;
	newcpu->cpuinfo.index = cpu_count;
	newcpu->cpuinfo.flags = 0;
	newcpu->cpuinfo.map = map;
	newcpu->cpuinfo.local_time = 0;

	cpu_list[cpu_count++] = newcpu;
}

void CPUManager::ResetCPU(int index)
{
	((PPC602*)(cpu_list[index]))->SoftReset();
	Resync();
}

void CPUManager::Resync()
{
	cpu_list[activecpu]->stolen_cycles = cpu_list[activecpu]->cycles_remaining;
	cpu_list[activecpu]->cycles_remaining = 0;

#if DEBUG_TIMING
	LOGERROR("CPU %d interrupted/reset. %d cycles were remaining\n", activecpu, cpu_list[activecpu]->stolen_cycles);
#endif
}

void CPUManager::SignalInterrupt(int index)
{
	((PPC602*)(cpu_list[index]))->Interrupt();
	Resync();
}

emu_time CPUManager::GetExecutedTime()
{
	return (cpu_list[activecpu]->cycles_to_run - cpu_list[activecpu]->cycles_remaining) / (double)cpu_list[activecpu]->cpuinfo.clock;
}

// Return absolute time until 
emu_time CPUManager::GetTimeslice()
{
	return timeslice;
}



void CPUManager::RunCPUs()
{
	Timer *t;
	INT32 cycles_ran;

	for (t = Core.timer_list; t; t = t->next)
	{
		// Find the first active timer
		if (t->flags & TIMER_ACTIVE)
			break;
	}

	/* Find the time until expiry */
	timeslice = t->expiry_time - Core.GetEmulationTime();

#if DEBUG_TIMING
	LOGERROR("Timeslice: %.12f\n", timeslice);
#endif

	// Loop over the CPUs
	for (int i = 0; i < cpu_count; ++i)
	{
		if (Core.GetState() == EMULATOR_SHUTDOWN)
			break;

		CPU *cpu = cpu_list[i];
		activecpu = i;

		cpu->stolen_cycles = 0;

		if (!(cpu->cpuinfo.flags & CPU_FLAG_SUSPENDED))
		{
			/* Compute the number of cycles to run for */
			INT32 cycles_to_run = (INT32)(timeslice * (double)cpu->cpuinfo.clock);

			if (cycles_to_run > 0)
			{
#if DEBUG_TIMING
				LOGERROR("Run CPU %d for %d cycles (time:%f)\n", i, cycles_to_run, cpu->cpuinfo.local_time);
#endif
				cycles_ran = cpu->Execute(cycles_to_run);

#if DEBUG_TIMING
				LOGERROR("Execution finished. Ran for %d cycles\n", cycles_ran);
#endif
				cpu->total_cycles += cycles_ran;

				/* Over or under run? */
				if (cycles_ran != cycles_to_run)
				{
					emu_time time_diff = (cycles_ran - cycles_to_run) / (double)cpu->cpuinfo.clock;
					timeslice += time_diff;
#if DEBUG_TIMING
					LOGERROR("Adjusting timeslice to %f\n", timeslice);
#endif
				}
			}
			else
				LOGERROR("ZERO CYc\n");
		}
		// Update the CPU's local time
		cpu->cpuinfo.local_time += timeslice;
		cpu->cycles_to_run = 0;
		cpu->cycles_remaining = 0;

#if DEBUG_TIMING
	LOGERROR("CPU time is %.12f\n", cpu->cpuinfo.local_time);
#endif
	}

	// Update our emulation time
	// TODO: Move this to core?
	Core.emulated_time += timeslice;
	UpdateTimers();
}


void CPUManager::UpdateTimers()
{
	int i = 0;
	Timer *t = Core.timer_list;
	Timer insert_list[16];		// TODO

	while (t)
	{
		Timer *t_next = t->next;

		if (t->flags & TIMER_ACTIVE)
		{
			if (t->expiry_time <= Core.GetEmulationTime())
			{
				/* Call the timer's callback function */
				if (t->callback)
					t->callback();

				/* Remove if one-shot */
				if (t->flags & TIMER_ONESHOT)
					RemoveTimer(&Core.timer_list, t);
				else
				{
					insert_list[i++] = *t;
					RemoveTimer(&Core.timer_list, t);
				}
			}
		}

		t = t_next;
	}

	for (int j = 0; j < i; ++j)
	{
		InsertTimer(&Core.timer_list, insert_list[j].period, insert_list[j].flags, true, insert_list[j].callback);
	}
}
