#pragma once
#include "M2x.h"
//#include "memory.h"
#include "core.h"
#include "timer.h"

#define MAX_CPU		4

// TOOD: Roll into the CPU class
struct cpu_info
{
	UINT32				clock;
	const char			*name;
	const memory_map	*map;
	UINT32				flags;
	UINT32				index;
	emu_time			local_time;
};


class CPU
{
private:
	UINT32		breakpoints[16];
	int			breakpoint_count;

	UINT32		watchpoints[16];
	int			watchpoint_count;
public:
	friend class DebugManager;
	UINT64		total_cycles;
	INT32		cycles_to_run;
	INT32		cycles_remaining;
	INT32		stolen_cycles;
	cpu_info	cpuinfo;

	virtual UINT32 Execute(UINT32 cycles) = 0;
	virtual void Reset(void) = 0;
	virtual UINT32 GetPC() = 0;
	virtual UINT32 GetReg(UINT32 num) = 0;
	CPU(void) :
		breakpoint_count(0),
		watchpoint_count(0),
		total_cycles(0),
		cycles_to_run(0),
		cycles_remaining(0),
		stolen_cycles(0)
	{
		ZeroMemory(breakpoints, sizeof(breakpoints));
		ZeroMemory(watchpoints, sizeof(watchpoints));
	}

	const char *GetName()
	{
		return cpuinfo.name;
	}

	UINT64 GetTotalCycles()
	{
		return total_cycles + (cycles_to_run - cycles_remaining);
	}
};

class CPUManager
{
private:
	int activecpu;
	int cpu_count;
	CPU *cpu_list[MAX_CPU];
	emu_time	timeslice;

	void UpdateTimers();

public:
#define CPU_FLAG_SUSPENDED	(1 << 0)
#define CPU_FLAG_RESET		(1 << 1)

	emu_time GetTimeslice(void);
	CPUManager() : cpu_count(0), activecpu(0)
	{
		ZeroMemory(cpu_list, sizeof(cpu_list));
	}
	int GetActiveCPU() { return activecpu; }
	UINT32 ActiveCPUGetPC() { return cpu_list[activecpu]->GetPC(); }
	void AddCPU(const char *name, UINT32 clock, memory_map *map);
	void RunCPUs();

	void AbortActiveCPUTimeSlice()
	{
		cpu_list[activecpu]->cycles_remaining = 0;
	}

	void ResetAllCPUs()
	{
		for (int i = 0; i < cpu_count; ++i)
		{
			cpu_list[i]->Reset();
		}
	}

	void Resync();

	void ResetCPU(int index);
	void SignalInterrupt(int index);

	emu_time GetExecutedTime(void);

	const char *ActiveCPUGetName()
	{
		return cpu_list[activecpu]->GetName();
	}
};
