#include "M2x.h"
#include "core.h"
#include "cpuexec.h"
#include "memory.h"
#include "log.h"
#include "video.h"
#include "konamim2.h"
#include "m2defs.h"

/* Change these to suit the CD image in use */
#define SECT_VAL	(2448)
#define CDOFFS		0


static UINT32 packet_r_idx;
static UINT32 packet_w_idx;
static UINT8 packet[2048];
static UINT8 packet_in[16];
static UINT32 packet_bytes;

static UINT8 atapi_irr;
static UINT8 atapi_error;
static UINT8 atapi_drive_sel;
static UINT8 atapi_features;
static UINT8 atapi_status;


typedef struct
{
	UINT32 num;
	UINT32 flags;
	UINT32 start;
	UINT32 length;
} track_info;


#define ATAPI_STATUS_BSY		(1 << 7)
#define ATAPI_STATUS_DRDY		(1 << 6)
#define ATAPI_STATUS_DMAREADY	(1 << 5)
#define ATAPI_STATUS_DF			(1 << 5)
#define ATAPI_STATUS_SERVICE	(1 << 4)
#define ATAPI_STATUS_DSC		(1 << 4)
#define ATAPI_STATUS_DRQ		(1 << 3)
#define ATAPI_STATUS_CORR		(1 << 2)
#define ATAPI_STATUS_CHECK		(1 << 0)

#define ATAPI_IRR_RELEASE		(1 << 2)
#define ATAPI_IRR_IO			(1 << 1)
#define ATAPI_IRR_COD			(1 << 0)

// TODO:
#define CD_BLOCK_LENGTH			2048



// TODO: SHIT!

static const track_info polystars_track_info[28] = 
{
	{ 1,   4, 0x000000, 0x061935 },
	{ 2,   0, 0x061935, 0x004367 },
	{ 3,   0, 0x070327, 0x035163 },
	{ 4,   0, 0x105515, 0x014266 },
	{ 5,   0, 0x123806, 0x015236 },
	{ 6,   0, 0x143042, 0x015626 },
	{ 7,   0, 0x162668, 0x012510 },
	{ 8,   0, 0x175203, 0x015443 },
	{ 9,   0, 0x194646, 0x012755 },
	{ 10,  0, 0x211426, 0x010426 },
	{ 11,  0, 0x221852, 0x011311 },
	{ 12,  0, 0x233163, 0x013148 },
	{ 13,  0, 0x250336, 0x031318 },
	{ 14,  0, 0x281654, 0x012525 },
	{ 15,  0, 0x294204, 0x025766 },
	{ 16,  0, 0x323970, 0x031825 },
	{ 17,  0, 0x355820, 0x020405 },
	{ 18,  0, 0x380225, 0x010109 },
	{ 19,  0, 0x390334, 0x005857 },
	{ 20,  0, 0x400216, 0x012735 },
	{ 21,  0, 0x412951, 0x014248 },
	{ 22,  0, 0x431224, 0x013972 },
	{ 23,  0, 0x445221, 0x011616 },
	{ 24,  0, 0x460837, 0x013703 },
	{ 25,  0, 0x474540, 0x001305 },
	{ 26,  0, 0x475845, 0x001551 },
	{ 27,  0, 0x481421, 0x003000 },
	{ 170, 0, 0x484421, 0x000000 },
};


enum _atapi_state
{
	IDLE,
	CMD_PACKET,
	RELEASE,
	DATA_TRANSFER_TO_HOST,
	DATA_TRANSFER_TO_DEVICE,
	DATA_TRANSFER,
	SERVICE,
	STATUS,
} atapi_state;

// Just in case, here's the Konami CD drive details
static const char *cd_model = "MATSHITACD-ROM CR-583";
static const char *cd_firmware = "1.07";
static const char *cd_serial = "96072613";

HANDLE hCDMemMap;
UINT32 CDFileSize;
UINT32 AllocGranularity;

UINT32 map_start;
UINT32 map_end;

UINT8 *cdmap_ptr;

#define MAP_BLOCK_SIZE		(1024*1024)


struct
{
	union
	{
		struct { UINT8 l; UINT8 h; };
		UINT16 w;
	};
} atapi_byte_cnt;


UINT32 lba_to_msf(UINT32 lba)
{
	UINT8 m, s, f;

	m = lba / (60 * 75);
	lba -= m * (60 * 75);
	s = lba / 75;
	f = lba % 75;

	return ((m / 10) << 20) | ((m % 10) << 16) |
	       ((s / 10) << 12) | ((s % 10) <<  8) |
	       ((f / 10) <<  4) | ((f % 10) <<  0);
}

UINT32 msf_to_lba(UINT32 msf)
{
	UINT32 f = 10 * ((msf >> 4) & 0xf) + (msf & 0xf);
	UINT32 s = 10 * ((msf >> 12) & 0xf) + ((msf >> 8) & 0xf);
	UINT32 m = 10 * ((msf >> 20) & 0xf) + ((msf >> 16) & 0xf);

	return (m * 60 + s) * 75 + f;
}


inline bool SectorInRange(UINT32 offs)
{
	return (offs >= map_start) && (offs + SECT_VAL-1 <= map_end);
}

UINT8 *GetCDSectorPtr(UINT32 lba)
{
	// Convert LBA into file offset
	UINT32 fileoffs = lba * SECT_VAL;

	if (!cdmap_ptr || !SectorInRange(fileoffs))
	{
		UINT32 map_length;

		if (cdmap_ptr)
			UnmapViewOfFile(cdmap_ptr);

		map_start = fileoffs & ~(AllocGranularity - 1);
		map_length = min(CDFileSize - map_start, MAP_BLOCK_SIZE);
		map_end = map_start + map_length - 1;

		cdmap_ptr = (UINT8 *)MapViewOfFile(hCDMemMap, FILE_MAP_READ, 0, map_start, map_length);
	}

	// 16 bytes for the header
	return cdmap_ptr + (fileoffs - map_start) + CDOFFS;
}






void SetATAPIInterrupt(int state)
{
	if (state == 1)
		SetCDEInterrupt(CDE_INT_ATAPIDRQ);
	else
		ClearCDEStatus(CDE_INT_ATAPIDRQ);
}

void SetATAPIState(enum _atapi_state newstate)
{
	switch (newstate)
	{
		case IDLE:
		{
			packet_r_idx = 0;
			packet_w_idx = 0;
			atapi_irr &= ~(ATAPI_IRR_IO | ATAPI_IRR_RELEASE | ATAPI_IRR_COD);
			atapi_status &= ~(ATAPI_STATUS_BSY | ATAPI_STATUS_DRQ);
			atapi_status |= ATAPI_STATUS_DRDY;
			break;
		}
		case CMD_PACKET:
		{
			atapi_status &= ~ATAPI_STATUS_BSY;
			atapi_status |= ATAPI_STATUS_DRQ;
			atapi_irr &= ~ATAPI_IRR_IO;
			atapi_irr |= ATAPI_IRR_COD;
			break;
		}
		case DATA_TRANSFER_TO_HOST:
		{
			atapi_state = DATA_TRANSFER;
			atapi_irr &= ~(ATAPI_IRR_COD | ATAPI_IRR_RELEASE);
			atapi_irr |= ATAPI_IRR_IO;
			atapi_status &= ~ATAPI_STATUS_BSY;
			atapi_status |= ATAPI_STATUS_DRQ;
			SetATAPIInterrupt(1);
			break;
		}
		case DATA_TRANSFER_TO_DEVICE:
		{
			atapi_state = DATA_TRANSFER;
			atapi_irr &= ~(ATAPI_IRR_COD | ATAPI_IRR_IO | ATAPI_IRR_RELEASE);
			atapi_status &= ~ATAPI_STATUS_BSY;
			atapi_status |= ATAPI_STATUS_DRQ;
			SetATAPIInterrupt(1);
			break;
		}
		case STATUS:
		{
			// TODO: Completition status in status register!
			atapi_irr |= (ATAPI_IRR_IO | ATAPI_IRR_COD);
			atapi_status &= ~(ATAPI_STATUS_BSY | ATAPI_STATUS_DRQ);
			atapi_status |= ATAPI_STATUS_DRDY;
			SetATAPIInterrupt(1);
			break;
		}
	}
	atapi_state = newstate;
}

void ATAPICopyString(char *dest, const char *src, const int field_len)
{
	int i;
	int str_len = strlen(src);

	for (i = 0; i < field_len; ++i)
	{
		if (i < str_len)
			*dest++ = *src++;
		else
			*dest++ = 0x20;
	}
}

void CreateATAPIIdentifyPacket()
{
	// TODO
	packet[0] = 0;

	ATAPICopyString((char *)&packet[20], cd_serial, 20);
	ATAPICopyString((char *)&packet[46], cd_firmware, 8);
	ATAPICopyString((char *)&packet[54], cd_model, 40);

	atapi_byte_cnt.w = 256;
}

READ32_HANDLER( atapi_r )
{
//	0E = Ready Reg.   Mask 0x88 - '1' = Ready
//	02 = Data Register
//	00 = Data Read Register

/*
  00 (R) |xxxxxxxx|xxxxxxxx|xxxxxxxx|xxxxxxxx| Data Read
  00 (W) |........|........|........|........|

  04 (R) |........|........|........|........|
  04 (W) |........|........|........|........|

  08 (R) |........|........|........|........|
  08 (W) |........|........|........|xxxxxxxx| Byte count

  0C (R) |........|........|........|........| ATAPI Status
                                     x.......| Busy
                                      .......| DRDY

  0C (W) |........|........|CCCCCCCC|xxxxxxxx| ATAPI Packet
         |xxxxxxxx|xxxxxxxx|........|........| ATAPI Command
*/
	assert(mask == 0xffff0000 || mask == 0x0000ffff);

	UINT32 val;
	int reg = (offset << 1) + ((mask == 0x0000ffff) ? 1 : 0);
	int shift = (mask == 0xffff0000) ? 16 : 0;

	switch (reg)
	{
		case 0x0:
		{
			val = (packet[packet_r_idx] << 8) | packet[packet_r_idx + 1];
			packet_r_idx += 2;

			if (packet_r_idx == atapi_byte_cnt.w)
				SetATAPIState(STATUS);

			break;
		}
		case 0x1: val = atapi_error << 8;			break;
		case 0x2: val = atapi_irr << 8;				break;
		case 0x4: val = atapi_byte_cnt.l << 8;		break;
		case 0x5: val = atapi_byte_cnt.h << 8;		break;
		case 0x6: val = atapi_drive_sel << 8;		break;
		case 0x7:
		{

			val = atapi_status << 8;
			if (atapi_state == STATUS)
			{
				SetATAPIInterrupt(0);
				SetATAPIState(IDLE);
			}
			break;
		}
		default:
			LOGERROR("ATAPI R:%.4x\n", reg);
			val = 0;
	}

	return val << shift;
}

// TODO
void CreateSensePage()
{
	atapi_byte_cnt.w = 16;
}

void CreateTOC()
{
	atapi_byte_cnt.w = 4 + 0 + 8;

	UINT32 track = packet_in[6];
	UINT32 alloc_len = (packet_in[7] << 8) | packet_in[8];
	UINT32 format = (packet_in[9] >> 6) & 3;
	UINT32 msf = packet_in[1] & 2;

	LOGERROR("ATAPI Read TOC: Track %x Format: %d\n", track, format);

	packet[0] = 0; // TOC data length MSB
	packet[1] = 8; // TOC data length LSB
	packet[2] = 1; // First track #
	packet[3] = 27; // Last track #

	packet[4] = 0;		// Reserved
	packet[5] = polystars_track_info[track - 1].flags;		// ADR and Control
	packet[6] = track;	// Track
	packet[7] = 0;		// Reserved

	if (track == 0xaa)
		track = 27;

	UINT32 abs_addr = msf ? polystars_track_info[track - 1].start : msf_to_lba(polystars_track_info[track - 1].start);
	WRITE32_BE(packet, 8, abs_addr);
}

void GetCDCapacity()
{
	// TODO!
	UINT32 lba = msf_to_lba(0x513623); // TODO
	atapi_byte_cnt.w = 8;
	packet[0] = lba >> 24;
	packet[1] = lba >> 16;
	packet[2] = lba >> 8;
	packet[3] = lba & 0xff;
	packet[4] = 0;
	packet[5] = 8;
	packet[6] = 0;
	packet[7] = 0;
}

// TODO: This only works for data!!!!!!!
void ReadCDData()
{
	UINT32 lba = READ32_BE(packet_in, 2);
	UINT32 len = READ32_BE(packet_in, 6);

	//LOGERROR("ATAPI Read Command: LBA:%.5x (%x) %x block(s)\n", lba, lba * 2352, len);

	// Return the bytes read
	atapi_byte_cnt.w = len * 2048;

	if (len > 1)
		CLOGERROR(GREEN, "MULTIPLE SECTOR READ\n");

	// Read in the sectors
	for (UINT32 i = 0; i < len; ++i)
		memcpy(packet + (2048*i), GetCDSectorPtr(lba + i), 2048);
}

void HandleATAPICommand(int code)
{
	switch(code)
	{
		case 0x00:
			// Test unit ready
			SetATAPIState(STATUS);
			break;
		case 0x03:
			// Request sense
			SetATAPIState(STATUS);
			break;
		case 0x25:
			// Read CD-ROM capacity
			GetCDCapacity();
			SetATAPIState(DATA_TRANSFER_TO_HOST);
			break;
		case 0x43:
			// Read TOC
			CreateTOC();
			SetATAPIState(DATA_TRANSFER_TO_HOST);
			break;
		case 0x5a:
			// Mode sense
			CreateSensePage();
			SetATAPIState(DATA_TRANSFER_TO_HOST);
			break;
		case 0xa8:
			// Read data
			ReadCDData();
			SetATAPIState(DATA_TRANSFER_TO_HOST);
			break;
		default:
			LOGERROR("Unknown ATAPI Command: %x\n", code);
	}
}

WRITE32_HANDLER( atapi_w )
{
	int reg = (offset << 1) + ((mask == 0x0000ffff) ? 1 : 0);
	int shift = (mask == 0xffff0000) ? 16 : 0;

	data >>= shift;
	UINT8 ldata = data >> 8;

	switch (reg)
	{
		case 0x0:
		{
			//LOGERROR("Packet [%.2x]: %x\n", packet_w_idx, ldata);
			packet_in[packet_w_idx++] = ldata;
			//LOGERROR("Packet [%.2x]: %x\n", packet_w_idx, data & 0xff);
			packet_in[packet_w_idx++] = data & 0xff;

			if (packet_w_idx == 12)
			{
				//atapi_status &= ~ATAPI_STATUS_DRQ;
				//atapi_status |= ATAPI_STATUS_BSY;
				HandleATAPICommand(packet_in[0]);
			}
			break;
		}
		case 0x1:
			//LOGERROR("Features:       %.2x\n", ldata);
			atapi_features = ldata;
			break;
		case 0x4:
			//LOGERROR("Byte count LSB: %.2x\n", ldata);
			atapi_byte_cnt.l = ldata;
			break;
		case 0x5:
			//LOGERROR("Byte count MSB: %.2x\n", ldata);
			atapi_byte_cnt.h = ldata;
			break;
		case 0x6:
			//LOGERROR("Drive Select:   %.2x\n", ldata);
			atapi_drive_sel = ldata;
			break;
		case 0x7:
		{
			data >>= 8;
			//LOGERROR("Command:        %.2x\n", data);
			packet_r_idx = 0;
			packet_w_idx = 0;

			if (data == 0xa1)
			{
				CreateATAPIIdentifyPacket();
				atapi_irr |= ATAPI_IRR_IO;
				SetATAPIState(DATA_TRANSFER_TO_HOST);
			}
			else if (data == 0xa0)
			{
				// Command packet
				//	atapi_status |= ATAPI_STATUS_BSY;
				// TODO: Should be a delay here really
				SetATAPIState(CMD_PACKET);
			}
			else
				LOGERROR("Unknown ATAPI command %x\n", data);
			break;
		}
		case 0xe:
		{
			if (ldata & 4)
				SetATAPIState(IDLE);
			LOGERROR("Device control: %.2x\n", ldata); break;
		}
		default:
			LOGERROR("ATAPI W:%.4x :%.4x\n", reg, data);
			break;
	}
}

WRITE32_HANDLER( atapi_dma_w )
{
	UINT32 src = 3;
	//LOGERROR("ATAPI DMA W: %x %x\n", offset, data);
}



/////////////////////////
//
//  CD Image Handling
//
/////////////////////////


bool LoadCDImage(const ROMFile file_desc)
{
	HANDLE hTemp;
	LARGE_INTEGER FileSize;
	char filepath[32];

	sprintf(filepath, "roms\\%s\\%s", Core.GetActiveSetName(), file_desc.name);

	/* Convert to UTF-16 format */
	WCHAR *tmp = CharToWString(filepath);


	// Note: Need better file checking
	hTemp = CreateFile( tmp, GENERIC_READ,
						FILE_SHARE_READ,
						0,
						OPEN_EXISTING,
						0,
						NULL );

	if (hTemp == INVALID_HANDLE_VALUE)
		goto FileError;

	GetFileSizeEx(hTemp, &FileSize);

	if (FileSize.LowPart == 0)
		goto FileError;

	// Close the handle and save some globals
	hCDMemMap = hTemp;
//	CloseHandle(hTemp);
	CDFileSize = FileSize.LowPart;

	// Create the file mapping
	hCDMemMap = CreateFileMapping(hTemp, 0, PAGE_READONLY, 0, 0, 0);

	// TODO: This should be part of a utility package
	SYSTEM_INFO SystemInfo;
	GetSystemInfo(&SystemInfo);
	AllocGranularity = SystemInfo.dwAllocationGranularity;

	return true;

FileError:
	ErrorBox("Could not open CD Image!");
	CloseHandle(hTemp);
	return false;
}


