{ Serial bus interface KA1835VG4, keyboard interface KA1835VG1, SMP modules }

unit IoSystem;

interface

  procedure IoInit;
  procedure IoClose;
  function IoWrPtr (index: word) : pointer;
  function IoRdPtr (index: word) : pointer;
  procedure TimerIrq;
  procedure KeyIrq;

implementation

uses Def, Keyboard, Smp;

var
  iodata: array [0..3] of word;
  ioword: word;		{ data to exchange with the host }
  ioindex: word;	{ index of the iodata location that should be written
			  at next CPU cycle, otherwise $FFFF }
  shiftreg: word;	{ serial bus shift register }
  RGRQ: word;		{ interrupt request latches }
  select: boolean;	{ serial bus SELECT signal }


{ write pending data to the ioindex memory location }
procedure IoWrite;
begin
  ioindex := ioindex and $03;		{safety measure}
  iodata[ioindex] := ioword;
  case ioindex of
    0: begin				{data register}
         shiftreg := ioword;
         RGRQ := RGRQ or (1 shl (iodata[2] and 7));
         if select then
         begin
           if not CpuStop and		{if the system isn't in the debug mode}
           ((iodata[2] and $20) = 0)	{if data register interrupts are enabled}
           then VIRQ_C4 := true;
           case iodata[2] and $0F of
             $00: shiftreg := SmpData (0, 0);
             $01: shiftreg := SmpData (1, 0);
             $02: shiftreg := KeyScanCode;
             $08: SmpData (0, shiftreg);
             $09: SmpData (1, shiftreg);
           end {case};
         end {if};
       end;
{   1: transfer rate specification ignored by the emulator }
    2: begin				{control register}
         if select then
         begin
           if not CpuStop and		{if the system isn't in the debug mode}
           ((iodata[2] and $20) = 0)	{if data register interrupts are enabled}
           then VIRQ_C4 := true;
           case iodata[2] and $0F of
             $00: shiftreg := SmpData (0, 0);
             $01: shiftreg := SmpData (1, 0);
             $02: shiftreg := KeyScanCode;
           end {case};
         end {if};
       end;
    3: begin				{command register}
         select := true;
         shiftreg := ioword;
         RGRQ := RGRQ or (1 shl (iodata[2] and 7));
         if (iodata[2] and $20) = 0	{if data register interrupts are enabled}
         then VIRQ_C4 := true;
         case iodata[2] and $0F of
           $00: shiftreg := SmpCmd (0, 0);
           $01: shiftreg := SmpCmd (1, 0);
           $02: shiftreg := KeyScanCode;
           $08: SmpCmd (0, shiftreg);
           $09: SmpCmd (1, shiftreg);
         end {case};
       end;
  end {case};
  ioindex := $FFFF;
end {IoWrite};


procedure IoInit;
begin
  ioindex := $FFFF;
  shiftreg := $FFFF;
  RGRQ := $FFFF;
  select := false;
  SmpOpen (0);
  SmpOpen (1);
end {IoInit};


procedure IoClose;
begin
  SmpClose (0);
  SmpClose (1);
end {IoClose};


function IoWrPtr (index: word) : pointer;
begin
  ioindex := (index shr 1) and $03;
  IoWrPtr := @ioword;
  procptr := @IoWrite;
end {IoWrPtr};


function IoRdPtr (index: word) : pointer;
begin
  index := (index shr 1) and $03;
  case index of
    0: begin				{data register}
         ioword := shiftreg;
         shiftreg := $FFFF;		{default value}
         RGRQ := RGRQ or (1 shl (iodata[2] and 7));
         if select then
         begin
           if not CpuStop and		{if the system isn't in the debug mode}
           ((iodata[2] and $20) = 0)	{if data register interrupts are enabled}
           then VIRQ_C4 := true;
           case iodata[2] and $0F of
             $00: shiftreg := SmpData (0, 0);
             $01: shiftreg := SmpData (1, 0);
             $02: shiftreg := KeyScanCode;
           end {case};
         end {if};
       end;
    1: ioword := RGRQ;			{interrupt request latches}
    2: begin
         ioword := (iodata[2] and $70) or $FF84;	{always ready}
	 if not select then ioword := ioword or $08;
       end;
    3: begin				{command register}
         if select and
         not CpuStop and		{if the system isn't in the debug mode}
         ((iodata[2] and $20) = 0)	{if data register interrupts are enabled}
         then VIRQ_C4 := true;
         ioword := shiftreg;
         select := false;
       end;
  end {case};
  IoRdPtr := @ioword;
end {IoRdPtr};


{ timer interrupt from the INRT input }
procedure TimerIrq;
begin
  if not CpuStop		{if the system isn't in the debug mode}
  and ((iodata[2] and $40) = 0)	{if timer interrupts are enabled}
  then VIRQ_C0 := true;
end {TimerIrq};


{ keyboard interrupt }
procedure KeyIrq;
begin
  if not CpuStop then		{if the system isn't in the debug mode}
  begin
    RGRQ := RGRQ and not 4;
    if (iodata[2] and $10) = 0	{if external interrupts are enabled}
	then VIRQ_C8 := true;
  end {if};
end {KeyIrq};


end.
