{ Real Time Clock and RAM with battery backup KA512VI1 = MC146818 }

unit Rtc;

interface

  procedure RtcInit;
  procedure RtcClose;
  procedure RtcUpdate;
  procedure RtcIrq;
  function RtcWrPtr (index: word) : pointer;
  function RtcRdPtr (index: word) : pointer;
  function PeriodicInterruptRate : integer;

implementation

uses SysUtils, Def, IoSystem;

const
  RtcFile: string = 'rtc.bin';

var
  rtcdata: array [0..63] of byte;
  rtcword: word;		{ to exchange data with the host }
  rtcindex: word;		{ if in range 0..63 then rtcword should be
				  written to the rtcdata memory location at
				  next access of the RTC system }

{ write pending data to the rtcdata memory }
procedure RtcWrite;
begin
  if rtcindex > 63 then Exit;
  if (rtcindex <> $0C) and (rtcindex <> $0D) then
  begin
    rtcdata[rtcindex] := rtcword shr 1;
  end {if};
  rtcindex := $FFFF;
end {RtcWrite};


procedure RtcInit;
var
//  f: file;
  t: TDateTime;
  x1,x2,x3,x4: word;
begin
{ File loading disabled in the public release, because a corrupted RtcFile can
  hang the emulated calculator (just as corrupted non-volatile RAM contents
  can hang the real one). In such case the user would have to delete the file
  (equivalent to removing the batteries in a real MK-90). I'm afraid that this
  would be asking too much and decided to avoid the hassle... }
//  if FileExists (RtcFile) then
//  begin
//    AssignFile (f, RtcFile);
//    Reset (f, 1);
//    BlockRead (f, rtcdata, 64);
//    CloseFile (f);
//  end; {if}

  rtcindex := $FFFF;
  rtcdata[$0D] := $00;	{clear the VRT bit}
  rtcdata[$0C] := $00;
  rtcdata[$0B] := $07;	{ $87 in the real machine, but then the emulated one
 expects the time and date to be set, instead of showing the current values
 copied by the code below }
  rtcdata[$0A] := $7F;
  rtcdata[5] := $00;
  rtcdata[3] := $00;
  rtcdata[1] := $00;

{ copy current date and time to rtcdata }
  t := Now;
  DecodeDate(t, x1, x2, x3);
  rtcdata[9] := byte(x1 mod 100);
  rtcdata[8] := byte(x2);
  rtcdata[7] := byte(x3);
  x3 := DayOfWeek(t) - 1;
  if x3 <= 0 then x3 := 7;
  rtcdata[6] := byte(x3);
  DecodeTime(t, x1, x2, x3, x4);
  rtcdata[4] := byte(x1);
  rtcdata[2] := byte(x2);
  rtcdata[0] := byte(x3);
end {RtcInit};


procedure RtcClose;
var
  f: file;
begin
  RtcWrite;
  AssignFile (f, RtcFile);
  Rewrite (f, 1);
  BlockWrite (f, rtcdata, 64);
  CloseFile (f);
end {RtcClose};


{ increment a BCD encoded byte }
procedure Binc (var x:byte);
begin
  Inc (x);
  if (x and $0F) > 9 then Inc (x,6);
end {Binc};


{ should be called every second }
procedure RtcUpdate;
const
  days: array[0..11] of integer = (31,28,31,30,31,30,31,31,30,31,30,31);
begin
  RtcWrite;

  if (rtcdata[$0B] and $80) <> 0 then	{SET bit}
  begin
    rtcdata[$0C] := rtcdata[$0C] and not $10;	{clear the update-ended interrupt flag}
    Exit;
  end {if};

{ time and date registers }
  if (rtcdata[$0B] and $04) = 0 then
{ BCD mode }
  begin
    Binc (rtcdata[0]);				{seconds}
    if rtcdata[0] >= $60 then
    begin
      rtcdata[0] := 0;
      Binc (rtcdata[2]);			{minutes}
      if rtcdata[2] >= $60 then
      begin
        rtcdata[2] := 0;
        Binc (rtcdata[4]);			{hours}
        if rtcdata[4] >= $24 then
	begin
          rtcdata[4] := 0;
          Binc (rtcdata[6]);			{day of the week};
          if rtcdata[6] > 7 then rtcdata[6] := 1;
          Binc (rtcdata[7]);			{day of the month};
          if rtcdata[7] > days[(rtcdata[8] - 1) mod 12] then
	  begin
            rtcdata[7] := 1;
            Binc (rtcdata[8]);			{month};
            if rtcdata[8] > $12 then
	    begin
              rtcdata[8] := 1;
              Binc (rtcdata[9]);		{year};
              if rtcdata[8] > $99 then rtcdata[8] := 0;
            end {if};
          end {if};
        end {if};
      end {if};
    end {if};
  end
  else
{ binary mode }
  begin
    Inc (rtcdata[0]);				{seconds}
    if rtcdata[0] >= 60 then
    begin
      rtcdata[0] := 0;
      Inc (rtcdata[2]);				{minutes}
      if rtcdata[2] >= 60 then
      begin
        rtcdata[2] := 0;
        Inc (rtcdata[4]);			{hours}
        if rtcdata[4] >= 24 then
	begin
          rtcdata[4] := 0;
          Inc (rtcdata[6]);			{day of the week};
          if rtcdata[6] > 7 then rtcdata[6] := 1;
          Inc (rtcdata[7]);			{day of the month};
          if rtcdata[7] > days[(rtcdata[8] - 1) mod 12] then
	  begin
            rtcdata[7] := 1;
            Inc (rtcdata[8]);			{month};
            if rtcdata[8] > 12 then
            begin
              rtcdata[8] := 1;
              Inc (rtcdata[9]);			{year};
              if rtcdata[8] > 99 then rtcdata[8] := 0;
            end {if};
          end {if};
        end {if};
      end {if};
    end {if};
  end {if};

{ alarm }
  if	((rtcdata[1] = rtcdata[0]) or ((rtcdata[1] and $C0) <> 0)) and
	((rtcdata[3] = rtcdata[2]) or ((rtcdata[3] and $C0) <> 0)) and
	((rtcdata[5] = rtcdata[4]) or ((rtcdata[5] and $C0) <> 0)) then
    rtcdata[$0C] := rtcdata[$0C] or $20	{set the alarm interrupt flag AF}
  else
    rtcdata[$0C] := rtcdata[$0C] and not $20;	{clear the flag AF}

{ update ended }
  rtcdata[$0C] := rtcdata[$0C] or $10;	{set the update-ended interrupt flag UF}

{ IRQF interrupt flag and IRQ pin = (PF and PIE) or (AF and AIE) or (UF and UIE) }
  if (rtcdata[$0B] and rtcdata[$0C] and $70) <> 0 then
  begin
    rtcdata[$0C] := rtcdata[$0C] or $80;	{set IRQF}
    TimerIrq;					{drive the IRQ pin low}
  end
  else
  begin
    rtcdata[$0C] := rtcdata[$0C] and not $80;	{clear IRQF}
    {perhaps here some code to drive the IRQ pin high}
  end {if};
end {RtcUpdate};


{ should be called at the periodic interrupt rate, default = 31.25ms }
procedure RtcIrq;
begin
  if (not CpuStop) and			{the system isn't in the debug mode}
  ((rtcdata[$0B] and $08) <> 0) then	{bit SQWE in the register B is set}
    EVNT_i := true;
  rtcdata[$0C] := rtcdata[$0C] or $40;	{periodic interrupt flag}
{ set the IRQF interrupt flag and drive the IRQ pin low, if the periodic
  interrupt is enabled }
  if (rtcdata[$0B] and $40) <> 0 then
  begin
    rtcdata[$0C] := rtcdata[$0C] or $80;
    TimerIrq;				{drive the IRQ pin low}
  end {if};
end {RtcIrq};


function RtcWrPtr (index: word) : pointer;
begin
  RtcWrite;
  rtcindex := (index shr 1) and $3F;
  RtcWrPtr := @rtcword;
end {RtcWrPtr};


procedure SetVrt;
begin
  rtcdata[$0D] := $80;	{set the VRT bit}
end {SetVrt};


procedure ClrIntFlags;
begin
  rtcdata[$0C] := 0;	{clear all interrupt flags}
end {ClrIntFlags};


function RtcRdPtr (index: word) : pointer;
begin
  RtcWrite;
  index := (index shr 1) and $3F;
  rtcword := word(rtcdata[index]) shl 1;
  if index = $0D then procptr := @SetVrt
  else if index = $0C then procptr := @ClrIntFlags;
  RtcRdPtr := @rtcword;
end {RtcRdPtr};


{ value in microseconds }
function PeriodicInterruptRate : integer;
const
  table: array [0..15] of integer =
	( 0, 3906, 7812, 122, 244, 488, 977, 1953, 3906, 7812, 15625, 31250,
	62500, 125000, 250000, 500000 );
begin
  PeriodicInterruptRate := table[rtcdata[$0A] and $0F];
end {PeriodicInterruptRate};


end.
