/************************************************************************
 *
 * MunkDisk.c
 *
 *	This file contains the disk-related portion of Udo Munk's
 *	I/O simulation.
 *
 *	Most of this code is Copyright (C) 1987-2007 by Udo Munk. I have
 *	shamelessly stolen it.
 *
 * revisions:
 *
 *	2007-04-12 rli: Began futzing with it.
 *
 ************************************************************************/

/****************************
 *
 *	Udo Munk's CODE
 *
 ****************************/

/*
 *	This module contains the I/O handlers for a simulation
 *	of the hardware required for a CP/M / MP/M system.
 *
 *	Used I/O ports:
 *
 *	10 - FDC drive
 *	11 - FDC track
 *	12 - FDC sector
 *	13 - FDC command
 *	14 - FDC status
 *
 *	15 - DMA destination address low
 *	16 - DMA destination address high
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "sim.h"
#include "simglb.h"
#include "WinIo.h"
#include "MunkDisk.h"

/*
 *	Structure to describe an emulated disk drive:
 *		pointer to filename
 *		pointer to file descriptor
 *		number of tracks
 *		number of sectors
 */
struct dskdef {
	char *fn;
	int *fd;
	unsigned int tracks;
	unsigned int sectors;
};

static BYTE drive;		/* current drive A..P (0..15) */
static BYTE track;		/* current track (0..255) */
static BYTE sector;		/* current sektor (0..255) */
static BYTE status;		/* status of last I/O operation on FDC */
static BYTE dmadl;		/* current DMA address destination low */
static BYTE dmadh;		/* current DMA address destination high */
static BYTE clkcmd;		/* clock command */
static BYTE timer;		/* 10ms timer */
static int drivea;		/* fd for file "drivea.cpm" */
static int driveb;		/* fd for file "driveb.cpm" */
static int drivec;		/* fd for file "drivec.cpm" */
static int drived;		/* fd for file "drived.cpm" */
static int drivee;		/* fd for file "drivee.cpm" */
static int drivef;		/* fd for file "drivef.cpm" */
static int driveg;		/* fd for file "driveg.cpm" */
static int driveh;		/* fd for file "driveh.cpm" */
static int drivei;		/* fd for file "drivei.cpm" */
static int drivej;		/* fd for file "drivej.cpm" */
static int drivek;		/* fd for file "drivek.cpm" */
static int drivel;		/* fd for file "drivel.cpm" */
static int drivem;		/* fd for file "drivem.cpm" */
static int driven;		/* fd for file "driven.cpm" */
static int driveo;		/* fd for file "driveo.cpm" */
static int drivep;		/* fd for file "drivep.cpm" */

static struct dskdef disks[16] = {
	{ "disks/drivea.cpm", &drivea, 77, 26 },
	{ "disks/driveb.cpm", &driveb, 77, 26 },
	{ "disks/drivec.cpm", &drivec, 77, 26 },
	{ "disks/drived.cpm", &drived, 77, 26 },
	{ "disks/drivee.cpm", &drivee, -1, -1 },
	{ "disks/drivef.cpm", &drivef, -1, -1 },
	{ "disks/driveg.cpm", &driveg, -1, -1 },
	{ "disks/driveh.cpm", &driveh, -1, -1 },
	{ "disks/drivei.cpm", &drivei, 255, 128 },
	{ "disks/drivej.cpm", &drivej, 255, 128 },
	{ "disks/drivek.cpm", &drivek, -1, -1 },
	{ "disks/drivel.cpm", &drivel, -1, -1 },
	{ "disks/drivem.cpm", &drivem, -1, -1 },
	{ "disks/driven.cpm", &driven, -1, -1 },
	{ "disks/driveo.cpm", &driveo, -1, -1 },
	{ "disks/drivep.cpm", &drivep, -1, -1 }
};

/*
 *	Forward declaration of the I/O handlers for all used ports
 *
 * revisions:
 *
 *	2007-04-12 rli: Tweaked parameter lists.
 */

static BYTE fdcd_in( BYTE, void * ), fdcd_out( BYTE, BYTE, void * );
static BYTE fdct_in( BYTE, void * ), fdct_out( BYTE, BYTE, void * );
static BYTE fdcs_in( BYTE, void * ), fdcs_out( BYTE, BYTE, void * );
static BYTE fdco_in( BYTE, void * ), fdco_out( BYTE, BYTE, void * );
static BYTE fdcx_in( BYTE, void * ), fdcx_out( BYTE, BYTE, void * );
static BYTE dmal_in( BYTE, void * ), dmal_out( BYTE, BYTE, void * );
static BYTE dmah_in( BYTE, void * ), dmah_out( BYTE, BYTE, void * );

#if 0  

/* rli: Not using this array, but want to keep I/O addresses around */

/*
 *	This array contains two function pointers for every
 *	active port, one for input and one for output.
 */
static BYTE (*port[256][2]) () = {
	{ cons_in, cons_out },		/* port 0 */
	{ cond_in, cond_out },		/* port 1 */
	{ prts_in, prts_out },		/* port 2 */
	{ prtd_in, prtd_out },		/* port 3 */
	{ auxs_in, auxs_out },		/* port 4 */
	{ auxd_in, auxd_out },		/* port 5 */
	{ io_trap, io_trap  },		/* port 6 */
	{ io_trap, io_trap  },		/* port 7 */
	{ io_trap, io_trap  },		/* port 8 */
	{ io_trap, io_trap  },		/* port 9 */
	{ fdcd_in, fdcd_out },		/* port 10 */
	{ fdct_in, fdct_out },		/* port 11 */
	{ fdcs_in, fdcs_out },		/* port 12 */
	{ fdco_in, fdco_out },		/* port 13 */
	{ fdcx_in, fdcx_out },		/* port 14 */
	{ dmal_in, dmal_out },		/* port 15 */
	{ dmah_in, dmah_out },		/* port 16 */
	{ io_trap, io_trap  },		/* port 17 */
	{ io_trap, io_trap  },		/* port 18 */
	{ io_trap, io_trap  },		/* port 19 */
	{ mmui_in, mmui_out },		/* port 20 */
	{ mmus_in, mmus_out },		/* port 21 */
	{ mmuc_in, mmuc_out },		/* port 22 */
	{ io_trap, io_trap  },		/* port 23 */
	{ io_trap, io_trap  },		/* port 24 */
	{ clkc_in, clkc_out },		/* port 25 */
	{ clkd_in, clkd_out },		/* port 26 */
	{ time_in, time_out },		/* port 27 */
	{ delay_in, delay_out  },	/* port 28 */
	{ io_trap, io_trap  },		/* port 29 */
	{ io_trap, io_trap  },		/* port 30 */
	{ io_trap, io_trap  },		/* port 31 */
	{ io_trap, io_trap  },		/* port 32 */
	{ io_trap, io_trap  },		/* port 33 */
	{ io_trap, io_trap  },		/* port 34 */
	{ io_trap, io_trap  },		/* port 35 */
	{ io_trap, io_trap  },		/* port 36 */
	{ io_trap, io_trap  },		/* port 37 */
	{ io_trap, io_trap  },		/* port 38 */
	{ io_trap, io_trap  },		/* port 39 */
	{ cons1_in, cons1_out  },	/* port 40 */
	{ cond1_in, cond1_out  },	/* port 41 */
	{ cons2_in, cons2_out  },	/* port 42 */
	{ cond2_in, cond2_out  },	/* port 43 */
	{ cons3_in, cons3_out  },	/* port 44 */
	{ cond3_in, cond3_out  },	/* port 45 */
	{ cons4_in, cons4_out  },	/* port 46 */
	{ cond4_in, cond4_out  },	/* port 47 */
	{ io_trap, io_trap  },		/* port 48 */
	{ io_trap, io_trap  },		/* port 49 */
	{ nets1_in, nets1_out  },	/* port 50 */
	{ netd1_in, netd1_out  },	/* port 51 */
	{ io_trap, io_trap  },		/* port 52 */
	{ io_trap, io_trap  },		/* port 53 */
	{ io_trap, io_trap  }		/* port 54 */
};
#endif

/*
 *	This function initializes the I/O handlers:
 *
 *	3. Open the files which emulates the disk drives. The file
 *	   for drive A must be opened, or CP/M can't be booted.
 *	   Errors for opening one of the other 15 drives results
 *	   in a NULL pointer for fd in the dskdef structure,
 *	   so that this drive can't be used.
 *
 * revisions:
 *
 *	2007-04-12 rli: Stripped out non-disk related stuff. Made certain
 *	  drives are opened in binary mode. Reformatted. Renamed.
 *
 */

void MunkDisk_init_io(void)
{
  register int i;

  if ((*disks[0].fd = open(disks[0].fn, O_RDWR | O_BINARY )) == -1) {
    perror("file disks/drivea.cpm");
    exit(1);
  }

  for (i = 1; i <= 15; i++)
    if ((*disks[i].fd = open(disks[i].fn, O_RDWR | O_BINARY )) == -1)
      disks[i].fd = NULL;
}

/*
 *	This function stops the I/O handlers:
 *
 *	1. The files emulating the disk drives are closed.
 *
 * revisions:
 *
 *	2007-04-12 rli: Stripped non-disk-related stuff. Reformatted.
 *	  Renamed.
 */
void MunkDisk_Exit(void)
{
  register int i;

  for (i = 0; i <= 15; i++)
    if (disks[i].fd != NULL)
      close(*disks[i].fd);
}

/*
 *	I/O handler for read FDC drive:
 *	return the current drive
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdcd_in( BYTE Address, void *Cookie )
{
  return((BYTE) drive);
}

/*
 *	I/O handler for write FDC drive:
 *	set the current drive
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdcd_out(BYTE Address, BYTE data, void *Cookie )
{
  drive = data;
  return((BYTE) 0);
}

/*
 *	I/O handler for read FDC track:
 *	return the current track
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdct_in( BYTE Address, void *Cookie )
{
  return((BYTE) track);
}

/*
 *	I/O handler for write FDC track:
 *	set the current track
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdct_out( BYTE Address, BYTE data, void *Cookie )
{
  track = data;
  return((BYTE) 0);
}

/*
 *	I/O handler for read FDC sector
 *	return the current sector
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdcs_in( BYTE Address, void *Cookie )
{
  return((BYTE) sector);
}

/*
 *	I/O handler for write FDC sector:
 *	set the current sector
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdcs_out(BYTE Address, BYTE data, void *Cookie )
{
  sector = data;
  return((BYTE) 0);
}

/*
 *	I/O handler for read FDC command:
 *	always returns 0
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdco_in( BYTE Address, void *Cookie )
{
  return((BYTE) 0);
}

/*
 *	I/O handler for write FDC command:
 *	transfer one sector in the wanted direction,
 *	0 = read, 1 = write
 *
 *	The status byte of the FDC is set as follows:
 *	  0 - ok
 *	  1 - illegal drive
 *	  2 - illegal track
 *	  3 - illegal sector
 *	  4 - seek error
 *	  5 - read error
 *	  6 - write error
 *	  7 - illegal command to FDC
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE fdco_out(BYTE Address, BYTE data, void *Cookie )
{
  register long pos;

  if (disks[drive].fd == NULL) {
    status = 1;
    return((BYTE) 0);
  }
  if (track > disks[drive].tracks) {
    status = 2;
    return((BYTE) 0);
  }
  if (sector > disks[drive].sectors) {
    status = 3;
    return((BYTE) 0);
  }
  pos = (((long)track) * ((long)disks[drive].sectors) + sector - 1) << 7;
    if (lseek(*disks[drive].fd, pos, 0) == -1L) {
    status = 4;
    return((BYTE) 0);
  }
  switch (data) {
    case 0:			/* read */

      if (read(*disks[drive].fd, (char *) ram + (dmadh << 8) +
          dmadl, 128) != 128)
            status = 5;
      else
            status = 0;
      break;

    case 1:			/* write */
      if (write(*disks[drive].fd, (char *) ram + (dmadh << 8) +
          dmadl, 128) != 128)
            status = 6;
      else
            status = 0;
      break;
    default:		/* illegal command */
      status = 7;
      break;
  }

  return((BYTE) 0);
}

/*
 *	I/O handler for read FDC status:
 *	returns status of last FDC operation,
 *	0 = ok, else some error
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak, reformatted.
 */
static BYTE fdcx_in(BYTE Address, void *Cookie )
{
  return((BYTE) status);
}

/*
 *	I/O handler for write FDC status:
 *	no reaction
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak, reformatted.
 *
 */
static BYTE fdcx_out(BYTE Address, BYTE data, void *Cookie )
{
  data = data;
  return((BYTE) 0);
}

/*
 *	I/O handler for read lower byte of DMA address:
 *	return lower byte of current DMA address
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE dmal_in( BYTE Address, void *Cookie )
{
  return((BYTE) dmadl);
}

/*
 *	I/O handler for write lower byte of DMA address:
 *	set lower byte of DMA address
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE dmal_out(BYTE Address, BYTE data, void *Cookie )
{
  dmadl = data;
  return((BYTE) 0);
}

/*
 *	I/O handler for read higher byte of DMA address:
 *	return higher byte of current DMA address
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE dmah_in( BYTE Address, void *Cookie )
{
  return((BYTE) dmadh);
}

/*
 *	I/O handler for write higher byte of DMA address:
 *	set higher byte of the DMA address
 *
 * revisions:
 *
 *	2007-04-12 rli: Parameter list tweak. Reformatted.
 *
 */
static BYTE dmah_out(BYTE Address, BYTE data, void *Cookie )
{
  dmadh = data;
  return((BYTE) 0);
}

/*****************************************
 *
 *	rli's CODE
 *
 *****************************************/


/***
 *
 * MunkDisk_Register
 *
 *	This routine registers the I/O ports used by Udo Munk's disk
 *	simulation, then calls his initialization routine.
 *
 * revisions:
 *
 *	2007-04-12 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- Port addresses are hard-coded.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	- An error message is displayed and execution is aborted if
 *	  any of the ports could not be registered.
 *
 ***/

void MunkDisk_Register( void )
{
  int Failures;

 
  /* { fdcd_in, fdcd_out },		/* port 10 */

  Failures = WinIo_RegisterReader( 10, fdcd_in, NULL );
  Failures += WinIo_RegisterWriter( 10, fdcd_out, NULL );

  /* { fdct_in, fdct_out },		/* port 11 */

  Failures += WinIo_RegisterReader( 11, fdct_in, NULL );
  Failures += WinIo_RegisterWriter( 11, fdct_out, NULL );

  /* { fdcs_in, fdcs_out },		/* port 12 */

  Failures += WinIo_RegisterReader( 12, fdcs_in, NULL );
  Failures += WinIo_RegisterWriter( 12, fdcs_out, NULL );

  /* { fdco_in, fdco_out },		/* port 13 */

  Failures += WinIo_RegisterReader( 13, fdco_in, NULL );
  Failures += WinIo_RegisterWriter( 13, fdco_out, NULL );

  /* { fdcx_in, fdcx_out },		/* port 14 */

  Failures += WinIo_RegisterReader( 14, fdcx_in, NULL );
  Failures += WinIo_RegisterWriter( 14, fdcx_out, NULL );

  /* { dmal_in, dmal_out },		/* port 15 */

  Failures += WinIo_RegisterReader( 15, dmal_in, NULL );
  Failures += WinIo_RegisterWriter( 15, dmal_out, NULL );

  /* { dmah_in, dmah_out },		/* port 16 */

  Failures += WinIo_RegisterReader( 16, dmah_in, NULL );
  Failures += WinIo_RegisterWriter( 16, dmah_out, NULL );

  /* If we couldn't register any of our ports, complain and die.
   */

  if( Failures != 0 ) {
    fprintf( stderr, "MunkDisk: Could not register.\n" );
    exit( EXIT_FAILURE );
  }

  /* We got all our ports. Initialize and exit.
   */

  MunkDisk_init_io();
  return;

}
