/*****************************************
 *
 * WinCons.c
 *
 *	This file provides a VT52ish terminal emulation in the console 
 *	window.
 *
 * revisions:
 *
 *	2007-04-11 rli: original version.
 *
 *	2007-04-14 rli: Liberated DumbCons and started hacking it over.
 *
 *	2007-04-30 rli: Began fiddling with actual escape sequences.
 *
 *	2007-05-10 rli: Video attributes.
 *
 *****************************************/

#define _WIN32_WINNT 0x0500

/******************
 *
 *	INCLUDES
 *
 ******************/

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <wincon.h>

#include "sim.h"
#include "WinIo.h"
#include "WinCons.h"

/******************
 *
 *	GLOBALS intended for PRIVATE use
 *
 ******************/

/***
 *
 * WinCons_ScreenInfo_g
 *
 *	This structure holds a description of the screen buffer as of
 *	when the program was started. This is needed primarily to supply
 *	the attributes that should be used for reverse and normal
 *	video.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 ***/

CONSOLE_SCREEN_BUFFER_INFO WinCons_ScreenInfo_g;

/***
 *
 * WinCons_Reply_g
 *
 *	This structure holds an escape sequence reply being sent to the
 *	console port. Characters are sent from this string until they'v
 *	all been sent.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * fields:
 *
 *	- Head: Supplies the index of the next character to be sent. If
 *	  this equal Tail, the reply has been sent.
 *
 *	- Tail: Supplies the index of the character AFTER the last one
 *	  to be sent. 
 *
 *	- String: Holds the characters to be sent.
 *
 ***/

struct {
  int Head;
  int Tail;
  char String[ 5 ];
} WinCons_Reply_g = {
  0, 0, ""
};

/******************
 *
 *	CODE
 *
 ******************/

/***
 *
 * WinCons_ClearEOL
 *
 *	This routine clears the screen from (and including) the current
 *	cursor position to the end of the current line.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	- Both the character and attribute are written. The attribute is
 *	  the current attribute used for screen writes (i.e., reversing
 *	  video and clearing the screen will fill the screen with reversed
 *	  spaces).
 *
 ***/

void WinCons_ClearEOL( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO ScreenBufferInfo;
  int CharsToWrite;
  DWORD CharsWritten;

  /* Form a handle to the screen buffer.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out where the cursor is and what the current display attribute
   * is.
   */

  GetConsoleScreenBufferInfo( Console, &ScreenBufferInfo );

  /* Figure out how many characters are left in the current line. 
   *
   * I think the screen size is a count, so it will be 80 for an 80-column
   * window.
   *
   * I think the cursor position is zero-based, so it will be 0 if the
   * cursor is at the left-hand edge.
   *
   * Which means we should be able to just subtract the cursor position
   * from the window size to find out how many characters to write.
   */

  CharsToWrite = ScreenBufferInfo.dwSize.X - 
    ScreenBufferInfo.dwCursorPosition.X;

  /* Write the characters and the attributes.
   */

  FillConsoleOutputCharacter( Console, ' ', CharsToWrite,
    ScreenBufferInfo.dwCursorPosition, &CharsWritten );
  FillConsoleOutputAttribute( Console, ScreenBufferInfo.wAttributes,
    CharsToWrite, ScreenBufferInfo.dwCursorPosition, &CharsWritten );

  return;
}

/***
 *
 * WinCons_ClearEOS
 *
 *	This routine clears the screen from (and including) the current
 *	cursor position to the end of the screen.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	- Both the character and attribute are written. The attribute is
 *	  the current attribute used for screen writes (i.e., reversing
 *	  video and clearing the screen will fill the screen with reversed
 *	  spaces).
 *
 ***/

void WinCons_ClearEOS( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO ScreenBufferInfo;
  int CharsToWrite;
  DWORD CharsWritten;

  /* Form a handle to the screen buffer.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out where the cursor is and what the current display attribute
   * is.
   */

  GetConsoleScreenBufferInfo( Console, &ScreenBufferInfo );

  /* Figure out how many characters are left in the current line. 
   *
   * I think the screen size is a count, so it will be 80 for an 80-column
   * window.
   *
   * I think the cursor position is zero-based, so it will be 0 if the
   * cursor is at the left-hand edge.
   *
   * Which means we should be able to just subtract the cursor position
   * from the window size to find out how many characters to write.
   */

  CharsToWrite = ScreenBufferInfo.dwSize.X - 
    ScreenBufferInfo.dwCursorPosition.X;

  /* Find out how many lines are left, and add those lines to the character
   * count. We want to begin counting with the line AFTER the one containing
   * cursor. Again, I think we have a 1-based count and a 0-based coordinate.
   */

  CharsToWrite += ( ScreenBufferInfo.dwSize.Y - 
    ( ScreenBufferInfo.dwCursorPosition.Y + 1 ) ) * 
    ScreenBufferInfo.dwSize.X;

  /* Write the characters and the attributes.
   */

  FillConsoleOutputCharacter( Console, ' ', CharsToWrite,
    ScreenBufferInfo.dwCursorPosition, &CharsWritten );
  FillConsoleOutputAttribute( Console, ScreenBufferInfo.wAttributes,
    CharsToWrite, ScreenBufferInfo.dwCursorPosition, &CharsWritten );

  return;
}

/***
 *
 * WinCons_ClearScreen
 *
 *	This routine homes the cursor and clears the screen.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ClearScreen( void )
{
  WinCons_GotoXY( 0, 0 );
  WinCons_ClearEOS();
  return;
}

/***
 *
 * WinCons_ConIn
 *
 *	This routine waits for a character to arrive at the console
 *	and returns that character. The character is not echoed.
 *
 *	For special keys, like the arrow keys, Watcom gives us a NUL
 *	followed by an additional code describing the keys. For the
 *	arrow key's we'll send the appropriate escape sequence. Other
 *	keys, we'll just return the NUL and stuff the other character
 *	in the reply message.
 *
 *	NOTE:	As far as I can tell, there isn't a keystroke in Windows
 *		that will give you a NUL. In particular, ^@ gives you a
 *		NUL followed by a ^C.
 *
 *	The sequences are:
 *
 *	- Up Arrow:    0 72 -> ESC A
 *	- Down Arrow:  0 80 -> ESC B
 *	- Right Arrow: 0 77 -> ESC C
 *	- Left Arrow:  0 75 -> ESC D 
 *
 * revisions:
 *
 *	2007-04-11 rli: original version.
 *
 *	2007-04-14 rli: Name change.
 *
 *	2007-05-06 rli: If a character is available from the
 *	  reply string, send it.
 *
 *	2007-05-07 rli: Arrow keys.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address of the port being read. This is
 *	  not used.
 *
 *	- Cookie: Not used.
 *
 * informal parameters:
 *
 *	- WinCons_Reply_g: If we're sending a reply (like an answerback
 *	  string), it's stored here.
 *
 * return value:
 *
 *	- A character read from the console.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE WinCons_ConIn( BYTE Address, void *Cookie )
{
  int c;

  /* If there's a character available in the reply, send it.
   */

  if( WinCons_Reply_g.Head != WinCons_Reply_g.Tail ) {
    return WinCons_Reply_g.String[ WinCons_Reply_g.Head++ ];
  }

  /* The reply string is empty. Get something from the keyboard.
   */

  c = getch();

  /* If the character we got isn't a NULL, we don't have a special
   * character; just return it.
   */

  if( c != 0 ) return c;

  /* We have a special key. Deal with it.
   */

  c = getch();

  switch( c ) {

    /***
     *
     * 0 72: Up Arrow
     *
     * We need to translate this into ESC A. Put the A in the answerback
     * message and return the ESC.
     *
     ***/

    case 72:

      WinCons_Reply_g.String[ 0 ] = 'A';
      WinCons_Reply_g.Head = 0;
      WinCons_Reply_g.Tail = 1;
      return 27;

    /***
     *
     * 0 80: Down Arrow
     *
     * We need to translate this into ESC B. Put the B in the answerback
     * message and return the ESC.
     *
     ***/

    case 80:

      WinCons_Reply_g.String[ 0 ] = 'B';
      WinCons_Reply_g.Head = 0;
      WinCons_Reply_g.Tail = 1;
      return 27;

    /***
     *
     * 0 77: Right Arrow
     *
     * We need to translate this into ESC C. Put the C in the answerback
     * message and return the ESC.
     *
     ***/

    case 77:

      WinCons_Reply_g.String[ 0 ] = 'C';
      WinCons_Reply_g.Head = 0;
      WinCons_Reply_g.Tail = 1;
      return 27;

    /***
     *
     * 0 75: Left Arrow
     *
     * We need to translate this into ESC D. Put the D in the answerback
     * message and return the ESC.
     *
     ***/

    case 75:

      WinCons_Reply_g.String[ 0 ] = 'D';
      WinCons_Reply_g.Head = 0;
      WinCons_Reply_g.Tail = 1;
      return 27;

  }

  /* This is a keystroke we don't recognize. Stuff the second character
   * in the answerback and return the NUL; this at least gives CP/M some
   * hope of handling it.
   */

  WinCons_Reply_g.String[ 0 ] = c;
  WinCons_Reply_g.Head = 0;
  WinCons_Reply_g.Tail = 1;
  return 0;

}

/***
 *
 * WinCons_ConSt
 *
 *	This routine checks to see if a character is available at the
 *	console.
 *
 * revisions:
 *
 *	2007-04-11 rli: original version.
 *
 *	2007-04-14 rli: Name change.
 *
 *	2007-05-06 rli: Check the reply string.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address of the port being read. This is
 *	  not used.
 *
 *	- Cookie: Not used.
 *
 * informal parameters:
 *
 *	- WinCons_Reply_g: If we're trying to send a reply (like an
 *	  answerback string), it's stored here. If any characters are
 *	  available from this reply, one is available from the console.
 *
 * return value:
 *
 *	- 0: No characters are available from the console.
 *
 *	- 0xff: A character is available from the console.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE WinCons_ConSt( BYTE Address, void *Cookie )
{

  /* Look to see if a character is available from the reply.
   */

  if( WinCons_Reply_g.Head != WinCons_Reply_g.Tail ) {
    return 0xff;
  }

  /* The reply string is empty; check the keyboard.
   */

  if( kbhit() == 0 ) return 0;
  return 0xff;
}

/***
 *
 * WinCons_DisableCtrlC
 *
 *	This routine disablles CTRL+C handling for the console window.
 *	CTRL+BREAK is still handled.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 *	2007-05-06 rli: Turn off line wrapping.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_DisableCtrlC( void )
{
  HANDLE Console;

  /* Get a handle to stdin's connection to the console window.
   */

  Console = GetStdHandle( STD_INPUT_HANDLE );

  /* We want to disable special handling for ^C, but we can't individually
   * disable it; we have to specify *all* input modes. The only one that
   * looks particularly useful to me is Quick Edit mode, which I use a lot.
   */

  SetConsoleMode( Console, ENABLE_QUICK_EDIT_MODE );

  /* Turn off line wrapping for VT-52 compatibility. We still want Windows
   * to deal with LF, BS, TAB, etc.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  SetConsoleMode( Console, ENABLE_PROCESSED_OUTPUT );

  return;
}

/***
 *
 * WinCons_GotoXY
 *
 *	This routine moves the cursor to a specific position in the window.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 *	2007-05-06 rli: Don't move the cursor along an axis if the coordinate
 *	  for that axis is off-screen. RT-11 K52 relies on this behavior;
 *	  I don't know if any CP/M software does.
 *
 * formal parameters:
 *
 *	- X: Supplies the column to which the cursor should be moved. This
 *	  is not range-checked.
 *
 *	- Y: Supplies the row to which the cursor should be moved. This is
 *	  not range-checked.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_GotoXY( int X, int Y )
{
  HANDLE Console;
  COORD Position;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the screen.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find the current cursor position and size of the window.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* If the X coordinate is off-screen, don't move in that axis.
   */

  if( ( X < 0 ) || ( X > ( Info.dwSize.X - 1 ) ) ) {
    X = Info.dwCursorPosition.X;
  }

  /* If the Y coordinate is off-screen, don't move in that axis.
   */

  if( ( Y < 0 ) || ( Y > ( Info.dwSize.Y - 1 ) ) ) {
    Y = Info.dwCursorPosition.Y;
  }

  /* We've clipped the coordinates; move the cursor.
   */

  Position.X = X;
  Position.Y = Y;
  SetConsoleCursorPosition( Console, Position );

  return;
}

/***
 *
 * WinCons_NormalVideo
 *
 *	This routine changes the character attributes that will be used
 *	for subsequent characters such that the characters will be displayed
 *	in normal video.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Contains information describing the screen
 *	  after it was resized. The character attributes are used.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_NormalVideo( void )
{
  HANDLE Console;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  SetConsoleTextAttribute( Console, WinCons_ScreenInfo_g.wAttributes );

  return;
}

/***
 *
 * WinCons_GetScreenInfo
 *
 *	This routine grabs info like the size of the window and current
 *	video parameters and stashes them somewhere globally.
 *
 *	Since the size of the window can change, I'm now primarily
 *	interested in the video parameters; reverse and normal video
 *	are based on the window parameters when we were entered.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Screen info is stashed here.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_GetScreenInfo( void )
{
  HANDLE Console;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Get the screen size.
   */

  GetConsoleScreenBufferInfo( Console, &WinCons_ScreenInfo_g );
}

/***
 *
 * WinCons_Register
 *
 *	This routine registers the dumb console. It occupies two read
 *	ports (0 and 1) and one write port (1).
 *
 * revisions:
 *
 *	2007-04-11 rli: original version.
 *
 *	2007-04-14 rli: Name change.
 *
 *	2007-04-20 rli: Disable CTRL+C handling.
 *
 *	2007-05-10 rli: Retired Resize in favor of SetSize.
 *
 *	2007-05-10 rli: Forgot I need global screen parameters, which was
 *	  being done by Resize.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- port numbers are hard-coded.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	- An error message is displayed and program execution stops if
 *	  a port could not be registered.
 *
 ***/

void WinCons_Register( void )
{
  int Failures;

  Failures = WinIo_RegisterReader( 0, WinCons_ConSt, NULL );
  Failures += WinIo_RegisterReader( 1, WinCons_ConIn, NULL );
  Failures += WinIo_RegisterWriter( 1, WinCons_ConOt, NULL );

  if( Failures != 0 ) {
    fprintf( stderr, "WinCons: Could not register at port 0.\n" );
    exit( EXIT_FAILURE );
  }

  /* Set up the console window.
   */

  WinCons_SetSize( 80, 24 );
  WinCons_GetScreenInfo();
  WinCons_DisableCtrlC();

  return;
}

/***
 *
 * WinCons_ReverseVideo
 *
 *	This routine changes the character attributes that will be used
 *	for subsequent characters such that the characters will be displayed
 *	in reverse video.
 *
 * revisions:
 *
 *	2007-04-20 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Contains information describing the screen
 *	  after it was resized. The character attributes are used.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ReverseVideo( void )
{
  HANDLE Console;
  WORD NewAttributes;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* We're supposed to have an attribute that can be used to reverse
   * the video (COMMON_LVB_REVERSE_VIDEO), but it's not available in
   * the header files provided with Watcom. Instead, I'll be doing evil
   * things to the attributes.
   *
   * Looking over wincon.h, I see that the foreground attributes are in
   * the least-significant four bits of the attribute and the background
   * in the next four bits. Switch them around. Probably ought to copy over
   * attribute bits that Watcom doesn't know about.
   */

  NewAttributes = ( ( WinCons_ScreenInfo_g.wAttributes & 0xf ) << 4 ) |
     ( ( WinCons_ScreenInfo_g.wAttributes & 0xf0 ) >> 4 ) |
     ( WinCons_ScreenInfo_g.wAttributes & 0xff00 );
  SetConsoleTextAttribute( Console, NewAttributes );

  return;
}

/***
 *
 * WinCons_MoveCursorUp
 *
 *	This routine moves the cursor up one line unless it's at the top of
 *	of the screen; if it's at the top of the screen, the move is
 *	ignored.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_MoveCursorUp( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  GetConsoleScreenBufferInfo( Console, &Info );
  if( Info.dwCursorPosition.Y == 0 ) return;
  Info.dwCursorPosition.Y--;
  SetConsoleCursorPosition( Console, Info.dwCursorPosition );
  return;

}

/***
 *
 * WinCons_MoveCursorLeft
 *
 *	This routine moves the cursor left one column unless it's at the 
 *	edge of the window; if it's at the edge, the move is ignored.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_MoveCursorLeft( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  GetConsoleScreenBufferInfo( Console, &Info );
  if( Info.dwCursorPosition.X == 0 ) return;
  Info.dwCursorPosition.X--;
  SetConsoleCursorPosition( Console, Info.dwCursorPosition );
  return;

}

/***
 *
 * WinCons_MoveCursorRight
 *
 *	This routine moves the cursor right one column unless it's at the 
 *	edge of the window; if it's at the edge, the move is ignored.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_MoveCursorRight( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  GetConsoleScreenBufferInfo( Console, &Info );
  if( !( Info.dwCursorPosition.X < ( Info.dwSize.X - 1 ) ) ) return;
  Info.dwCursorPosition.X++;
  SetConsoleCursorPosition( Console, Info.dwCursorPosition );
  return;

}

/***
 *
 * WinCons_MoveCursorDown
 *
 *	This routine moves the cursor down one row unless it's at the 
 *	bottom of the window; if it's at the bottom, the move is ignored.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_MoveCursorDown( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  Console = GetStdHandle( STD_OUTPUT_HANDLE );
  GetConsoleScreenBufferInfo( Console, &Info );
  if( !( Info.dwCursorPosition.Y < ( Info.dwSize.Y - 1 ) ) ) return;
  Info.dwCursorPosition.Y++;
  SetConsoleCursorPosition( Console, Info.dwCursorPosition );
  return;

}

/***
 *
 * WinCons_ScrollDown
 *
 *	This routine scrolls the console window down one line. It is used
 *	when a reverse index is performed while the cursor is on the tope
 *	line of the screen.
 *
 *	The top line of the screen is filled with normal video blanks.
 *
 *	This is pretty much copied verbatim from the MSDN scrolling
 *	example.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Supplies the character attributes to be
 *	  used for normal video.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ScrollDown( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;
  SMALL_RECT ScrollRect;
  SMALL_RECT ClipRect;
  CHAR_INFO FillChar;
  COORD Dest;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Get the screen size.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* Our scrolling region is from the top line to the next-to-bottom line.
   */

  ScrollRect.Top = 0;
  ScrollRect.Bottom = Info.dwSize.Y - 2;
  ScrollRect.Left = 0;
  ScrollRect.Right = Info.dwSize.X - 1;

  /* The destination is the second line.
   */

  Dest.X = 0;
  Dest.Y = 1;

  /* The entire window is affected.
   */

  ClipRect.Top = 0;
  ClipRect.Bottom = Info.dwSize.Y - 1;
  ClipRect.Left = 0;
  ClipRect.Right = Info.dwSize.X - 1;

  /* We want normal video spaces inserted in the vacant row.
   */

  FillChar.Attributes = WinCons_ScreenInfo_g.wAttributes;
  FillChar.Char.AsciiChar = ' ';

  /* Do the scroll.
   */

  ScrollConsoleScreenBuffer( Console, &ScrollRect, &ClipRect, Dest, &FillChar );

  return;

}

/***
 *
 * WinCons_InsertLine
 *
 *	This routine inserts a line at the current cursor position. After
 *	the line is inserted, the cursor is moved to the left margin.
 *	Normal video spaces are inserted in the new line.
 *
 *	This is an H19/Z19 function used by WordStar 4.0. The manual for
 *	the Z19 is not clear about whether the inserted blanks are normal
 *	video or use the currently active video mode. I don't have a Z19
 *	to check.
 *
 * revisions:
 *
 *	2007-05-12 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Supplies the character attributes to be
 *	  used for normal video.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_InsertLine( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;
  SMALL_RECT ScrollRect;
  SMALL_RECT ClipRect;
  CHAR_INFO FillChar;
  COORD Dest;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Get the screen size.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* If the cursor is in the bottom line, just do a carriage return and
   * clear to end of line. I don't know what Windows will do if I try
   * to scroll the bottom line down.
   */

  if( Info.dwCursorPosition.Y == Info.dwSize.Y - 1 ) {
    printf( "\r" );
    WinCons_ClearEOL();
    return;
  }

  /* The cursor is not in the bottom line. Our scrolling region is from the 
   * current line to the next-to-bottom line.
   */

  ScrollRect.Top = Info.dwCursorPosition.Y;
  ScrollRect.Bottom = Info.dwSize.Y - 2;
  ScrollRect.Left = 0;
  ScrollRect.Right = Info.dwSize.X - 1;

  /* The destination is the line after the current line.
   */

  Dest.X = 0;
  Dest.Y = ScrollRect.Top + 1;

  /* The portion of the window from the current line to the bottom is
   * affected. I think Windows uses this information to figure out which
   * part of the window needs to be redrawn.
   */

  ClipRect.Top = ScrollRect.Top;
  ClipRect.Bottom = Info.dwSize.Y - 1;
  ClipRect.Left = 0;
  ClipRect.Right = Info.dwSize.X - 1;

  /* We want normal video spaces inserted in the vacant row.
   */

  FillChar.Attributes = WinCons_ScreenInfo_g.wAttributes;
  FillChar.Char.AsciiChar = ' ';

  /* Do the scroll.
   */

  ScrollConsoleScreenBuffer( Console, &ScrollRect, &ClipRect, Dest, &FillChar );

  /* We need to move the cursor to the beginning of the inserted line. The
   * easiest way to do this is by printing a carriage return.
   */

  printf( "\r" );

  return;

}

/***
 *
 * WinCons_DeleteLine
 *
 *	This routine deletes the line at the cursor position. The portion
 *	of the screen below the cursor is scrolled up and blanks fill the
 *	bottom line.
 *
 *	This is an H19/Z19 function used by WordStar 4.0. The manual for
 *	the Z19 is not clear about whether the inserted blanks are normal
 *	video or use the currently active video mode. I don't have a Z19
 *	to check.
 *
 *	The Z19 manual also doesn't say what happens to the cursor position
 *	after the line is deleted. I'm moving it to the left margin, as
 *	insert does.
 *
 * revisions:
 *
 *	2007-05-12 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_ScreenInfo_g: Supplies the character attributes to be
 *	  used for normal video.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_DeleteLine( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;
  SMALL_RECT ScrollRect;
  SMALL_RECT ClipRect;
  CHAR_INFO FillChar;
  COORD Dest;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Get the screen size.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* If the cursor is in the bottom line, just do a carriage return and
   * clear to end of line. Deleting this line would involve scrolling
   * zero lines; I don't know how Windows would like that.
   */

  if( Info.dwCursorPosition.Y == Info.dwSize.Y - 1 ) {
    printf( "\r" );
    WinCons_ClearEOL();
    return;
  }

  /* The cursor is not in the bottom line. Our scrolling region is from the 
   * the line following the current line to the bottom line.
   */

  ScrollRect.Top = Info.dwCursorPosition.Y + 1;
  ScrollRect.Bottom = Info.dwSize.Y - 1;
  ScrollRect.Left = 0;
  ScrollRect.Right = Info.dwSize.X - 1;

  /* The destination is the current line.
   */

  Dest.X = 0;
  Dest.Y = Info.dwCursorPosition.Y;

  /* The portion of the window from the current line to the bottom is
   * affected. I think Windows uses this information to figure out which
   * part of the window needs to be redrawn.
   */

  ClipRect.Top = Info.dwCursorPosition.Y;
  ClipRect.Bottom = Info.dwSize.Y - 1;
  ClipRect.Left = 0;
  ClipRect.Right = Info.dwSize.X - 1;

  /* We want normal video spaces inserted in the vacant row.
   */

  FillChar.Attributes = WinCons_ScreenInfo_g.wAttributes;
  FillChar.Char.AsciiChar = ' ';

  /* Do the scroll.
   */

  ScrollConsoleScreenBuffer( Console, &ScrollRect, &ClipRect, Dest, &FillChar );

  /* Move the cursor to the beginning of the new line. The easiest way to
   * do this is by printing a carriage return.
   */

  printf( "\r" );

  return;

}

/***
 *
 * WinCons_ReverseIndex
 *
 *	This is a reverse linefeed. The cursor is moved up one line, unless
 *	it is at the top of the window. If it's at the top of the window,
 *	the window is scrolled down.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ReverseIndex( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out where the cursor is.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* If the cursor's at the top, scroll the window.
   */

  if( Info.dwCursorPosition.Y == 0 ) {
    WinCons_ScrollDown();
    return;
  }

  /* The cursor's not at the top. Move it up one.
   */

  WinCons_MoveCursorUp();
  return;

}

/***
 *
 * WinCons_Identify
 *
 *	This routine sends a reply that indicates the terminal is a
 *	VT-52.
 *
 *	The string sent is ESC / K, indicating we're a VT-52 without
 *	a copier.
 *
 * revisions:
 *
 *	2007-05-06 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_Reply_g: The reply string is stored here.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_Identify( void )
{
  WinCons_Reply_g.String[ 0 ] = 27;
  WinCons_Reply_g.String[ 1 ] = '/';
  WinCons_Reply_g.String[ 2 ] = 'K';
  WinCons_Reply_g.Head = 0;
  WinCons_Reply_g.Tail = 3;
  return;
} 

/***
 *
 * WinCons_ShowAttributes
 *
 *	Generates a reply showing the current video attributes. The reply
 *	is an escape sequence of the form ESC b attr; that is, a sequence
 *	that can be used to set the attributes.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_Reply_g: The reply is stored here.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ShowAttributes( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out where the cursor is.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* Generate the reply.
   */

  WinCons_Reply_g.String[ 0 ] = 27;
  WinCons_Reply_g.String[ 1 ] = 'b';
  WinCons_Reply_g.String[ 2 ] = Info.wAttributes;
  WinCons_Reply_g.Head = 0;
  WinCons_Reply_g.Tail = 3;
  return;
}

/***
 *
 * WinCons_SetAttributes
 *
 *	Changes the attributes that will be used to display subsequent
 *	characters.
 *
 *	The notion of "normal" and "reverse" video does not change; you
 *	can return to the original video attributes by issuing ESC U.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Attributes: Supplies the attributes to be used. These are
 *	  stuffed into the least-significant byte of the video attributes;
 *	  that is, it changes the colors, but leaves the other attributes
 *	  unchanged.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_SetAttributes( char Attributes )
{

  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Get the current attributes so we can fold the color in.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* Update the color; carefully, since we've not said whether the
   * attributes are signed or unsigned.
   */

  Info.wAttributes = ( Info.wAttributes & ~0xff ) |
    ( (int)Attributes & 0xff );

  /* Update the attributes.
   */

  SetConsoleTextAttribute( Console, Info.wAttributes );

  return;
}

/***
 *
 * WinCons_ShowSize
 *
 *	Returns an escape sequence of the form ESC Y row+32 col+32 that
 *	moves the cursor to the bottom right corner of the screen. This
 *	is intended to allow a CP/M program to detect the size of the 
 *	console window.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinCons_Reply_g: The escape sequence is stored here.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_ShowSize( void )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out how big the console window is.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* Construct the escape sequence.
   */

  WinCons_Reply_g.String[ 0 ] = 27;
  WinCons_Reply_g.String[ 1 ] = 'Y';
  WinCons_Reply_g.String[ 2 ] = ( Info.dwSize.Y - 1 ) + 32;
  WinCons_Reply_g.String[ 3 ] = ( Info.dwSize.X - 1 ) + 32;
  WinCons_Reply_g.Head = 0;
  WinCons_Reply_g.Tail = 4;
  return;

}

/***
 *
 * WinCons_SetSizeBigger
 *
 *	Resize the console window so it's bigger.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Cols: The number of columns that should be in the window.
 *
 *	- Rows: The number of rows that should be in the window.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_SetSizeBigger( int Cols, int Rows )
{
  HANDLE Console;
  SMALL_RECT WindowSize;
  COORD BufferSize;
  int Status;

  /* Get a handle to stdout's current screen buffer.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Since we're making the window bigger, we have to resize the screen
   * buffer first.
   *
   * If the buffer resize fails, don't change the window size.
   */

  BufferSize.X = (SHORT)Cols;
  BufferSize.Y = (SHORT)Rows;
  Status = SetConsoleScreenBufferSize( Console, BufferSize );
  if( Status == 0 ) return;

  /* The buffer's been resized. Resize the window.
   */

  WindowSize.Top = 0;
  WindowSize.Left = 0;
  WindowSize.Bottom = Rows - 1;
  WindowSize.Right = Cols - 1;

  SetConsoleWindowInfo( Console, TRUE, &WindowSize );

  return;
}

/***
 *
 * WinCons_SetSizeSmaller
 *
 *	Resize the console window so it's Smaller.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Cols: The number of columns that should be in the window.
 *
 *	- Rows: The number of rows that should be in the window.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_SetSizeSmaller( int Cols, int Rows )
{
  HANDLE Console;
  SMALL_RECT WindowSize;
  COORD BufferSize;
  int Status;

  /* Get a handle to stdout's current screen buffer.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Since we're making the window smaller, we have to change the window
   * size first. If the window size doesn't change, don't resize the buffer.
   */

  WindowSize.Top = 0;
  WindowSize.Left = 0;
  WindowSize.Bottom = Rows - 1;
  WindowSize.Right = Cols - 1;

  Status = SetConsoleWindowInfo( Console, TRUE, &WindowSize );
  if( Status == 0 ) return;
  
  /* The window's been resized. Resize the screen buffer.
   */

  BufferSize.X = (SHORT)Cols;
  BufferSize.Y = (SHORT)Rows;
  SetConsoleScreenBufferSize( Console, BufferSize );

  return;
}

/***
 *
 * WinCons_SetSize
 *
 *	Change the size of the window. Precisely how this is done
 *	depends on whether we're making it bigger or smaller; there are
 *	two syscalls involved, each of which will fail if you're trying to
 *	make it smaller than the other.
 *
 * revisions:
 *
 *	2007-05-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Cols: The number of columns that should be in the window.
 *
 *	- Rows: The number of rows that should be in the window.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void WinCons_SetSize( int Cols, int Rows )
{
  HANDLE Console;
  CONSOLE_SCREEN_BUFFER_INFO Info;

  /* Get a handle to the console window.
   */

  Console = GetStdHandle( STD_OUTPUT_HANDLE );

  /* Find out how big the console window is.
   */

  GetConsoleScreenBufferInfo( Console, &Info );

  /* I've just realized that I don't really know what "bigger" means. What
   * if one coordinate is bigger, but the other is smaller? We'd better
   * tackle them one at at time.
   *
   * Let's do the X coordinate first.
   */

  if( Cols < Info.dwSize.X ) WinCons_SetSizeSmaller( Cols, Info.dwSize.Y );
  if( Cols > Info.dwSize.X ) WinCons_SetSizeBigger( Cols, Info.dwSize.Y );

  /* Now do the Y coordinate.
   */

  if( Rows < Info.dwSize.Y ) WinCons_SetSizeSmaller( Cols, Rows );
  if( Rows > Info.dwSize.Y ) WinCons_SetSizeBigger( Cols, Rows );

  return;
} 

/***
 *
 * WinCons_HandleEscape
 *
 *	This routine handles escape sequences. It is called by ConsOt
 *	when we're in the middle of a sequence so it can take a whack
 *	at the character.
 *
 * revisions:
 *
 *	2007-04-30 rli: Original do-nothing version.
 *
 *	2007-05-12 rli: Accept H19 escape sequences for reverse and
 *	  normal video as synonyms for the undocumented K52 sequences.
 *
 *	2007-05-12 rli: H19 insert line. H19 delete line.
 *
 * formal parameters:
 *
 *	- Victim: Supplies the character to be processed.
 *
 * informal parameters:
 *
 *	None.
 *
 * return value:
 *
 *	- 0: The sequence is complete; normal processing of characters
 *	  should commence.
 *
 *	- else: Further characters remain in this sequence; escape
 *	  processing remains in effect.
 *
 * other effects:
 *
 *	none.
 *
 ***/

int WinCons_HandleEscape( char Victim )
{
  static int CharsLeft = 0;
  static char Command;
  static char Parameters[ 2 ];

  /* If we're not in the middle of an extended sequence like cursor
   * movement, just handle the new command.
   */

  if( CharsLeft == 0 ) {

    /* Dispatch on the command.
     */

    switch( Victim ) {

      /***
       *
       * ESC A - Move up one line.
       *
       ***/

      case 'A':
 
        WinCons_MoveCursorUp();
        return 0;

      /***
       *
       * ESC B - Move down one line.
       *
       ***/

      case 'B':
 
        WinCons_MoveCursorDown();
        return 0;

      /***
       *
       * ESC C - Move right one position.
       *
       ***/

      case 'C':
 
        WinCons_MoveCursorRight();
        return 0;

      /***
       *
       * ESC D - Move left one position.
       *
       ***/

      case 'D':
 
        WinCons_MoveCursorLeft();
        return 0;

      /***
       *
       * ESC H - Home the cursor.
       *
       ***/

      case 'H':

        WinCons_GotoXY( 0, 0 );
        return 0;

      /***
       *
       * ESC I - Reverse Index.
       *
       ***/

      case 'I':

        WinCons_ReverseIndex();
        return 0;

      /***
       *
       * ESC J - Clear to end of screen.
       *
       ***/

      case 'J':
 
        WinCons_ClearEOS();
        return 0;

      /***
       *
       * ESC K - Clear to end of line.
       *
       ***/

      case 'K':

        WinCons_ClearEOL();
        return 0;

      /***
       *
       * ESC L - Insert line
       *
       * This H19 sequence inserts a line at the current cursor position.
       * It's used by WordStar 4.0.
       *
       ***/

      case 'L':

        WinCons_InsertLine();
        return 0;

      /***
       *
       * ESC M - Delete line
       *
       * This H19 sequence deletes the line at the current cursor position.
       * It's used by WordStar 4.0.
       *
       ***/

      case 'M':

        WinCons_DeleteLine();
        return 0;

      /***
       *
       * ESC T - Reverse Video
       * ESC p - Reverse video
       *
       * RT-11 K52 uses this for highlighting. A real VT52 ignores it. I
       * really don't know whether ESC T is supposed to start or stop
       * highlighting.
       * 
       * ESC p is the Heathkit H19 sequence.
       *
       ***/

      case 'T':
      case 'p':

        WinCons_ReverseVideo();
        return 0;

      /***
       *
       * ESC U - Normal video
       * ESC q - Normal video
       *
       * RT-11 K52 uses this for highlighting. A real VT52 ignores it. I
       * really don't know whether ESC U is supposed to start or stop
       * highlighting.
       *
       * ESC q is the Heathkit H19 sequence.
       *
       ***/

      case 'U':
      case 'q':

        WinCons_NormalVideo();
        return 0;

      /***
       *
       * ESC Y - Cursor movement
       *
       * We need to collect two more characters, the row and the column.
       *
       ***/

      case 'Y':

        Command = 'Y';
        CharsLeft = 2;
        return 1;

      /***
       *
       * ESC Z - Identify
       *
       * Returns an escape sequence identifying the features that are
       * available.
       *
       ***/

      case 'Z':

        WinCons_Identify();
        return 0;

      /***
       *
       * ESC a - Return an escape sequence describing the current
       * video attributes. VT52 can't do this.
       *
       ***/

      case 'a':

        WinCons_ShowAttributes();
        return 0;

      /***
       *
       * ESC b - Set video attribute
       *
       * We need to collect a character that will be used to update
       * the video attributes. VT52 can't do this.
       *
       ***/

      case 'b':

	Command = 'b';
        CharsLeft = 1;
        return 1;

      /***
       *
       * ESC c - Report size of console window.
       *
       * Returns an ESC Y sequence that moves the cursor to the lower left
       * corner of the screen. VT52 can't do this.
       *
       ***/

      case 'c':

        WinCons_ShowSize();
        return 0;

      /***
       *
       * ESC d row+32 col+32 - Set size of console window.
       *
       * Resize the console window such that ESC Y row+32 col+32 will
       * move the cursor to the lower right corner of the window. We need
       * to collect two more characters. VT52 can't do this.
       *
       ***/

      case 'd':

        Command = 'd';
        CharsLeft = 2;
        return 1;

      /***
       *
       * ESC z - Reset terminal
       *
       * Resize the terminal to 80x24, clear the screen, home the cursor,
       * and return to normal video.
       *
       * This is an H19 sequence, but I've got my own idea about what
       * precisely it should do. I've implemented it primarily is a short
       * sequence that can be used by a WordStar that changes the console
       * window size to get back to normal.
       *
       ***/

      case 'z':

        WinCons_SetSize( 80, 24 );
        WinCons_ClearScreen();
        WinCons_NormalVideo();
        return 0;

      /***
       *
       * Unknown sequence - Exit, turning off escape processing.
       *
       ***/

      default:
        return 0;

    }
  }

  /* Since we got here, there is at least one character left in the
   * sequence. Decrement the count of remaining characters and store
   * the character.
   *
   * When we find the end of the sequence, the terminating character for
   * the sequence will be stored in Parameters[ 0 ]. If there was an
   * additional character (like for cursor positioning), it will be
   * in Parameters[ 1 ].
   */

  CharsLeft--;
  Parameters[ CharsLeft ] = Victim;

  /* If that was not the final character, keep escape processing going.
   */

  if( CharsLeft != 0 ) {
    return 1;
  }

  /* That was the final character. Process the command.
   */

  switch( Command ) {

    /***
     *
     * ESC Y row+32 col+32 - Cursor movement.
     *
     * Row is in Parameters[ 1 ] and col is in Parameters[ 0 ].
     *
     * Need to be a bit careful to deal with large window sizes. We want
     * row 0 to refer to row 224, not row -32.
     *
     ***/

    case 'Y':

      WinCons_GotoXY( ( (int)Parameters[ 0 ] - 32 ) & 0xff, 
        ( (int)Parameters[ 1 ] - 32 ) & 0xff );
      return 0;

    /***
     *
     * ESC b attr - Set video attributes
     *
     * Sets the video attributes to be used for subsequent characters.
     * The attr byte is used as-is. It's in Parameters[ 0 ].
     *
     ***/

    case 'b':

      WinCons_SetAttributes( Parameters[ 0 ] );
      return 0;

    /***
     *
     * ESC d row+32 col+32 - Set size of console window.
     *
     * Row is in Parameters[ 1 ] and col is in Parameters[ 0 ]. We need
     * to give SetSize counts, so we have to add one.
     *
     ***/

    case 'd':

      WinCons_SetSize( ( ( (int)Parameters[ 0 ] - 32 ) & 0xff ) + 1, 
        ( ( (int)Parameters[ 1 ] - 32 ) & 0xff ) + 1 );
      return 0;
 
    /***
     *
     * Unknown sequence - ignore it and turn off escape processing.
     *
     ***/

    default:
      return 0;

  }

}

/***
 *
 * WinCons_ConOt
 *
 *	This routine displays a character on the console.
 *
 * revisions:
 *
 *	2007-04-11 rli: original version.
 *
 *	2007-04-14 rli: Name change.
 *
 *	2007-04-20 rli: Begin fiddling with control processing.
 *
 *	2007-04-30 rli: Pawn ESCape processing off on HandleEscape.
 *
 *	2007-05-06 rli: Hmm. TAB is among the characters processed by
 *	  Windows. Wonder what it does.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address of the port being written. This
 *	  is not used.
 *
 *	- Value: Supplies the character to write.
 *
 *	- Cookie: Not used.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	- Zero is always returned.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE WinCons_ConOt( BYTE Address, BYTE Value, void *Cookie )
{
  static int CharsLeft = 0;
  static int EscapeSeen = 0;
  static int X;
  static int Y;

  /* If we're in the middle of an escape sequence, pass the character
   * on for further processing.
   */

  if( EscapeSeen ) {
    EscapeSeen = WinCons_HandleEscape( Value );
    return 0;
  }

  /* If we're not collecting parameters for a control sequence, we
   * need to print the character.
   */

  if( CharsLeft == 0 ) {

    /* Control characters require special processing. I'm going to treat
     * all 8-bit characters as printable, since I don't know the rules
     * for that half of the character set (CP/M usually strips bit 7,
     * anyway).
     */

    if( ( Value < 32 ) || ( Value == 127 ) ) {

      switch( Value ) {

        /* BS, LF, TAB, and CR can be handled by Windows.
         */

        case 8: case 10: case 13: case 9:
          putch( Value );
          return 0;

        /* FF homes the cursor and clears the screen.
         */

        case 12:
          WinCons_ClearScreen();
          return 0;

        /* ^B clears to end of line.
         */

        case 2:
          WinCons_ClearEOL();
          return 0;

        /* ^C clears to end of screen.
         */

        case 3:
          WinCons_ClearEOS();
          return 0;

        /* ^O (Shift In) turns on reverse video.
         */

        case 15:
          WinCons_ReverseVideo();
          return 0;

        /* ^N (Shift Out) returns to normal video.
         */

        case 14:
          WinCons_NormalVideo();
          return 0;

        /* ^Axy will be cursor movement. When we see ^A, remember that
         * we have a couple of characters left.
         */

        case 1:
          CharsLeft = 2;
          return 0;

        /* ^[ (Escape) begins escape processing. Subsequent characters
         * will be given to HandleEscape until further notice.
         */

        case 27:
          EscapeSeen = 1;
          return 0;

    
      }

      /* Ignore anything we haven't handled.
       */

      return 0;

    }

    /* We must have a printable character. Print it and exit.
     */

    putch( Value );
    return 0;

  }

  /* We're in the middle of a cursor movement sequence. CharsLeft will be:
   *
   *   - 2 = Collect X coordinate, expect one more.
   *   - 1 = Collect Y coordinat, move cursor, back to normal.
   */

  if( CharsLeft == 2 ) {
    X = Value;
    CharsLeft = 1;
    return 0;
  }

  Y = Value;
  WinCons_GotoXY( X, Y );

  CharsLeft = 0;
  return 0;

}

