/********************
 *
 * rli_io.c
 *
 *	This file contains the I/O glue between the simulation and the
 *	I/O system. The simulation calls these functions to perform I/O.
 *
 * revisions:
 *
 *	2007-04-08 rli: Initial do-nothing version.
 *
 *	2007-04-09 rli: Starting to build an I/O infrastructure.
 *
 *	2007-04-11 rli: DumbCons.
 *
 *	2007-04-12 rli: MunkStubs.
 *
 *	2007-04-12 rli: MunkDisk.
 *
 *	2007-04-14 rli: Trade WinCons for DumbCons. Register it last so the
 *	  size of the window won't be changed if there are errors before
 *	  it is registered.
 *
 ********************/

/***************
 *
 *	INCLUDES
 *
 ***************/

#include <stdio.h>
#include <stdlib.h>
#include "sim.h"
#include "simglb.h"
#include "WinIo.h"
#include "WinCons.h"
#include "MunkStubs.h"
#include "MunkDisk.h"

/***************************
 *
 *	GLOBALS intended for PRIVATE use
 *
 ***************************/

/***
 *
 * WinIo_PortReaders_g
 *
 *	This array contains pointers to the input port handlers. There is
 *	one handler for each possible port. When a port is read, the
 *	appropriate pointer is used to call the reader.
 *
 *	I/O handlers can be registered using WinIo_RegisterReader; that
 *	routine updates a port's entry. At system initialization, all
 *	entries are initialized to point to WinIo_DefaultReader, which
 *	aborts the simulation due to an I/O trap.
 *
 * revisions:
 *
 *	2007-04-09 rli: original version.
 *
 ***/

WinIo_PortReader_t WinIo_PortReaders_g[ 256 ];

/***
 *
 * WinIo_PortWriters_g
 *
 *	This array contains pointers to the output port handlers. There is
 *	one handler for each possible port. When a port is written, the
 *	appropriate pointer is used to call the writer.
 *
 *	I/O handlers can be registered using WinIo_RegisterWriter; that
 *	routine updates a port's entry. At system initialization, all
 *	entries are initialized to point to WinIo_DefaultWriter, which
 *	aborts the simulation due to an I/O trap.
 *
 * revisions:
 *
 *	2007-04-09 rli: Original version.
 *
 ***/

WinIo_PortWriter_t WinIo_PortWriters_g[ 256 ];


/***************************
 *
 *	CODE
 *
 ***************************/

/***
 *
 * init_io
 *
 *	This routine is called during simulation startup to initialize
 *	the I/O system.
 *
 * revisions:
 *
 *	2007-04-08 rli: Initial do-nothing version.
 *
 *	2007-04-09 rli: Initialization of port handler arrays.
 *
 *	2007-04-11 rli: DumbCons.
 *
 *	2007-04-12 rli: MunkStubs.
 *
 *	2007-04-12 rli: MunkDisk.
 *
 *	2007-04-14 rli: Trade WinCons for DumbCons.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	- WinIo_PortReaders_g: The array of input port handlers. This is
 *	  initialized such that all entries refer to WinIo_DefaultReader.
 *
 *	- WinIo_PortWriters_g: This array of output port handlers. This is
 *	  initialized such that all entries refer to WinIo_DefaultWriter.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void init_io( void )
{
  int Port;

  /* Initialize the port handler arrays. We don't have to initialize the
   * cookies because the default handlers don't use them.
   */

  for( Port = 0; Port < 256; Port++ ) {
    WinIo_PortReaders_g[ Port ].Handler = WinIo_DefaultReader;
    WinIo_PortWriters_g[ Port ].Handler = WinIo_DefaultWriter;
  }

  /* MunkStubs at ports 2, 3, and 5. */

  MunkStubs_Register();

  /* MukDisk at ports 10 through 16. */

  MunkDisk_Register();

  /* WinCons at ports 0 and 1. */

  WinCons_Register();

  return;
}

/***
 *
 * exit_io
 *
 *	This routine is called when it's time for the simulation to
 *	exit. It cleans up the I/O system.
 *
 * revisions:
 *
 *	2007-04-08 rli: Initial do-nothing version.
 *
 *	2007-04-12 rli: MunkDisk.
 *
 * formal parameters:
 *
 *	none.
 *
 * informal parameters:
 *
 *	none.
 *
 * return value:
 *
 *	none.
 *
 * other effects:
 *
 *	none.
 *
 ***/

void exit_io( void )
{
  MunkDisk_Exit();
  return;
}

/***
 *
 * io_in
 *
 *	This routine is called by the simulation to read a byte from an
 *	I/O port.
 *
 * revisions:
 *
 *	2007-04-08 rli: original do-nothing version.
 *
 *	2007-04-09 rli: Call the appropriate reader.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address from which a value is to be read.
 *
 * informal parameters:
 *
 *	- WinIo_PortReaders_g: This array describes the input ports. There
 *	  is one entry for each possible port. Each entry supplies a function
 *	  pointer for the routine to handle the port and an opaque cookie
 *	  to be passed to the routine.
 *
 * return value:
 *
 *	- The byte read from the port.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE io_in( BYTE Address )
{
  return WinIo_PortReaders_g[ Address ].Handler( Address,
   WinIo_PortReaders_g[ Address ].Cookie );
}

/***
 *
 * io_out
 *
 *	This routine is called by the simulation to write a byte to an
 *	I/O port.
 *
 * revisions:
 *
 *	2007-04-08 rli: Original do-nothing version.
 *
 *	2007-04-09 rli: Call the appropriate handler.
 *
 * formal parameters:
 *
 *	- Address: Supplies the port address to which a byte is to be
 *	  written.
 *
 *	- Value: Supplies the value to be written to that port.
 *
 * informal parameters:
 *
 *	- WinIo_PortWriters_g: This array describes the output ports. There
 *	  is one entry for each possible port. Each entry supplies a function
 *	  pointer for the routine to handle the port and an opaque cookie
 *	  to be passed to the routine.
 *
 * return value:
 *
 *	- Zero is always returned. Although this routine is declared as
 *	  returning a value, the simulation does not use the value
 *	  returned. 
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE io_out( BYTE Address, BYTE Value )
{
  return WinIo_PortWriters_g[ Address ].Handler( Address, Value,
    WinIo_PortWriters_g[ Address ].Cookie );
}

/***
 *
 * WinIo_DefaultReader
 *
 *	This is the default input port handler. It is used to read an input
 *	port unless an alternate routine has been registered.
 *
 *	The simulation is stopped due to an I/O trap.
 *
 * revisions:
 *
 *	2007-04-09 rli: original version.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address of the port to be read. This is
 *	  not used.
 *
 *	- Cookie: An opaque value passed to this routine from the
 *	  port handler table. This parameter is not used.
 *
 * informal parameters:
 *
 *	- cpu_state: Is set to STOPPED to force the simulation to stop.
 *
 *	- cpu_error: Is set to IOTRAP to indicate that an attempt was
 *	  made to reference an unused port.
 *
 * return value:
 *
 *	- 0 is always returned.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE WinIo_DefaultReader( BYTE Address, void *Cookie )
{
  cpu_state = STOPPED;
  cpu_error = IOTRAP;
  return 0;
}

/***
 *
 * WinIo_DefaultWriter
 *
 *	This is the default output port handler. It is used to write an
 *	output port unless an alternate routine has been registered.
 *
 *	The simulation is stopped due to an I/O trap.
 *
 * revisions:
 *
 *	2007-04-09 rli: original version.
 *
 * formal parameters:
 *
 *	- Address: Supplies the address of the port to be read. This is
 *	  not used.
 *
 *	- Value: Supplies the value to be written to the port. This is
 *	  not used.
 *
 *	- Cookie: An opaque value passed to this routine from the handler
 *	  table. This parameter is not used.
 *
 * informal parameters:
 *
 *	- cpu_state: Set to STOPPED to force the simulation to stop.
 *
 *	- cpu_error: Set to IOTRAP to indicate that an attempt was made
 *	  to reference an unused port.
 *
 * return value:
 *
 *	- 0 is always returned.
 *
 * other effects:
 *
 *	none.
 *
 ***/

BYTE WinIo_DefaultWriter( BYTE Address, BYTE Value, void *Cookie )
{
  cpu_state = STOPPED;
  cpu_error = IOTRAP;
  return 0;
}

/***
 *
 * WinIo_RegisterReader
 *
 *	This routine registers a handler for an input port.
 *
 * revisions:
 *
 *	2007-04-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Address: Supplies the port address that is to be registered.
 *
 *	- Reader: Supplies a pointer to the function that should be called
 *	  when that port is read.
 *
 *	- Cookie: Supplies an opaque value that is to be passed, unmodified,
 *	  to the Reader function when the port is read. This can be a pointer
 *	  to a data structure holding context for the reader.
 *
 * informal parameters:
 *
 *	- WinIo_PortReaders_g: The array of port readers. The entry 
 *	  corresponding to Address is overwritten.
 *
 *	- WinIo_DefaultReader: The default handler for input ports. An
 *	  unused entry in WinIo_PortReaders_g is expected to refer to
 *	  this reader.
 *
 * return value:
 *
 *	- 0: Success. The reader has been registered; it will be called
 *	  when the simulation reads the specified I/O port.
 *
 *	- 1: Failure. The port address is already in use. The reader has
 *	  NOT been registered.
 *
 * other effects:
 *
 *	- An error message is displayed if the port is in used.
 *
 ***/

int WinIo_RegisterReader( BYTE Address, 
  BYTE (*Reader)( BYTE Address, void *Cookie ),
  void *Cookie )
{

  /* If the port is already in use, complain and exit, indicating failure.
   */

  if( WinIo_PortReaders_g[ Address ].Handler != WinIo_DefaultReader ) {
    fprintf( stderr, "WinIo: Input port 0x%x is already in use!\n",
      Address );
    return 1;
  }

  /* The port is available. Register the reader and exit, indicating success.
   */

  WinIo_PortReaders_g[ Address ].Handler = Reader;
  WinIo_PortReaders_g[ Address ].Cookie = Cookie;
  return 0;

}

/***
 *
 * WinIo_RegisterWriter
 *
 *	This routine registers a handler for an output port.
 *
 * revisions:
 *
 *	2007-04-10 rli: original version.
 *
 * formal parameters:
 *
 *	- Address: Supplies the port address that is to be registered.
 *
 *	- Writer: Supplies a pointer to the function that should be called
 *	  when that port is written.
 *
 *	- Cookie: Supplies an opaque value that is to be passed, unmodified,
 *	  to the Writer function when the port is read. This can be a pointer
 *	  to a data structure holding context for the writer.
 *
 * informal parameters:
 *
 *	- WinIo_PortWriters_g: The array of port writers. The entry 
 *	  corresponding to Address is overwritten.
 *
 *	- WinIo_DefaultWriter: The default handler for output ports. An
 *	  unused entry in WinIo_PortWriters_g is expected to refer to
 *	  this writer.
 *
 * return value:
 *
 *	- 0: Success. The writer has been registered; it will be called
 *	  when the simulation writes the specified I/O port.
 *
 *	- 1: Failure. The port address is already in use. The writer has
 *	  NOT been registered.
 *
 * other effects:
 *
 *	- An error message is displayed if the port is in used.
 *
 ***/

int WinIo_RegisterWriter( BYTE Address, 
  BYTE (*Writer)( BYTE Address, BYTE Value, void *Cookie ),
  void *Cookie )
{

  /* If the port is already in use, complain and exit, indicating failure.
   */

  if( WinIo_PortWriters_g[ Address ].Handler != WinIo_DefaultWriter ) {
    fprintf( stderr, "WinIo: Output port 0x%x is already in use!\n",
      Address );
    return 1;
  }

  /* The port is available. Register the writer and exit, indicating success.
   */

  WinIo_PortWriters_g[ Address ].Handler = Writer;
  WinIo_PortWriters_g[ Address ].Cookie = Cookie;
  return 0;

}





