// Centipede Emulator 0.2
// ----------------------
// by Ivan Mackintosh email: ivan@rcp.co.uk
//
//
// Still to be completed:
// ----------------------
// Colour Palettes
// A way of speeding up the sound and adding a volume control
// A way of controlling the game speed
//
// Thanks to:
// ----------
// Peter Rittwage - for help getting past the trackball checking
// Dave Spicer - for answering general queries.
// Allard van der Bas - providing the repository
// Neil Bradley for supplying 6502 code
// Ron Fries for supplying excellent Pokey emulator
//
// Change History
// --------------
// 24 Feb 97 Version 0.2
// Added sprite flipping and animation.
// Added Dipswtich option (-switch)
// Changed from Marats 6502 engine to Neil Bradleys (modified)
// Removed all 6502 code hacks (-hack to put them back)
//     without the hack sometimes the emulation freezes!!!
// Added 2 player support



#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <conio.h>
#include <ctype.h>
#include <pc.h>
#include <allegro.h>  
#include "6502.h"
#include "pokey.h"
#include "dipswtch.h"

/* prototypes */
void DisplayMessage(void);
int LoadRom(BYTE * pAddress, char * Name, size_t Size);
void DisplayCharSet(void);
void DrawSingleChar(int CharNo, int x, int y, int palette);
void DrawSprite(int Offset, int x, int y);
int GetBit(int);
void setupvideo(void);
void WarpChars(void);
void RedrawScreen(void);
void LoadStatus(void);
void ReadKeyboard(void);
void SaveStatus(void);
void DipSwitches(void);


#define SAMPLE_SIZE	(3000)
#define SAVEFILE	"centiped.sav"
#define CODE_ROM1	"136001.307"
#define CODE_ROM2	"136001.308"
#define CODE_ROM3	"136001.309"
#define CODE_ROM4	"136001.310"
#define GRPX_ROM1	"136001.211"
#define GRPX_ROM2	"136001.212"

#define CODE_START	(0x2000)
#define	HISCORE_BYTES	(0x2F)
#define RAM_HISCORES	(0x2)
#define	ROM_HISCORES	(0x3A69)
#define DIPSW1		(0x800)
#define DIPSW2		(0x801)
#define SPRITE_OFFSET	(0x7C0)
#define SPRITE_XPOS	(0x7D0)
#define SPRITE_YPOS	(0x7E0)
#define SPRITE_COL	(0x7F0)
#define SCREEN_MEM	(0x400)


int addrmask = 0x3fff;
BYTE * gameImage = NULL;
BYTE CharSet[4096];
BYTE WarpCharSet[16384];
BITMAP *piccie;
BYTE pal[768];
WORD ExitAddress;
extern volatile char key[128];

int DisplayChars = 0;
int nojoy = 0;
int nosound = 0;
int CPURunning = 0;

BYTE pSample[SAMPLE_SIZE];
static SAMPLE Sound = {8, 15960, SAMPLE_SIZE, pSample};


int main(int argc, char * argv[])
{
   int i;
   int EditDS = 0;
   int Hack = 0;
   
   DisplayMessage();
   
   // parse command line
   for (i = 1; i < argc; i ++)
   {
      if ((argv[i][0] == '-') || (argv[i][0] == '/'))
      {
	 if (strcmp(&argv[i][1], "chars") == 0)
	    DisplayChars = 1;

	 else if(strcmp(&argv[i][1], "nojoy") == 0)
	    nojoy = 1;
	 
	 else if(strcmp(&argv[i][1], "nosound") == 0)
	    nosound = 1;
	 
	 else if (strcmp(&argv[i][1], "switch") == 0)
	    EditDS = 1;
	 
	 else if (strcmp(&argv[i][1], "hack") == 0)
	    Hack = 1;
      }
   }

   /* create memory map */
   if ((gameImage = (BYTE *)malloc(0x4000)) == NULL)
   { printf("Memory Allocation failed\n"); exit(1); }

   // clear memory
   for (i = 0; i < 0x4000; i ++)
      gameImage[i] = 0;
   
   /* load roms */
   printf("Loading Roms...\n");
   if (!LoadRom(&gameImage[CODE_START], CODE_ROM1, 0x800)) 
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x800], CODE_ROM2, 0x800))
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x1000], CODE_ROM3, 0x800))
      { free (gameImage); exit(1); }
   if (!LoadRom(&gameImage[CODE_START + 0x1800], CODE_ROM4, 0x800))
      { free (gameImage); exit(1); }
   if (!LoadRom(&CharSet[0], GRPX_ROM1, 0x800))
      { free (gameImage); exit(1); }
   if (!LoadRom(&CharSet[0x800], GRPX_ROM2, 0x800))
      { free (gameImage); exit(1); }
   
   WarpChars();

   init6502();
   reset6502();

   // has the hack param been specified? if so NOP out the code
   // that locks the emulation
   if (Hack)
   {
      gameImage[0x38a8] = 0xea;
      gameImage[0x38a9] = 0xea;
      gameImage[0x38ae] = 0xea;
      gameImage[0x38af] = 0xea;
   }
      
   // detect joystick
   if (!nojoy) 
   {
      int det = initialise_joystick();
      
      if (det == 0)
	 printf("Joystick detected\n");
      else
	 nojoy = 1;
   }

   if (!nosound)
   {
      int det = install_sound(-1, 0,NULL);
      
      if (det == 0)
      {
	 printf("Digital sound device detected\n");
	 Pokey_sound_init(FREQ_17_APPROX, 15960);
	 for (i = 0; i < SAMPLE_SIZE; i ++)
	    pSample[i] = 0;
	 play_sample(&Sound,128,128,1000,1);
      }
      else
	 nosound = 1;
   }
      
   install_keyboard();
   
   printf("\nPress a key to start\n");
   readkey();
   
   setupvideo();

   // if -d switch is set then display chars and exit
   if (DisplayChars)
   {
      DisplayCharSet();
      readkey();
      set_gfx_mode(GFX_TEXT,80,25,0,0);
      destroy_bitmap(piccie);
      allegro_exit();
      free(gameImage);
      exit(1);
   }

   // set service mode off
   gameImage[0xC00] = 0x60;

   LoadStatus();

   // if -switch param display edit dip switch menu
   if (EditDS) DipSwitches();
   
   CPURunning = 1;
   do
   {
      ExitAddress = exec6502(INSTRUCTIONS_PER_USER_INPUT);
      ReadKeyboard();
   } 
   while(CPURunning);


   // save highscores + dipswitches
   SaveStatus();
         
   free (gameImage);
   destroy_bitmap(piccie);
   set_gfx_mode(GFX_TEXT,80,25,0,0);
   allegro_exit();
   printf("Exit Address = %x\n", ExitAddress);
   exit(0);
}




int LoadRom(BYTE * pAddress, char * Name, size_t Size)
{
   FILE * fp;

   if ((fp = fopen(Name, "rb")) == NULL)
   {
      printf("Failed to open %s\n", Name);
      return 0;
   }

   if (fread(pAddress, 1, Size, fp) != Size)
   {
      printf("Failed to read %s\n", Name);
      free(gameImage);
      fclose(fp);
      return 0;
   }

   fclose(fp);
   return 1;
}






void DrawSingleChar(int CharNo, int x, int y, int palette)
{
   int cx, cy;
   int Start = CharNo * 64;
   
   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 8; cx ++)
      {
	 piccie->line[(y * 8) + cy][(x * 8) + cx] = 
	    (BYTE)(WarpCharSet[Start] | (palette << 2));
	 Start++;
      }
   }
}


void DrawSprite(int Offset, int x, int y)
{
   int cx, cy;
   int Start;
   int SpOffset = Offset & 0x3E;
   int Flip = 1;
   Start = SpOffset * 64;
   
   // use second set of sprite data?
   if ((Offset & 0x01) == 0x01)
      Start += 8192; 

   // flip sprite horizontally?
   if ((Offset & 0x80) == 0x80)
      Flip = 0;
   
   for (cy = 7; cy >=0; cy--)
   {
      for (cx = 0; cx < 8; cx++)
      {
	 BYTE ypos = 255-(y+cy);
	 
	 if (!Flip)
	 {
	    BYTE xpos = 238 - (x - (8 - cx));

	    if ((WarpCharSet[Start] != 0) && ((BYTE)(xpos+8) < 0xf0))
	       piccie->line[ypos][(BYTE)(xpos+8)] = WarpCharSet[Start];

	    if ((WarpCharSet[Start+64] != 0) && (xpos < 0xf0))
	       piccie->line[ypos][xpos] = WarpCharSet[Start+64];
	 }
	 else
	 {
	    BYTE xpos = 238 - (x - cx);
	    
	    if ((WarpCharSet[Start] != 0) && (xpos < 0xf0))
	       piccie->line[ypos][xpos] = WarpCharSet[Start];

	    if ((WarpCharSet[Start+64] != 0) && ((BYTE)(xpos+8) < 0xf0))
	       piccie->line[ypos][(BYTE)(xpos+8)] = WarpCharSet[Start+64];
	    
	 }
	 Start++;
      }
   }
}



void DisplayCharSet(void)
{
   int x, y;
   for (x = 0; x < 32; x ++)
   {
      for (y = 0; y < 8; y ++)
      {
	 DrawSingleChar((y * 32) + x, x, y, 0);	 
      }
   }

   blit(piccie, screen, 0,0, 0,0, 256,256);
}



void setcolor(char red, char green, char blue, unsigned short color)
{
   color = 3*color;
   pal[color++]=red;
   pal[color++]=green;
   pal[color]=blue;
}

void setpallete ()
{
   int count;

   outportb(0x3c8,0x00);
   for(count=0;count<768;count++)
      outportb(0x3c9,pal[count]);
}

void setupvideo ()
{
   WORD teller;
   WORD count1,count2;

   set_gfx_mode(GFX_MODEX,256,256,0,0);
   for (teller=0;teller<256;teller++)
      setcolor(0,0,0,teller);

   setcolor(0,0,0,0);		// Black
   setcolor(0,63,0,1);		// Green
   setcolor(63,0,0,2);		// Red
   setcolor(63,63,63,3);	// White

   setcolor(0,0,0,4);		// Black
   setcolor(0,63,0,5);		// Green
   setcolor(32,32,32,6);	// Blue
   setcolor(63,63,63,7);	// White

   setcolor(0,0,0,8);		// Black
   setcolor(0,63,0,9);		// Green
   setcolor(0,63,63,10);	// Light Blue
   setcolor(63,63,63,11);	// White

   piccie = create_bitmap(256,256);
   for (count1=0;count1<256;count1++)
      for (count2=0;count2<256;count2++)
         piccie->line[count1][count2] = 0;

   setpallete();
}




int GetBit(int startbit)
{
   int nbyte = startbit/8;
   int nbit = startbit%8;
   int mask = 0x80 >> nbit;
   
   return ((CharSet[nbyte] & mask) == mask);
}

void WarpChars(void)
{
   int x,cx,cy, bit;
   BYTE result = 0;
   int Count = 0;

   for (cy = 0; cy < 8; cy ++)
   {
      for (cx = 0; cx < 32; cx ++)
      {
         for (bit = 7; bit >= 0; bit --)
         {
            for (x = 0; x < 8; x ++)
            {
               result = 0;
               if (GetBit(bit + (x * 8) + (cx * 64) + (cy * 2048))) result = 1;
               if (GetBit(bit + (x * 8) + (cx * 64) + (cy * 2048) + 16384)) result ^= 2;
         
	       WarpCharSet[Count] = result;
	       Count++;
            }
         }
      }
   }
}


void RedrawScreen(void)
{
   int x, y;
   int count = 0;

   for (x = 0; x < 30; x++)
   {
      for (y = 31; y >= 0; y--)
      {
	 DrawSingleChar(gameImage[0x400 + count] + 64, x, y, 0);
	 count ++;
      }
   }

   // draw sprites
   for (x = 0; x < 16; x ++)
   {
      int xpos = gameImage[SPRITE_XPOS + x];
      int ypos = gameImage[SPRITE_YPOS + x];
      int offset = gameImage[SPRITE_OFFSET + x];

      if (ypos < 0xf8)
	 DrawSprite(offset, xpos, ypos);
   }

   blit(piccie, screen, 0,0, 0,0, 256,256);
}



void ReadKeyboard(void)
{
   if (key[KEY_ESC])
      CPURunning = 0;

   RedrawScreen();

   if (!nojoy)
      poll_joystick();
   
   if (!nosound)
   {
      Pokey_process(pSample, SAMPLE_SIZE);
      adjust_sample(&Sound, 128, 128, 1000, 1);
   }
}



void M_WRMEM(WORD A,BYTE V)
{
   if (A == 0x1800)
      clockticks6502 = 0;
   
   // don't allow code to overwrite itself
   if (A >= CODE_START)
      return;
   
   if (!nosound)
   {
      if ((A >= 0x1000) && (A <= 0x1008))
      {
	 Update_pokey_sound(0xC200+A, V);
      }
   }
   
   gameImage[A] = V;
}


BYTE M_RDMEM(WORD A)
{
   if (A == 0xC01)
   {
      BYTE ValC01 = 0xFF;
      
      if (key[KEY_3])
	 ValC01 &= ~0x40;  // insert coin

      if (key[KEY_1])
	 ValC01 &= ~0x01;  // 1 player start
      
      if (key[KEY_2])
	 ValC01 &= ~0x02;  // 2 player start 
      
      if (key[KEY_CONTROL])
	 ValC01 &= ~0x0C;  // player fire (keyboard)
      
      if (!nojoy && joy_b1)
	 ValC01 &= ~0x0C;  // player fire (joystick)
	 
      return ValC01;
   }

   if (A == 0xC03)
   {
      BYTE ValC03 = 0xFF;
      
      // player1 move (keyboard)
      if (key[KEY_UP])
	 ValC03 &= ~0x11;
      if (key[KEY_DOWN])
	 ValC03 &= ~0x22;
      if (key[KEY_LEFT])
	 ValC03 &= ~0x44;
      if (key[KEY_RIGHT])
	 ValC03 &= ~0x88;

      // player1 move (joystick)
      if (!nojoy)
      {
	 if (joy_up)
	    ValC03 &= ~0x11;
	 if (joy_down)
	    ValC03 &= ~0x22;
	 if (joy_left)
	    ValC03 &= ~0x44;
	 if (joy_right)
	    ValC03 &= ~0x88;
      }
      
      
      return ValC03;
      
   }
            
   return(gameImage[(A & addrmask)]);
}



// If the status file centiped.sav exists then load the
// hiscore table and dip switch settings from it
void LoadStatus(void)
{
   FILE * fp;
   int i;
   
   // open save file if exists
   fp = fopen(SAVEFILE, "rb");
   if (fp == NULL) return;
   
   // load in high score over ROM high scores
   for (i = 0; i < HISCORE_BYTES; i ++)
      gameImage[ROM_HISCORES + i] = (BYTE)fgetc(fp);
   
   // load in dipswitch settings
   gameImage[DIPSW1] = (BYTE)fgetc(fp);
   gameImage[DIPSW2] = (BYTE)fgetc(fp);
   
   fclose(fp);
}

// Saves the RAM based high scores and the dipswitch
// settings - called on emulator exit
void SaveStatus(void)
{
   FILE * fp;
   int i;
   
   // open/create save file
   fp = fopen(SAVEFILE, "wb");
   if (fp == NULL) return;
   
   // save the high scores from RAM
   for (i = 0; i < HISCORE_BYTES; i ++)
      fputc(gameImage[RAM_HISCORES + i], fp);
   
   // save dipswitch settings
   fputc(gameImage[DIPSW1], fp);
   fputc(gameImage[DIPSW2], fp);
   
   fclose(fp);
}




void DisplayMessage(void)
{
   printf("\nPLEASE DO NOT DISTRIBUTE THE SOURCE FILES OR THE EXECUTABLE WITH"
      " ROM IMAGES.\nDOING SO WILL HARM FURTHER EMULATOR DEVELOPMENT AND WILL"
      " CONSIDERABLY ANNOY\nTHE RIGHTFUL COPYRIGHT HOLDERS OF THOSE ROM"
      " IMAGES AND CAN RESULT IN LEGAL\nACTION UNDERTAKEN BY EARLIER"
      " MENTIONED COPYRIGHT HOLDERS.\n\n\n");

   printf("Command line switches:\n");
   printf("-chars    Display character set\n");
   printf("-nojoy    Disable joystick support\n");
   printf("-nosound  Disable sound support\n");
   printf("-switch   Display dip switch menu\n");
   printf("-hack     Use this if the emulation normally locks up on you\n\n");
   
   printf("Keys:\n");
   printf("Pressing 1 or 2 starts the game. Move around using the"
      " joystick or\ncursor keys and <ctrl> to fire. Pressing <esc> quits"
      " the emulator.\n\n");
}




// Clears the bitmap that represents the screen
void ClearBitmap(void)
{
   int x, y;
   for (x = 0; x < 256; x ++)
      for (y = 0; y < 256; y ++)
         piccie->line[y][x] = 0;
}



// Draws a given alphanumeric string using the centipede chars
// at a specified place on the screen.
void DrawString(int x, int y, char * Text, int TextLen, int Palette)
{
   int i;

   for (i = 0; i < TextLen; i ++)
   {
      if (Text[i] == ' ')
         DrawSingleChar(64, x + i, y, Palette);
      else if (isalpha(Text[i]))
         DrawSingleChar(Text[i], x + i , y, Palette);
      else if (isdigit(Text[i]))
         DrawSingleChar((Text[i] - '0') + 96, x + i, y, Palette);
   }
}





// DipSwitches
// -----------
// Decodes and displays the current dip switch setting and allows
// these setting to be modified. This function is invoked on start
// up if the -switch parameter is specified.
void DipSwitches(void)
{
   int i, t, ExitDS = 0;
   // debounce 
   int DBUp = 0, DBDown = 0, DBFire = 0;

   int Setting = 0;
   int CurValues[NUM_SWITCHES];
   
   // don't do anything until the user lets go of <return>
   // otherwise the menu will exit straight away
   while (key[KEY_SPACE])
      ;
   
   ClearBitmap();
   DrawString(6, 2, "DIP SWITCH SETTINGS", 19, 1);

   // dipswitch settings
   for (i = 0; i < NUM_SWITCHES; i ++)
   {
      DrawString(5, 7 + (i * 2), Dipswitch[i].Heading, HEADING_SIZE, 
      (Setting == i)?2:0);

      for (t = 0; t < Dipswitch[i].NumOptions; t ++)
      {
         if ((gameImage[DIPSW1 + Dipswitch[i].Switch] & ~Dipswitch[i].Mask) ==
	    Dipswitch[i].Value[t])
	    break;
      }

      // if not found set default 
      if (t == Dipswitch[i].NumOptions)
         t = Dipswitch[i].Default; 
      DrawString(6 + HEADING_SIZE, 7 + (i * 2), Dipswitch[i].Options[t],
		 OPTION_SIZE, (Setting == i)?2:0);

      CurValues[i] = t;
      gameImage[DIPSW1 + Dipswitch[i].Switch] &= Dipswitch[i].Mask;
      gameImage[DIPSW1 + Dipswitch[i].Switch] |= Dipswitch[i].Value[t];
   }
   
   DrawString(1, 25, "USE UP DOWN AND FIRE TO CYCLE", 30, 1);
   DrawString(2, 26, "THROUGH DIP SWITCH SETTINGS", 27, 1);
   DrawString(0, 29, "PRESS SPACE TO START CENTIPEDE", 31, 1);
   
   blit(piccie, screen, 0,0, 0,0, 256,256);

   while(!ExitDS)
   {
      if (key[KEY_UP] && (Setting > 0) && !DBUp)
      {
         // remove old highlight
	 
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 0);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 0);

	 Setting --;

	 // add new highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 2);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);

         DBUp = 1;
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_DOWN] && (Setting < NUM_SWITCHES - 1) && !DBDown)
      {
         // remove old highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 0);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 0);

	 Setting ++;

	 // add new highlight
	 DrawString(5, 7 + (Setting * 2), Dipswitch[Setting].Heading, 
	    HEADING_SIZE, 2);
	 DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
	    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);

	 DBDown = 1;
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_CONTROL] && !DBFire)
      {
         CurValues[Setting]++;
         if (CurValues[Setting] == Dipswitch[Setting].NumOptions)
            CurValues[Setting] = 0;

         gameImage[DIPSW1 + Dipswitch[Setting].Switch] &= Dipswitch[Setting].Mask;
         gameImage[DIPSW1 + Dipswitch[Setting].Switch] |= Dipswitch[Setting].Value[CurValues[Setting]];
         
         DBFire = 1;
         DrawString(6 + HEADING_SIZE, 7 + (Setting * 2),
                    Dipswitch[Setting].Options[CurValues[Setting]], OPTION_SIZE, 2);
         blit(piccie, screen, 0,0, 0,0, 256,256);
      }
      if (key[KEY_SPACE])
         ExitDS = 1;

      // debounce reset
      if (!key[KEY_UP]) DBUp = 0;
      if (!key[KEY_DOWN]) DBDown = 0;
      if (!key[KEY_CONTROL]) DBFire = 0;
   }
}

