/*
 * Altogether: Xerox Alto microcode-level simulator
 * Main CPU simulation
 * $Id: cpu.c 129 2007-04-20 22:25:10Z eric $
 * Copyright 2001, 2003, 2004, 2005, 2007 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-infile-zip.h>
#include <gsf/gsf-input.h>

#include "util.h"
#include "alto.h"
#include "debug.h"
#include "scheduler.h"
#include "cpu.h"
#include "main_mem.h"
#include "emulator.h"  /* we need the skip flag */


#ifdef DEFAULT_PATH
char *default_path = MAKESTR(DEFAULT_PATH);
#else
char *default_path = NULL;
#endif


#define UCODE_PAGE_SIZE 1024

#define UCODE_ROM_SIZE (UCODE_ROM_PAGES * UCODE_PAGE_SIZE)
#define UCODE_RAM_SIZE (UCODE_RAM_PAGES * UCODE_PAGE_SIZE)
#define UCODE_RAM_BASE (UCODE_ROM_SIZE)
#define UCODE_SIZE (UCODE_ROM_SIZE + UCODE_RAM_SIZE)

uint32_t ucode_raw [UCODE_SIZE];  // undecoded 32-bit microcode word
uinst_t  ucode     [UCODE_SIZE];  // microcode word decoded into individual fields
bool     ubreak    [UCODE_SIZE];  // breakpoints


uint16_t constant_rom [CONSTANT_ROM_SIZE];

uint16_t l, m, t, cram_addr;
int l_alu_carry;

int rsel;  /* from rsel field of microinstruction, but may be
	      modified by emulator acsource and acdest uops */
uint16_t r [32];
uint16_t s [S_REG_BANKS][32];


uint16_t reset_mode_register;  /* bit 1<<n set if task n starts in ROM */

char *task_name [] =
{
  "emu", "t01", "t02", "t03", "dsc", "t05", "t06", "eth",
  "mrt", "dwt", "cur", "dht", "dvt", "par", "dwd", "t17"
};

int current_task, next_task, next2_task;
uint16_t task_request;  /* bit 1<<n set if task n requesting service */
uint16_t task_autoblock;  /* bit 1<<n set if task n autoblocks */

bool task_switch;  /* set when task switch happens */


int rdram_flag;  /* set by rdram, action happens on next cycle */
int wrtram_flag;  /* set by wrtram, action happens on next cycle */

uint16_t address_modifier;
uint16_t upc [TASK_COUNT];
bool ram_related [TASK_COUNT];
int s_reg_bank [TASK_COUNT];

int normal_ram_bank [TASK_COUNT];
int extended_ram_bank [TASK_COUNT];

uint16_t current_upc;  /* adddress of microinstruction currently being executed,
		   * used only for trace output, etc.
		   */
uinst_t *current_uinst;

uint16_t bus;  /* wire-AND */

uint16_t alu_out;
bool alu_carry_out;

uint16_t shifter_out;
bool shifter_carry_out;


int illegal_instruction;


dispatch_fn_t bus_select_early_fn [TASK_COUNT][16];
dispatch_fn_t bus_select_late_fn [TASK_COUNT][16];
dispatch_fn_t f1_early_fn [TASK_COUNT][16];
dispatch_fn_t f1_late_fn [TASK_COUNT][16];
dispatch_fn_t f2_early_fn [TASK_COUNT][16];
dispatch_fn_t f2_late_fn [TASK_COUNT][16];


#define bit_mask(bits) ((1 << bits) - 1)


void decode_uinst (uint16_t addr)
{
  uint32_t val;

  val = ucode_raw [addr];
  ucode [addr].rsel = (val >> 27) & bit_mask (5);
  ucode [addr].aluf = (val >> 23) & bit_mask (4);
  ucode [addr].bs   = (val >> 20) & bit_mask (3);
  ucode [addr].f1   = (val >> 16) & bit_mask (4);
  ucode [addr].f2   = (val >> 12) & bit_mask (4);
  ucode [addr].t    = (val >> 11) & bit_mask (1);
  ucode [addr].l    = (val >> 10) & bit_mask (1);
  ucode [addr].next = (val >>  0) & bit_mask (10);
}


#define UCODE_INVERTED_BITS 0x00088400


static char *ucode_rom_filenames [16] =
{
  "55x.3", "64x.3", "65x.3", "63x.3", "53x.3", "60x.3", "61x.3", "62x.3",
  "54x.3", "74x.3", "75x.3", "73x.3", "52x.3", "70x.3", "71x.3", "72x.3"
};

static void load_ucode_rom (GsfInfile *zip_infile)
{
  int page, segment, addr;
  char *fn;
  GsfInput *f;
  unsigned char buf [UCODE_PAGE_SIZE];

  for (page = 0; page < UCODE_ROM_PAGES; page++)
    {
      for (addr = 0; addr < UCODE_PAGE_SIZE; addr++)
	{
	  ucode_raw [(page << 10) + addr] = 0;  /* should this be all-ones? */
	}
      for (segment = 0; segment <= 7; segment++)
	{
	  fn = ucode_rom_filenames [(page << 3) + segment];
	  f = gsf_infile_child_by_name (zip_infile, fn);
	  if (! f)
	    fatal (3, "can't open ucode file %s\n", fn);
	  gsf_input_read (f, UCODE_PAGE_SIZE, buf);
	  g_object_unref (G_OBJECT (f));
	  for (addr = 0; addr < UCODE_PAGE_SIZE; addr++)
	    {
	      /* addresses are complemented */
	      ucode_raw [(page << 10) + addr] |=
		(buf [(UCODE_PAGE_SIZE - 1) - addr] << ((7 - segment) * 4));
	    }
	}
      for (addr = 0; addr < UCODE_PAGE_SIZE; addr++)
	{
	  ucode_raw [(page << 10) + addr] ^= ~ UCODE_INVERTED_BITS;
	  decode_uinst ((page << 10) + addr);
#if 0
	  printf ("%05o: %08x\n", (page << 10) + addr, ucode [(page << 10) + addr]);
#endif
	}
    }
}


bool check_ubreak (void)
{
  return (ubreak [upc [current_task]]);
}


void set_ubreak (uint16_t addr, bool value)
{
  ubreak [addr] = value;
}


static char *const_rom_filenames [4] =
{
  "c0.3", "c1.3", "c2.3", "c3.3"
};

static int unscramble_constant_address (int a)
{
  return (((a & 0x80) >> 7) |
	  ((a & 0x78) << 1) |
	  ((a & 0x04) >> 1) |
	  ((a & 0x02) << 1) |
	  ((a & 0x01) << 3));
}

static int unscramble_constant_data (int d)
{
  int i;
  int d2 = 0;
  for (i = 0; i <= 7; i++)
    d2 |= ((d & (1 << i)) << (15 - 2 * i));
  for (i = 8; i <= 15; i++)
    d2 |= ((d & (1 << i)) >> ((i - 8) * 2 + 1));
  return (d2);
}

static void load_const_rom (GsfInfile *zip_infile)
{
  int segment, addr;
  char *fn;
  GsfInput *f;
  unsigned char buf [CONSTANT_ROM_SIZE];

  for (addr = 0; addr < CONSTANT_ROM_SIZE; addr++)
    constant_rom [addr] = 0;
  for (segment = 0; segment < 4; segment++)
    {
      fn = const_rom_filenames [segment];
      f = gsf_infile_child_by_name (zip_infile, fn);
      if (! f)
	fatal (3, "can't open constants file %s\n", fn);
      gsf_input_read (f, CONSTANT_ROM_SIZE, buf);
      g_object_unref (G_OBJECT (f));
      for (addr = 0; addr < CONSTANT_ROM_SIZE; addr++)
	{
	  constant_rom [unscramble_constant_address (addr)] |=
	      (buf [addr] ^017) << (4 * segment);
	}
    }
  for (addr = 0; addr < CONSTANT_ROM_SIZE; addr++)
    {
      constant_rom [addr] = unscramble_constant_data (constant_rom [addr]);
    }
}


void load_microcode (char *zip_fn)
{
  char *fn;
  GsfInput *zip_input;
  GsfInfile *zip_infile;
  GError *err = NULL;

  fn = find_file_in_path_list (zip_fn, NULL, default_path);
  if (! fn)
    fatal (3, "can't find file '%s'\n", zip_fn);

  zip_input = gsf_input_stdio_new (fn, & err);
  if (! zip_input)
    fatal (3, "can't open file '%s': %s\n", zip_fn, err->message);

  zip_infile = gsf_infile_zip_new (zip_input, & err);
  if (! zip_infile)
    fatal (3, "'%s' not a ZIP file: %s\n", zip_fn, err->message);

  g_object_unref (G_OBJECT (zip_input));

  load_ucode_rom (zip_infile);
  load_const_rom (zip_infile);

  g_object_unref (G_OBJECT (zip_infile));
}


uint16_t bank_register_read (uint16_t address)
{
  return (0177760 |
	  (normal_ram_bank [address & 017] << 2) |
	  extended_ram_bank [address & 017]);
}


void bank_register_write (uint16_t address, uint16_t data)
{
  normal_ram_bank [address & 017] = (data >> 2) & 3;
  extended_ram_bank [address & 017] = data & 3;
}


void init_cpu_mmio (void)
{
  install_memory_mapped_io (0177740, 0177757,
			    bank_register_read, bank_register_write);
}


void soft_reset (void)
{
  int i;

  current_task = 0;
  for (i = 0; i < TASK_COUNT; i++)
    {
      upc [i] = i;
      if (! (reset_mode_register & (1 << i)))
	upc [i] += UCODE_RAM_BASE;
      s_reg_bank [i] = 0;
    }

  task_request = 0;  /* only emulator requesting */
  set_task_request (task_emulator);

  reset_mode_register = 0xffff;

  address_modifier = 0;
  rdram_flag = 0;
  wrtram_flag = 0;
}


void hard_reset (void)
{
  reset_mode_register = 0xffff;
  soft_reset ();
}


void bs_fn_bad (int task, int code)
{
  illegal_instruction = 1;
  printf ("bad bus source field %d\n", code);
}

void bs_fn_read_r_early (int task, int code)
{
  char regname [4] = { 'r', '0', '0', '\0' };
  regname [1] += rsel >> 3;
  regname [2] += rsel & 7;
  drive_bus (regname, r [rsel]);
}

void bs_fn_load_r_early (int task, int code)
{
  drive_bus ("load_r", 0);
}

void bs_fn_load_r_late (int task, int code)
{
  r [rsel] = shifter_out;
  tprintf (1, "r%02o <= %06o\n", rsel, shifter_out);
}



void bs_fn_mem_data_early (int task, int code)
{
  drive_bus ("read_mem", read_main_mem ());
}


void bs_fn_mouse_early (int task, int code)
{
  /* ??? */
}


void f1_fn_bad (int task, int code)
{
  illegal_instruction = 1;
  printf ("bad f1 field %d\n", code);
}


void f1_fn_load_mar_late (int task, int code)
{
  int bank;
  if (rsel == 037)
    {
      // starting a memory refresh cycle - currently we don't do
      // anything special
      ;
    }
  if (current_uinst->f2 == f2_load_md)
    bank = extended_ram_bank [current_task];
  else
    bank = normal_ram_bank [current_task];
  load_main_mem_addr ((bank << 16) | alu_out);
}


void f1_fn_task_early (int task, int code)
{
  int i;
  int mask;
  for (i = 15, mask = 0100000; i >= 0; i--, mask >>= 1)
    if (task_request & mask)
      {
	next2_task = i;
	tprintf (1, "task switch to %s\n", task_name [next2_task]);
	if (task_autoblock & (1 << i))
	  clear_task_request (i);
	if (next2_task != next_task)
	  task_switch = true;
	return;
      }
  fatal (3, "no tasks requesting service\n");
}


void f1_fn_block_early (int task, int code)
{
  clear_task_request (task);
}


void f2_fn_bad (int task, int code)
{
  illegal_instruction = 1;
  printf ("bad f2 field %d\n", code);
}

void f2_fn_bus_eq_0_late (int task, int code)
{
  branch ("bus_eq_0", bus == 0);
}

void f2_fn_shifter_lt_0_late (int task, int code)
{
  branch ("shifter_lt_0", (shifter_out & 0100000) != 0);
}

void f2_fn_shifter_eq_0_late (int task, int code)
{
  branch ("shifter_eq_0", shifter_out == 0);
}

void f2_fn_bus_late (int task, int code)
{
  branch ("bus", bus & 01777);
}

void f2_fn_alucy_late (int task, int code)
{
  branch ("alucy", l_alu_carry);
}

void f2_fn_load_md_late (int task, int code)
{
  if (current_uinst->f1 == f1_load_mar)
    ; /* it's just part of an XMAR */
  else
    write_main_mem (bus);
}




void install_bus_select_fn (int task,
			    int code,
			    dispatch_fn_t early_fn,
			    dispatch_fn_t late_fn)
{
  bus_select_early_fn [task][code] = early_fn;
  bus_select_late_fn [task][code] = late_fn;
}

void install_f1_fn (int task,
		    int code,
		    dispatch_fn_t early_fn,
		    dispatch_fn_t late_fn)
{
  f1_early_fn [task][code] = early_fn;
  f1_late_fn [task][code] = late_fn;
}

void install_f2_fn (int task,
		    int code,
		    dispatch_fn_t early_fn,
		    dispatch_fn_t late_fn)
{
  f2_early_fn [task][code] = early_fn;
  f2_late_fn [task][code] = late_fn;
}


void uninstall_bus_select_fn (int task, int code)
{
  bus_select_early_fn [task][code] = bs_fn_bad;
  bus_select_late_fn [task][code] = bs_fn_bad;
}

void uninstall_f1_fn (int task, int code)
{
  f1_early_fn [task][code] = f1_fn_bad;
  f1_late_fn [task][code] = f1_fn_bad;
}

void uninstall_f2_fn (int task, int code)
{
  f2_early_fn [task][code] = f2_fn_bad;
  f2_late_fn [task][code] = f2_fn_bad;
}


void init_function_dispatch (void)
{
  int task, i;

  for (task = 0; task < TASK_COUNT; task++)
    {
      ram_related [task] = false;

      for (i = 0; i < 8; i++)
	{
	  install_bus_select_fn (task, i, & bs_fn_bad, & bs_fn_bad);
	}

      install_bus_select_fn (task, bs_read_r, & bs_fn_read_r_early, NULL);
      install_bus_select_fn (task, bs_load_r, & bs_fn_load_r_early, & bs_fn_load_r_late);
      install_bus_select_fn (task, bs_no_source, NULL, NULL);
      install_bus_select_fn (task, bs_mem_data, & bs_fn_mem_data_early, NULL);
      install_bus_select_fn (task, bs_mouse, & bs_fn_mouse_early, NULL);

      for (i = 0; i < 16; i++)
	{
	  install_f1_fn (task, i, & f1_fn_bad, & f1_fn_bad);
	  install_f2_fn (task, i, & f2_fn_bad, & f2_fn_bad);
	}

      install_f1_fn (task, f1_nop,          NULL, NULL);
      install_f1_fn (task, f1_load_mar,     NULL, & f1_fn_load_mar_late);
      install_f1_fn (task, f1_task,         & f1_fn_task_early, NULL);
#if 0
      install_f1_fn (task, f1_block,        & f1_fn_block_early, NULL);
      /* don't install f1_block here - it's not appropriate for all tasks */
#endif

      // shifts and constants are handled elsewhere
      install_f1_fn (task, f1_l_lsh_1,      NULL, NULL);
      install_f1_fn (task, f1_l_rsh_1,      NULL, NULL);
      install_f1_fn (task, f1_l_lcy_8,      NULL, NULL);
      install_f1_fn (task, f1_constant,     NULL, NULL);

      install_f2_fn (task, f2_nop,          NULL, NULL);
      install_f2_fn (task, f2_bus_eq_0,     NULL, & f2_fn_bus_eq_0_late);
      install_f2_fn (task, f2_shifter_lt_0, NULL, & f2_fn_shifter_lt_0_late);
      install_f2_fn (task, f2_shifter_eq_0, NULL, & f2_fn_shifter_eq_0_late);
      install_f2_fn (task, f2_bus,          NULL, & f2_fn_bus_late);
      install_f2_fn (task, f2_alucy,        NULL, & f2_fn_alucy_late);
      install_f2_fn (task, f2_load_md,      NULL, & f2_fn_load_md_late);

      // constants are handled elsewhere
      install_f2_fn (task, f2_constant,     NULL, NULL);
    }
}


void print_uinst (uinst_t *uinst)
{
  printf ("rsel %02o, aluf %02o, bs: %o, f1: %02o, f2: %02o, t %d, l %d, next %04o\n",
	  uinst->rsel,
	  uinst->aluf,
	  uinst->bs,
	  uinst->f1,
	  uinst->f2,
	  uinst->t,
	  uinst->l,
	  uinst->next);
}


static void alu (void)
{
  uint32_t result;
  switch (current_uinst->aluf)
    {
    case aluf_bus__alut:
      result = bus;
      break;
    case aluf_treg:
      result = t;
      break;
    case aluf_bus_or_t__alut:
      result = bus | t;
      break;
    case aluf_bus_and_t:
      result = bus & t;
      break;
    case aluf_bus_xor_t:
      result = bus ^ t;
      break;
    case aluf_bus_plus_1__alut:
      result = bus + 1;
      break;
    case aluf_bus_minus_1__alut:
      /* result = bus - 1; */
      result = bus + 0177777;
      break;
    case aluf_bus_plus_t:
      result = bus + t;
      break;
    case aluf_bus_minus_t:
      /* result = bus - t; */
      result = bus + (t ^ 0177777) + 1;
      break;
    case aluf_bus_minus_t_minus_1:
      /* result = (bus - t) - 1; */
      result = bus + (t ^ 0177777);
      break;
    case aluf_bus_plus_t_plus_1__alut:
      result = bus + t + 1;
      break;
    case aluf_bus_plus_skip__alut:
      result = bus + skip;
      break;
    case aluf_bus_and_t__alut:
      result = bus & t;
      break;
    case aluf_bus_and_not_t:
      result = bus & ~ t;
      break;
    case aluf_zero:
      result = 0;
      break;
    case aluf_undef:
      illegal_instruction = 1;
      result = 0;
    }
  alu_out = (result & 0177777);
  alu_carry_out = ((result & 0200000) != 0);
}

static uint16_t shifter (void)
{
  uint16_t result;
  int shifter_function;
  int magic, dns;

  magic = (current_task == task_emulator) && (current_uinst->f2 == f2_em_magic);
  dns   = (current_task == task_emulator) && (current_uinst->f2 == f2_em_dns);

  if (dns)
    shifter_function = (ir >> 6) & 2;
  else
    switch (current_uinst->f1)
      {
      case f1_l_lsh_1:	shifter_function = 1;  break;
      case f1_l_rsh_1:	shifter_function = 2;  break;
      case f1_l_lcy_8:	shifter_function = 3;  break;
      default:          shifter_function = 0;  break;
      }

  switch (shifter_function)
    {
    case 0:
      shifter_carry_out = l_alu_carry;
      result = l;
      break;
    case 1:
      shifter_carry_out = l >> 15;
      result = (l << 1) & 0177776;
      if (magic && (t & 0100000))
	result |= 1;
      else if (dns)
	result |= l_alu_carry;
      break;
    case 2:
      shifter_carry_out = l & 1;
      result = (l >> 1);
      if (magic && (t & 1))
	result |= 0100000;
      else if (dns)
	result |= (l_alu_carry << 15);
      break;
    case 3:
      shifter_carry_out = l_alu_carry;
      result = (l >> 8) | ((l & 0377) << 8);
      break;
    }
  return (result);
}


static void rdram (void)
{
  int addr;
  uint32_t val;

  if (cram_addr & 004000)
    {
      /* read ROM 0 at current upc */
      addr = current_upc & 01777;
    }
  else
    {
      /* read RAM at cram_addr */
      addr = (cram_addr & 001777) +
	UCODE_RAM_BASE + ((cram_addr >> 2) & 006000);
    }
  val = ucode_raw [addr];
  if (cram_addr & 002000)
    drive_bus ("rdram", (val >> 16) ^ (UCODE_INVERTED_BITS >> 16));
  else
    drive_bus ("rdram", (val & 0177777) ^ (UCODE_INVERTED_BITS & 0xffff));
  rdram_flag = 0;
}


static void wrtram (void)
{
  int addr;
  addr = (cram_addr & 001777) +
    UCODE_RAM_BASE + ((cram_addr >> 2) & 006000);
  ucode_raw [addr] = (((l ^ (UCODE_INVERTED_BITS >> 16)) << 16) |
		      (alu_out ^ (UCODE_INVERTED_BITS & 0xffff)));
  decode_uinst (addr);
  wrtram_flag = 0;
}


void start_rdram (void)
{
  rdram_flag = 1;
}


void start_wrtram (void)
{
  wrtram_flag = 1;
}


// Indexed by ALUF field of microinstruction.  If true, and T register is
// loaded, the data source for the T register will be the ALU output rather
// than the bus.
static bool t_source [16] =
{
  1,  // bus 
  0,  // T
  1,  // bus or T
  0,  // bus and T
  0,  // bus xor T
  1,  // bus + 1
  1,  // bus - 1
  0,  // bus + T
  0,  // bus - T
  0,  // bus - T - 1
  1,  // bus + T + 1
  1,  // bus + skip
  1,  // bus and T
  0,  // bus and not T
  0,  // undefined
  0   // undefined
};


void pre_instruction_trace (void)
{
  printf ("time %" PRI_u_SIM_TIME ", %s-%04o: ",
	  current_time,
	  task_name [next_task],
	  upc [next_task]);

  print_uinst (& ucode [upc [next_task]]);
}


void cpu_cycle (void)
{
  bool do_bs;
  dispatch_fn_t f;

  /* possible task switching */
  current_task = next_task;
  next_task = next2_task;

  /* "fetch" microinstruction */
  current_upc = upc [current_task];

  current_uinst = & ucode [current_upc];

  rsel = current_uinst->rsel;
#if 0
  if (current_trace_level >= 1)
    print_uinst ();
#endif

  do_bs = ((current_uinst->f1 != f1_constant) &&
	   (current_uinst->f2 != f2_constant));

  if (do_bs && (current_uinst->bs == bs_mem_data) &&
      check_read_main_mem_stall ())
    {
      tprintf (1, "memory read stall\n");
      return;
    }

  if ((current_uinst->f2 == f2_load_md) &&
      (current_uinst->f1 != f1_load_mar) &&
      check_write_main_mem_stall ())
    {
      tprintf (1, "memory write stall\n");
      return;
    }

  /*
   * point upc to next instruction, with any address modification from
   * previous instruction
   */
  upc [current_task] = (current_upc & ~ 01777) |
    current_uinst->next | address_modifier;

  address_modifier = 0;  /* no address modifier yet for this instruction */
  bus = 0xffff;  /* precharge the wire-AND data bus */
  illegal_instruction = 0;  /* assume legal */

  if (rdram_flag)
    rdram ();

  do_bs = ((current_uinst->f1 != f1_constant) &&
	   (current_uinst->f2 != f2_constant));

  if ((current_uinst->f1 == f1_constant) ||
      (current_uinst->f2 == f2_constant) ||
      (current_uinst->bs >= 4))
    drive_bus ("constant", constant_rom [(current_uinst->rsel << 3) + current_uinst->bs]);

  /* early-f2 has to be done before early-bs because emulator f2 acsource or
     acdest may change rsel */
  f = f2_early_fn [current_task][current_uinst->f2];
  if (f)
    (* f) (current_task, current_uinst->f2);

  if (do_bs)
    {
      f = bus_select_early_fn [current_task][current_uinst->bs];
      if (f)
	(* f) (current_task, current_uinst->bs);
    }

  f = f1_early_fn [current_task][current_uinst->f1];
  if (f)
    (* f) (current_task, current_uinst->f1);

  alu ();
  shifter_out = shifter ();

  if (do_bs)
    {
      f = bus_select_late_fn [current_task][current_uinst->bs];
      if (f)
	(* f) (current_task, current_uinst->bs);
    }

  f = f1_late_fn [current_task][current_uinst->f1]; 
  if (f)
    (* f) (current_task, current_uinst->f1);

  f = f2_late_fn [current_task][current_uinst->f2];
  if (f)
    (* f) (current_task, current_uinst->f2);

  /* wrtram uses the last L register, must do this before updating L */
  if (wrtram_flag)
    wrtram ();

  if (current_uinst->l)
    {
      l = alu_out;
      if (ram_related [current_task])
	m = alu_out;
      l_alu_carry = alu_carry_out;
      tprintf (1, "alucy, l <= %o, %06o\n", l_alu_carry, l);
    }

  if (current_uinst->t)
    {
      if (t_source [current_uinst->aluf])
	t = alu_out;
      else
	t = bus;
      cram_addr = alu_out;
      tprintf (1, "t <= %06o\n", t);
    }
}


void set_task_request (int task)
{
  task_request |= (1 << task);
}

void clear_task_request (int task)
{
  task_request &= ~ (1 << task);
}

void set_task_autoblock (int task)
{
  task_autoblock |= (1 << task);
}

