/*
 * Altogether: Xerox Alto microcode-level simulator
 * GUI main
 * $Id: gui.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "alto.h"
#include "debug.h"
#include "gui.h"
#include "cpu.h"
#include "main.h"
#include "display.h"


GdkColor white = { 0, 0xffff, 0xffff, 0xffff };
GdkColor black = { 0, 0x0000, 0x0000, 0x0000 };

static GdkPixmap *screen_pixmap;

GtkWidget *main_window;
GtkWidget *screen_drawing_area_widget;


static gint screen_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
  if (screen_pixmap)
    {
      gdk_draw_pixmap (widget->window,
		       widget->style->fg_gc [GTK_WIDGET_STATE (widget)],
		       screen_pixmap,
		       event->area.x, event->area.y,
		       event->area.x, event->area.y,
		       event->area.width, event->area.height);
    }
  return (FALSE);
}


void gui_quit ()
{
  set_debugger_state (false);
  gtk_exit (0);
}


static void mi_quit (gpointer data,
		     guint action,
		     GtkWidget *widget)
{
  gtk_main_quit ();
}

static void mi_go (gpointer data,
		   guint action,
		   GtkWidget *widget)
{
  sim_go ();
}


static void mi_step (gpointer data,
		     guint action,
		     GtkWidget *widget)
{
  sim_step (1);
}


static void mi_trace (gpointer data,
		      guint action,
		      GtkWidget *widget)
{
  sim_trace ();
}

static void mi_stop (gpointer data,
		     guint action,
		     GtkWidget *widget)
{
  sim_stop ();
}



static void mi_about (gpointer data,
		      guint action,
		      GtkWidget *widget)
{
  g_message ("about");
}


static GtkItemFactoryEntry menu_items [] =
{
  { "/_File",           NULL, 0,        0, "<Branch>" },
  { "/File/Quit",       NULL, mi_quit,  0, NULL }, 
  { "/_Execution",      NULL, 0,        0, "<Branch>" },
  { "/Execution/Go",    NULL, mi_go,    0, NULL },
  { "/Execution/Step",  NULL, mi_step,  0, NULL },
  { "/Execution/Trace", NULL, mi_trace, 0, NULL },
  { "/Execution/Stop",  NULL, mi_stop,  0, NULL },
  { "/_Help",           NULL, 0,        0, "<Branch>" },
  { "/Help/About",      NULL, mi_about, 0, NULL }
};

static int menu_item_count = sizeof (menu_items) / sizeof (menu_items [0]);


void gui_create_main_window (void)
{
  GtkWidget *vbox;
  GtkAccelGroup *accel;
  GtkItemFactory *menu_factory;

  main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
		      GTK_SIGNAL_FUNC (gui_quit), NULL);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (main_window), vbox);

  accel = gtk_accel_group_new ();

  menu_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
				       "<foo>",
				       accel);

  gtk_item_factory_create_items (menu_factory,
				 menu_item_count,
				 menu_items,
				 NULL);

  gtk_window_add_accel_group (GTK_WINDOW (main_window), accel);

  gtk_box_pack_start (GTK_BOX (vbox),
		      gtk_item_factory_get_widget (menu_factory, "<foo>"),
		      FALSE, FALSE, 0);

  screen_drawing_area_widget = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA (screen_drawing_area_widget),
			 PIXELS_PER_LINE,
			 ACTIVE_LINE_COUNT);
  gtk_box_pack_start (GTK_BOX (vbox), screen_drawing_area_widget,
		      TRUE, TRUE, 0);

  gtk_signal_connect (GTK_OBJECT (screen_drawing_area_widget), "expose_event",
		      (GtkSignalFunc) screen_expose_event, NULL);

  gtk_widget_show_all (main_window);
}


static gint memory_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
  return (FALSE);
}


void gui_create_memory_window (void)
{
  GtkWidget *window;
  GtkWidget *drawing_area;
  GtkWidget *vbox;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  vbox = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (gui_quit), NULL);

  drawing_area = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area),
			 20, 20);
  gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

  gtk_widget_show (drawing_area);

  gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
		      (GtkSignalFunc) memory_expose_event, NULL);

  gtk_widget_show (window);
}


void gui_update_frame (uint8_t *frame_buffer)
{
  GdkRectangle update_rect;

  if (screen_pixmap)
    gdk_pixmap_unref (screen_pixmap);
  screen_pixmap = gdk_pixmap_create_from_data (main_window->window,
					       frame_buffer,
					       PIXELS_PER_LINE,
					       ACTIVE_LINE_COUNT,
					       -1, /* depth */
					       & white,  /* fg color */
					       & black);  /* bg color */

  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width  = screen_drawing_area_widget->allocation.width;
  update_rect.height = screen_drawing_area_widget->allocation.height;
  gtk_widget_draw (screen_drawing_area_widget, & update_rect);
}


void init_gui (int *argc, char **argv [])
{
  gtk_init (argc, argv);

  gdk_color_alloc (gdk_colormap_get_system (), & white);
  gdk_color_alloc (gdk_colormap_get_system (), & black);

  gui_create_main_window ();
  install_frame_ready_fn (& gui_update_frame);

  /* gui_create_memory_window (); */
}


void gui_main (void)
{
  gtk_main ();
}


static idle_fn_t idle_fn;
static bool idle_installed;
static guint idle_handler_id;

static bool in_idle_handler;


static gboolean idle_caller (gpointer data)
{
  in_idle_handler = true;
  if (idle_fn)
    idle_fn ();
  in_idle_handler = false;
  /*
   * If someone called remove_idle() while the idle handler was
   * executing, it wasn't actually removed, so we return false
   * here and it will be removed for us.
   */
  return (idle_installed);
}


static void install_idle (void)
{
#ifdef GTK_IDLE_DEBUG
  printf ("installing idle function\n");
#endif /* GTK_IDLE_DEBUG */
  idle_handler_id = gtk_idle_add (& idle_caller, NULL);
  idle_installed = true;
}

static void remove_idle (void)
{
#ifdef GTK_IDLE_DEBUG
  printf ("removing idle function\n");
#endif /* GTK_IDLE_DEBUG */
  /*
   * Note: if we're currently executing in the idle handler,
   * we're not allowed to call gtk_idle_remove because it will
   * corrupt the list.  So instead we just note that the idle
   * function is (or more properly, should be) removed.
   */
  if (! in_idle_handler)
    gtk_idle_remove (idle_handler_id);
  idle_installed = false;
}

/* install the function that the GUI may call when idle */
void gui_set_idle_function (void (* fn)(void))
{
  if (idle_installed)
    {
      remove_idle ();
      idle_fn = fn;
      if (idle_fn)
	install_idle ();
    }
  else
    idle_fn = fn;
}


/* enable/disable the idle function */
void gui_set_idle (bool enable)
{
  if (enable)
    {
      if (! idle_installed)
	install_idle ();
    }
  else
    {
      if (idle_installed)
	remove_idle ();
    }
}

/* the idle function should periodically call back into the GUI to
   check for events */
void gui_idle_callback (void)
{
  while (gtk_events_pending ())
    gtk_main_iteration();
}


int gui_set_io_callback (int fd, int events, io_fn_t fn, void *data)
{
  return (gdk_input_add (fd, events, (GdkInputFunction) fn, data));
}

void gui_clear_io_callback (int tag)
{
  gdk_input_remove (tag);
}
