/*
 * Altogether: Xerox Alto microcode-level simulator
 * Scheduler
 * $Id: scheduler.c 121 2005-01-20 02:03:33Z eric $
 * Copyright 2001 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "priority_queue.h"
#include "scheduler.h"


sim_time_t current_time;

static priority_queue_t *q;


typedef struct
{
  event_fn_t event_fn;
  uint32_t arg1;
  void *arg2;  /* used in free list */
} event_t;


static event_t *event_free_list;


static event_t *alloc_event (void)
{
  event_t *event = event_free_list;
  if (event)
    event_free_list = event->arg2;
  else
    {
      event = malloc (sizeof (event_t));
      if (! event)
	return (NULL);
    }
  memset (event, 0, sizeof (event_t));
  return (event);
}


static void free_event (event_t *event)
{
  event->arg2 = event_free_list;
  event_free_list = event;
}


void init_scheduler (void)
{
  current_time = 0;
  event_free_list = NULL;
  q = new_priority_queue (MAX_QUEUED_EVENTS);
  if (! q)
    {
      fprintf (stderr, "scheduler failed to create priority queue\n");
      exit (3);
    }
}


sim_time_t next_scheduled_event_time (void)
{
  return (priority_queue_head (q));
}


void run_scheduler_until (sim_time_t time)
{
  event_t *event;
  sim_time_t new_time;

  while ((new_time = priority_queue_head (q)) <= time)
    {
      current_time = new_time;
      priority_queue_remove (q, (priority_t *) & new_time, (void **) & event);
      (* event->event_fn)(event->arg1, event->arg2);
      free_event (event);
    }
  current_time = time;
}


bool schedule_event (uint64_t time, event_fn_t event_fn, uint32_t arg1, void *arg2)
{
  event_t *event;

  if (time <= current_time)
    return (false);

  event = alloc_event ();
  if (! event)
    {
      fprintf (stderr, "scheduler failed to allocate event\n");
      exit (3);
    }
  event->event_fn = event_fn;
  event->arg1 = arg1;
  event->arg2 = arg2;
  if (! priority_queue_add (q, time, event))
    {
      fprintf (stderr, "scheduler failed to insert event into priority queue\n");
      exit (3);
    }
  return (true);
}
