/*
 * Altogether: Xerox Alto microcode-level simulator
 * Main CPU simulation
 * $Id: cpu.h 125 2005-04-09 07:11:14Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

/* no ethernet installed */
#define ETHERNET_NODE_ID 0377


#define CPU_MICROCYCLE_TIME 170 /* ns */


#undef MICROINSTRUCTION_USE_BITFIELDS
#if defined(MICROINSTRUCTION_USE_BITFIELDS)
  #define declare_uinst_field(name,bits) unsigned int name:bits
#else
  #define declare_uinst_field(name,bits) unsigned int name
#endif

typedef struct
{
  declare_uinst_field (rsel, 5);
  declare_uinst_field (aluf, 4);
  declare_uinst_field (bs,   3);
  declare_uinst_field (f1,   4);
  declare_uinst_field (f2,   4);
  declare_uinst_field (t,    1);
  declare_uinst_field (l,    1);
  declare_uinst_field (next, 10);
} uinst_t;


#define TASK_COUNT 16

typedef enum
{
  task_emulator,
  task_1, task_2, task_3,
  task_disk_sector,
  task_5, task_6,
  task_ethernet,
  task_mem_refresh,
  task_display_word,
  task_cursor,
  task_display_horizontal,
  task_display_vertical,
  task_parity,
  task_disk_word,
  task_17
} task_t;


#if ALTO_2K
#define UCODE_ROM_PAGES 1
#define UCODE_RAM_PAGES 2
#define S_REG_BANKS 1
#elsif ALTO_3K
#define UCODE_ROM_PAGES 1
#define UCODE_RAM_PAGES 3
#define S_REG_BANKS 8
#else
#define UCODE_ROM_PAGES 1
#define UCODE_RAM_PAGES 1
#define S_REG_BANKS 1
#endif


extern uint64_t current_cycle;  /* count of 170 ns microinstruction cycles */


#define CONSTANT_ROM_SIZE 256

extern uint16_t l, m, t;
extern int l_alu_carry;

extern int rsel;  /* from rsel field of microinstruction, but may be
		     modified by emulator acsource and acdest uops */
extern uint16_t r [32];
extern uint16_t s [S_REG_BANKS][32];

extern uint16_t reset_mode_register;

extern char *task_name [];
extern int current_task;
extern int next_task;
extern int next2_task;

extern bool task_switch;


extern uinst_t *current_uinst;

extern bool ram_related [TASK_COUNT];
extern int s_reg_bank [TASK_COUNT];

extern uint16_t address_modifier;
#define branch(cause,val)  do { address_modifier |= ((val) & 01777); tprintf (1, "branch %o caused by %s\n", val, cause); } while (0)

uint16_t upc [TASK_COUNT];

extern uint16_t bus;  /* wire-AND */
#define drive_bus(source,val)  do { uint16_t x = (val); bus &= x; tprintf (1, "%s driving bus, val %06o\n", source, x); } while (0)

extern uint16_t shifter_out;  /* used by emulator dns function */
bool shifter_carry_out;


typedef void (* dispatch_fn_t)(int task, int code);

void f1_fn_block_early (int task, int code);

void install_bus_select_fn (int task,
			    int code,
			    dispatch_fn_t early_fn,
			    dispatch_fn_t late_fn);

void install_f1_fn (int task,
		    int code,
		    dispatch_fn_t early_fn,
		    dispatch_fn_t late_fn);

void install_f2_fn (int task,
		    int code,
		    dispatch_fn_t early_fn,
		    dispatch_fn_t late_fn);

void uninstall_bus_select_fn (int task, int code);
void uninstall_f1_fn (int task, int code);
void uninstall_f2_fn (int task, int code);


typedef enum {
  aluf_bus__alut,
  aluf_treg,
  aluf_bus_or_t__alut,
  aluf_bus_and_t,
  aluf_bus_xor_t,
  aluf_bus_plus_1__alut,
  aluf_bus_minus_1__alut,
  aluf_bus_plus_t,
  aluf_bus_minus_t,
  aluf_bus_minus_t_minus_1,
  aluf_bus_plus_t_plus_1__alut,
  aluf_bus_plus_skip__alut,
  aluf_bus_and_t__alut,
  aluf_bus_and_not_t,
  aluf_zero,
  aluf_undef
} aluf_t;


typedef enum {
  bs_read_r,
  bs_load_r,
  bs_no_source,
  bs_task_3,
  bs_task_4,
  bs_mem_data,
  bs_mouse,
  bs_disp
} bs_t;


typedef enum {
  f1_nop,
  f1_load_mar,
  f1_task,
  f1_block,
  f1_l_lsh_1,
  f1_l_rsh_1,
  f1_l_lcy_8,
  f1_constant,
  f1_task_10, f1_task_11, f1_task_12, f1_task_13,
  f1_task_14, f1_task_15, f1_task_16, f1_task_17
} f1_t;


typedef enum {
  f2_nop,
  f2_bus_eq_0,
  f2_shifter_lt_0,
  f2_shifter_eq_0,
  f2_bus,
  f2_alucy,
  f2_load_md,
  f2_constant,
  f2_task_10, f2_task_11, f2_task_12, f2_task_13,
  f2_task_14, f2_task_15, f2_task_16, f2_task_17
} f2_t;


/* a task set to autoblock will clear its own request once the task
   gets control */
void set_task_autoblock (int task);

void set_task_request (int task);
void clear_task_request (int task);


void start_rdram (void);
void start_wrtram (void);


void load_microcode (char *zip_fn);


bool check_ubreak (void);
void set_ubreak (uint16_t addr, bool value);

void init_function_dispatch (void);
void init_cpu_mmio (void);
void soft_reset (void);
void hard_reset (void);

void print_uinst (uinst_t *uinst);


void pre_instruction_trace (void);
void cpu_cycle (void);
