/*
 * Altogether: Xerox Alto microcode-level simulator
 * Debug support
 * $Id: debug.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 1996, 2001, 2003 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */


#include <ctype.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef USE_BENCHMARK
#include <signal.h>
#include <sys/time.h>
#endif /* USE_BENCHMARK */

#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#ifdef USE_TCL
#include <tcl.h>
#endif

#include "alto.h"
#include "debug.h"
#include "scheduler.h"
#include "cpu.h"
#include "main.h"
#include "main_mem.h"

#ifdef USE_GUI
#include "gui.h"
#endif /* USE_GUI */


#ifdef USE_TCL
Tcl_Interp *tcl_interp;
#endif /* USE_TCL */

static char *debugger_prompt = "> ";

int stepflag;
bool traceflag;


static int next_addr = 0;


int current_trace_level;


void tprintf (int trace_level, char *fmt, ...)
{
  va_list ap;

  if (traceflag && (current_trace_level >= trace_level))
    {
      va_start (ap, fmt);
      vfprintf (stdout, fmt, ap);
      va_end (ap);
    }
}

int current_debug_level;
void dprintf (int debug_level, char *fmt, ...)
{
  va_list ap;

  if (current_debug_level >= debug_level)
    {
      va_start (ap, fmt);
      vfprintf (stdout, fmt, ap);
      va_end (ap);
    }
}


static unsigned long parse_address (char *s)
{
  unsigned long val;
  char *p2;
  
  val = strtoul (s, & p2, 0);
  if (*p2 != '\0')
    return (-1);

  if ((val >= 0) && (val <= 0xffff))
    return (val);
  
  return (-1);
}


static void list_ubreak (void)
{
  printf ("not yet implemented\n");
}


#ifdef USE_TCL
#define CMD_ARGS ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]
#else
#define CMD_ARGS int argc, char *argv[]
#endif

static int ubreak_cmd (CMD_ARGS)
{
  int uaddr;
  if ((argc < 1) || (argc > 2))
    return (-1);
  if (argc == 1)
    {
      list_ubreak ();
      return (0);
    }
  uaddr = parse_address (argv [1]);
  if (uaddr < 0)
    return (-1);
  set_ubreak (uaddr, true);
  return (0);
}


static int uclear_cmd (CMD_ARGS)
{
  int uaddr;
  if (argc != 2)
    return (-1);
  uaddr = parse_address (argv [1]);
  if (uaddr < 0)
    return (-1);
  set_ubreak (uaddr, false);
  return (0);
}


static void examine_range (int addr1, int addr2)
{
  int addr;
  int data;

  for (; addr1 <= addr2; addr1 += 8)
    {
      printf ("%06o: ", addr1);
      for (addr = addr1; addr < (addr1 + 8); addr++)
	{
	  if (addr <= addr2)
	    {
	      data = debug_read_mem (addr);
	      printf ("%06o ", data);
	    }
	  else
	    printf ("       ");
	}
      for (addr = addr1; addr < (addr1 + 8); addr++)
	{
	  if (addr <= addr2)
	    {
	      uint8_t c1, c2;
	      data = debug_read_mem (addr);
	      c1 = data >> 8;
	      c2 = data & 0xff;
	      if ((c1 >= 0x20) && (c1 <= 0x7e))
		printf ("%c", c1);
	      else
		printf (".");
	      if ((c2 >= 0x20) && (c2 <= 0x7e))
		printf ("%c", c2);
	      else
		printf (".");
	    }
	  else
	    printf (" ");
	}
      printf ("\n");
    }
}


static int examine_cmd (CMD_ARGS)
{
  int first, last;

  if ((argc < 1) || (argc > 3))
    return (-1);

  first = next_addr;
  if (argc >= 2)
    first = parse_address (argv [1]);
  last = first + 63;
  if (argc == 3)
    last = parse_address (argv [2]);

  if ((first < 0) || (last < first))
    return (-1);

  examine_range (first, last);

  next_addr = last + 1;

  return (0);
}


static int deposit_cmd (CMD_ARGS)
{
  int addr, data;

  if (argc < 3)
    return (-1);

  addr = parse_address (argv [1]);
  if (addr < 0)
    return (-1);

  argc -= 2;
  argv += 2;

  while (argc--)
    {
      data = parse_address (argv [0]);
      if (data < 0)
	return (-1);
      debug_write_mem (addr++, data);
      argv++;
    }

  return (0);
}


static void skip_whitespace (char **p)
{
  while (isspace (**p))
    (*p)++;
}


static int load_from_file (FILE *f)
{
  char *p, *p2;
  int len;
  unsigned long int addr, data;
  char buffer [300];

  while (fgets (buffer, sizeof (buffer), f))
    {
      len = strlen (buffer);
      if (buffer [len - 1] == '\n')
	buffer [len - 1] = '\0';
      p = & buffer [0];
      skip_whitespace (& p);
      if ((! *p) || (*p == '#'))
	continue;
      addr = strtoul (p, & p2, 8);
      if (p2 == p)
	return (1);
      p = p2;
      skip_whitespace (& p);
      if (*p != ':')
	return (2);
      p++;
      while (*p)
	{
	  skip_whitespace (& p);
	  if (*p == '#')
	    break;
	  data = strtoul (p, & p2, 8);
	  if (p2 == p)
	    return (3);
	  p = p2;
	  debug_write_mem (addr++, data);
	}
    }

  return (0);
}


static int load_cmd (CMD_ARGS)
{
  FILE *f;
  int result;

  if (argc != 2)
    return (-1);

  f = fopen (argv [1], "r");
  if (! f)
    {
      fprintf (stderr, "error opening file\n");
      return (1);
    }

  result = load_from_file (f);
  if (result)
    fprintf (stderr, "error %d loading file\n", result);

  fclose (f);
  return (result);
}


static int load_binary_file (FILE *f, unsigned long addr)
{
  uint8_t data [2];
  unsigned long count = 0;

  while (2 == fread (& data [0], 1, 2, f))
    {
      debug_write_mem (addr++, (data [0] << 8) | data [1]);
      count++;
    }

  printf ("loaded %lu words\n", count);

  return (0);
}


static int load_bin_cmd (CMD_ARGS)
{
  FILE *f;
  int result;
  unsigned long addr;
  char *p2;

  if ((argc < 2) || (argc > 3))
    return (-1);

  if (argc == 3)
    {
      addr = strtoul (argv [2], & p2, 0);
      if (*p2 != '\0')
	return (-1);
    }
  else
    addr = 1;

  f = fopen (argv [1], "rb");
  if (! f)
    {
      fprintf (stderr, "error opening file\n");
      return (1);
    }

  result = load_binary_file (f, addr);
  if (result)
    fprintf (stderr, "error %d loading file\n", result);

  fclose (f);
  return (result);
}


#ifdef USE_BENCHMARK
static void alarm_handler (int signo)
{
  clear_runflag (INTBREAK);
}


static void set_alarm (unsigned long sec)
{
  struct itimerval itv;

  itv.it_interval.tv_sec = sec;
  itv.it_interval.tv_usec = 0;
  itv.it_value.tv_sec = sec;
  itv.it_value.tv_usec = 0;
  setitimer (ITIMER_REAL, &itv, NULL);
  signal (SIGALRM, alarm_handler);
}


static int benchmark_cmd (CMD_ARGS)
{
  unsigned long sec;
  char *p2;

  if (argc != 2)
    return (-1);

  sec = strtoul (argv [1], & p2, 0);
  if (*p2 != '\0')
    return (-1);

  set_alarm (sec);

  stepflag = 0;
  traceflag = false;
  set_runflag();
  
  return (0);
} 
#endif /* USE_BENCHMARK */


static int go_cmd (CMD_ARGS)
{
  if (argc != 1)
    return (-1);
  
  sim_go ();
  
  return (0);
} 


static int step_cmd (CMD_ARGS)
{
  int count = 1;
  if ((argc < 1) || (argc > 2))
    return (-1);

  if (argc == 2)
    count = parse_address (argv [1]);

  sim_step (count);
  
  return (0);
}


static int trace_cmd (CMD_ARGS)
{
  if (argc != 1)
    return (-1);
  
  sim_trace ();

  return (0);
}


static int task_trace_cmd (CMD_ARGS)
{
  unsigned long task, val;
  char *p2;

  if ((argc < 2) || (argc > 3))
    return (-1);

  task = strtoul (argv [1], &p2, 0);
  if (*p2 != '\0')
    return (-1);

  if (task >= TASK_COUNT)
    return (-1);

  if (argc == 3)
    {
      val = strtoul (argv [2], &p2, 0);
      if (*p2 != '\0')
	return (-1);
    }
  else
    val = ~ task_switch_break [task];

  task_switch_break [task] = val;
  printf ("task %s (%lu) break %s\n", task_name [task], task,
	  val ? "enabled" : "disabled");
  
  return (0);
}


static int quit_cmd (CMD_ARGS)
{ 
  if (argc != 1)
    return (-1);
  exit (0);
}


static int xyzzy_cmd (CMD_ARGS)
{
  printf ("Nothing happens here.\n");
  return (0);
}


static int help_cmd (CMD_ARGS);

#ifdef USE_TCL
#define cmd_handler Tcl_CmdProc *
#else
typedef int (*cmd_handler)(CMD_ARGS);
#endif

typedef struct
{
  char *name;
  cmd_handler handler;
  int min_chr;
  char *usage;
} cmd_entry;

cmd_entry cmd_table [] =
{
  { "benchmark", benchmark_cmd, 2, "Benchmark <seconds>             benchmark\n" },
  { "deposit",   deposit_cmd,   1, "Deposit <addr> <value>...       deposit values into memory\n" },
#if 0
  { "disassemble", dis_cmd,     2, "DIsassemble [<addr> [<addr>]]   disassemble from addr\n" },
#endif
  { "examine", examine_cmd,     1, "Examine [<addr> [<addr>]]       examine memory\n" },
  { "go", go_cmd,           1, "Go                              go\n" },
  { "help", help_cmd,       1, "Help                            list commands\n\
" },
  { "lb", load_bin_cmd,     2, "LBinary <filename> [<addr>]     load a binary file\n" },
  { "load", load_cmd,       1, "Load <filename>                 load a file\n" },
  { "quit", quit_cmd,       4, "QUIT                            quit simulator\n" },
#if 0
  { "read", read_cmd,       3, "REAd <addr>                     read memory at addr\n" },
  { "register", reg_cmd,    1, "Register [<reg> <val>]          display or set registers\n" },
  { "save", save_cmd,       2, "SAve <filename>                 save the world\n" },
#endif
  { "step", step_cmd,       1, "Step                            single-step one cycle\n" },
  { "trace", trace_cmd,     1, "Trace                           go in trace mode\n" },
  { "tasktrace", task_trace_cmd,   2, "TAsktrace <task> [<bool>]       enable/disable task switch tracing\n" },
  { "ubreak", ubreak_cmd,   2, "UBreak [<addr>]                 list or set ucode breakpoints\n" },
  { "uclear", uclear_cmd,   2, "UClear [<addr>]                 clear ucode breakpoints\n" },
#if 0
  { "write", write_cmd,     1, "Write <addr> <val>              write val to memory at addr\n" },
#endif
  { "xyzzy", xyzzy_cmd,     1, "Xyzzy\n" },
  { NULL, NULL, 0, NULL }
};


static int find_cmd (char *s)
{
  int i;
  int len = strlen (s);

  for (i = 0; cmd_table [i].name; i++)
    {
      if ((len >= cmd_table [i].min_chr) &&
          (strncasecmp (s, cmd_table [i].name, len) == 0))
        return (i);
    }

  return (-1);
}


static int help_cmd (CMD_ARGS)
{
  int i;

  if (argc == 1)
    {
      for (i = 0; cmd_table [i].name; i++)
        fprintf (stderr, cmd_table [i].usage);
      fprintf (stderr, "\n"
                       "Commands may be abbreviated to the portion listed in caps.\n");
      return (0);
    }

  if (argc != 2)
    return (-1);

  i = find_cmd (argv [1]);
  if (i < 0)
    {
      fprintf (stderr, "unrecognized command\n");
      return (1);
    }
  
  fprintf (stderr, cmd_table [i].usage);
  return (0);
}


#ifndef USE_TCL
static void execute_command (CMD_ARGS)
{
  int i;
  
  i = find_cmd (argv [0]);

  if (i < 0)
    {
      fprintf (stderr, "unrecognized command\n");
      return;
    }
  
  if ((* cmd_table [i].handler)(argc, argv) < 0)
    fprintf (stderr, "Usage: %s", cmd_table [i].usage);
}
#endif


#ifndef USE_READLINE

#define MAX_LINE 200
/*
 * print prompt, get a line of input, return a copy
 * caller must free
 */
char *readline (char *prompt)
{
  char inbuf [MAX_LINE];

  if (prompt)
    printf (prompt);
  fgets (inbuf, MAX_LINE, stdin);
  return (strdup (& inbuf [0]));
}
#endif /* USE_READLINE */


#define MAX_ARGS 100


static void debugger_command (char *cmd)
{
#ifdef USE_TCL
  int result;
  char *result_string;
#else
  char *s;
  int argc;
  char *argv [MAX_ARGS];
#endif

#ifdef USE_READLINE
  if (*cmd)
    add_history (cmd);
#endif

#ifdef USE_TCL
  result = Tcl_Eval (tcl_interp, cmd);
  result_string = Tcl_GetStringResult (tcl_interp);
  if (result != TCL_OK)
    fprintf (stderr, "TCL error ");
  if (result_string && strlen (result_string))
    fprintf (stderr, "%s\n", result_string);
#else
  argc = 0;
  for (s = cmd; (argc < MAX_ARGS) && ((s = strtok (s, " \t\n")) != NULL); s = NULL)
    argv [argc++] = s;
  
  if (argc)
    execute_command (argc, argv);
#endif
}


void debugger (int type)
{
  char *cmd;

  stepflag = 0;
  if (type != STEP)
    traceflag = false;

  switch (type)
    {
    case INTBREAK: printf("User break\n");  break;
    case ERRORBRK: printf("Error\n");       break;
    case UBREAKPT: printf("ubreak\n");      break;
    case STEP:     break;
    default:       printf("badbrkpt\n"); break;
    }

#if 0
  dumpregs();
#endif

  while (! runflag)
    {
      pre_instruction_trace ();
      cmd = readline (debugger_prompt);
      if (cmd)
	{
	  debugger_command (cmd);
	  free (cmd);
	}
    }
}


static bool debugger_state;

#ifdef USE_GUI
static int io_callback_tag;
#endif /* USE_GUI */


#ifdef USE_READLINE
void stdin_read_callback (void *data, int fd, int event)
{
  rl_callback_read_char ();
}
#endif /* USE_READLINE */


void set_debugger_state (bool enable)
{
  if (enable == debugger_state)
    return;
  debugger_state = enable;
  if (enable)
    {
#ifdef USE_READLINE
      /* pre_instruction_trace (); */
      rl_callback_handler_install (debugger_prompt, & debugger_command);
#ifdef USE_GUI
      io_callback_tag = gui_set_io_callback (STDIN_FILENO,
					     GUI_IO_READ_EVENT,
					     stdin_read_callback,
					     NULL);
#endif /* USE_GUI */
#endif /* USE_READLINE */
    }
  else
    {
#ifdef USE_READLINE
      rl_callback_handler_remove ();
#ifdef USE_GUI
      gui_clear_io_callback (io_callback_tag);
#endif /* USE_GUI */
#endif /* USE_READLINE */
    }
}


char *init_files [] =
{
  ".altorc",
  "~/.altorc"
};


void debug_init (void)
{
#ifdef USE_TCL
  cmd_entry *ce;
  int i;
  int result;
  char *result_string;
  char buf [200];

  tcl_interp = Tcl_CreateInterp ();
  for (ce = & cmd_table [0]; ce->name != NULL; ce++)
    Tcl_CreateCommand (tcl_interp,
		       ce->name,
		       ce->handler,
		       NULL, 
		       NULL);
  for (i = 0; init_files [i]; i++)
    {
      sprintf (buf, "if [ file isfile \"%s\" ] { source \"%s\" }",
	       init_files [i], init_files [i]);
      result = Tcl_Eval (tcl_interp, buf);
      result_string = Tcl_GetStringResult (tcl_interp);
      if (result != TCL_OK)
	fprintf (stderr, "TCL error ");
      if (result_string && strlen (result_string))
	fprintf (stderr, "%s\n", result_string);
    }
#endif /* USE_TCL */
}


void debug_term (void)
{
#ifdef USE_TCL
  Tcl_DeleteInterp (tcl_interp);
#endif /* USE_TCL */
}
