/*
 * Altogether: Xerox Alto microcode-level simulator
 * Bit-mapped display simulation
 * $Id: display.c 126 2005-04-09 07:20:18Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <inttypes.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "alto.h"
#include "debug.h"
#include "cpu.h"
#include "scheduler.h"
#include "display.h"


#define f2_dwt_load_ddr   f2_task_10
#define f2_dht_evenfield  f2_task_10
#define f2_dvt_evenfield  f2_task_10
#define f2_dht_setmode    f2_task_11


#define DISPLAY_FIFO_SIZE 16 /* words */

#define TOTAL_LINES 875
#define FIRST_ACTIVE_LINE 32  /* picked arbitrarily */


#define PIXEL_TIME 50 /* ns */
#define NORMAL_WORD_TIME (16 * PIXEL_TIME)
#define WIDE_WORD_TIME (2 * (NORMAL_WORD_TIME))


#define LINE_TIME 38095 /* 1e9 / (30 * TOTAL_LINES) ns */
#define ACTIVE_LINE_TIME (PIXELS_PER_LINE * PIXEL_TIME) /* 30400 ns */

/*
 * The exact timing of events within a horizontal line is unknown.  I
 * need to either study the schematics or look at the real thing on a
 * scope.  So the following two constants are just wild guesses.
 */
#define ACTIVE_LINE_FETCH_TIME 170
#define ACTIVE_LINE_START_TIME (LINE_TIME - ((PIXELS_PER_LINE + 18) * PIXEL_TIME))


int display_trace;


#define dtprintf(lev,args) do { if (display_trace >= lev) printf args; } while (0)


uint64_t next_line_time;
int scan_line;
bool even_field;

static bool next_double_width;
static bool double_width;
static uint16_t inverse;


bool dwt_enable;
bool dht_enable;


uint16_t display_fifo [DISPLAY_FIFO_SIZE];
int display_fifo_in_ptr;
int display_fifo_out_ptr;
int display_fifo_depth;


#ifdef NATIVE_DISPLAY_MSB_LEFT
static uint16_t expand_bit [256];
#endif

#ifdef NATIVE_DISPLAY_LSB_LEFT
static uint8_t bit_reverse [256];
static uint16_t expand_bit_reverse [256];
#endif


static uint8_t *frame_buffer;
static uint8_t *line_ptr;
static uint8_t *word_ptr;
static int line_word_count;  /* space (words) remaining in frame buffer
					for current scan line */
static frame_ready_fn_t frame_ready_fn;



static void setup_bit_reverse (void)
{
  int i;

#ifdef NATIVE_DISPLAY_MSB_LEFT
  for (i = 0; i < 256; i++)
    {
      expand_bit [i] = (((i & 0x80) << 8) |   /* 7 => 15 */
			((i & 0x80) << 7) |   /* 7 => 14 */
			((i & 0x40) << 7) |   /* 6 => 13 */
			((i & 0x40) << 6) |   /* 6 => 12 */
			((i & 0x20) << 6) |   /* 5 => 11 */
			((i & 0x20) << 5) |   /* 5 => 10 */
			((i & 0x10) << 5) |   /* 4 => 9 */
			((i & 0x10) << 4) |   /* 4 => 8 */
			((i & 0x08) << 4) |   /* 3 => 7 */
			((i & 0x08) << 3) |   /* 3 => 6 */
			((i & 0x04) << 3) |   /* 2 => 5 */
			((i & 0x04) << 2) |   /* 2 => 4 */
			((i & 0x02) << 2) |   /* 1 => 3 */
			((i & 0x02) << 1) |   /* 1 => 2 */
			((i & 0x01) << 1) |   /* 0 => 1 */
			((i & 0x01)));        /* 0 => 0 */

#ifdef NATIVE_DISPLAY_MIN_IS_WHITE
      expand_bit [i] ^= 0xffff;
#endif /* NATIVE_DISPLAY_MIN_IS_WHITE */
    }
#endif /* NATIVE_DISPLAY_MSB_LEFT */

#ifdef NATIVE_DISPLAY_LSB_LEFT
  for (i = 0; i < 256; i++)
    {
      bit_reverse [i] = (((i & 0x01) << 7) |
			 ((i & 0x02) << 5) |
			 ((i & 0x04) << 3) |
			 ((i & 0x08) << 1) |
			 ((i & 0x10) >> 1) |
			 ((i & 0x20) >> 3) |
			 ((i & 0x40) >> 5) |
			 ((i & 0x80) >> 7));

      expand_bit_reverse [i] = (((i & 0x01) << 15) |  /* 0 => 15 */
				((i & 0x01) << 14) |  /* 0 => 14 */
				((i & 0x02) << 12) |  /* 1 => 13 */
				((i & 0x02) << 11) |  /* 1 => 12 */
				((i & 0x04) << 9) |   /* 2 => 11 */
				((i & 0x04) << 8) |   /* 2 => 10 */
				((i & 0x08) << 6) |   /* 3 => 9 */
				((i & 0x08) << 5) |   /* 3 => 8 */
				((i & 0x10) << 3) |   /* 4 => 7 */
				((i & 0x10) << 2) |   /* 4 => 6 */
				(i & 0x20) |          /* 5 => 5 */
				((i & 0x20) >> 1) |   /* 5 => 4 */
				((i & 0x40) >> 3) |   /* 6 => 3 */
				((i & 0x40) >> 4) |   /* 6 => 2 */
				((i & 0x80) >> 6) |   /* 7 => 1 */
				((i & 0x80) >> 7));   /* 7 => 0 */

#ifdef NATIVE_DISPLAY_MIN_IS_WHITE
      bit_reverse [i] ^= 0xff;
      expand_bit_reverse [i] ^= 0xffff;
#endif /* NATIVE_DISPLAY_MIN_IS_WHITE */
    }
#endif /* NATIVE_DISPLAY_LSB_LEFT */
}


void f1_fn_dwt_block_early (int task, int code)
{
  dwt_enable = false;
  clear_task_request (task_display_word);
  dtprintf (1, ("dwt blocks\n"));
}


void f1_fn_dht_block_early (int task, int code)
{
  dht_enable = false;
  clear_task_request (task_display_horizontal);
  dtprintf (1, ("dht blocks\n"));
  f1_fn_dwt_block_early (task, code);
}


void f2_fn_load_ddr_late (int task, int code)
{
  if (display_fifo_depth >= DISPLAY_FIFO_SIZE)
    {
      fprintf (stderr, "display FIFO overrun\n");
      clear_task_request (task_display_word);
      return;
    }
  display_fifo [display_fifo_in_ptr++] = bus;
  if (display_fifo_in_ptr == DISPLAY_FIFO_SIZE)
    display_fifo_in_ptr = 0;
  if (++display_fifo_depth == DISPLAY_FIFO_SIZE)
    clear_task_request (task_display_word);
  dtprintf (2, ("dwt stores %04x in FIFO, depth %d\n", bus, display_fifo_depth));
}

void f2_fn_evenfield_late (int task, int code)
{
  branch ("evenfield", even_field);
}

void f2_fn_setmode_late (int task, int code)
{
  next_double_width = bus >> 15;
  inverse = ((bus >> 14) & 1) ? 0xffff : 0x0000;
  branch ("setmode", next_double_width);
}


static void active_video_word (uint32_t arg1, void *arg2)
{
  int word;
  if (display_fifo_depth == 0)
    {
      dtprintf (1, ("display FIFO underrun\n"));
      word = inverse;
    }
  else
    {
      word = display_fifo [display_fifo_out_ptr++] ^ inverse;
      if (display_fifo_out_ptr == DISPLAY_FIFO_SIZE)
	display_fifo_out_ptr = 0;
      display_fifo_depth--;
      dtprintf (1, ("display pulls %04x from FIFO, depth %d\n", word, display_fifo_depth));
    }

  /* the FIFO obviously can't be full at this point */
  if (dwt_enable)
    set_task_request (task_display_word);

  if (line_word_count)
    {
#ifdef NATIVE_DISPLAY_MIN_IS_BLACK
      word ^= 0xffff;
#endif /* NATIVE_DISPLAY_MIN_IS_BLACK */
      if (double_width)
	{
	  uint16_t temp;
#ifdef NATIVE_DISPLAY_MSB_LEFT
	  temp = expand_bit [word >> 8];
#endif /* NATIVE_DISPLAY_MSB_LEFT */
#ifdef NATIVE_DISPLAY_LSB_LEFT
	  temp = expand_bit_reverse [word & 0xff];
#endif /* NATIVE_DISPLAY_LSB_LEFT */
	  *(word_ptr++) = (temp >> 8);
	  *(word_ptr++) = (temp & 0xff);

#ifdef NATIVE_DISPLAY_MSB_LEFT
	  temp = expand_bit [word & 0xff];
#endif /* NATIVE_DISPLAY_MSB_LEFT */
#ifdef NATIVE_DISPLAY_LSB_LEFT
	  temp = expand_bit_reverse [word >> 8];
#endif /* NATIVE_DISPLAY_LSB_LEFT */
	  *(word_ptr++) = (temp >> 8);
	  *(word_ptr++) = (temp & 0xff);
	  line_word_count -= 2;
	}
      else
	{
	  uint8_t b1, b2;
#ifdef NATIVE_DISPLAY_MSB_LEFT
	  b1 = word >> 8;
	  b2 = word & 0xff;
#endif /* NATIVE_DISPLAY_MSB_LEFT */
#ifdef NATIVE_DISPLAY_LSB_LEFT
	  b1 = bit_reverse [word & 0xff];
	  b2 = bit_reverse [word >> 8];
#endif /* NATIVE_DISPLAY_LSB_LEFT */
	  *(word_ptr++) = b1;
	  *(word_ptr++) = b2;
	  line_word_count--;
	}
    }
  if (line_word_count /*  && (display_fifo_depth || dht_enable) */)
    {
      dtprintf (2, ("display line has %d words left\n", line_word_count));
      schedule_event (current_time + (double_width ? WIDE_WORD_TIME : NORMAL_WORD_TIME),
		      active_video_word, 0, NULL);
    }
  else
    dtprintf (2, ("display line finished\n"));
}


static void start_line_fetch (uint32_t arg1, void *arg2)
{
  display_fifo_in_ptr = 0;
  display_fifo_out_ptr = 0;
  display_fifo_depth = 0;

  line_word_count = BYTES_PER_LINE / 2;
  
  set_task_request (task_display_word);
  dwt_enable = true;
  dtprintf (1, ("start_line_fetch\n"));
}


static void start_scan_line (uint32_t arg1, void *arg2)
{
  double_width = next_double_width;

  scan_line += 2;
  if (scan_line > TOTAL_LINES)
    {
      scan_line -= TOTAL_LINES;
      even_field = ! (scan_line & 1);
      if (even_field)
	{
	  line_ptr = frame_buffer + BYTES_PER_LINE;
	}
      else
	{
	  memset (frame_buffer, (inverse & 0xff) ^ 0xff, FRAME_BUFFER_SIZE);
	  line_ptr = frame_buffer;
	}
      set_task_request (task_display_vertical);
      dwt_enable = false;
      dht_enable = true;
    }

  dtprintf (1, ("%" PRI_u_SIM_TIME ": start of scan line %d\n", current_time, scan_line));

  if ((scan_line >= FIRST_ACTIVE_LINE) &&
      (scan_line < (FIRST_ACTIVE_LINE + ACTIVE_LINE_COUNT)))
    {
      word_ptr = line_ptr;
      line_ptr += (2 * BYTES_PER_LINE);
      if (dht_enable)
	{
	  set_task_request (task_display_horizontal);
	  schedule_event (next_line_time + ACTIVE_LINE_FETCH_TIME,
			  start_line_fetch, 0, NULL);
	  /* the following should really be outside the dht cond */
	  schedule_event (next_line_time + ACTIVE_LINE_START_TIME,
			  active_video_word, 0, NULL);
	}
    }
  else if (scan_line == (FIRST_ACTIVE_LINE + ACTIVE_LINE_COUNT))
    {
      dtprintf (1, ("frame complete\n"));
      if (frame_ready_fn)
	frame_ready_fn (frame_buffer);
    }

  next_line_time += LINE_TIME;
  schedule_event (next_line_time, start_scan_line, 0, NULL);
}


void install_display (void)
{
  install_f1_fn (task_display_word,       f1_block,         & f1_fn_dwt_block_early, NULL);
  install_f1_fn (task_display_horizontal, f1_block,         & f1_fn_dht_block_early, NULL);
  install_f2_fn (task_display_word,       f2_dwt_load_ddr,  NULL, & f2_fn_load_ddr_late);
  install_f2_fn (task_display_vertical,   f2_dvt_evenfield, NULL, & f2_fn_evenfield_late);
  install_f2_fn (task_display_horizontal, f2_dht_evenfield, NULL, & f2_fn_evenfield_late);
  install_f2_fn (task_display_horizontal, f2_dht_setmode,   NULL, & f2_fn_setmode_late);

  setup_bit_reverse ();

  frame_buffer = malloc (FRAME_BUFFER_SIZE);
  if (! frame_buffer)
    {
      fprintf (stderr, "failed to malloc frame buffer\n");
      exit (3);
    }

  set_task_autoblock (task_display_vertical);
  set_task_autoblock (task_display_horizontal);

  scan_line = TOTAL_LINES - 1;
  /* the first scan line will be 1 (after start_scan_line() adds 2) */

  next_line_time = LINE_TIME;
  schedule_event (next_line_time, start_scan_line, 0, NULL);
}


void install_frame_ready_fn (frame_ready_fn_t fn)
{
  frame_ready_fn = fn;
}
