/*
 * Altogether: Xerox Alto microcode-level simulator
 * Nova Macroinstruction Emulator simulation
 * $Id: emulator.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include "alto.h"
#include "debug.h"
#include "cpu.h"
#include "emulator.h"
#include "ram_related.h"


uint16_t ir;
int skip, carry;


void bs_fn_disp_early (int task, int code)
{
  if ((ir & 01400) && (ir & 02000))
    drive_bus ("disp", ir | 0177400);
  else
    drive_bus ("disp", ir & 0377);
}


void f1_fn_load_rmr_late (int task, int code)
{
  reset_mode_register = bus;
}

void f1_fn_esrb_late (int task, int code)
{
  s_reg_bank [task] = (bus >> 1) & 07;
}

void f1_fn_rsnf_early (int task, int code)
{
  drive_bus ("rsn", 0177400 | ETHERNET_NODE_ID);
}

void f1_fn_startf_late (int task, int code)
{
  /* ??? */
}

void f2_fn_busodd_late (int task, int code)
{
  branch ("busodd", bus & 1);
}

/* Do Novel Shifts */
void f2_fn_dns_early (int task, int code)
{
  rsel &= ~3;
  rsel |= (((ir >> 11) & 3) ^ 3);
}

void f2_fn_dns_late (int task, int code)
{
  bool new_carry;
  switch ((ir >> 4) & 3)
    {
    case 0:
      new_carry = carry;
      break;
    case 1:
      new_carry = 0;
      break;
    case 2:
      new_carry = 1;
      break;
    case 3:
      new_carry = ! carry;
      break;
    }
  new_carry ^= l_alu_carry;
  if (! (ir & 000010))
    {
      r [rsel] = shifter_out;
      carry = new_carry;
    }
  skip = ((ir & 1) ^
	  ((((ir >> 2) & 1) & (shifter_out == 0)) |
	   ((((ir >> 1) & 1) != 0) & (new_carry == 0))));
}

void f2_fn_acdest_early (int task, int code)
{
  rsel &= ~3;
  rsel |= (((ir >> 11) & 3) ^ 3);
}

void f2_fn_load_ir_late (int task, int code)
{
  ir = bus;
  skip = 0;
  branch ("load_ir", ((ir & 0100000) >> 12) | ((ir & 0003400) >> 8));
}

void f2_fn_idisp_late (int task, int code)
{
  int bval;
  /*
   * this all could be made more efficient by using an array
   * indexed by IR[0-9]
   */
  if (ir & 0100000)
    bval = 3 - ((ir >> 6) & 3);  /* complement of SH field of IR */
  else if ((ir & 0060000) == 0)
    bval = (ir >> 11) & 3;  /* JMP, JSR, ISZ, DSZ */
  else if ((ir & 0060000) == 0020000)
    bval = 4;  /* LDA */
  else if ((ir & 0060000) == 0040000)
    bval = 5;  /* STA */
  else if ((ir & 0007400) == 0)
    bval = 1;
  else if ((ir & 0007400) == 0000400)
    bval = 0;
  else if ((ir & 0007400) == 0003000)
    bval = 016;  /* CONVERT */
  else if ((ir & 0007400) == 0007000)
    bval = 6;
  else
    bval = (ir >> 8) & 017;
  branch ("idisp", bval);
}

void f2_fn_acsource_early (int task, int code)
{
  rsel &= ~3;
  rsel |= (((ir >> 13) & 3) ^ 3);
}

void f2_fn_acsource_late (int task, int code)
{
  int bval;
  if (ir & 0100000)
    bval = 3 - ((ir >> 6) & 3);  /* complement of SH field */
  else if ((ir & 0060000) != 0060000)
    bval = (ir >> 10) & 1;  /* indirect bit */
  else
    switch ((ir >> 8) & 037)
      {
      case 000:  bval = 002;  break;  /* CYCLE */
      case 001:  bval = 005;  break;  /* RAMTRAP */
      case 002:  bval = 003;  break;  /* NOPAR -- parameterless group */
      case 003:  bval = 006;  break;  /* RAMTRAP */
      case 004:  bval = 007;  break;  /* RAMTRAP */
      case 011:  bval = 004;  break;  /* JSRII */
      case 012:  bval = 004;  break;  /* JSRIS */
      case 016:  bval = 001;  break;  /* CONVERT */
      case 037:  bval = 017;  break;  /* ROMTRAP - used by Swat debugger */
      default:   bval = 016;  break;  /* RAMTRAP */
      }
  branch ("acsource", bval);
}


void install_emulator (void)
{
  install_ram_related (task_emulator);

  install_bus_select_fn (task_emulator, bs_disp, & bs_fn_disp_early, NULL);

  // For emulator, f1_fn_load_rmr replaces ram-related f1_fn_srb
  install_f1_fn (task_emulator, f1_em_load_rmr, NULL, & f1_fn_load_rmr_late);
  install_f1_fn (task_emulator, f1_em_esrb,     NULL, & f1_fn_esrb_late);
  install_f1_fn (task_emulator, f1_em_rsnf,     & f1_fn_rsnf_early, NULL);
  install_f1_fn (task_emulator, f1_em_startf,   NULL, & f1_fn_startf_late);

  install_f2_fn (task_emulator, f2_em_busodd,   NULL, & f2_fn_busodd_late);

  // magic handled in cpu.c in shifter()
  install_f2_fn (task_emulator, f2_em_magic,    NULL, NULL);

  install_f2_fn (task_emulator, f2_em_dns,      & f2_fn_dns_early, & f2_fn_dns_late);
  install_f2_fn (task_emulator, f2_em_acdest,   & f2_fn_acdest_early, NULL);
  install_f2_fn (task_emulator, f2_em_load_ir,  NULL, & f2_fn_load_ir_late);
  install_f2_fn (task_emulator, f2_em_idisp,    NULL, & f2_fn_idisp_late);
  install_f2_fn (task_emulator, f2_em_acsource, & f2_fn_acsource_early, & f2_fn_acsource_late);
}
