/*
 * Altogether: Xerox Alto microcode-level simulator
 * Main program
 * $Id: main.c 129 2007-04-20 22:25:10Z eric $
 * Copyright 2001, 2003 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>

#ifdef USE_READLINE
#include <readline/history.h>
#endif

#include "util.h"
#include "alto.h"
#include "debug.h"
#include "scheduler.h"
#include "main_mem.h"
#include "cpu.h"
#include "main.h"
#include "emulator.h"
#include "mem_refresh.h"
#include "display.h"
#include "disk.h"
#include "ethernet.h"

#ifdef USE_GUI
#include "gui.h"
#endif /* USE_GUI */


static bool no_window;

static bool runflag_initialized;
bool runflag;
int stepflag;
bool traceflag;
bool task_switch_break [TASK_COUNT];

int reason;


static void int_handler (int x)
{
  clear_runflag (INTBREAK);
}


void clear_runflag (int why)
{
  reason = why;
  if (runflag || (! runflag_initialized))
    {
#ifdef USE_GUI
      gui_set_idle (false);
#endif
      signal (SIGINT, int_handler);
      set_debugger_state (true);
    }
  runflag = false;
  runflag_initialized = true;
}


void set_runflag (void)
{
  if ((! runflag) || (! runflag_initialized))
    {
#ifdef USE_GUI
      gui_set_idle (true);
#endif
      set_debugger_state (false);
    }
  runflag = true;
  runflag_initialized = true;
}


static bool check_arg (int *argc, char ***argv, char *str)
{
  int i;

  for (i = 1; i < *argc; i++)
    {
      if (strcmp ((*argv) [i], str) == 0)
	{
	  for (; i < *argc; i++)
	    (*argv) [i] = (*argv) [i+1];
	  (*argc)--;
	  return (true);
	}
    }
  return (false);
}


sim_time_t next_uinst_time = CPU_MICROCYCLE_TIME;


void run_simulation (void)
{
#ifdef USE_GUI
  int iteration = 0;
#endif /* USE_GUI */
  sim_time_t next_event_time;

  next_event_time = next_scheduled_event_time ();
  while (runflag)
    {
#ifdef USE_GUI
      if (! no_window)
	{
	  iteration++;
	  if ((iteration & 0xff) == 0)
	    return;
	}
#endif /* USE_GUI */
      if (stepflag && ! --stepflag)
	{
	  clear_runflag (STEP);
	}
      if (check_ubreak ())
	{
	  clear_runflag (UBREAKPT);
	  return;
	}
      if (task_switch && task_switch_break [next2_task])
	{
	  clear_runflag (TASK_SWITCH);
	  task_switch = false;
	  return;
	}
      if (traceflag)
	pre_instruction_trace ();

      if (next_event_time < next_uinst_time)
	{
	  run_scheduler_until (next_uinst_time);
	  next_event_time = next_scheduled_event_time ();
	}
      else
	current_time = next_uinst_time;
      cpu_cycle ();
      next_uinst_time += CPU_MICROCYCLE_TIME;
    }
}


static void print_args (int argc, char **argv)
{
  int i;
  for (i = 1; i < argc; i++)
    printf ("arg %d: \"%s\"\n", i, argv [i]);
}


void usage (FILE *f)
{
  fprintf (f, "%s: Xerox Alto microcode-level simulator\n", progname);
  fprintf (f, "Copyright 2001-2007 Eric Smith <eric@brouhaha.com>\n");
  fprintf (f, "http://altogether.brouhaha.com/\n");
  fprintf (f, "\n");
  fprintf (f, "usage: %s [options...]\n", progname);
  fprintf (f, "\n");
  fprintf (f, "options:\n");
  fprintf (f, "-d    debug:\n");
  fprintf (f, "-nw   no window:\n");
}


int main (int argc, char *argv [])
{
  char *zip_fn = "AltoIIXM_ucode.zip";

  progname = newstr (argv [0]);

#ifdef USE_READLINE
  using_history ();
#endif

  debug_init ();

  current_debug_level = 1;
  current_trace_level = 1;  /* init to 1 while debugging.  really should
				 init to 0 and have cmd to change */

  no_window = check_arg (& argc, & argv, "-nw");

#ifdef USE_GUI
  if (! no_window)
    init_gui (& argc, & argv);
#endif /* USE_GUI */

  /* check for debug argument */
  if (check_arg (& argc, & argv, "-d"))
    clear_runflag (INTBREAK);  /* ??? better reason? */
  else
    set_runflag ();

#ifdef USE_GUI
  if (! no_window)
    {
      gui_set_idle_function (run_simulation);
      gui_set_idle (runflag);
    }
#endif /* USE_GUI */

  init_scheduler ();

  load_microcode (zip_fn);

  init_main_mem ();

  init_cpu_mmio ();

  init_function_dispatch ();

  install_emulator ();
  install_mem_refresh ();
  install_display ();
  install_disk ();
  install_ethernet (task_ethernet);

  hard_reset ();

#ifdef USE_GUI
  if (! no_window)
    {
      gui_main ();
    }
  else
#endif /* USE_GUI */
    while (1)
      {
	if (runflag)
	  run_simulation ();
	else
	  debugger (reason);
      }
  debug_term ();
  exit (0);
}

void sim_go (void)
{
  stepflag = 0;
  traceflag = false;
  set_runflag();
}

void sim_step (int count)
{
  stepflag = count;
  traceflag = true;
  set_runflag ();
}

void sim_trace (void)
{
  stepflag = 0;
  traceflag = true;
  set_runflag ();
}

void sim_stop (void)
{
  clear_runflag (USTOP);
}
