/*
 * Altogether: Xerox Alto microcode-level simulator
 * Priority Queue
 * $Id: priority_queue.c 121 2005-01-20 02:03:33Z eric $
 * Copyright 2001 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>

#include "priority_queue.h"


typedef struct 
{
  priority_t priority;
  void *data;
} element_t;


struct priority_queue_t
{
  int max;
  int n;
  element_t *element;
};


priority_queue_t *new_priority_queue (int max_entries)
{
  priority_queue_t *q;

  q = calloc (1, sizeof (struct priority_queue_t));
  if (! q)
    goto fail;

  q->element = calloc (max_entries, sizeof (element_t));
  if (! q->element)
    goto fail;

  q->max = max_entries;
  return (q);

 fail:
  if (q)
    {
      if (q->element)
	free (q->element);
      free (q);
    }
  return (NULL);
}

bool priority_queue_empty (priority_queue_t *q)
{
  return (q->n == 0);
}

bool priority_queue_full (priority_queue_t *q)
{
  return (q->n == q->max);
}

int priority_queue_size (priority_queue_t *q)
{
  return (q->n);
}


#define left_child_of(n) (2 * (n) + 1)
#define right_child_of(n) (2 * (n) + 1)
#define parent_of(n) (((n) - 1) / 2)


static void swap (element_t *e1, element_t *e2)
{
  element_t temp;
  temp = *e1;
  *e1 = *e2;
  *e2 = temp;
}


bool priority_queue_add (priority_queue_t *q,
			 priority_t priority, void *data)
{
  int i;

  if (q->n == q->max)
    return (false);
  q->element [q->n].priority = priority;
  q->element [q->n].data = data;

  i = q->n;
  while ((i != 0) &&
	 (q->element [i].priority < q->element [parent_of (i)].priority))
    {
      swap (& q->element [i], & q->element [parent_of (i)]);
      i = parent_of (i);
    }
  q->n++;
  return (true);
}


static void balance_down (priority_queue_t *q, int i)
{
  int l;
  int r;

  for (;;)
    {
      l = left_child_of (i);
      if (l >= q->n)
	return;

      r = right_child_of (i);

      if ((q->element [l].priority < q->element [i].priority) &&
	  ((r >= q->n) || (q->element [l].priority < q->element [r].priority)))
	{
	  swap (& q->element [i], & q->element [l]);
	  i = l;
	  continue;
	}

      if ((r >= q->n) || (q->element [i].priority < q->element [r].priority))
	return;

      swap (& q->element [i], & q->element [r]);
      i = r;
    }
}


bool priority_queue_peek (priority_queue_t *q,
			  priority_t *priority, void **data)
{
  element_t result;

  if (q->n == 0)
    return (false);

  result = q->element [0];

  (* priority) = result.priority;
  (* data) = result.data;
  return (true);
}


bool priority_queue_remove (priority_queue_t *q,
			    priority_t *priority, void **data)
{
  element_t result;

  if (q->n == 0)
    return (false);

  result = q->element [0];
  q->n--;
  if (q->n)
    {
      q->element [0] = q->element [q->n];
      balance_down (q, 0);
    }

  (* priority) = result.priority;
  (* data) = result.data;
  return (true);
}


priority_t priority_queue_head (priority_queue_t *q)
{
  if (q->n == 0)
    return (PQ_LOWEST_PRIORITY);
  return (q->element [0].priority);
}
