/*
 * Altogether: Xerox Alto microcode-level simulator
 * RAM-related task microinstruction simulation
 * $Id: ram_related.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include "alto.h"
#include "debug.h"
#include "cpu.h"
#include "emulator.h"
#include "ram_related.h"


#define bs_rrt_read_sreg  bs_task_3
#define bs_rrt_load_sreg  bs_task_4

#define f1_rrt_swmode     f1_task_10
#define f1_rrt_wrtram     f1_task_11
#define f1_rrt_rdram      f1_task_12
#define f1_rrt_srb        f1_task_13
#define f1_em_esrb        f1_task_15


void bs_fn_read_sreg_early (int task, int code)
{
  if (current_uinst->rsel)
    drive_bus ("sreg", s [s_reg_bank [task]][current_uinst->rsel]);
  else
    drive_bus ("m", m);
}

void bs_fn_load_sreg_early (int task, int code)
{
  // real hardware drives bus with undefined value (vs. 0 for R registers)
  drive_bus ("load_s", 0);
}

void bs_fn_load_sreg_late (int task, int code)
{
  // unlike R registers, there is no shifter in the data path
  s [s_reg_bank [task]][current_uinst->rsel] = m;
}


void f1_fn_swmode_late (int task, int code)
{
  /* ??? */
}

void f1_fn_wrtram_late (int task, int code)
{
  start_wrtram ();
}

void f1_fn_rdram_late (int task, int code)
{
  start_rdram ();
}

void f1_fn_srb_late (int task, int code)
{
  s_reg_bank [task] = (bus >> 1) & 07;
}

void install_ram_related (int task)
{
  ram_related [task] = true;

  install_bus_select_fn (task, bs_rrt_read_sreg, & bs_fn_read_sreg_early, NULL);
  install_bus_select_fn (task, bs_rrt_load_sreg, & bs_fn_load_sreg_early, & bs_fn_load_sreg_late);

  install_f1_fn (task, f1_rrt_swmode, NULL, & f1_fn_swmode_late);
  install_f1_fn (task, f1_rrt_wrtram, NULL, & f1_fn_wrtram_late);
  install_f1_fn (task, f1_rrt_rdram, NULL, & f1_fn_rdram_late);
#ifdef ALTO_3K
  if (task == task_emulator)
    install_f1_fn (task, f1_em_esrb, NULL, & f1_fn_srb_late);
  else
    install_f1_fn (task, f1_rrt_srb, NULL, & f1_fn_srb_late);
#else
  install_f1_fn (task, f1_em_load_rmr, NULL, & f1_fn_load_rmr_late);
#endif
}

void uninstall_ram_related (int task)
{
  ram_related [task] = false;
}
