/*
 * Altogether: Xerox Alto microcode-level simulator
 * Diablo 31, 44 disk and controller simulation
 * $Id: disk.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 2001, 2003, 2004 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include "alto.h"
#include "debug.h"
#include "cpu.h"
#include "scheduler.h"
#include "disk.h"


typedef struct
{
  char *name;
  int sectors_per_track;
  double rpm;
  double bit_rate;
} drive_type_info_t;


static drive_type_info_t drive_type_info [MAX_DRIVE_TYPE] =
{
  { "Diablo 31", 12, 1500, (3.330e6 / 2) },
  { "Diablo 44", 12, 2400, (5.000e6 / 2) },
};


static drive_type_t drive_type;
static sim_time_t word_time, sector_time;


static int selected_drive;
static bool disk_in_place [MAX_DRIVE];
static int current_sector [MAX_DRIVE];


static uint16_t kstat;
// kstat [0-3] = current sector number
// kstat [4-7] = 017 octal
// kstat [8] = 1 if seek failed, possibly to illegal cylinder address
// kstat [9] = 1 if seek in progress
// kstat [10] = 1 if disk unit not ready
// kstat [11] = 1 if data or sector porcessing was late during last sector
// kstat [12] = 1 if disk interface did not transfer data during last sector
// kstat [13] = 1 if checksum error
// kstat [14-15] = completion code
//                 0 if command completed correctly
//                 1 if hardware error (see bits 8-11) or sector overflow
//                 2 if check error, command terminated instantly
//                 3 if disk command specified illegal sector


// static uint16_t kcmd;
// kcmd [0-7] = 0110 octal (magic number)
// kcmd [8-9] = 0 for header read, 1 for check, 2 or 3 for write
// kcmd [10-11] = 0 for label read, 1 for check, 2 or 3 for write
// kcmd [12-13] = 0 for data block read, 1 for check, 2 or 3 for write
// kcmd [14] = 1 for seek only 
// kcmd [15] is XORed with kcmd [14] to yield hardware disk number

static uint16_t kadr;
// kadr [0-3] = sector (0-11 decimal)
// kadr [4-12] = cylinder (0-202 for model 31, 0-405 for model 44)
// kadr [13] = head number
// kadr [14] = disk number, 0 is removable pack of model 44
// kadr [15] = 1 for hardware positioner restore

static uint16_t kcom;
// kcom [1] = XFEROFF, if 1 inhibits data transmission to/from the disk
// kcom [2] = WDINHIB, if 1 prevents the disk word task from awakening
// kcom [3] = BCLKSRC, if 1 takes the bit clock from the disk input or
//                     crystal clock as appropriate, if 1 forces the use of
//                     crystal clock (that's what the HW manual says!)
// kcom [4] = WFFO, if 0 holds the disk bit counter at -1 until a 1 bit is
//                  read, if 1 allows the bit counter to proceed normally
// kcom [5] = SENDADR, if 1 causes KDATA [4-12] and KDATA [15] to be
//                     transmitted to disk unit as track address

static uint16_t kdata;

static bool strobe;

static int current_record;

static bool current_command_wants_data_transfer;  // $$$
static bool disk_not_ready_for_command;  // $$$
static bool fatal_error_in_latches;  // $$$

static bool write_current_record;   // $$$
static bool verify_current_record;  // $$$


#define bs_disk_kstat       bs_task_3
#define bs_disk_kdata       bs_task_4

#define f1_disk_strobe      f1_task_11
#define f1_disk_load_kstat  f1_task_12
#define f1_disk_increcno    f1_task_13
#define f1_disk_clrstat     f1_task_14
#define f1_disk_load_kcom   f1_task_15
#define f1_disk_load_kadr   f1_task_16
#define f1_disk_load_kdata  f1_task_17

#define f2_disk_init        f2_task_10
#define f2_disk_rwc         f2_task_11
#define f2_disk_recno       f2_task_12
#define f2_disk_xfrdat      f2_task_13
#define f2_disk_swrnrdy     f2_task_14
#define f2_disk_nfer        f2_task_15
#define f2_disk_strobon     f2_task_16


static void sector_mark (uint32_t arg1, void *arg2)
{
  if (! disk_in_place [arg1])
    return;
  current_sector [arg1]++;
  if (current_sector [arg1] >= drive_type_info [drive_type].sectors_per_track)
    current_sector [arg1] = 0;
  if (arg1 == selected_drive)
    set_task_request (task_disk_sector);
  schedule_event (current_time + sector_time, sector_mark, arg1, NULL);
}


static void select_drive (int drive)
{
  selected_drive = drive;
}


static void insert_disk (int drive)
{
  if (! disk_in_place [drive])
    schedule_event (sector_time, sector_mark, drive, NULL);
  disk_in_place [drive] = true;
}


static void remove_disk (int drive)
{
#if 0
  if (disk_in_place [drive])
    unschedule_event (?);
#endif
  disk_in_place [drive] = false;
}


void bs_fn_disk_kstat_early (int task, int code)
{
  // 003: put KSTAT on BUS
  drive_bus ("kstat", kstat);  /* $$$ merge in current sector */
  // $$$
}

void bs_fn_disk_kdata_early (int task, int code)
{
  // 004: put KDATA on BUS
  drive_bus ("kdata", kdata);
  // $$$ should have separate register for input data?
  // $$$
}

void f1_fn_disk_strobe_late (int task, int code)
{
  // 011:  Initiate a disk seek operation.
  strobe = 1;
  // $$$
}

void f1_fn_disk_load_kstat_late (int task, int code)
{
  // 012:  KSTAT[12-15] are loaded from BUS[12-15], except that
  // BUS[13] is ored into KSTAT[13].
  kstat &= ~ 013;      // clear KSTAT[12], KSTAT[14-15]
  kstat |= bus & 017;  // or BUS[12-15] into KSTAT[12-15]
}

void f1_fn_disk_increcno_late (int task, int code)
{
  // 013:  Advance the shift registers holding the KADR register so that
  // they present the number and read/write/check status of the next record
  // to the hardware
  // $$$
}

void f1_fn_disk_clrstat_late (int task, int code)
{
  // 014:  Reset all error latches and KSTAT[13].
  kstat &= ~ 004;
  // $$$
}

void f1_fn_disk_load_kcom_late (int task, int code)
{
  // 015:  The KCOM register is loaded from BUS[1-5]
  kcom = (bus >> 10) & 037;
}

void f1_fn_disk_load_kadr_late (int task, int code)
{
  // 016:  The KADR register is loaded from BUS[8-14].
  kadr = (bus >> 1) & 0177;
}

void f1_fn_disk_load_kdata_late (int task, int code)
{
  // 017:  The KDATA register is loaded from BUS[0-15].
  kdata = bus;
}

void f2_fn_disk_init_late (int task, int code)
{
  // 010:  NEXT = NEXT | ((WDTASKACT & WDINIT) ? 037 : 0);
  // if (WDTASKACT & WDINIT)
  //   branch ("f2_disk_init", 037);
}

void f2_fn_disk_rwc_late (int task, int code)
{
  // 011:  NEXT = NEXT | (current record to be writen ? 3 :
  //                     (current record to be checked ? 2 : 0))
  if (write_current_record)
    branch ("f2_fn_disk_rwc", 3);
  else if (verify_current_record)
    branch ("f2_fn_disk_rwc", 2);
}

void f2_fn_disk_recno_late (int task, int code)
{
  // 012:  NEXT |= map [current_record];
  // map [5] = { 0, 2, 3, 1 }
  switch (current_record)
    {
    case 0:
      break;
    case 1:
      branch ("f2_fn_disk_recno", 2);
      break;
    case 2:
      branch ("f2_fn_disk_recno", 3);
      break;
    case 3:
      branch ("f2_fn_disk_recno", 1);
      break;
    }
}

void f2_fn_disk_xfrdat_late (int task, int code)
{
  // 013: NEXT |= (current command wants data transfer ? 1 : 0)
  if (current_command_wants_data_transfer)
    branch ("f2_fn_disk_xfrdat", 1);
}

void f2_fn_disk_swrnrdy_late (int task, int code)
{
  // 014: NEXT |= (disk not ready to accept command ? 1 : 0)
  if (disk_not_ready_for_command)
    branch ("f2_fn_disk_swrnrdy", 1);
}

void f2_fn_disk_nfer_late (int task, int code)
{
  // 015: NEXT |= (fatal error in latches ? 0 : 1)
  if (! fatal_error_in_latches)
    branch ("f2_fn_disk_nfer", 1);
}

void f2_fn_disk_strobon_late (int task, int code)
{
  // 016: NEXT |= (if seek strobe still on ? 1 : 0)
  if (strobe)
    branch ("f2_fn_disk_strobon", 1);
}


void install_disk (void)
{
  install_bus_select_fn (task_disk_word, bs_disk_kstat, & bs_fn_disk_kstat_early, NULL);
  install_bus_select_fn (task_disk_word, bs_disk_kdata, & bs_fn_disk_kdata_early, NULL);

  install_f1_fn (task_disk_word, f1_disk_strobe,     NULL, & f1_fn_disk_strobe_late);
  install_f1_fn (task_disk_word, f1_disk_load_kstat, NULL, & f1_fn_disk_load_kstat_late);
  install_f1_fn (task_disk_word, f1_disk_increcno,   NULL, & f1_fn_disk_increcno_late);
  install_f1_fn (task_disk_word, f1_disk_clrstat,    NULL, & f1_fn_disk_clrstat_late);
  install_f1_fn (task_disk_word, f1_disk_load_kcom,  NULL, & f1_fn_disk_load_kcom_late);
  install_f1_fn (task_disk_word, f1_disk_load_kadr,  NULL, & f1_fn_disk_load_kadr_late);
  install_f1_fn (task_disk_word, f1_disk_load_kdata, NULL, & f1_fn_disk_load_kdata_late);

  install_f2_fn (task_disk_word, f2_disk_init,       NULL, & f2_fn_disk_init_late);
  install_f2_fn (task_disk_word, f2_disk_rwc,        NULL, & f2_fn_disk_rwc_late);
  install_f2_fn (task_disk_word, f2_disk_recno,      NULL, & f2_fn_disk_recno_late);
  install_f2_fn (task_disk_word, f2_disk_xfrdat,     NULL, & f2_fn_disk_xfrdat_late);
  install_f2_fn (task_disk_word, f2_disk_swrnrdy,    NULL, & f2_fn_disk_swrnrdy_late);
  install_f2_fn (task_disk_word, f2_disk_nfer,       NULL, & f2_fn_disk_nfer_late);
  install_f2_fn (task_disk_word, f2_disk_strobon,    NULL, & f2_fn_disk_strobon_late);

  drive_type = DIABLO_31;

  word_time = 1e9 * (16 / drive_type_info [drive_type].bit_rate);
  sector_time = 1e9 * ((60.0 / drive_type_info [drive_type].rpm) /
		       drive_type_info [drive_type].sectors_per_track);

#if 0
  insert_disk (0);
#endif
}
