/*
 * Altogether: Xerox Alto microcode-level simulator
 * Main memory simulation
 * $Id: main_mem.c 117 2004-12-31 19:03:27Z eric $
 * Copyright 2001, 2003 Eric Smith <eric@brouhaha.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.  Note that permission is
 * not granted to redistribute this program under the terms of any
 * other version of the General Public License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111  USA
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include "alto.h"
#include "debug.h"
#include "scheduler.h"
#include "cpu.h"
#include "main_mem.h"


bool mem_start_flag;
uint64_t mem_start_cycle;
int mem_cycle_second_word;
uint16_t mar;


static uint16_t ram [RAM_SIZE];

static mmio_read_fn_t  mmio_read_fn  [IO_PAGE_SIZE];
static mmio_write_fn_t mmio_write_fn [IO_PAGE_SIZE];


static uint16_t bad_mmio_read_fn (uint16_t address)
{
  printf ("stray I/O read of address %06o\n", address);
  return (0177777);
}


static void bad_mmio_write_fn (uint16_t address, uint16_t data)
{
  printf ("stray I/O write of address %06o, data %06o\n", address, data);
}


void init_main_mem (void)
{
  uint32_t addr;
  for (addr = 0; addr < IO_PAGE_SIZE; addr++)
    {
      mmio_read_fn  [addr] = & bad_mmio_read_fn;
      mmio_write_fn [addr] = & bad_mmio_write_fn;
    }
}


void load_main_mem_addr (int addr)
{
  tprintf (1, "mar <= %06o\n", addr);
  mar = addr;
  mem_start_flag = 1;
  mem_start_cycle = current_time;
  mem_cycle_second_word = 0;
}


bool check_read_main_mem_stall (void)
{
  if (! mem_start_flag)
    {
      printf ("mem read without cycle start!\n");
      return (false);
    }
  return (current_time < (mem_start_cycle + 4 * CPU_MICROCYCLE_TIME));
}


bool check_write_main_mem_stall (void)
{
  if (! mem_start_flag)
    {
      printf ("mem write without cycle start!\n");
      return (false);
    }
  return (current_time < (mem_start_cycle + 3 * CPU_MICROCYCLE_TIME));
}


uint16_t read_main_mem (void)
{
  uint16_t data;
  uint16_t base_addr;

  if (! mem_start_flag)
    {
      fprintf (stderr, "fatal error - mem read with no preceding address\n");
      return (0177777);
    }
  if (current_time > (mem_start_cycle + 4 * CPU_MICROCYCLE_TIME))
    {
      fprintf (stderr, "fatal error - mem read too late\n");
      return (0177777);
    }
  base_addr = mar & 0177777;
  if (base_addr < IO_PAGE_BASE)
    data = ram [mar];
  else
    data = mmio_read_fn [mar - IO_PAGE_BASE] (base_addr);
  tprintf (1, "read RAM addr %06o, data = %06o\n", mar, data);
  if (mem_cycle_second_word)
    {
      mem_cycle_second_word = 0;
      mem_start_flag = 0;
    }
  else
    {
      mem_cycle_second_word = 1;
      mar ^= 1;
      mem_start_cycle += CPU_MICROCYCLE_TIME;
    }
  return (data);
}

void write_main_mem (uint16_t data)
{
  uint16_t base_addr;
  if (! mem_start_flag)
    {
      fprintf (stderr, "fatal error - mem write with no preceding address\n");
      return;
    }
  if (current_time > (mem_start_cycle + 3 * CPU_MICROCYCLE_TIME))
    {
      fprintf (stderr, "fatal error - mem write too late\n");
      return;
    }
  base_addr = mar & 0177777;
  if (base_addr < IO_PAGE_BASE)
    ram [mar] = bus;
  else
    mmio_write_fn [mar - IO_PAGE_BASE] (base_addr, bus);
  tprintf (1, "write RAM addr %06o, data = %06o\n", mar, bus);
  if (mem_cycle_second_word)
    {
      mem_cycle_second_word = 0;
      mem_start_flag = 0;
    }
  else
    {
      mem_cycle_second_word = 1;
      mar ^= 1;
      mem_start_cycle += CPU_MICROCYCLE_TIME;
    }
}


void install_memory_mapped_io (uint16_t first_address,
			       uint16_t last_address,
			       mmio_read_fn_t read_fn,
			       mmio_write_fn_t write_fn)
{
  uint32_t address;

  if ((first_address <= IO_PAGE_BASE) ||
      (last_address >= (IO_PAGE_BASE + IO_PAGE_SIZE)) ||
      (first_address > last_address))
    {
      fprintf (stderr, "internal error - bad memory-mapped I/O address\n");
      exit (3);
    }

  for (address = first_address; address <= last_address; address++)
    {
      mmio_read_fn  [address - IO_PAGE_BASE] = read_fn ? read_fn : & bad_mmio_read_fn;
      mmio_write_fn [address - IO_PAGE_BASE] = write_fn ? write_fn : & bad_mmio_write_fn;
    }
}


uint16_t debug_read_mem (int addr)
{
  return (ram [addr]);
}


void debug_write_mem (int addr, uint16_t data)
{
  ram [addr] = data;
}

