//******************************************************************************
//  IBM 7094 Emulator - Central Processing Unit
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains the registers and higher level functions for the
//  emulated IBM 7094 Central Processing Unit.
//  The actual instruction execution functions are contained in other units
//  named B709Ixxx.
//------------------------------------------------------------------------------
Unit B709CPU;

Interface

Uses Windows,SysUtils,Forms,
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Inst, // CPU Instruction definitions
     B709IBIS, // Instruction Execution - Basic Instructions
     B709Misc, // Miscellaneous utility functions
     B709Stop, // Process Stops Panel
     B709Trce, // Log/Trace functions
     B709Core, // Core Storage functions
     B709Plot, // Core Usage Plot display
     B709Chan; // I/O Channel and Device functions

// Simulated CPU operating mode control flags
Var CPURunning: Boolean;     // CPU is Running
    ChnlActive: Boolean;     // Channel(s) are running
    StopPend:   Boolean;     // Stop requested
    ChanWait:   Boolean;     // Waiting for channel (LCHA & RCHA)
    ChWCAddr:   TAddr;       // Address of waiting channel command from LCHA

Var CPUCheck:   Boolean;     // Machine Check in CPU
    AutoMode:   Boolean;     // Auto run enabled
    SlowMode:   Boolean;     // Slow run enabled

// 7094 CPU operating mode control flags
Var TrapMode:   Boolean;     // Transfer Trap Mode enabled
    MTagMode:   Boolean;     // Multiple Tag Mode (Off=7094 On=709x)
    FTrpMode:   Boolean;     // Floating Trap Mode (On=Normal Off=704 Mode)
    STrpMode:   Boolean;     // Select Trap Mode (Off=Normal On=704 Mode)
    CTrpMode:   Boolean;     // Copy Trap Mode (Off=Normal On=704 Mode)
    SNulMode:   Boolean;     // Storage Nullification Mode (Off=Normal On=704 Mode)

// CPU Registers - Control
Var InstCtr:  TAddr;         // Instruction counter
    PrevICtr: TAddr;         // Previous instruction address
    StorReg:  TWord;         // Storage register (data from memory)
    OrigStorReg:  TWord;
    InstReg:  TInst;         // Instruction OpCode field
    FlagReg:  Byte;          // Instruction Flag field
    TagsFld:  Byte;          // Instruction Index tags field
    OAdrReg:  TAddr;         // Instruction Address field - Original
    AddrFld:  TAddr;         // Instruction Address field - After Mods
    DecrFld:  TAddr;         // Instruction Decrement field

// CPU Registers - Programmer
Var RegAC:  TWord;                     // Accumulator
    SgnAC:  Boolean;                   // Accumulator Sign Bit
    RegMQ:  TWord;                     // Multiplier Quotient (includes sign)
    XegXR:  Array[1..7] Of TAddr;      // Index registers
    RegSI:  TWord;                     // Sense Indicators
    RegSL:  Array[1..4] Of Boolean;    // Sense Lights
    RegSS:  Array[1..6] Of Boolean;    // Switch Register
    OVFInd: Boolean;                   // AC Overflow indicator
    MQOInd: Boolean;                   // MQ Overflow Indicator
    DVCInd: Boolean;                   // Divide Check indicator
    IOCInd: Boolean;                   // Input/Output Check indicator

Var  HTRAddr:   TAddr;       // Address saved from last HTR instruction
     XECAddr:   TAddr;       // Address of any executing XEC instruction
     TrapInhib: Boolean;     // To inhibit channel traps after ENB,RST,XEC

// Transfer reason codes - Reason that IC value was altered
// Mainly used by XEC instruction, but may prove useful elsewhere
Type TTranCode=(
       TCNone,
//     TCXEC1,     // XEC Instruction - Set target address
//     TCXEC2,     // XEC Instruction -
//     TCXEC3,     // XEC Instruction -
//     TCChTr,     // Channel trap
       TCFPSp,     // Floating Point trap
       TCTrap,     // Transfer trap
       TCTran,     // Transfer instruction
       TCTest);    // Test instruction

Var TranCode:  TTranCode;              // Reason code for last transfer
    StopMesg:  String;                 // Reason processor stopped message

// Modification flags for tracing (more flags in implementation section below)
// These flags indicate that an emulated 7094 value has changed and the
// altered value should be shown in a trace entry
Var INDTrcFlg: Boolean;                // Indicators have changed

// Statistics, Tracing and Display update
Var InstCount: Int64;                  // Number of instructions executed
    SwtfCount: Int64;                  // Instruction count at last SWT
    SSAlert:   Boolean;                // Alert when program executes SWT
    LoopCtr:   Word;                   // Increments each cycle (for TrueFact)
    TrueFact:  Word;                   // True display update factor
    DispIPtr:  TAddr;                  // Address of last instruction traced

// Type for setting of RegAC sign bit value in SetAS procedure
Type TACSign=(SA0,SA1,SAX);            // 0, 1, or unchanged

Procedure CPUReset;
Procedure SetStopPending(SM: String);
Procedure SetIC(NI: TAddr; TC: TTranCode);
Procedure SetAC(AC: TWord);
Procedure SetAS(AC: TWord; SB: TACSign);
Procedure SetSB(SB: Boolean);
Function  GetXR(RI: Byte): TAddr;
Procedure SetMQ(MQ: TWord);
Procedure SetXR(RI: Byte; XV: Integer);
Procedure SetSI(NI: TWord);
Procedure XECute(ST: TAddr; TE: Boolean);
Function  HashValue(HA: LongWord): Word;
Procedure Process;
Procedure ProcessCPU;          // For access by XEC
Procedure ProcessChannels;

Implementation

Uses B709Main; // For buttons and displays

// Modification flags for tracing
Var RACTrcFlg: Boolean;                // Trace AC register changes
    RMQTrcFlg: Boolean;                // Trace MQ register changes
    RXRTrcFlg: Boolean;                // Trace Index register changes
    RSITrcFlg: Boolean;                // Trace Sense Indicator register changes

Var InstRec:   TInstruction;           // Current instruction record
    DispCount: Integer;                // Number of times instruction repeated
    DispStr:   ShortString;            // Instruction trace string

//******************************************************************************
//  Instruction Tracing and Logging functions
//------------------------------------------------------------------------------
// Display instruction details after fetch, Before execution
Procedure ShowInstruction;
Var AP: Integer;
Begin
  // Same Instruction as last time?
  If DispIPtr<>PrevICtr then begin
    // New Instruction. If previous was repeated, show count
    If DispCount>1 then begin
      Trace(TIINS,DispStr);
      Trace(TIINS,'     Above instruction was repeated '+IntToStr(DispCount)+' times');
    End;
    // Note new Instruction address, Reset repeat count
    DispIPtr:=PrevICtr; DispCount:=1;
    // Build instruction display string
    DispStr:=IntToOct(PrevICtr,5)+' '+       // Execution Address
             InstOct(InstReg)+' '+           // Opcode
             IntToOct(OAdrReg,5)+' '+        // Operand
             InstRec.Mnemonic+' '+           // Mnemonic and Description
             InstRec.Desc;
    // Append Modification tags to description string
    If (TagsFld<>0) And (InstRec.IdxModEn) then
      DispStr:=DispStr+' (Tag='+IntToOct(TagsFld,1)+')';
    // Insert Operand address into description string
    AP:=Pos('@@@@@',DispStr);
    If AP>0 then begin
      Delete(DispStr,AP,5);
      Insert(IntToOct(AddrFld,5),DispStr,AP);
    End;
    // Insert Operand value into description string
    AP:=Pos('%%%%%%%%%%%%',DispStr);
    If AP>0 then begin
      Delete(DispStr,AP,12);
      Insert(IntToOct(GetCore(AddrFld),12),DispStr,AP);
    End;
    // Insert Index register tag value into description string
    AP:=Pos('$',DispStr);
    If AP>0 then
      DispStr[AP]:=Chr($30+TagsFld);
    // Insert Index instruction decrement field into description string
    AP:=Pos('!!!!!',DispStr);
    If AP>0 then begin
      Delete(DispStr,AP,5);
      Insert(IntToOct(DecrFld,5),DispStr,AP);
    End;
    // Indicate if being XEC'd
    If XECAddr<>$8000 then
      DispStr:='>>>>> '+DispStr;
    // Show me the whole instruction word (and indirect addressing)
    If StorReg=OrigStorReg then begin
      DispStr:=DispStr+' ['+IntToOct(StorReg,12)+']';
    End else Begin
      DispStr:=DispStr+' [*'+IntToOct(OrigStorReg,12)+' -> '+IntToOct(StorReg,12)+']';
    End;
    // Show string
    Trace(TIIns,DispStr);
  End else Begin
    // Same Instruction. Count times repeated
    Inc(DispCount);
    // But show it if we are in step mode
    If Not AutoMode then Trace(TIINS,DispStr);
  End;
End;

// Display effects of instruction after execution
Procedure DisplayEffects;
Var AW: TWord;
    SS: String;
Begin
  If RACTrcFlg then begin
    SS:=IntToOct(RegAC,12);
    If (RegAC And QBitMask)>0 then SS:='Q'+SS else SS:=' '+SS;
    If SgnAC then SS:='S'+SS else SS:=' '+SS;
    SS:='AC='+SS;
    If OVFInd then SS:=SS+' OV=1' else SS:=SS+' OV=0';
SS:=SS+' '+WordDump(RegAC);
    Trace(TIREG,SS);
    IndTrcFlg:=False;           // Prevent repeat show below
    RACTrcFlg:=False;
  End;
  If RMQTrcFlg then begin
    AW:=RegMQ;
    Trace(TIREG,'MQ=  '+IntToOct(AW,12)+' '+WordDump(AW));
    RMQTrcFlg:=False;
  End;
  If RXRTrcFlg then begin
    Trace(TIREG,'X'+Chr(TagsFld+$30)+'='+IntToOct(GetXR(TagsFld),5));
    RXRTrcFlg:=False;
  End;
  If RSITrcFlg then begin
    Trace(TIREG,'SI='+IntToOct(RegSI,12));
    RSITrcFlg:=False;
  End;
  If IndTrcFlg then begin
    If OVFInd then Trace(TIREG,'OV=1')
              else Trace(TIREG,'OV=0');
    IndTrcFlg:=False;
  End;
  If (CorTrcAdr>=0) And (TIMEM In TraceRecdFlags) then begin
    Trace(TIMEM,'Core['+IntToOct(CorTrcAdr,5)+']='+IntToOct(GetCore(CorTrcAdr),12)+' '+WordDump(GetCore(CorTrcAdr)));
    CorTrcAdr:=-1;
  End;
  RACTrcFlg:=False;
  RMQTrcFlg:=False;
  RXRTrcFlg:=False;
  RSITrcFlg:=False;
  INDTrcFlg:=False;
  CorTrcAdr:=-1;
End;

//******************************************************************************
// Store/Read values To/From various CPU registers
//------------------------------------------------------------------------------
// IC - Instruction Counter
Procedure SetIC(NI: TAddr; TC: TTranCode);
Begin
  InstCtr:=NI;               // Set new value
  TranCode:=TC;              // Note reason for change
End;

// AC - Accumulator register And/Or Sign bit
Procedure SetAS(AC: TWord; SB: TACSign);
Begin
  RegAC:=AC;
  If SB<>SAX then
    SgnAC:=Boolean(SB);
  RACTrcFlg:=True;
End;

Procedure SetAC(AC: TWord);
Begin
  RegAC:=AC;
  RACTrcFlg:=True;
End;

Procedure SetSB(SB: Boolean);
Begin
  SgnAC:=SB;
  RACTrcFlg:=True;
End;

Procedure SetMQ(MQ: TWord);
Begin
  RegMQ:=MQ;
  RMQTrcFlg:=True;
End;

// XR - Index Registers, Allowing for Multiple Tag Mode also
Procedure SetXR(RI: Byte; XV: Integer);
Begin
  If RI=0 then Exit;
  XV:=XV And AddrMask;
  If MTagMode then begin
    If (RI And 1)>0 then XegXR[1]:=XV;
    If (RI And 2)>0 then XegXR[2]:=XV;
    If (RI And 4)>0 then XegXR[4]:=XV;
  End else
    XegXR[RI]:=XV;
  RXRTrcFlg:=True;
End;

Function GetXR(RI: Byte): TAddr;
Var SF: Boolean;
Begin
  If RI=0 then begin
    Result:=0;
  End else Begin
    If MTagMode then begin
      Result:=0;
      If (RI And 1)>0 then Result:=XegXR[1];
      If (RI And 2)>0 then Result:=Result Or XegXR[2];
      If (RI And 4)>0 then Result:=Result Or XegXR[4];
    End else
      Result:=XegXR[RI];
  End;
  // Thanks to Leif Harcke for pointing out the need for this:-
  SF:=RXRTrcFlg;
  SetXR(RI,Result);          // Store result back to XR's
  RXRTrcFlg:=SF;
End;

// SI - Sense Indicators
Procedure SetSI(NI: TWord);
Begin
  RegSI:=(NI And Wd36Mask);
  RSITrcFlg:=True;
End;

//******************************************************************************
// Central Processing Unit major functions
//------------------------------------------------------------------------------
// Reset the Processor
Procedure CPUReset;
begin
  ClearTrace;
  CPUCheck:=False;
  SetIC(0,TCNone);
  PrevICtr:=$8000;
  StorReg:=0;
  InstReg:=0;
  AddrFld:=0;
  SetAS(0,SA0);
  RegMQ:=0;
  FillChar(XegXR,SizeOf(XegXR),#0);
  FillChar(RegSL,SizeOf(RegSL),#0);
  TrapMode:=False;
  SNulMode:=False;
  OVFInd:=False;
  DVCInd:=False;
  IOCInd:=False;
  InstCount:=0;
  SwtfCount:=0;
  DispIPtr:=1;
  DispCount:=0;
  RACTrcFlg:=False;
  RMQTrcFlg:=False;
  RXRTrcFlg:=False;
  RSITrcFlg:=False;
  INDTrcFlg:=False;
  CorTrcAdr:=-1;
  XECAddr:=$8000;
  HTRAddr:=$8000;
  FetchStopCtr:=0;
end;

// Stop the Processor
Procedure SetStopPending(SM: String);
begin
  StopPend:=True;
  StopMesg:='Stopped - '+SM;
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,StopMesg);
End;

// Execute a single instruction at the specified address
// Used by the XEC instruction and Channel trap.
Procedure XECute(ST: TAddr; TE: Boolean);
Var SI: TAddr;
begin
  // Must save SI locally, as well as publishing in XECAddr for use by TSX
  // 9M71B uses chained XEC's, restoring from XECAddr would cause recursion
  SI:=InstCtr;          // Save XEC instruction address locally
  XECAddr:=SI;          // Make XEC instruction address known for TSX
  SetIC(ST,TCNone);     // Setup target instruction address
  B709CPU.ProcessCPU;   // Execute the target instruction
  // See if it changed the IC
  Case TranCode Of
    TCNone: SetIC(SI,TCNone);                   // Restore IC of XEC+1
    TCTest: If TE then                          // Allow Tests to transfer?
              SetIC(SI+(InstCtr-(ST+1)),TCNone) // Yes, Make skip relative to XEC
            Else
              SetIC(SI,TCNone);                 // No, Restore IC
    Else    Begin End;                          // Proceed with altered address
  End;
  XECAddr:=$8000;       // Clear XEC instruction address
  TrapInhib:=True;      // Inhibit channel traps for one CPU cycle
End;

// Execute Trap for the specified Channel
Procedure PerformChannelTrap(CI: Byte);
Var TA: TAddr;
    SD: TWord;
    SS: String[3];
Begin
  TA:=8+(CI*2);                   // Form address of trap locations
  // Form trap information word: Reason Bit and Current IC
  SD:=0;
  With Channels[CI] do begin
    If LCHInd then begin               // No IOxP or IOxT pending?
      SS:='LCH';                            // Note for logging
      SD:=$000040000; LCHInd:=False;        // Set trap reason code in decrement
    End;
    If TCKInd then begin               // Tape Check Indicator?
      SS:='TCK';
      SD:=$000080000; TCKInd:=False;        // And turn off causing indicator
    End;
    If EOFInd then begin               // End of File Indicator?
      SS:='EOF';
      SD:=$000100000; EOFInd:=False;
    End;
    TrapCtrlInd:=False;           // Disable further traps
  End;
  SD:=SD Or InstCtr;              // Insert IC in Address field
  SetCore(TA,SD);                 // Store trap data at even address
  // Log, Show altered core, Then XECute the instruction at odd address
  If (TICCW In TraceRecdFlags) then
    Trace(TICCW,'Channel '+Chr($40+CI)+' Trap ('+SS+') execute at: '+IntToOct(TA+1,5));
  If (TIREG In TraceRecdFlags) Or (TIMEM In TraceRecdFlags) then DisplayEffects;
  XECute(TA+1,False);
End;

// Generate an Instruction Table index from the Opcode hash argument
// Dropping down to assembler could be the go here...
Function HashValue(HA: LongWord): Word;
Var RR: Longword;
    NI: Byte;
Begin
  RR:=0;
  For NI:=1 to 28 do begin
    RR:=RR Xor HA;
    RR:=RR Shl 1;
    Inc(RR,7);
  End;
  Result:=RR Mod HashPrim;
End;

// Load Instruction Control Record for given Opcode hash argument
Function LoadICR(HA: LongWord): TInstruction;
Var HV: Word;
    SC: Byte;
Begin
  HV:=HashValue(HA);                   // Get home slot from hashed argument
  For SC:=1 to 4 do begin              // Allow up to 4 hash synonyms
    Result:=InstructionTable[HV];      // Get ICR from table slot
    If Result=NIL then Break;          // Empty slot?!
    If Result.HashArg=HA then Exit;    // Correct instruction?
    Inc(HV,13);                        // No, bounce to next synonym slot
    HV:=HV Mod HashPrim;
  End;
  // Hit an empty slot or bounced 4 times, Must be bad opcode
  Raise Exception.Create('Invalid OpCode: '+InstOct(InstReg)+
                         '.'+IntToOct(AddrFld,5)+
                         ' at '+IntToOct(PrevICtr,5));
End;

// Fetch and Decode the next Instruction
Procedure Fetch;
Var HA: LongWord;
begin
  // Test for Stops
  If InstCtr=FetchStopAddr then begin
    Inc(FetchStopCtr);
    If FetchStopTrace And (FetchStopCtr>=FetchStopCount) then
      MainForm.SetTraceOn;
  End;
  If PlotDisp then PlotForm.PlotFetch(InstCtr);
  PrevICtr:=InstCtr;                        // Note previous address
  StorReg:=GetCore(InstCtr);                // Get next instruction from Core
  OrigStorReg:=StorReg;
  // Decode the Instruction - Extract instruction parts
  InstReg:=(StorReg And IOpCMask) Shr IOpCShft;
  FlagReg:=(StorReg And FlagMask) Shr FlagShft;
  TagsFld:=(StorReg And TagsMask) Shr TagsShft;
  DecrFld:=(StorReg And DecrMask) Shr DecrShft;
  AddrFld:=(StorReg And AddrMask);
  OAdrReg:=AddrFld;
  // Checkout OpCode, Form Opcode hash argument for ICR fetch
  If (InstReg And $600)<>0 then begin
    HA:=(InstReg And $E00) Shr 9;                     // Short Opcodes
  End else Begin
    If (InstReg=$1F0) Or (InstReg=$9F0) then begin
      HA:=(InstReg Shl 13) Or AddrFld;                // Long Opcodes
    End else Begin
      HA:=InstReg;                                    // Normal Opcodes
    End;
  End;
  // Load Instruction Control Record from Opcode hash argument
  InstRec:=LoadICR(HA);
  // Apply Address Modifications
  With InstRec do
    If IdxModEn then begin
      If TagsFld<>0 then begin
        // Indexing
        AddrFld:=AddrFld-GetXR(TagsFld);
        AddrFld:=AddrFld And AddrMask;
      End;
      If IndAdrEn And (FlagReg=3) then begin
        // Indirect Addressing
        StorReg:=GetCore(AddrFld);
        TagsFld:=(StorReg And TagsMask) Shr TagsShft;
        AddrFld:=(StorReg And AddrMask);
        AddrFld:=AddrFld-GetXR(TagsFld);
        AddrFld:=AddrFld And AddrMask;
      End;
    End;
  // Display instruction details
  If (TIINS In TraceRecdFlags) then ShowInstruction;
End;

// Execute the Fetched Instruction
Procedure Execute;
Begin
  // Check for Stops
  If InstCtr=FetchStopAddr then begin
    Inc(FetchStopCtr);
    If FetchStopStop And (FetchStopCtr>=FetchStopCount) then begin
      SetStopPending('Fetch from Address');
      Exit;
    End;
  End;
  // Set next instruction address
  Inc(InstCtr); InstCtr:=InstCtr And CoreWrap;
  // Execute current instruction
  With InstRec do begin
    ExProc(InstRec);              // Run the Execution proc for this instruction
    Inc(UseCount);                // Count times used
  End;
  Inc(InstCount);                 // Count instructions executed
  // Display effects of instruction after execution
  If (TIREG In TraceRecdFlags) Or (TIMEM In TraceRecdFlags) then DisplayEffects;
  // SlowMode?
  If SlowMode And AutoMode then begin
    ShowLastTrace;                // Make last trace entry visible
    MainForm.UpdateDisplays;      // Update console display
    Sleep(500);                   // Wait a bit
    Application.ProcessMessages;  // Needed to allow exit from slow mode
  End;
End;

// CPU phase of a processor cycle
procedure ProcessCPU;
begin
  If StopPend Or                  // If stop is pending, or
     ChanWait then Exit;          // waiting for channel, Don't execute
  Try
    Fetch;                        // Fetch & Decode instruction
    Execute;                      // Execute it
  Except
    On E: Exception do begin      // Screwed up?
      Trace(TIGEN,E.Message);     // Log it
      CPUCheck:=True;             // Set checked state
      CPURunning:=False;          // Mark no longer running
      Raise;                      // Pass the exception upwards
    End;
  End;
End;

// Channel phase of a processor cycle
Procedure ProcessChannels;
Var CI: Integer;
begin
  Try
    ChnlActive:=False;            // Reset any channel active flag
    For CI:=1 to LastChan do      // Try each channel
      With Channels[CI] do begin
        // If channel is active, Run one cycle
        If Active then begin
          If CurDevice.WRMode then OperateWR       // Do write operations
                              else OperateRD;      // Or read operations
          If Active then         // If it's still active now,
            ChnlActive:=True;    // note that one or more channels are active
        End;
        // Process any pending channel trap
        If TrapInhib then Continue;    // Traps inhibited for one instruction?
        If TrapCtrlInd And             // Traps enabled?
           (LCHTrapEnab And LCHInd) Or // Yes, Is indicator on and the trap
           (TCKTrapEnab And TCKInd) Or // type Enabled?
           (EOFTrapEnab And EOFInd) then begin
          PerformChannelTrap(CI);      // Yes, Setup channel trap
          Break;                       // then let it execute
        End;
      End;
  Except
    On E: Exception do begin      // Screwed up?
      Trace(TIGEN,E.Message);     // Log it
      ChnCheck:=True;             // Set checked state
      CPURunning:=False;          // Mark no longer running
      Raise;                      // Pass it up
    End;
  End;
  TrapInhib:=False;               // Let any traps occur next cycle
  // Update console display regularly
  Inc(LoopCtr);
  If (LoopCtr Mod TrueFact)=0 then MainForm.UpdateDisplays;
End;

// Main Processor Loop - Interleaves CPU and Channel cycles
procedure Process;
begin
  CPURunning:=True;
  Repeat
    // Let other stuff get a look in sometimes
    If (LoopCtr Mod $8000)=0 then Application.ProcessMessages;
    // Fetch, Decode and Execute next instruction (if no stop pending)
    ProcessCPU;
    // Run Channel cycles (also updates console display)
    ProcessChannels;
    // Continue until Stop is pending, then run channels only until all inactive
  Until StopPend And (Not ChnlActive);
  MainForm.UpdateDisplays;            // Ensure display is up to date
  CPURunning:=False;
End;

End.
