//******************************************************************************
//  IBM 7094 Emulator - Specialized Display components
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains specialized display components used by the emulator.
//  Before displaying any emulator forms in Delphi, this unit must be
//  first installed to the Delphi component pallete. (or it'll screw things up)
//------------------------------------------------------------------------------
Unit B709Cmps;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics,
     Controls, Forms, Dialogs,ExtCtrls,StdCtrls;

Type TBitData=Record
       Invs: Boolean;
       Valu: Boolean;
       XPos: Integer;
       XTxt: Integer;
       Colr: TColor;
       Text: String[2];
     End;

//******************************************************************************
//  TDispReg - Register display component
//------------------------------------------------------------------------------
Type
  TDispReg = class(TCustomPanel)
    Constructor Create(OW: TComponent); Override;
    Procedure Paint;                    Override;
    Procedure Loaded;                   Override;
  private
    Radix:    Byte;
    NumBits:  Byte;
    ActBits:  Byte;
    ValDigs:  Byte;
    FHexMode: Boolean;
    Force:    Boolean;
    FLegend:  ShortString;
    Bits:     Array[0..40] of TBitData;
    BitsTop:  Integer;
    LedsTop:  Integer;
    ValPosn: Integer;
    ValWid:  Integer;
    ValHgt:  Integer;
    BitHgt:  Integer;
    FValue:   Int64;
    FCompact: Boolean;
    FBitWidth: Integer;
    FLedWidth: Integer;
    FShowLgd: Boolean;
    FLedHeight: Integer;
    FBitHeight: Integer;
    FValFont: TFont;
    FBitFont: TFont;
    procedure SetBitWidth(NV: Integer);
    procedure SetLedWidth(NV: Integer);
    procedure SetBitHeight(NV: Integer);
    procedure SetLedHeight(NV: Integer);
    Procedure SetHexMode(const HM: Boolean);
    procedure SetCompact(const NV: Boolean);
    Procedure SetLegend(const LG: ShortString);
    procedure SetShowLgd(const NV: Boolean);
    Procedure SetValue(const NV: Int64);
    procedure SetupLEDS;
    procedure PaintLED(BD: TBitData);
    procedure SetBitFont(const NV: TFont);
    procedure SetValFont(const NV: TFont);
    procedure SetValueLED;
    procedure SetValueTXT;
  protected
  public
  published
    Property HexMode:  Boolean      Read FHexMode   Write SetHexMode;
    Property ShowLgd:  Boolean      Read FShowLgd   Write SetShowLgd;
    Property Legend:   ShortString  Read FLegend    Write SetLegend;
    Property Value:    Int64        Read FValue     Write SetValue;
    Property Compact:  Boolean      Read FCompact   Write SetCompact;
    Property BitWidth: Integer      Read FBitWidth  Write SetBitWidth;
    Property LedWidth: Integer      Read FLedWidth  Write SetLedWidth;
    Property BitHeight: Integer     Read FBitHeight Write SetBitHeight;
    Property LedHeight: Integer     Read FLedHeight Write SetLedHeight;
    Property BitFont: TFont Read FBitFont Write SetBitFont;
    Property ValFont: TFont read FValFont write SetValFont;
    Property Enabled;
    Property Anchors;
    Property Color;
    Property ParentColor;
  end;

//******************************************************************************
//  TButnInd - Button/Indicator component
//------------------------------------------------------------------------------
Type
  TButnInd=class(TCustomControl)
    Constructor Create(OW: TComponent); Override;
    Procedure Paint;                    Override;
    Procedure Loaded;                   Override;
  private
    FRoundX:    Integer;
    FRoundY:    Integer;
    FState:     Boolean;
    FONCaption,
    FOFCaption: ShortString;
    FONColor,
    FOFColor:   TColor;
    FOnClick:   TNotifyEvent;
    TextBegY:   Integer;
    procedure SetOfCaption(const NV: ShortString);
    procedure SetState(const NV: Boolean);
    procedure SetOnCaption(const NV: ShortString);
    procedure SetRoundX(Const RX: Integer);
    procedure SetRoundY(Const RY: Integer);
  protected
//  procedure BIClick(Sender: TObject);
  public
    procedure Click; override;
  published
    Property Enabled;
    Property RoundX:     Integer      Read FRoundX     Write SetRoundX;
    Property RoundY:     Integer      Read FRoundY     Write SetRoundY;
    Property OnCaption:  ShortString  Read FOnCaption  Write SetOnCaption;
    Property OfCaption:  ShortString  Read FOfCaption  Write SetOfCaption;
    Property ONColor:    TColor       Read FONColor    Write FONColor;
    Property OFColor:    TColor       Read FOFColor    Write FOFColor;
    Property State:      Boolean      Read FState      Write SetState;
    Property Anchors;
    Property Font;
    Property TabOrder;
    Property OnClick;
  end;

//******************************************************************************
//  TUnitRecordDisplay - Unit Record display component
//------------------------------------------------------------------------------
Type TCard=String[80];
     TLine=String[132];

Type TURMode=(URMRDR,URMPUN,URMPRN);

Type TEditEvent=Procedure(SE: TObject; RN: Word) Of Object;

Type TUnitRecordDisplay = Class(TCustomControl)
        constructor Create(AOwner: TComponent);            Override;
        destructor Destroy;                                Override;
        Procedure Loaded;                                  Override;
      private
        DrwBitMap: TBitMap;   // Offscreen bitmap (to avoid flicker)
        DrwCanvas: TCanvas;   // And its canvas
        DrwRect:   TRect;
        SBVert:    TScrollBar;
        CharWid,
        CharHgt:  Integer;
        LftChar:  Integer;
        FURMode:  TURMode;
        SetOrLock: Boolean;
        FRecords: TStringList;
        FOnPositionChange:   TEditEvent;
        procedure SBVertChange(Sender: TObject);
        procedure FontChanged(Sender: TObject);
        procedure SetURMode(const Value: TURMode);
        procedure SetVSBMax;
        procedure SetRecords(Const SL: TStringList);
      protected
        Procedure WMERASEBKGND(Var WM: TWMERASEBKGND);     Message WM_ERASEBKGND;
        procedure Paint;                                   Override;
        procedure Resize; Override;
      public
        RecSize:      Integer;
        LinesPerPage: Integer;
        BegRec:       Integer;
        procedure SetSize;
        procedure SetDisplayPosition(CO,RO: Integer);
        Property  Records: TStringList Read FRecords Write SetRecords;
      published
        Property OnPositionChange:   TEditEvent Read FOnPositionChange   Write FOnPositionChange;
        Property URMode: TURMode Read FURMode Write SetURMode;
        Property Color;
        Property Align;
        Property Font;
      end;

Var ShowTextValues: Boolean;

Procedure Register;

Implementation

//******************************************************************************
// Delphi Component Registration procedure
//------------------------------------------------------------------------------
procedure Register;
begin
  RegisterComponents('B709Comps',[TDispReg,TButnInd,TUnitRecordDisplay]);
end;

//******************************************************************************
//  TDispReg - Register display component
//------------------------------------------------------------------------------
// This component implements a binary register display.
// Originally, it was intended to be more general purpose, but it should be
// pared down to just those functions needed in the emulator, making it much
// faster in the process.
//******************************************************************************
Const HGap=4; VGap=4;

Function IntToOct(IV: Int64; ND: Byte): ShortString;
Var DC: Byte;
Begin
  Result:='';
  For DC:=1 to ND do begin
    Result:=Chr((IV And 7)+$30)+Result;
    IV:=IV Shr 3;
  End;
End;

Constructor TDispReg.Create(OW: TComponent);
Begin
  Inherited;
  Parent:=TWinControl(OW);
  FBitFont:=TFont.Create;
  FValFont:=TFont.Create;
  FBitWidth:=12; FLedWidth:=10;
  FHexMode:=False;
  FLedWidth:=8;
  FBitWidth:=10;
  FLedHeight:=8;
  FBitHeight:=10;
  FValue:=$5555;
  FLegend:='0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15';
  FShowLgd:=True;
  Radix:=4;
End;

procedure TDispReg.Loaded;
begin
  Inherited;
  SetupLEDS;
end;

procedure TDispReg.SetBitFont(const NV: TFont);
begin
  FBitFont:=NV; SetupLEDS;
end;

procedure TDispReg.SetValFont(const NV: TFont);
begin
  FValFont:=NV; SetupLEDS;
end;

procedure TDispReg.SetShowLgd(const NV: Boolean);
begin
  FShowLgd:=NV; SetupLEDS;
end;

procedure TDispReg.SetCompact(Const NV: Boolean);
begin
  FCompact:=NV; SetupLEDS;
end;

procedure TDispReg.SetLegend(Const LG: ShortString);
begin
  FLegend:=LG;  SetupLEDS;
end;

procedure TDispReg.SetHexMode(Const HM: Boolean);
begin
  If HM then Radix:=4 else Radix:=3;
  FHexMode:=HM; SetupLEDS;
  Application.ProcessMessages;
  Force:=True;
  SetValue(FValue);
End;

procedure TDispReg.SetBitWidth(NV: Integer);
begin
  If NV<4 then Exit;
  If NV>200 then Exit;
  If NV<FLedWidth then NV:=FLedWidth;
  FBitWidth:=NV; SetupLEDS;
end;

procedure TDispReg.SetLedWidth(NV: Integer);
begin
  If NV<4 then Exit;
  If NV>200 then Exit;
  If NV<4 then Exit;
  If NV>FBitWidth then NV:=FBitWidth;
  FLedWidth:=NV; SetupLEDS;
end;

procedure TDispReg.SetBitHeight(NV: Integer);
begin
  If NV<4 then Exit;
  If NV>200 then Exit;
  If NV<FLedHeight then NV:=FLedHeight;
  FBitHeight:=NV; SetupLEDS;
end;

procedure TDispReg.SetLedHeight(NV: Integer);
begin
  If NV<4 then Exit;
  If NV>200 then Exit;
  If NV<4 then Exit;
  If NV>FBitHeight then NV:=FBitHeight;
  FLedHeight:=NV; SetupLEDS;
end;

Procedure TDispReg.SetupLEDS;
Var LL,LP,BC,XL:  Integer;
Begin
  If FLegend='' then Exit;
  // Form H/V sizes of Value/Bit text fonts
  With Canvas do begin
    Font.Assign(FBitFont);
    BitHgt:=TextHeight('X');
    Font.Assign(FValFont);
    With TextExtent('X') do begin
      ValWid:=CX; ValHgt:=CY;
    End;
  End;
  // Process Legend to get Bit texts and Number of bits
  NumBits:=0; ActBits:=0;
  LL:=Length(FLegend);
  While LL>1 do begin
    LP:=LL; Dec(LL);
    While (FLegend[LL]<>' ') And (LL>0) do Dec(LL);
    Inc(NumBits);
    With Bits[NumBits] do begin
      Text:=Trim(Copy(FLegend,LL+1,LP-LL));
      Invs:=(Text='-') Or (Text='.');
      If Text<>'-' then Inc(ActBits);  // Count actual bits
    End;
  End;
  // Form number of text digits
  ValDigs:=(NumBits Div Radix);
  If (NumBits Mod Radix)>0 then Inc(ValDigs);
  // Set component width
  ValPosn:=HGap+(ActBits*BitWidth)+HGap;
  Width:=ValPosn+(ValWid*ValDigs)+HGap+HGap;
  // Form Left offset for each bit and its text, Set "Off" colour
  XL:=ValPosn-HGap;
  For BC:=1 to NumBits do begin
    With Bits[BC] do begin
      If Text<>'-' then Dec(XL,FBitWidth);  // Skip "missing" bits
      XPos:=XL;
      If Length(Text)>1 then XTxt:=XPos+2
                        else XTxt:=XPos+4;
      Colr:=CLLtGray;
    End;
  End;
  // Display it all
  Invalidate;
End;

Procedure TDispReg.Paint;
Var BC,BR,BG,X1,X2,RC: Integer;
    FF: Boolean;
Begin
  With Canvas do begin
    BG:=(FBitHeight-FLedHeight) Div 2;
    If FShowLgd then begin
      BitsTop:=VGap+BitHgt;//+VGap;
      Height:=BitsTop+BitHeight+VGap;
    End else Begin
      BitsTop:=VGap;
      Height:=BitsTop+BitHeight+VGap;
    End;
    LedsTop:=BitsTop+BG;
    // Clear display area
    Brush.Color:=Color;
    FillRect(ClientRect);
    // Paint Hex/Oct background panels
    BC:=0; RC:=Radix-1; FF:=True;
    BG:=(BitWidth-LedWidth) Div 2;
    Repeat
      Inc(BC);
      X2:=Bits[BC].XPos+LedWidth+BG;
      BR:=0;
      Repeat
        If BC=NumBits then Break;
        Inc(BC);
        If Not Bits[BC].Invs then Inc(BR);
      Until BR=RC;
      If FF then begin
        X1:=Bits[BC].XPos-BG;
        Brush.Color:=CLAqua;
        FillRect(Rect(X1,BitsTop,X2,BitsTop+FBitHeight));
      End;
      FF:=Not FF;
    Until BC=NumBits;
    // Set font for bit texts
    Font.Assign(FBitFont);
    // Show bit legend
    If FShowLgd then begin
      Brush.Color:=Color;
      For BC:=1 to NumBits do
        With Bits[BC] do
          If Not Invs then
            Canvas.TextOut(XTxt,VGap,Text);
    End;
  End;
  Force:=True;
  // Paint bit panels
  For BC:=1 to NumBits do
    PaintLED(Bits[BC]);
End;

procedure TDispReg.SetValue(Const NV: Int64);
begin
  If (NV=FValue) And (Not Force) then Exit;
  FValue:=NV;
  If FCompact then Font.Assign(FBitFont);
  SetValueLED;
  If ShowTextValues then SetValueTXT;
  Force:=False;
End;

procedure TDispReg.SetValueLED;
Var BC: Integer;
    BM: Int64;
    NB: Boolean;
Begin
  BM:=1;
  For BC:=1 to Numbits do begin
    With Bits[BC] do begin
      If Invs then Continue;
      NB:=(FValue And BM)>0;
      If (NB<>Valu) Or Force then begin
        If NB then Colr:=CLYellow else Colr:=CLLtGray;
        PaintLED(Bits[BC]);
      End;
      Valu:=NB;
    End;
    BM:=BM Shl 1;
  End;
End;

procedure TDispReg.SetValueTXT;
Var VT: Integer;
Begin
  // Display value text
  With Canvas do begin
    Brush.Color:=Color;
    Font.Assign(FValFont);
    VT:=LedsTop+VGap;
    If FHexMode then TextOut(ValPosn,VT,IntToHex(FValue,ValDigs))
                else TextOut(ValPosn,VT,IntToOct(FValue,ValDigs));
  End;
End;

Procedure TDispReg.PaintLED(BD: TBitData);
Begin
  With Canvas,BD do begin
    If Invs then Exit;
    Brush.Color:=Colr;
    Rectangle(XPos,LedsTop,XPos+LedWidth,LedsTop+FLedHeight);  // Paint bit panel
    If Not FCompact then TextOut(XTxt,LedsTop+3,Text);         // Paint bit text
  End;
End;

//******************************************************************************
//  TButnInd - Button/Indicator component
//------------------------------------------------------------------------------
// This component implements a combined button and indicator.
//******************************************************************************
constructor TButnInd.Create(OW: TComponent);
begin
  Inherited;
  Parent:=TWinControl(OW);
  Width:=50; Height:=20;
  Caption:='';
  FOnCaption:='on';
  FOfCaption:='off';
  FONColor:=ClYellow;
  FOFColor:=ClBlue;
  FState:=False;
//OnClick:=Click;
end;

procedure TButnInd.Loaded;
begin
  Inherited;
  With Canvas do begin
    Font.Assign(Self.Font);
    TextBegY:=(Height-TextHeight('W')) Div 2;
  End;
  Invalidate;
end;

procedure TButnInd.SetRoundX(Const RX: Integer);
begin
  FRoundX:=RX; Invalidate;
end;

procedure TButnInd.SetRoundY(Const RY: Integer);
begin
  FRoundY:=RY; Invalidate;
end;

procedure TButnInd.SetOfCaption(const NV: ShortString);
begin
  FOfCaption:=NV;
  Invalidate;
end;

procedure TButnInd.SetOnCaption(const NV: ShortString);
begin
  FOnCaption:=NV;
  Invalidate;
end;

procedure TButnInd.SetState(const NV: Boolean);
begin
  If NV=State then Exit;
  FState:=NV;
  Invalidate;
end;

Procedure TButnInd.Paint;
Var CS: String;
    CO: TColor;
    BX: Integer;
Begin
  Visible:=Enabled;
  If FState then begin
    CS:=ONCaption; CO:=ONColor;
  End else Begin
    CS:=OFCaption; CO:=OFColor;
  End;
  With Canvas do begin
    Brush.Color:=CO;
    If (FRoundX<>0) Or (FRoundY<>0) then begin
      Pen.Color:=CO;
      With ClientRect do RoundRect(Left,Top,Right,Bottom,FRoundX,FRoundY);
      Pen.Color:=Font.Color;
    End else Begin
      Pen.Color:=ClBlue;
      With ClientRect do Rectangle(Left,Top,Right,Bottom);
//      FillRect(ClientRect);
      Pen.Color:=Font.Color;
    End;
//  Pen.Style:=PSSolid; Pen.Color:=CLBlack;
    BX:=(ClientWidth-TextWidth(CS)) Div 2;
    TextOut(BX,TextBegY,CS);
  End;
  Inherited;
end;
{
procedure TButnInd.BIClick;
begin
  If Assigned(FOnClick) then
    FOnClick(Self);
end;
}
procedure TButnInd.Click;
begin
  Inherited;
  If Assigned(FOnClick) then
    FOnClick(Self);
end;

//******************************************************************************
//  TUnitRecordDisplay - Unit Record display component
//------------------------------------------------------------------------------
// This component implements a display for list of unit records, (card images
// and print lines).
// It exists mainly to allow card decks to be displayed in reverse order with
// first card at the bottom of the list. But it does provide other functions
// that are useful for both unit record types.
//******************************************************************************
Const SBSize=25;

Constructor TUnitRecordDisplay.Create(AOwner: TComponent);
Begin
  Inherited;
  Name:='UnitRecord';
  URMode:=URMRDR; RecSize:=80;
  Color:=CLWhite;
  Width:=100; Height:=100;
  LftChar:=1; BegRec:=1;
  With Font do begin
    Name:='lucidia console'; Size:=6; Pitch:=FPFixed;
  End;
//OnClick:=MOClick;
//OnDblClick:=DBClick;
  DrwBitMap:=TBitMap.Create;
  SBVert:=TScrollBar.Create(Self);
  With SBVert do begin
    Parent:=Self;       TabStop:=False;
    Kind:=SBVertical;   Align:=ALRight;
    Width:=SBSize;      OnChange:=SBVertChange;
  End;
  Font.OnChange:=FontChanged;
End;

destructor TUnitRecordDisplay.Destroy;
begin
  SBVert.Free;
  DrwBitMap.Free;
  Inherited;
end;

procedure TUnitRecordDisplay.Loaded;
begin
  Inherited;
  Resize;
End;

procedure TUnitRecordDisplay.Resize;
begin
  Inherited;
  With DrwBitMap do begin
    Width:=ClientWidth-SBSize+8;        // Set to display area size,
    Height:=ClientHeight;//-SBSize;      // reduced by scrollbar width
    DrwCanvas:=Canvas;                  // Make it available for drawing on
  End;
  With DrwCanvas do begin
    DrwRect:=ClientRect;                // Make size available
    Font.Assign(Self.Font);             // Set to same font as screen
  End;
end;

procedure TUnitRecordDisplay.WMERASEBKGND(Var WM: TWMERASEBKGND);
begin
  WM.Result:=1;
end;

procedure TUnitRecordDisplay.SetRecords(Const SL: TStringList);
begin
  FRecords:=SL;
end;

procedure TUnitRecordDisplay.SetURMode(const Value: TURMode);
begin
  FURMode:=Value;
  If FRecords<>NIL then FRecords.Clear;
  If Value=URMPRN then RecSize:=100 else RecSize:=80;
  SetSize;
end;

procedure TUnitRecordDisplay.FontChanged(Sender: TObject);
begin
  SetSize;
end;

Procedure TUnitRecordDisplay.SetSize;
Begin
  If DrwBitMap=NIL then Exit;
  With DrwBitMap do begin
    With Canvas do begin
      Font.Assign(Self.Font);             // Set to same font as screen
      With TextExtent('WM') do begin
        CharWid:=CX Div 2; CharHgt:=CY;
      End;
    End;
    Height:=Self.Height;
    LinesPerPage:=(Height Div CharHgt)+1;
    DrwCanvas:=Canvas;                  // Make it available for drawing on
  End;
  SetDisplayPosition(1,1);
  Invalidate;
end;

procedure TUnitRecordDisplay.Paint;
Var PY: Integer;
    RN: Word;
    RS: String[132];
begin
  Inherited;
  If CSLoading in ComponentState then Exit;
  If DrwBitMap=NIL then Exit;
  If DrwCanvas=NIL then Exit;
  // Work with offscreen bitmap's canvas
  With DrwCanvas do begin
    // Clear it
    Brush.Color:=Self.Color;
    FillRect(DrwRect);
    Pen.Style:=PSSolid; Pen.Color:=CLBlack;
    If FRecords<>NIL then
      If FURMode=URMRDR then begin
        RN:=BegRec;
        PY:=Height-CharHgt;
        While (PY>0) And (RN<=FRecords.Count) do begin
          RS:=FRecords[RN-1];
          RS:=Copy(RS,LftChar,RecSize);
          TextOut(0,PY,RS);
          Dec(PY,CharHgt); Inc(RN);
        End;
      End else Begin
        RN:=BegRec;
        If RN>FRecords.Count then RN:=FRecords.Count;
        PY:=0;
        While (PY<=Height) And (RN<=FRecords.Count) And (RN>0) do begin
          RS:=FRecords[RN-1];
          RS:=Copy(RS,LftChar,RecSize);
          TextOut(0,PY,RS);
          Inc(PY,CharHgt); Inc(RN);
        End;
      End;
  End;
  // Update screen canvas from offscreen canvas
  Canvas.Draw(0,0,DrwBitMap);
End;

procedure TUnitRecordDisplay.SBVertChange(Sender: TObject);
Var SP: Integer;
begin
  SP:=SBVert.Position;
  If FURMode=URMRDR then SP:=-SP;
  SetDisplayPosition(LftChar,SP);
  Invalidate;
end;

procedure TUnitRecordDisplay.SetDisplayPosition(CO,RO: Integer);
Var RC: Integer;
begin
  If FRecords=NIL then Exit;
  If SetOrLock then Exit;
  SetOrLock:=True;
  LftChar:=CO;
  RC:=FRecords.Count;
  If RO>RC then RO:=RC;
  If RO<1 then RO:=1;
  BegRec:=RO;
  SetVSBMax;
  If FURMode=URMRDR then RO:=-RO;
  SBVert.Position:=RO;
  If Assigned(FOnPositionChange) then
    FOnPositionChange(Self,BegRec);
  SetOrLock:=False;
end;

procedure TUnitRecordDisplay.SetVSBMax;
Var ML: Integer;
begin
  With SBVert do begin
    SmallChange:=1;
    LargeChange:=LinesPerPage;
    Min:=1;
    If FRecords=NIL then begin
      ML:=1;
    End else Begin
      ML:=FRecords.Count-LinesPerPage+2;
      If ML<1 then ML:=1;
    End;
    If FURMode=URMRDR then begin
      Max:=1; Min:=-ML;
    End else Begin
      Min:=1; Max:=ML;
    End;
    PageSize:=LinesPerPage;
  End;
End;

Initialization
  ShowTextValues:=True;
End.
