//******************************************************************************
//  IBM 7094 Emulator - Line Printer device
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit defines the TDevLST object which descends from TDevice
//  TDevLST provides the functions for the IBM 716 line printer
//  There is only one instance of TDevLST, "LSTDev", which has an
//  associated form window "PrinterForm" to display its data.
//  This unit also contains the definition and methods of PrinterForm.
//------------------------------------------------------------------------------
Unit B709DLST;

Interface

Uses SysUtils,Windows,Forms,Messages, StdCtrls,
     Controls, Classes, ExtCtrls, Dialogs,
     B709Cmps, // Display Components
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Misc, // Miscellaneous utility functions
     B709Trce, // Log/Trace functions
     B709Chan; // I/O Channel and Device functions

Type TDevLST=Class(TDevice)
       Constructor Create(DT: TDevType; CH: TChannel; DA: TAddr);      Override;
       Destructor  Destroy;                                            Override;
     Private
       CardImage: Array[1..80] Of Word;
       PunchMask: Word;
//;    EndOfLine:  Boolean;
       Procedure GeneratePrintLine;
       Procedure ResetLine;
//     Procedure WriteWordFudge(WW: TWord);
     Protected
       Function  DeviceSelect(WR: Boolean): Boolean;                   Override;
//     Function  TestEOR: Boolean;                                     Override;
       Procedure WriteWord(WW: TWord);                                 Override;
     Public
       LineString: String;
     End;

Const Heading='IBM 716 Line Printer';

Type
  TPrinterForm = class(TForm)
    Panel1: TPanel;
    LALineNum: TLabel;
    Label1: TLabel;
    CBSize: TCheckBox;
    Page: TUnitRecordDisplay;
    Label2: TLabel;
    LALineCount: TLabel;
    Timer: TTimer;
    CBLUpd: TCheckBox;
    BIClear: TButnInd;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure CBSizeClick(Sender: TObject);
    procedure PagePositionChange(SE: TObject; RN: Word);
    procedure TimerTimer(Sender: TObject);
    procedure CBLUpdClick(Sender: TObject);
    procedure BIClearClick(Sender: TObject);
  private
    PrintOut: TStringList;
    UpdFlag:  Boolean;         // Data has changed since last timer tick
    TopLine:  Integer;
    Procedure ShowFileDetails(Sender: TObject);
    Procedure RefreshDisplay(Sender: TObject);
    Procedure PrintLine(Sender: TObject);
  public
    Procedure Initialize;
    Procedure Clear;
  end;

Var LSTDev:      TDevLST;              // Device instance
    PrinterForm: TPrinterForm;         // Form instance

Implementation

Uses B709Main,B709DRDR, B709DTAP; // For form positioning

{$R *.DFM}

// 24 words make up a 72 character print line
// The words are loaded to a buffer, which is in the form of a card image,
// The 12 bits in each column are interpreted as card code chars for printing
//      00000000011111111112222222222333333333344444444445555555555666666666677777777778
//      12345678901234567890123456789012345678901234567890123456789012345678901234567890
// 12   +-------------WORD 23--------------++-------------WORD 24--------------+ 8
// 11   +-------------WORD 21--------------++-------------WORD 22--------------+ 4
//  0   +-------------WORD 19--------------++-------------WORD 20--------------+ 2
//  1   +-------------WORD 17--------------++-------------WORD 18--------------+ 1
//  2   +-------------WORD 15--------------++-------------WORD 16--------------+ 8
//  3   +-------------WORD 13--------------++-------------WORD 14--------------+ 4
//  4   +-------------WORD 11--------------++-------------WORD 12--------------+ 2
//  5   +-------------WORD 9---------------++-------------WORD 10--------------+ 1
//  6   +-------------WORD 7---------------++-------------WORD 8---------------+ 8
//  7   +-------------WORD 5---------------++-------------WORD 6---------------+ 4
//  8   +-------------WORD 3---------------++-------------WORD 4---------------+ 2
//  9   +-------------WORD 1---------------++-------------WORD 2---------------+ 1

//******************************************************************************
//  TDevLST Methods
//------------------------------------------------------------------------------
constructor TDevLST.Create(DT: TDevType; CH: TChannel; DA: TAddr);
begin
  Inherited Create(DT,CH,DA);
  BinAddr:=DA+$01;                     // Form address for Binary mode
  FileNam:='B716.PRN';
  WRProt:=False;
  ResetLine;
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' 716 Line Printer defined');
end;

destructor TDevLST.Destroy;
begin
  Inherited;
end;

Function  TDevLST.DeviceSelect(WR: Boolean): Boolean;
begin
If FBinMode then Showmessage('LST in Bin Mode');
  Result:=Inherited DeviceSelect(WR);
{  If Not WR then begin                  // For now
    Result:=False;
//  Error('Read selected for Printer');
  End;}
  BinMode:=(DevAddr And $02)<>0;
ResetLine;
EndOfRec:=False;
End;

Procedure TDevLST.WriteWord(WW: TWord);
Var CI,BI: Byte;
    BM: TWord;
begin
//XTrace('LST word='+IntToOct(WW,12)+' '+WordDump(WW));
{If FudgeMode then begin
  WriteWordFudge(WW);
  Exit;
End;}
  // Build Hollerith (Row binary) card image of 72 char print line
  // Load incoming words two to a card row,
  // going left to right, bottom to top
  If EndOfRec then
     ResetLine;                    // Prepare for new line?
  If Odd(WordNum) then CI:=37 else CI:=1;        // Starting bit column - left/right word?
  BM:=$800000000;                                // Start mask over leftmost bit in word
  For BI:=1 to 36 do begin                       // For each bit in the word/column
    If (WW And BM)<>0 then                       // If word bit is on, set column bit
      CardImage[CI]:=CardImage[CI] Or PunchMask;
    BM:=BM Shr 1;                                // Move mask to next word bit
    Inc(CI);                                     // and move to next card column
  End;
  Inc(WordNum);                                  // Count words done,
  If Not Odd(WordNum) then                       // Every second word, advance
    PunchMask:=PunchMask Shl 1;                  // the punch mask to row above
  If WordNum=24 then begin                       // When all words done,
    GeneratePrintLine;                           // Form ASCII text line
//    ProcessEndOfRecord;                          // Tell form to print it
//    ResetLine;                                   // Prepare for next line
  End;
End;
{
Procedure TDevLST.WriteWordFudge(WW: TWord);
Begin
  LineString:=LineString+WordDump(WW);
  Inc(WordNum);
  If WordNum=14 then begin                       // When all words done,
    ProcessEndOfRecord;                          // Tell form to print it
  End;
End;
}
procedure TDevLST.GeneratePrintLine;
Var CI: Byte;
    CW: Word;
    CB: Byte;
    CH: Char;
Begin
  LineString:=StringOfChar(' ',72);//+#$D+#$A;   // Blank output line
  For CI:=1 to 72 do begin
    CW:=CardImage[CI];                 // Get punch word for column
    CB:=TabCRDtoBCD[CW];               // Convert to BCD 9-Code
    CH:=TabBCDtoASC[CB+1];             // then to ASCII
    LineString[CI]:=CH;                        // Insert char into output line
  End;
  PutBytes(LineString[1],72);                 // Store line bytes to file
  Inc(RecdNum);                        // Count records
  // Update associated display
//  PrinterForm.PrintLine(LS);           // Print the line
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' Write Line: '+LineString);
  ProcessEndOfRecord;                          // Tell form to print it
  ResetLine;                                   // Prepare for next line
End;

Procedure TDevLst.ResetLine;
Begin
  // Reset for new line
  WordNum:=0;                               // Start at row 9 left
  PunchMask:=1;                             // Start punch mask on bottom (9) row
  EndOfRec:=False;                          // Not at end of line
  FillChar(CardImage,SizeOf(CardImage),#0); // Clear ASCII print line
End;

//******************************************************************************
//  Form Functions
//------------------------------------------------------------------------------
procedure TPrinterForm.FormCreate(Sender: TObject);
begin
  FormHandles[FIPrinter]:=Handle;
  Caption:=Heading;
  LALineNum.Caption:='';
  LALineCount.Caption:='';
  PrintOut:=TStringList.Create;
  Page.Records:=PrintOut;
  TopLine:=1;
end;

procedure TPrinterForm.FormDestroy(Sender: TObject);
begin
//Printout.SaveToFile('B716.PRN');
  Printout.SaveToFile(LSTDev.FileNam);
  PrintOut.Free;
end;

procedure TPrinterForm.Initialize;
//r TH: Integer;
Begin
//Width:=Screen.Width-Left-2;
//TH:=TapesForm.Height;
//If TH<300 then begin
//  Left:=MainForm.Width;
//  Top:=TapesForm.Top+TapesForm.Height
//End else Begin
//  Left:=MainForm.Width+200;
//  Top:=TapesForm.Top+50;
//End;
//Height:=Screen.Height-Top-2-25;
//Show;
  // Set up event handlers
  LSTDev.OnFileChange:=ShowFileDetails;
  LSTDev.OnStatusChange:=RefreshDisplay;
  LSTDev.OnEndOfRecord:=PrintLine;
end;

Procedure TPrinterForm.FormResize(Sender: TObject);
begin
  Page.SetSize;
end;

Procedure TPrinterForm.CBSizeClick(Sender: TObject);
begin
  With Page do begin
    If CBSize.Checked then Font.Size:=8
                      else Font.Size:=6;
    SetSize;
  End;
End;

procedure TPrinterForm.PagePositionChange(SE: TObject; RN: Word);
begin
  TopLine:=RN;
  LALineNum.Caption:=IntToStr(RN);
end;

procedure TPrinterForm.CBLUpdClick(Sender: TObject);
begin
  If CBLUpd.Checked then Timer.Interval:=5000
                    else Timer.Interval:=250;
end;

procedure TPrinterForm.BIClearClick(Sender: TObject);
begin
  Clear;
end;

procedure TPrinterForm.Clear;
begin
  Page.Records.Clear;
  Page.SetDisplayPosition(1,1);
  RefreshDisplay(Self);
end;

procedure TPrinterForm.ShowFileDetails(Sender: TObject);
begin
  Caption:=Heading+' - '+LSTDev.FileNam;
end;

procedure TPrinterForm.RefreshDisplay(Sender: TObject);
begin
  UpdFlag:=True;
  LALineCount.Caption:=IntToStr(PrintOut.Count);
End;

procedure TPrinterForm.PrintLine(Sender: TObject);
Var TL: Integer;
Begin
  PrintOut.Add(LSTDev.LineString);            // Print the line,
  If PrintOut.Count=1 then             // Update line number label when lines appear
    RefreshDisplay(Self);
  TL:=PrintOut.Count-Page.LinesPerPage+2;
  If TL<1 then TL:=1;
  TopLine:=TL;
  UpdFlag:=True;
  If CBLUpd.Checked then TimerTimer(Self);
End;
{
function TPrinterForm.LineCount: Word;
begin
  Result:=PrintOut.Count;
End;
}
procedure TPrinterForm.TimerTimer(Sender: TObject);
begin
  If UpdFlag then begin
    Page.SetDisplayPosition(1,TopLine);
    Page.Invalidate;
    UpdFlag:=False;
  End;
End;
{//;
function TDevLST.TestEOR: Boolean;
begin
  Result:=EndOfLine;
end;}

End.
