//******************************************************************************
//  IBM 7094 Emulator - Global definitions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains global definitions for used by most other units:-
//    Bit masks to extract various fields from instructions and commands
//    Shift counts to move these fields into required positions
//------------------------------------------------------------------------------
Unit B709Defs;

Interface

Uses ExtCtrls,     // For TCanvasPanel
     Windows,
     Messages;     // For WM_APP

Const RelVer=3;    // Version identification
      RelLvl=0;
      RelMod='E';

// 70XX Data Types
Type TWord=Int64;       // Low 36 bits contain a 70XX Word
     TInst=Word;        // Low 12 bits contain a 70XX Instruction Operation Code
     TAddr=Integer;     // Low 15 bits contain a 70XX Address
     TComd=Byte;        // Low 4 bits contain a 70XX Instruction Operation Code

// S00000000011111111112222222222333333
// S12345678901234567890123456789012345
// XXXXXXXXXXXXXXXXXX.................. Left Half (Indicator operations)
// ..................XXXXXXXXXXXXXXXXXX Right Half (Indicator operations)
// X................................... Sign Field
// XXXXXXXXXXXX........................ Normal format Operation Code
// XXX................................. Short format Operation Code (also Prefix)
// ............XX...................... Flag Field
// ...............X.................... Multiple Tag Mode flag field
// ..................XXX............... Tags Field
// .....................XXXXXXXXXXXXXXX Address Field (also long format opcode)
// .......................XXXX......... Channel Number in Address Field
// ...........................XXXXXXXXX Device Address in Address Field
// ...XXXXXXXXXXXXXXX.................. Index Register Decrement field
// ..........XX........................ Channel Number in Operation Code
// XXX................................. Channel Command Operation Code
// ..................X................. Channel Command Indirection Flag
// ...................X................ Channel Command No Transmit Flag
// ...XXXXXXXXXXXXXXX.................. Channel Command Word Count
// ..........XXXXXXXX..........XXXXXXXX Channel Trap bits
// ..........XXXXXXXX.................. Convert Count
// XXXXXX.............................. Convert Character
// ............XXXXXX.................. VLM/VDH/VDP Count
// 842184218421842184218421842184218421

// Masks to extract fields from an Instruction Word
Const LfHaMask: TWord=$FFFFC0000; // Left Half Mask
      RtHaMask: TWord=$00003FFFF; // Right Half Mask
      IOpCMask: TWord=$FFF000000; // Instruction Operation Code - Normal format
      FlagMask: TWord=$000C00000; // Flag field
      TagsMask: TWord=$000038000; // Index Register Tags
      AddrMask: TWord=$000007FFF; // Address (Y Field)
      DecrMask: TWord=$1FFFC0000; // Index Register Decrement field
      PrfxMask: TWord=$E00000000; // Prefix field
      MTagMask: TWord=$000100000; // Multiple Tag Mode flag
      TrapMask: TWord=$003FC00FF; // Channel Trap bits
//    CnvCMask: TWord=$003FC0000; // Convert Count
      VLACMask: TWord=$000FC0000; // Variable Length Arithmetic Count

// Some inverses of the above
Const NotSignMask: TWord=$7FFFFFFFF;
      NotAddrMask: TWord=$FFFFF8000;
      NotDecrMask: TWord=$E0003FFFF;
      NotPrfxMask: TWord=$1FFFFFFFF;
      NotTagsMask: TWord=$FFFFC7FFF;

// Masks to extract Channel Command fields from a Word
Const COpCMask: TWord=$E00000000; // Command Operation Code
      COpFMask: TWord=$000020000; // Indirection Flag
      COpNMask: TWord=$000010000; // No Transmit Flag (NTF)
      COpWMask: TWord=$1FFFC0000; // Word Count
      DaCUMask: TWord=$000001FFF; // I/O Device Address
      DaChMask: TWord=$000001E00; // I/O Device Channel Number
      DaUnMask: TWord=$00000000F; // I/O Device Unit Number

// Masks to extract various fields from an Operation Code
Const IOpSMask: TInst=$800;       // Sign bit
      IOpXMask: TWord=$E00;       // Instruction Operation Code - Short format
      IOpNMask: TInst=$003;       // Channel Number

// Bit shift counts to move Instruction fields to/from LSB
Const IOpCShft=24;                // Operation code in a Word
      TagsShft=15;                // Tags Field in a Word
      FlagShft=22;                // Flag Field in a Word
      DecrShft=18;                // Index Register Decrement field
      HalfShft=18;                // Half Word shift

// Bit shift counts to move Channel Command fields to/from LSB
Const COpCShft=33;                // Operation Code
      COpNShft=15;                // No Transmit Flag
      COpWShft=18;                // Word Count

// Bit shift counts to move Device Addresses to/from LSB
Const DaChShft=9;                 // Channel Address

// Accumulator and Carry bits layout
// Accumulator sign bit is maintained in a separate boolean variable
// ..CQP000 00000011111111112222222222333333
// ..CQP123 45678901234567890123456789012345
// ........ ................................ Sign bit
// ..C..... ................................ Carry from Q
// ...Q.... ................................ Carry bit Q
// ....P... ................................ Carry bit P
// .....XXX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX Rest of register
// ...QPXXX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX AC - 37 bits (QP1..35)

// ....SXXX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX MQ & Storage - 36 bits (S1..35)
// .S.**XXX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX Octal Display form - 13 digits
// S..**XXX XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX Hex Display form - 10 digits
// 14214214 21421421421421421421421421421421 13.3 Octal digits
// 84218421 84218421842184218421842184218421 10 Hex digits

Const//SBitMask: TWord=$8000000000; // Accumulator Sign bit
     //CBitMask: TWord=$2000000000; // Accumulator carry out of bit Q
      QBitMask: TWord=$1000000000; // Accumulator carry bit Q
      PBitMask: TWord=$0800000000; // Accumulator carry bit P (MQ/Core Sign bit)
      Bit1Mask: TWord=$0400000000; // Accumulator bit 1
      LowBMask: TWord=$0000000001; // Accumulator bit 35
      Wd35Mask: TWord=$07FFFFFFFF; // 35 bits: 1..35
      Wd36Mask: TWord=$0FFFFFFFFF; // 36 bits: S1..35 (P1..35 if AC)
      Wd37Mask: TWord=$1FFFFFFFFF; // 37 bits of AC: QP1..35
      Wd38Mask: TWord=$9FFFFFFFFF; // All 38 bits of AC: S__QP1..35
      CharMask: TWord=$0FC0000000; // 6 MSBs, a character

Const Heading='IBM 7094 Emulator';

//******************************************************************************
// Character code mappings
//******************************************************************************
//     0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1
//     0 1 2 3 4 5 6 7 0 1 2 3 4 5 6 7
//
//  00   1 2 3 4 5 6 7 8 9 0 # @ : > (
//  20   / S T U V W X Y Z r , % ^ \ +
//  40 - J K L M N O P Q R ! $ * ] ; d
//  60 & A B C D E F G H I ? . ) [ < g

// IBM 9-Code character mapping table - BCD index value gives ASCII character
Const TabBCDtoASC: String=
//     0000000011111 111222222223333333344444444555555556666666677777777
//     0123456701234 567012345670123456701234567012345670123456701234567
      '0123456789b='':>t+ABCDEFGHI?.)[<g-JKLMNOPQR!$*];d /STUVWXYZr,(^\f';

// Inverse of the above - ASCII character value gives BCD code
Var   TabASCToBCD: Array[0..$FF] Of Byte;

// Card (Z/N punch) codes ordered by BCD (9-code) character value
// Each entry is a 12 bit value representing a card column layout:-
// 12 11 0 1 2 3 4 5 6 7 8 9     - Card columns
//  8  4 2 1 8 4 2 1 8 4 2 1     - Bit positions in word
// A BCD index gives the CARD code
Const TabBCDtoCRD: Array[0..$3F] Of Word=(

//      0    1    2    3    4    5    6    7
      $200,$100,$080,$040,$020,$010,$008,$004, // 0
      $002,$001,$082,$042,$022,$012,$00A,$006, // 1
      $800,$900,$880,$840,$820,$810,$808,$804, // 2
      $802,$801,$A00,$842,$822,$812,$80A,$806, // 3
      $400,$500,$480,$440,$420,$410,$408,$404, // 4
      $402,$401,$600,$442,$422,$412,$40A,$406, // 5
      $000,$300,$280,$240,$220,$210,$208,$204, // 6
      $202,$201,$282,$242,$222,$212,$20A,$206  // 7
      );

// Inverse of the above - CARD index gives BCD character value
Var   TabCRDtoBCD: Array[0..$FFF] Of Byte;

// A Delphi TPanel component with the Canvas property exposed
// General purpose, but actually used only by the core storage display
Type TCanvasPanel=Class(TPanel)
       Property Canvas;
     End;

//******************************************************************************
// Windows
//******************************************************************************
// Form/Window identifiers
Type TFormID=(
       FIMain,  FIConfig,FIFileEdit,FIScript,FICore,
       FIReader,FIPunch, FIPrinter, FITapes,
       FIStops, FITrace, FITapeView,FIPlot);

// Form/Window Names
Const FormNames: Array[TFormID] Of ShortString=(
        'Main','Config','Edit','Script','Core',
        'Reader','Punch','Printer','Tapes',
        'Stops','Trace','TapeView','CorePlot');

// Form/Window handles
Var FormHandles: Array[TFormID] Of HWnd;

//******************************************************************************
// Messaging
//******************************************************************************
// Messaging is used mainly by the scripting system. This allows scripts to
// operate as a reasonably disconnected function from the rest of the emulator.
// It controls the emulator by sending messages to the main console window,
// instructing it to simulate the actions that the user/operator may perform.
//
// The WM_EMCMsg message type is used to tell the main console window
// to perform an action as if it were done by the user/operator
// The WM_IOCMsg message type is used to tell an I/O device to perform some
// operation.

// Internally generated Message codes
Const WM_EMCMsg   =WM_APP+100;
      WM_IOCMsg   =WM_APP+101;

// Record structure used by WM_EMCMsg messages
Type PEMCMsg=^TEMCMsg;
     TEMCMsg=Packed Record
       FunCode:   Word;           // Function Code
       Parm1:     Integer;        // Parameter 1 value
       Parm2:     Integer;        // Parameter 2 value
       StrParm:   PString;        // Pointer to any required string parameter
     End;

// Parameter values for WM_EMCMsg Messages - General Emulation
Const
  // General
  CCENDCONFIG    =03;         // Config group done, update displays
  CCGENCHNS      =04;         // Generate channel set
  CCTRACE        =07;         // Set tracing on/off
  CCSAVTRACE     =08;         // Save trace list to file
  CCSCRIPT       =09;         // Script is active
  // Display
  CCSETSENSW     =17;         // Set a sense switch on/off
  // CPU
  CCPRESS        =20;         // Press a CPU console button
  // Core
  CCSETCORSIZ    =30;         // Set the core storage size
  CCSETCORVAL    =31;         // Set the value of a core storage location
  CCPLOTCLEAR    =32;         // Core usage plot clear
  CCPLOTONOFF    =33;         // Core usage plot ON/OFF
  // I/O
  CCTAPEDEF      =40;         // Define a tape drive
  CCTAPEDEL      =41;         // Delete a tape drive
  // Tape View
  CCTAPVADR      =50;         // Tape view window - Select Drive to display
  CCTAPVMOD      =51;         // Tape view window - Set Display Mode
  CCTAPVLIN      =52;         // Tape view window - Set Line number
  CCTAPVSAV      =53;         // Tape view window - Save To File
  // Debugging
  // Scripting
  CCSCRIPTRESUME =61;         // Resume Scripter
  CCDISPSHOW     =63;         // Show Scripter dialog

// Device types
Type  TDevType=(
        DTXXX,          // No device
        DT711,          // Card Reader
        DT721,          // Card Punch
        DT716,          // Line Printer
        DT729,          // Tape Drive
        DT1301          // Disk Drive
      );

Const DevTypeStrings: Array[TDevType] Of String=(
        'Unknown','Reader','Punch','Printer','Tape','Disk');

// Action codes used by WM_IOCMsg messages
Const FCRESET  =1;                // Reset I/O device
      FCCLEAR  =2;                // Clear I/O device
      FCOPEN   =3;                // Open a Windows file for I/O
      FCCLOSE  =4;                // Close Windows file
      FCSAVE   =6;                // Save I/O data to PC file
      FCSEEK   =12;               // Set new file position
      FCLODREC =13;               // Load specified record data
      // Reader specific
      FCCARD   =20;               // Insert card image
      FCLODFIL =21;               // Insert specified file data
      FCLODEOF =22;               // Insert EOF card 
      // Tape specific
      FCWRPROT=31;                // Set Write protect status
      FCREWIND=33;                // Rewind tape
      FCREWUNL=34;                // Rewind and Unload tape
      FCWTMARK=35;                // Write tape mark
      FCRELOAD=36;                // Reload last file used
      FCZAPA  =40;                // Set device data ZAP address
      FCZAPV  =41;                // Set device data ZAP value

// Record structure used by WM_IOCONTROL messages
Type PIOControl=^TIOControl;
     TIOControl=Packed Record
       DevAddr:   Word;           // Target device address
       Action:    Word;           // I/O operation action code
       IntParm:   Integer;        // Value of any required integer parameter
       StrParm:   PString;        // Pointer to any required string parameter
     End;

Procedure EMCMsg(FC: Word; P1,P2: Integer; SS: String);
Procedure IOCMsg(DA,AC: Word; IP: Integer; SS: String);
Function  WordDump(WD: TWord): String;

Implementation

//******************************************************************************
// Messaging - Build message record and send to main window
//******************************************************************************
Procedure EMCMsg(FC: Word; P1,P2: Integer; SS: String);
Var MR: TEMCMsg;
Begin
  With MR do begin
    FunCode:=FC;
    Parm1:=P1;
    Parm2:=P2;
    If SS='' then StrParm:=NIL
             else StrParm:=@SS;
  End;
  SendMessage(FormHandles[FIMain],WM_EMCMsg,0,Longint(@MR));
End;

Procedure IOCMsg(DA,AC: Word; IP: Integer; SS: String);
Var MR: TIOControl;
Begin
  With MR do begin
    DevAddr:=DA;
    Action:= AC;
    IntParm:=IP;
    If SS='' then StrParm:=NIL
             else StrParm:=@SS;
  End;
  SendMessage(FormHandles[FIMain],WM_IOCMsg,0,Longint(@MR));
End;

//******************************************************************************
// Initialize character code mappings
//******************************************************************************
// Load CARD to BCD lookup table from BCD to CARD table
Procedure   GenASC2BCDTab;
Var BI: Byte;
    AC: Char;
Begin
  // Fill table with BCD blank code
  FillChar(TabASCtoBCD,SizeOf(TabASCtoBCD),$0A);
  For BI:=0 to $3F do begin  // For each BCD value
    AC:=TabBCDtoASC[BI+1];   // Get ASCII value
    TabASCtoBCD[Ord(AC)]:=BI;// Store BCD value to inverse array at ASCII posn.
  End;
End;

// Load CARD to BCD lookup table from BCD to CARD table
Procedure GenCRD2BCDTab;
Var BI: Byte;
    ZN: Word;
Begin
  FillChar(TabCRDtoBCD,SizeOf(TabCRDtoBCD),$30);
  For BI:=0 to $3F do begin  // For each BCD value
    ZN:=TabBCDtoCRD[BI];     // Get CARD column coding
    TabCRDtoBCD[ZN]:=BI;     // Store BCD value to inverse array at CARD posn.
  End;
End;

Function WordDump(WD: TWord): String;
Var CC: Byte;
Begin
  Result:='';
  For CC:=1 to 6 do begin
    Result:=TabBCDtoASC[(WD And $3F)+1]+Result;
    WD:=WD Shr 6;
  End;
End;

Initialization
  GenASC2BCDTab;
  GenCRD2BCDTab;
End.
