//******************************************************************************
//  IBM 7094 Emulator - Instruction definitions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit defines the IBM 7094 instructions
//  It provides basic information on all CPU instructions.
//  This information is used to interpret opcodes and display trace entries
//------------------------------------------------------------------------------
Unit B709Inst;

Interface

Uses SysUtils,Dialogs,Classes,
     B709Defs, // General definitions
     B709Misc; // Miscellaneous utility functions

// Instruction format types
Type TIFormat=(
       IFNorm,     // Opcode is in the first 12 bits
       IFLong,     // Opcode is in the first 12 and the last 12 bits
       IFShrt      // Opcode is in the first 3 bits
     );

// Instruction Control Record
Type TInstruction=Class;                    // Forward ref
     TExProc=Procedure(IR: TInstruction);   // Type for Execution Proc

     TInstruction=Class
       HashArg:  LongWord;       // Hash argument formed from opcode
       InstForm: TIFormat;       // Instruction format
       OpCode:   TInst;          // Opcode value (first 12 bits of instruction)
       EoCode:   TAddr;          // Extended (Long) Opcode value (last 12 bits)
       ExProc:   TExProc;        // Procedure to process a group of instructions.
       ExRef:    Byte;           // Identifies specific opcode within a group
       ChanID:   Byte;           // Channel number (1-8)
       IdxModEn: Boolean;        // Indexing is possible
       IndAdrEn: Boolean;        // Indirect Addressing is possible
       Mnemonic: String[4];      // Instruction Mnemonic
       Desc:     String[50];     // Instruction Description
       // Statistics
       UseCount: LongWord;       // Number of times executed
     End;

Const HashPrim=1751;             // Smallest value that produces few synonyms

// Instruction Table - List of pointers to TInstruction record objects
Var InstructionTable:  Array[0..HashPrim-1] Of TInstruction;

Procedure InstInit;
Procedure What(IR: TInstruction);

Implementation

Uses B709CPU,  // For InstCtr
     B709IBIS, // Instruction Execution - Basic Instructions
     B709IINA, // Instruction Execution - Fixed Point Arithmetic Instructions
     B709IFPA, // Instruction Execution - Floating Point Arithmetic Instructions
     B709IIOC; // Instruction Execution - I/O Instructions

// Display message when an unimplemented instruction is encountered
Procedure What(IR: TInstruction);
Begin
  Error('Unhandled instruction at addr '+IntToOct(PrevICtr,5)+': '+IR.Mnemonic+' - '+IR.Desc);
End;

// Find a free slot in the Instruction Table for the given Control Record
Procedure StoreICR(IR: TInstruction);
Var HV: Word;
    SC: Byte;
Begin
  HV:=HashValue(IR.HashArg);                // Gen hash value from opcode arg
  For SC:=1 to 4 do begin                   // Allow up to 4 synonyms
    If InstructionTable[HV]=NIL then begin  // Is primary slot taken?
      InstructionTable[HV]:=IR;             // No, put it there
      Exit;                                 // and done.
    End;
    Inc(HV,13);                             // Slot taken, Bounce to another
    HV:=HV Mod HashPrim;                    // Stay inside table
  End;
  Error('Too many ICR synonyms for '+IR.Mnemonic);
End;

// Add an instruction to the instruction table
Procedure DIN(EP: TExProc;        // Procedure to execute the instruction
             SC: Byte;            // Identifier within an instruction group
             OS: String;          // Instruction Opcode string
             CN: Byte;            // Channel number
             AF: String;          // Addressing Mode Flags
             MN: String;          // Instruction Mnemonic
             ID: String);         // Description (contains display control strings)
Var IR: TInstruction;
Begin
  // Create & Load Instruction table record object. Store it to table.
  IR:=TInstruction.Create;
  With IR do begin
    ExProc:=EP;              // Procedure to process instruction group
    ExRef:=SC;               // Specific opcode within group
    Mnemonic:=MN;            // Instruction Mnemonic
    Desc:=ID;                // Instruction Description
    ChanID:=CN;              // Channel number
    OpCode:=OctStrToInt(Copy(OS,3,4));      // Form standard opcode value
    If OS[2]='-' then OpCode:=OpCode+$800;
    Case OS[1] Of                           // Short, Normal or Long format?
      'S': Begin
             InstForm:=IFShrt;              // Set S/N/L
             HashArg:=OpCode Shr 9;         // Form hash argument from opcode
             StoreICR(IR);                  // Store to table
           End;
      'A','B',
      ' ': Begin
             InstForm:=IFNorm;
             HashArg:=OpCode;
             StoreICR(IR);
           End;
      'C','D',
      'L':   Begin
             InstForm:=IFLong;
             EoCode:=OctStrToInt(Copy(OS,8,5));  // Get extended opcode
             HashArg:=(OpCode Shl 13) Or EoCode; // Include in hash argument
             StoreICR(IR);
           End;
      Else Error('Bad D format code');
    End;
    // Note instruction addressing modes
    IdxModEn:=AF[1]='X';
    IndAdrEn:=AF[2]='I';
  End;
End;

// Add a set of 8 channel related instructions to the instruction table
// Various formats are required, set by CN parameter
// Some formats generate sub instructions (eg. printer hubs)
Procedure CIN(EP: TExProc; SC: Byte; OS: String; CN: Byte; AF: String; MN: String; ID: String);
Var CI,SI,FB,HP,PP: Byte;
    BV: Word;
    DS: String;
    WS: String[3];
    FI: Char;
    CH: Char;
Begin
  FB:=Ord(OS[6]);            // Get Opcode base for format A,B
  HP:=Pos('#',ID);           // Get # posn in description string
  PP:=Pos('%',ID);           // Get % posn in description string
  // Generate 8 instructions
  For CI:=1 to 8 do begin
    CH:=Chr($40+CI);         // Form channel char
    MN[4]:=CH;               // Set channel char in Mnemonic
    ID[HP]:=CH;              // Set channel char in description
    DS:=OS;                  // Copy Opcode string
    FI:=OS[1];               // Format Code is in first char
    Case FI of
      // +VVVC and -VVVC // xxxxN.xxxxx // N=Channel number-1
      'A': Begin
            DS[6]:=Chr($30+(CI-1));         // Modify Opcode string
            DIN(EP,SC,DS,CI,AF,MN,ID);      // Generate one instruction
           End;
      // sVVV+CC/2 // SxxxN.xxxxx // S=Alternating sign, N counts by 2
      'B': Begin
             If Odd(CI) then DS[2]:='+'     // Modify Opcode string
                        else DS[2]:='-';
             DS[6]:=Chr(FB+((CI-1) Div 2));
             DIN(EP,SC,DS,CI,AF,MN,ID);     // Generate one instruction
           End;
      // +0760.CCVVV
      'C': Begin
             WS:=IntToOct(CI,2);            // Get channel number as octal string
             DS[8]:=WS[1]; DS[9]:=WS[2];    // Modify Opcode string
             DIN(EP,SC,DS,CI,AF,MN,ID);     // Generate one instruction
           End;
      // +0760.CCVVV+N
      'D': Begin
             WS:=IntToOct(CI,2);            // Get channel number as octal string
             DS[8]:=WS[1]; DS[9]:=WS[2];
             BV:=OctStrToInt(Copy(OS,10,3));// Get base value of extension
             For SI:=0 to CN do begin       // Generate hub codes
               WS:=IntToOct(BV+SI,3);
               DS[10]:=WS[1]; DS[11]:=WS[2];// Modify Opcode string
               DS[12]:=WS[3];
               ID[PP]:=Chr($31+SI);         // Set hub char in description
               DIN(EP,SC,DS,CI,AF,MN,ID);   // Generate one instruction
             End;
           End;
      Else Error('Bad C format code');
    End;
  End;
End;

//******************************************************************************
// Load instruction information to Instruction Control Table (IML!?)
// Sections and ordering as per "IBM 7094 Principles of Operation" GA22-6703-4
//------------------------------------------------------------------------------
// Display control string field replacement codes:-
// #             Channel ID
// $             Index register tag field
// @@@@@         Address field
// !!!!!         Index instruction decrement field
// %%%%%%%%%%%%  Contents of storage at Address field location
// ""            Duplicate previous instruction
Procedure LoadInstTable;
Begin
  // Fixed Point Operations p26
  DIN(@IXAddSub,01,' +0500.00000',0,'XI','CLA ','Clear and Add (%%%%%%%%%%%%)');
  DIN(@IXAddSub,02,' -0500.00000',0,'XI','CAL ','Clear and Add Logical Word (%%%%%%%%%%%%)');
  DIN(@IXAddSub,03,' +0502.00000',0,'XI','CLS ','Clear and Subtract (%%%%%%%%%%%%)');
  DIN(@IXAddSub,04,' +0400.00000',0,'XI','ADD ','Add (%%%%%%%%%%%%)');
  DIN(@IXAddSub,05,' +0401.00000',0,'XI','ADM ','Add Magnitude (%%%%%%%%%%%%)');
  DIN(@IXAddSub,06,' +0402.00000',0,'XI','SUB ','Subtract (%%%%%%%%%%%%)');
  DIN(@IXAddSub,07,' -0400.00000',0,'XI','SBM ','Subtract Magnitude (%%%%%%%%%%%%)');
  DIN(@IXAddSub,08,' +0361.00000',0,'XI','ACL ','Add and Carry Logical Word (%%%%%%%%%%%%)');
  DIN(@IXMulDiv,01,' +0200.00000',0,'XI','MPY ','Multiply by @@@@@');
  DIN(@IXMulDiv,02,' -0200.00000',0,'XI','MPR ','Multiply and Round by @@@@@');
  DIN(@IXMulDiv,03,'L+0760.00010',0,'X.','RND ','Round');
  DIN(@IXMulDiv,04,' +0204.00000',0,'X.','VLM ','Variable Length Multiply by @@@@@');
  DIN(@IXMulDiv,05,' +0220.00000',0,'XI','DVH ','Divide or Halt by @@@@@');
  DIN(@IXMulDiv,06,' +0221.00000',0,'XI','DVP ','Divide or Proceed by @@@@@');
  DIN(@IXMulDiv,07,' +0224.00000',0,'X.','VDH ','Variable Length Divide or Halt');
  DIN(@IXMulDiv,08,' +0225.00000',0,'X.','VDP ','Variable Length Divide or Proceed');
  // Floating Point Operations p31
  DIN(@IXFloatS,01,' +0300.00000',0,'XI','FAD ','Floating Add');
  DIN(@IXFloatS,02,' +0304.00000',0,'XI','FAM ','Floating Add Magnitude');
  DIN(@IXFloatS,03,' -0300.00000',0,'XI','UFA ','Unnormalized Floating Add');
  DIN(@IXFloatS,04,' +0302.00000',0,'XI','FSB ','Floating Subtract');
  DIN(@IXFloatS,05,' -0304.00000',0,'XI','UAM ','Unnormalized Add Magnitude');
  DIN(@IXFloatS,06,' +0306.00000',0,'XI','FSM ','Floating Subtract Magnitude');
  DIN(@IXFloatS,07,' -0302.00000',0,'XI','UFS ','Unnormalized Floating Subtract');
  DIN(@IXFloatS,08,' -0306.00000',0,'XI','USM ','Unnormalized Subtract Magnitude');
  DIN(@IXFloatS,09,'L+0760.00011',0,'X.','FRN ','Floating Round');
  DIN(@IXFloatS,10,' +0260.00000',0,'XI','FMP ','Floating Multiply');
  DIN(@IXFloatS,11,' -0260.00000',0,'XI','UFM ','Unnormalized Floating Multiply');
  DIN(@IXFloatS,12,' +0240.00000',0,'XI','FDH ','Floating Divide or Halt');
  DIN(@IXFloatS,13,' +0241.00000',0,'XI','FDP ','Floating Divide or Proceed');
  // Double-Precison Floating Point Operations p37
  DIN(@IXFloatD,01,' +0443.00000',0,'XI','DLD ','Double Load ');
  DIN(@IXFloatD,02,' -0603.00000',0,'XI','DST ','Double Store');
  DIN(@IXFloatD,03,' +0301.00000',0,'XI','DFAD','Double Floating Add');
  DIN(@IXFloatD,04,' -0301.00000',0,'XI','DUFA','Double Unnormalized Floating Add');
  DIN(@IXFloatD,05,' +0303.00000',0,'XI','DFSB','Double Floating Subtract');
  DIN(@IXFloatD,06,' -0303.00000',0,'XI','DUFS','Double Unnormalized Floating Subtract');
  DIN(@IXFloatD,07,' +0305.00000',0,'XI','DFAM','Double Unnormalized Add Magnitude');
  DIN(@IXFloatD,08,' -0305.00000',0,'XI','DUAM','Double Unnormalized Add Magnitude');
  DIN(@IXFloatD,09,' +0307.00000',0,'XI','DFSM','Double Floating Subtract Magnitude');
  DIN(@IXFloatD,10,' -0307.00000',0,'XI','DUSM','Double Unnormalized Subtract Magnitude');
  DIN(@IXFloatD,11,' +0261.00000',0,'XI','DFMP','Double Floating Multiply');
  DIN(@IXFloatD,12,' -0261.00000',0,'XI','DUFM','Double Unnormalized Floating Multiply');
  DIN(@IXFloatD,13,' -0240.00000',0,'XI','DFDH','Double Floating Divide or Halt');
  DIN(@IXFloatD,14,' -0241.00000',0,'XI','DFDP','Double Floating Divide or Proceed');
  // Shifting Operations p40
  DIN(@IXShiftR,01,' +0767.00000',0,'X.','ALS ','Accumulator Left Shift (@@@@@)');
  DIN(@IXShiftR,02,' +0771.00000',0,'X.','ARS ','Accumulator Right Shift (@@@@@)');
  DIN(@IXShiftR,03,' +0763.00000',0,'X.','LLS ','Long Left Shift (@@@@@)');
  DIN(@IXShiftR,04,' +0765.00000',0,'X.','LRS ','Long Right Shift (@@@@@)');
  DIN(@IXShiftR,05,' -0763.00000',0,'X.','LGL ','Logical Left Shift (@@@@@)');
  DIN(@IXShiftR,06,' -0765.00000',0,'X.','LGR ','Logical Right Shift (@@@@@)');
  DIN(@IXShiftR,07,' -0773.00000',0,'X.','RQL ','Rotate MQ Left');
  // Word transmission Operations p43
  DIN(@IXLodSto,01,' +0560.00000',0,'XI','LDQ ','Load MQ from @@@@@ (%%%%%%%%%%%%)');
  DIN(@IXLodSto,02,' -0600.00000',0,'XI','STQ ','Store MQ to @@@@@');
  DIN(@IXLodSto,03,' -0620.00000',0,'XI','SLQ ','Store Left half MQ to @@@@@');
  DIN(@IXLodSto,04,' +0601.00000',0,'XI','STO ','Store to @@@@@');
  DIN(@IXLodSto,05,' +0602.00000',0,'XI','SLW ','Store Logical Word to @@@@@');
  DIN(@IXLodSto,06,' +0630.00000',0,'XI','STP ','Store Prefix to @@@@@');
  DIN(@IXLodSto,07,' +0622.00000',0,'XI','STD ','Store Decrement to @@@@@');
  DIN(@IXLodSto,08,' +0625.00000',0,'XI','STT ','Store Tag to @@@@@');
  DIN(@IXLodSto,09,' +0621.00000',0,'XI','STA ','Store Address to @@@@@');
  DIN(@IXLodSto,10,' -0625.00000',0,'XI','STL ','Store Instruction Location Counter');
  DIN(@IXLodSto,11,'S-1000.00000',0,'..','STR ','Store Location and Trap');
  DIN(@IXLodSto,12,' +0600.00000',0,'XI','STZ ','Store Zero to @@@@@');
  DIN(@IXLodSto,13,' +0131.00000',0,'..','XCA ','Exchange AC and MQ');
  DIN(@IXLodSto,14,' -0130.00000',0,'..','XCL ','Exchange Logical AC and MQ');
  DIN(@IXLodSto,15,'L+0760.00004',0,'XI','ENK ','Enter Keys');
  // Control Instructions p45
  DIN(@IXContrl,01,' +0761.00000',0,'..','NOP ','No Operation');
  DIN(@IXContrl,02,' +0420.00000',0,'..','HPR ','Halt and Proceed');
  DIN(@IXTtmXfr,01,' +0000.00000',0,'XI','HTR ','Halt and Transfer to @@@@@');
  DIN(@IXContrl,03,' +0522.00000',0,'XI','XEC ','Execute (@@@@@)');
  DIN(@IXTtmXfr,02,' +0020.00000',0,'XI','TRA ','Transfer to @@@@@');
  DIN(@IXContrl,04,'L+0760.00007',0,'X.','ETM ','Enter Trapping Mode');
  DIN(@IXContrl,05,'L-0760.00007',0,'X.','LTM ','Leave Trapping Mode');
  DIN(@IXContrl,06,' +0021.00000',0,'XI','TTR ','Trap Transfer to @@@@@');
  DIN(@IXTtmXfr,03,' +0100.00000',0,'XI','TZE ','Transfer on Zero to @@@@@');
  DIN(@IXTtmXfr,04,' -0100.00000',0,'XI','TNZ ','Transfer on No Zero to @@@@@');
  DIN(@IXTtmXfr,05,' +0120.00000',0,'XI','TPL ','Transfer on Plus to @@@@@');
  DIN(@IXTtmXfr,06,' -0120.00000',0,'XI','TMI ','Transfer on Minus to @@@@@');
  DIN(@IXTtmXfr,07,' +0140.00000',0,'XI','TOV ','Transfer on Overflow to @@@@@');
  DIN(@IXTtmXfr,08,' -0140.00000',0,'XI','TNO ','Transfer on No Overflow to @@@@@');
  DIN(@IXTtmXfr,09,' +0162.00000',0,'XI','TQP ','Transfer on MQ Plus to @@@@@');
  DIN(@IXTtmXfr,10,' +0161.00000',0,'XI','TQO ','Transfer on MQ Overflow to @@@@@');
  DIN(@IXTtmXfr,11,' +0040.00000',0,'XI','TLQ ','Transfer on Low MQ to @@@@@');
  DIN(@IXTtmXfr,12,' +0074.00000',0,'..','TSX ','Transfer and set Index $ to (@@@@@)');
  DIN(@IXTtmXfr,13,'S+1000.00000',0,'..','TXI ','Transfer with Index $ Incremented (!!!!!)');
  DIN(@IXTtmXfr,14,'S+3000.00000',0,'..','TXH ','Transfer on Index $ High (!!!!!)');
  DIN(@IXTtmXfr,15,'S-3000.00000',0,'..','TXL ','Transfer on Index $ Low or Equal (!!!!!)');
  DIN(@IXTtmXfr,16,'S+2000.00000',0,'..','TIX ','Transfer on Index $ (!!!!!)');
  DIN(@IXTtmXfr,17,'S-2000.00000',0,'..','TNX ','Transfer on No Index $ (!!!!!)');
  // 'PSE - Plus Sense' sub functions p50
  DIN(@IXSensLS,01,'L+0760.00140',0,'X.','SLF ','Sense Lights Off');
  DIN(@IXSensLS,11,'L+0760.00141',0,'X.','SLN1','Sense Light On 1');
  DIN(@IXSensLS,12,'L+0760.00142',0,'X.','SLN2','Sense Light On 2');
  DIN(@IXSensLS,13,'L+0760.00143',0,'X.','SLN3','Sense Light On 3');
  DIN(@IXSensLS,14,'L+0760.00144',0,'X.','SLN4','Sense Light On 4');
  DIN(@IXSensLS,21,'L+0760.00161',0,'X.','SWT1','Sense Switch Test 1');
  DIN(@IXSensLS,22,'L+0760.00162',0,'X.','SWT2','Sense Switch Test 2');
  DIN(@IXSensLS,23,'L+0760.00163',0,'X.','SWT3','Sense Switch Test 3');
  DIN(@IXSensLS,24,'L+0760.00164',0,'X.','SWT4','Sense Switch Test 4');
  DIN(@IXSensLS,25,'L+0760.00165',0,'X.','SWT5','Sense Switch Test 5');
  DIN(@IXSensLS,26,'L+0760.00166',0,'X.','SWT6','Sense Switch Test 6');
  DIN(@IXSensLS,27,'L+0760.00167',0,'X.','SWT7','Sense Switch Test 7');
  CIN(@IXSensLS,41,'C+0760.CC360',0,'X.','SPTc','Sense Printer Test Ch#');
  CIN(@IXSensLS,51,'D+0760.CC361',9,'X.','SPRc','Sense Printer Ch# Hub %');
  CIN(@IXSensLS,61,'D+0760.CC341',1,'X.','SPUc','Sense Punch Ch# Hub %');
  // 'MSE - Minus Sense' sub functions p51
  DIN(@IXSensLS,31,'L-0760.00141',0,'X.','SLT1','Sense Light Test 1');
  DIN(@IXSensLS,32,'L-0760.00142',0,'X.','SLT2','Sense Light Test 2');
  DIN(@IXSensLS,33,'L-0760.00143',0,'X.','SLT3','Sense Light Test 3');
  DIN(@IXSensLS,34,'L-0760.00144',0,'X.','SLT4','Sense Light Test 4');
  DIN(@IXSensLS,35,'L-0760.00145',0,'X.','SLT5','Sense Light Test 5'); // 9M71B
  CIN(@IXTestIO,01,'C+0760.CC000',0,'X.','BTTc','Beginning of Tape Test Ch#');
  CIN(@IXTestIO,02,'C-0760.CC000',0,'X.','ETTc','End of Tape Test Ch#');
  DIN(@IXTestIO,03,'L+0760.00005',0,'XI','IOT ','Input-Output Check Test');
  DIN(@IXTestsO,01,'L-0760.00001',0,'X.','PBT ','P Bit Test');
  DIN(@IXTestsO,02,'L+0760.00001',0,'X.','LBT ','Low order Bit Test');
  DIN(@IXTestsO,03,'L+0760.00012',0,'X.','DCT ','Divide Check Test');
  DIN(@IXTestsO,04,' +0520.00000',0,'XI','ZET ','Storage Zero Test (@@@@@)');
  DIN(@IXTestsO,05,' -0520.00000',0,'XI','NZT ','Storage Not Zero Test (@@@@@)');
  DIN(@IXCompar,01,' +0340.00000',0,'XI','CAS ','Compare Accumulator with Storage (@@@@@)');
  DIN(@IXCompar,02,' -0340.00000',0,'XI','LAS ','Logical Compare AC with Storage (@@@@@)');
  CIN(@IXChnXfr,01,'A+0060.00000',0,'XI','TCOc','Transfer on Channel # Operating to @@@@@');
  CIN(@IXChnXfr,02,'A-0060.00000',0,'XI','TCNc','Transfer on Channel # Not Operating to @@@@@');
  CIN(@IXChnXfr,03,'B+0022.00000',0,'XI','TRCc','Transfer on Channel # Check to @@@@@');
  CIN(@IXChnXfr,04,'B+0030.00000',0,'XI','TEFc','Transfer on DSC # EOF to @@@@@');
  // Index Register transmission Operations p54
  DIN(@IXIdxReg,01,'L-0760.00016',0,'X.','EMTM','Enter Multiple Tag Mode');
  DIN(@IXIdxReg,02,'L+0760.00016',0,'X.','LMTM','Leave Multiple Tag mode');
  DIN(@IXIdxReg,03,' +0756.00000',0,'..','PCA ','Place Complement of Index in Address');
  DIN(@IXIdxReg,04,' -0756.00000',0,'..','PCD ','Place Complement of Index in Decrement');
  DIN(@IXIdxReg,05,' +0636.00000',0,'..','SCA ','Store Complement of Index in Address');
  DIN(@IXIdxReg,06,' -0636.00000',0,'..','SCD ','Store Complement of Index in Decrement');
  DIN(@IXIdxReg,07,' +0534.00000',0,'..','LXA ','Load Index $ from Address (@@@@@)');
  DIN(@IXIdxReg,08,' +0535.00000',0,'..','LAC ','Load Complement of Address in Index');
  DIN(@IXIdxReg,09,' -0534.00000',0,'..','LXD ','Load Index $ from Decrement (@@@@@)');
  DIN(@IXIdxReg,10,' -0535.00000',0,'..','LDC ','Load Complement of Decrement in Index');
  DIN(@IXIdxReg,11,' +0774.00000',0,'..','AXT ','Address to Index $ True (@@@@@)');
  DIN(@IXIdxReg,12,' -0774.00000',0,'..','AXC ','Address to Index $ Complemented (@@@@@)');
  DIN(@IXIdxReg,13,' +0734.00000',0,'..','PAX ','Place Address in Index $');
  DIN(@IXIdxReg,14,' +0737.00000',0,'..','PAC ','Place Complement of Address in Index');
  DIN(@IXIdxReg,15,' +0754.00000',0,'..','PXA ','Place Index $ in Address (@@@@@)');
  DIN(@IXIdxReg,16,' -0754.00000',0,'..','PXD ','Place Index $ in Decrement (@@@@@)');
  DIN(@IXIdxReg,17,' -0734.00000',0,'..','PDX ','Place Decrement in Index $');
  DIN(@IXIdxReg,18,' -0737.00000',0,'..','PDC ','Place Complement of Decrement in Index');
  DIN(@IXIdxReg,19,' +0634.00000',0,'..','SXA ','Store Index $ in Address (@@@@@)');
  DIN(@IXIdxReg,20,' -0634.00000',0,'..','SXD ','Store Index $ in Decrement (@@@@@)');
  // Logical Operations p59
  DIN(@IXLogico,01,' -0501.00000',0,'XI','ORA ','OR to Accumulator (@@@@@)');
  DIN(@IXLogico,02,' -0602.00000',0,'XI','ORS ','OR to Storage at (@@@@@)');
  DIN(@IXLogico,03,' -0320.00000',0,'XI','ANA ','AND to Accumulator (@@@@@)');
  DIN(@IXLogico,04,' +0320.00000',0,'XI','ANS ','AND to Storage at (@@@@@)');
  DIN(@IXLogico,05,' +0322.00000',0,'XI','ERA ','Exclusive OR to Accumulator (@@@@@)');
  DIN(@IXLogico,06,'L+0760.00006',0,'X.','COM ','Complement Magnitude');
  DIN(@IXLogico,07,'L+0760.00000',0,'X.','CLM ','Clear Magnitude');
  DIN(@IXLogico,08,'L+0760.00002',0,'X.','CHS ','Change Sign');
  DIN(@IXLogico,09,'L+0760.00003',0,'X.','SSP ','Set Sign Plus');
  DIN(@IXLogico,10,'L-0760.00003',0,'X.','SSM ','Set Sign Minus');
  // Sense Indicator Operations p61
  DIN(@IXSnsInd,01,' +0044.00000',0,'..','PAI ','Place Accumulator in Indicators');
  DIN(@IXSnsInd,02,' -0046.00000',0,'..','PIA ','Place Indicators in Accumulator');
  DIN(@IXSnsInd,03,' +0441.00000',0,'XI','LDI ','Load Indicators');
  DIN(@IXSnsInd,04,' +0604.00000',0,'XI','STI ','Store Indicators to @@@@@');
  DIN(@IXSnsInd,05,' +0043.00000',0,'..','OAI ','Or Accumulator to Indicators');
  DIN(@IXSnsInd,06,' +0442.00000',0,'XI','OSI ','Or Storage to Indicators');
  DIN(@IXSnsInd,07,' -0055.00000',0,'..','SIL ','Set Indicators of Left Half');
  DIN(@IXSnsInd,08,' +0055.00000',0,'..','SIR ','Set Indicators of Right Half');
  DIN(@IXSnsInd,09,' -0042.00000',0,'..','RIA ','Reset Indicators from Accumualtor');
  DIN(@IXSnsInd,10,' +0445.00000',0,'XI','RIS ','Reset Indicators from Storage');
  DIN(@IXSnsInd,11,' -0057.00000',0,'..','RIL ','Reset Indicators of Left Half');
  DIN(@IXSnsInd,12,' +0057.00000',0,'..','RIR ','Reset Indicators of Right Half');
  DIN(@IXSnsInd,13,' +0041.00000',0,'..','IIA ','Invert Indicators from AC');
  DIN(@IXSnsInd,14,' +0440.00000',0,'XI','IIS ','Invert Indicators from Storage');
  DIN(@IXSnsInd,15,' -0051.00000',0,'..','IIL ','Invert Indicators of Left Half');
  DIN(@IXSnsInd,16,' +0051.00000',0,'..','IIR ','Invert Indicators of Right Half');
  DIN(@IXTtmXfr,18,' +0042.00000',0,'XI','TIO ','Transfer if Indicators On');
  DIN(@IXTtmXfr,19,' +0046.00000',0,'XI','TIF ','Transfer if Indicators Off');
  DIN(@IXTestsO,06,' +0446.00000',0,'XI','ONT ','On Test for Indicators');
  DIN(@IXTestsO,07,' +0444.00000',0,'XI','OFT ','Off Test for Indicators');
  DIN(@IXSnsInd,17,' -0056.00000',0,'..','LNT ','Left Half Indicators On Test');
  DIN(@IXSnsInd,18,' +0056.00000',0,'..','RNT ','Right Half Indicators on Test');
  DIN(@IXSnsInd,19,' -0054.00000',0,'..','LFT ','Left Half Indicators Off Test');
  DIN(@IXSnsInd,20,' +0054.00000',0,'..','RFT ','Left Half Indicators on Test');
  // Convert Instructions p66
  DIN(@IXConvrt,01,' +0114.00000',0,'..','CVR ','Convert by Replacement from the AC');
  DIN(@IXConvrt,02,' -0154.00000',0,'..','CRQ ','Convert by Replacement from the MQ');
  DIN(@IXConvrt,03,' -0114.00000',0,'..','CAQ ','Convert by Addition from the MQ');
  // Input-Output Operations p68
  DIN(@IXInpOut,01,' +0762.00000',0,'X.','RDS ','Read Select device=@@@@@');
  DIN(@IXInpOut,02,' +0766.00000',0,'X.','WRS ','Write Select device=@@@@@');
  DIN(@IXInpOut,03,' +0764.00000',0,'X.','BSR ','Backspace Record');
  DIN(@IXInpOut,04,' -0764.00000',0,'X.','BSF ','Backspace File');
  DIN(@IXInpOut,05,' +0770.00000',0,'X.','WEF ','Write End of File');
  DIN(@IXInpOut,06,' +0772.00000',0,'X.','REW ','Rewind');
  DIN(@IXInpOut,07,' -0772.00000',0,'X.','RUN ','Rewind and Unload');
  DIN(@IXInpOut,08,' +0776.00000',0,'..','SDN ','Set Density');
  CIN(@IXIOTran,01,'C+0760.CC352',0,'X.','RDCc','Reset Data Channel #');
  // Input-Output Transmission Operations p71
  CIN(@IXIOTran,02,'B+0640.00000',0,'XI','SCHA','Store Channel # to @@@@@');
  CIN(@IXIOTran,03,'B+0540.00000',0,'XI','RCHA','Reset and Load Channel #');
  CIN(@IXIOTran,04,'B+0544.00000',0,'XI','LCHA','Load Channel #');
  // Channel Trap Operations p76
  DIN(@IXChTrap,01,' +0564.00000',0,'XI','ENB ','Enable from Y');
  DIN(@IXChTrap,02,'L+0760.00014',0,'X.','RCT ','Restore Channel Traps');
  // System Compatibility Operations p76
  DIN(@IXTtmXfr,30,' -0021.00000',0,'XI','ESNT','Enter Storage Nullification and Transfer');
  DIN(@IXTtmXfr,31,'L-0760.00010',0,'X.','LSNM','Leave Storage Nullification Mode');
  DIN(@IXSysCom,01,'L-0760.00005',0,'X.','ESTM','Enter Select Trap Mode');
  DIN(@IXSysCom,02,'L-0760.00006',0,'X.','ECTM','Enter Copy Trap Mode');
  DIN(@IXSysCom,03,'L-0760.00002',0,'X.','EFTM','Enter Floating Trap Mode');
  DIN(@IXSysCom,04,'L-0760.00004',0,'X.','LFTM','Leave Floating Trap Mode');
  // 7909 Data Channel instructions p78
  // RSCA STCA SCHA ENB RICA SCDA
  // Spurious stuff found in 9M71B diagnostic
  DIN(@IXMulDiv,04,' +0205.00000',0,'X.','VLM!','??? Variable Length Multiply by @@@@@');
  DIN(@IXMulDiv,08,' +0227.00000',0,'X.','VDP!','??? Variable Length Divide or Proceed');
  DIN(@IXContrl,01,'L+0760.00041',0,'..','NOP!','NOP according to 9M71B Diagnostic');
  DIN(@IXContrl,01,'L+0760.00101',0,'..','NOP!','""');
  DIN(@IXContrl,01,'L+0760.00341',0,'..','NOP!','""');
  DIN(@IXContrl,01,'L+0760.00145',0,'X.','SLN5','""');
  DIN(@IXContrl,01,'L+0760.00146',0,'X.','SLN6','""');
  DIN(@IXContrl,01,'L+0760.00147',0,'X.','SLN7','""');
  // From 9M71B. These are probably crap, Or caused by address modification
  DIN(@IXConvrt,03,' -0115.00000',0,'..','CAQ!','??? Convert by Addition from the MQ');
  DIN(@IXConvrt,03,' -0116.00000',0,'..','CAQ!','??? Convert by Addition from the MQ');
  DIN(@IXConvrt,03,' -0117.00000',0,'..','CAQ!','??? Convert by Addition from the MQ');
  DIN(@IXConvrt,01,' +0115.00000',0,'..','CVR ','??? Convert by Replacement from the AC');
  DIN(@IXConvrt,01,' +0116.00000',0,'..','CVR ','??? Convert by Replacement from the AC');
  DIN(@IXConvrt,01,' +0117.00000',0,'..','CVR ','??? Convert by Replacement from the AC');
End;

Procedure InstInit;
Begin
  // Clear instruction lookup table
  FillChar(InstructionTable,SizeOf(InstructionTable),0);
  // Load instructions to table
  LoadInstTable;
End;

Procedure DropInstTable;
Var SL: TStringList;
    II,SP: Integer;
    IR: TInstruction;
    SS: String;
    PC: Double;
Begin
  // Dump instruction usage stats
  SL:=TStringList.Create;
  With SL do begin
    Sorted:=True;
    For II:=0 to HashPrim-1 do begin
      IR:=InstructionTable[II];
      If IR<>NIL then
        With IR do begin
          // Remove trace tags from desc
          SS:=Desc;
          SP:=Pos('(',SS); If SP>0 then SS:=Copy(SS,1,SP-1);
          SP:=Pos('@',SS); If SP>0 then SS:=Copy(SS,1,SP-1);
          // Output: Percent Count Mnemonic Desc
          If InstCount>0 then PC:=100*(UseCount/InstCount)
                         else PC:=0;
          SS:=Format('%6.3f %10u ',[PC,UseCount])+Pad(Mnemonic,7)+SS;
          Add(SS);
        End;
    End;
    Sorted:=False;
    Insert(0,'%Total      Count Mnemon Desc');
    SaveToFile('B7094 Instruction Counts.Log');
    Free;
  End;
  // Drop table entry storage
  For II:=0 to HashPrim-1 do begin
    IR:=InstructionTable[II];
    If IR<>NIL then IR.Free;
  End;
End;

Initialization
Finalization
  DropInstTable;       // Release instruction objects
End.

