//******************************************************************************
//  IBM 7094 Emulator - Main Unit
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains the functions of the main IBM 7094 operator console
//  (It is actually based more on the appearance of an IBM 7044.)
//------------------------------------------------------------------------------
Unit B709Main;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics, ComCtrls,
     Controls, Forms, Dialogs, StdCtrls, ExtCtrls, Buttons,
     B709Cmps, // Display Components
     B709Defs, // General definitions
     B709Eror, // Error message display form
     B709Misc, // Miscellaneous utility functions
     B709Inst, // CPU Instruction definitions
     B709Core, // Core Storage functions
     B709Plot, // Core Usage Plot display
     B709Stop, // Process Stops Panel
     B709CPU,  // Central Processing Unit functions
     B709Chan, // I/O Channel and Device functions
     B709DRDR, // Reader display form
     B709DLST, // Printer display form
     B709DTAP, // Tape Drives display form
     B709TapF, // Tape Drive display subform
     B709TapV, // Tape data viewer
     B709OCLI, // Operator control language script Interpreter
     B709OCLD, // Script dialog display
     B709OCLE, // Simple text file editor
     B709CnFg, // Configuration form
     B709Spls, // Splash Form
     B709Trce; // Log/Trace functions

Var Heading: String='IBM 7094 Mainframe Emulator';

Type
  TMainForm = class(TForm)
    Panel1: TPanel;
    PARegs: TPanel;
    Label1: TLabel;
    Label3: TLabel;
    CBAuto: TCheckBox;
    StatusBar: TStatusBar;
    DRAC: TDispReg;
    DRIC: TDispReg;
    DRMQ: TDispReg;
    DRSR: TDispReg;
    Label2: TLabel;
    Label4: TLabel;
    PAChans: TPanel;
    Label8: TLabel;
    DRCLR: TDispReg;
    DRCAR: TDispReg;
    Label5: TLabel;
    DRCWR: TDispReg;
    Label6: TLabel;
    DRCCR: TDispReg;
    Label7: TLabel;
    RBChA: TRadioButton;
    RBChB: TRadioButton;
    RBChC: TRadioButton;
    RBChD: TRadioButton;
    Label9: TLabel;
    CBSlow: TCheckBox;
    BIRun: TButnInd;
    BIChnAct: TButnInd;
    BIChnChk: TButnInd;
    BICPUChk: TButnInd;
    BIHexMode: TButnInd;
    BIStart: TButnInd;
    BILoadCard: TButnInd;
    BIPwrOff: TButnInd;
    BIReset: TButnInd;
    BIClear: TButnInd;
    DRCDR: TDispReg;
    Label11: TLabel;
    PASense: TPanel;
    Label16: TLabel;
    CBSS1: TButnInd;
    CBSS2: TButnInd;
    CBSS3: TButnInd;
    CBSS4: TButnInd;
    CBSS5: TButnInd;
    CBSS6: TButnInd;
    CBSSAsk: TCheckBox;
    RBChE: TRadioButton;
    RBChF: TRadioButton;
    RBChG: TRadioButton;
    RBChH: TRadioButton;
    PAXRegs: TPanel;
    Label13: TLabel;
    Label14: TLabel;
    Label15: TLabel;
    Label17: TLabel;
    Label18: TLabel;
    Label19: TLabel;
    Label20: TLabel;
    DRIR1: TDispReg;
    DRIR2: TDispReg;
    DRIR3: TDispReg;
    DRIR4: TDispReg;
    DRIR5: TDispReg;
    DRIR6: TDispReg;
    DRIR7: TDispReg;
    BISL1: TButnInd;
    BISL2: TButnInd;
    BISL3: TButnInd;
    BISL4: TButnInd;
    Label12: TLabel;
    Label21: TLabel;
    DRSI: TDispReg;
    Label22: TLabel;
    BIOVFInd: TButnInd;
    BIDVCInd: TButnInd;
    BIIOCInd: TButnInd;
    BITTMode: TButnInd;
    BILoadTape: TButnInd;
    Label23: TLabel;
    Label24: TLabel;
    Label25: TLabel;
    Label26: TLabel;
    Label27: TLabel;
    Label28: TLabel;
    Label29: TLabel;
    Label30: TLabel;
    Label31: TLabel;
    Label32: TLabel;
    BIChnTrp: TButnInd;
    BIChnLCH: TButnInd;
    BIChnTCK: TButnInd;
    BIChnEOF: TButnInd;
    Panel2: TPanel;
    Label34: TLabel;
    STTrueFact: TStaticText;
    UDTrueFact: TUpDown;
    CBTrace: TCheckBox;
    CBCore: TCheckBox;
    CBStops: TCheckBox;
    Label10: TLabel;
    LAInstCount: TLabel;
    BIConfig: TButnInd;
    BIAbout: TButnInd;
    BIRunScript: TButnInd;
    BIEditScript: TButnInd;
    BIChnStop: TButnInd;
    CBPlot: TCheckBox;
    procedure BIPwrOffClick(Sender: TObject);
    procedure BILoadClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure BIStartClick(Sender: TObject);
    procedure BIHexModeClick(Sender: TObject);
    procedure CBCtrlClick(Sender: TObject);
    procedure BIClearClick(Sender: TObject);
    procedure BIResetClick(Sender: TObject);
    procedure RBChanClick(Sender: TObject);
    procedure CBSSClick(Sender: TObject);
    procedure CBSSAskClick(Sender: TObject);
    procedure BIAboutClick(Sender: TObject);
    procedure BIChnStopClick(Sender: TObject);
    procedure UDTrueFactClick(Sender: TObject; Button: TUDBtnType);
    procedure BIConfigClick(Sender: TObject);
    procedure BIRunScriptClick(Sender: TObject);
    procedure BIEditScriptClick(Sender: TObject);
  private
    Initialized: Boolean;
    HexMode:     Boolean;
    DispChan:    TChannel;
    SenseFlags:  Array[1..6] Of Boolean;     // Sense Switches Tested flags
    TickCtr:     Byte;
    procedure SetupSystem;
    procedure SetSenseSwitch(SN: Byte; SV: Boolean);
    procedure SetupChnDev(CC: Byte);
  protected
    procedure WMEMCMsg(Var Message: TMessage); Message WM_EMCMsg;
    procedure WMIOCMsg(Var Message: TMessage); Message WM_IOCMsg;
  public
    procedure SetControlFlags;
    procedure UpdateDisplays;
    Procedure ShowMsg(MS: ShortString);
    procedure SenseFlash(SN: Byte);
    procedure SetTraceOn;
  end;

Var MainForm: TMainForm;

Implementation

{$R *.DFM}

Const DefaultTF=1000;        // Default value for True Factor

//******************************************************************************
// Main Form Constructor/Destructor
//******************************************************************************
procedure TMainForm.FormCreate(Sender: TObject);
begin
  FormHandles[FIMain]:=Handle;         // Make window handle available
  Heading:=Heading+' V'+IntToStr(RelVer)+'.'+IntToStr(RelLvl)+RelMod;
  Caption:=Heading;
  Left:=0; Top:=0;
  HexMode:=False;
  LAInstCount.Caption:='';
end;

Procedure TMainForm.FormDestroy(Sender: TObject);
Begin
  DropChannels;
End;

//******************************************************************************
// Initialization & Termination
//******************************************************************************
// Performed once when Main Form shows
procedure TMainForm.FormActivate(Sender: TObject);
Var SS: String;
begin
  If Initialized then Exit;       // Only do this once
  Initialized:=True;
  ConfigForm.Initialize;          // Configuration load
  SetupSystem;                    // Setup according to loaded configuration
  While SplashForm.Visible do     // Let splash form die away
    Application.ProcessMessages;
  // AutoRun startup script from name in config file
  SS:=ConfigForm.GetStartScript;
  If SS<>'' then
    EditForm.RunScript(SS);
End;

// Setup system according to current configuration
// Called by inital activation and by configuration change
Procedure TMainForm.SetupSystem;
Begin
  TraceForm.Initialize;           // Trace display
  CoreForm.Initialize;            // Core Storage display
  PlotForm.Initialize;            // Core Usage Plot display
  // Initialize System
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,'Power ON');      // Announce startup
  InstInit;                       // Load Instruction Table
  MTagMode:=True;                 // Start in 7090 Mode
  TrapMode:=False;
  FTrpMode:=True;
  STrpMode:=False;
  CTrpMode:=False;
  SNulMode:=False;
  SetSI(0);                       // Clear sense indicators
  CoreInit(MaxCore);                // Set core size
  SetupChnDev(2);                 // Default to 2 channels
  BIResetClick(Self);             // Reset CPU

  TrueFact:=DefaultTF;            // Set True Factor update value
  UDTrueFact.Position:=DefaultTF;
  SetControlFlags;                // Pick up control settings
  // Now device displays can be initialized
  ReaderForm.Initialize;
//PunchForm.Initialize;
  TapesForm.Initialize;
  PrinterForm.Initialize;
  TapeViewForm.Initialize;
End;

// Create number of channels specified
Procedure TMainForm.SetupChnDev(CC: Byte);
Var CI: Byte;
    CH: TChannel;
Begin
  // Create Channel set
  For CI:=1 to 8 do begin
    CH:=Channels[CI];                    // Get current channel entry
    If (CI<=CC) And (CH=NIL) then begin  // Exists?
      CH:=TChannel.Create(CI);           // Create it
    End;
    If (CI>CC) And (CH<>NIL) then begin  // Exists, but beyond count?
      CH.Free; CH:=NIL;                  // Drop it
    End;
    Channels[CI]:=CH;                    // Update channel list
  End;
  LastChan:=CC;
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,'Channel count set to:  '+IntToStr(LastChan));
  DispChan:=Channels[1];          // Select Channel A for display
  // Create standard unit record devices
  If RDRDev=NIL then
    With Channels[1] do begin
      RDRDev:=TDevRDR(AddDevice(DT711,0));
//    PUNDev:=TDevPUN(AddDevice(DT721,0));
      LSTDev:=TDevLST(AddDevice(DT716,0));
    End;
End;

procedure TMainForm.BIPwrOffClick(Sender: TObject);
begin
  ConfigForm.SaveMiscData;
  Close;
end;

procedure TMainForm.BIEditScriptClick(Sender: TObject);
begin
  EditForm.Show;
end;

procedure TMainForm.BIRunScriptClick(Sender: TObject);
begin
  If BIRunSCript.State then begin
    OCLThread.TerminateAllScripts;
  End else Begin
    With EditForm do
      If EditFileName<>'' then
        RunScript(EditFileName);
  End;
End;

//******************************************************************************
// Event handlers for miscellaneous (minor) functions
//******************************************************************************
// Process click of window display option controls
procedure TMainForm.CBCtrlClick(Sender: TObject);
begin
  SetControlFlags;
end;

procedure TMainForm.SetControlFlags;
begin
  // Extract control settings
  AutoMode:=CBAuto.Checked;
  SlowMode:=CBSlow.Checked;
  TracDisp:=CBTrace.Checked;
  PlotDisp:=CBPlot.Checked;
  SSAlert:=CBSSAsk.Checked;
  // Adjust for new settings
  CBSlow.Enabled:=AutoMode;
  TraceForm.Visible:=TracDisp;
  PlotForm.Visible:=PlotDisp;
  CoreForm.Visible:=CBCore.Checked;
  StopsForm.Visible:=CBStops.Checked;
end;

// Debug - Enable Logging and Tracing
Procedure TMainForm.SetTraceOn;
begin
  CBTrace.Checked:=True;
End;

// Display text in status area
procedure TMainForm.ShowMsg(MS: ShortString);
Begin
  StatusBar.SimpleText:=MS;
End;

// Set/Reset Hex/Octal display mode
procedure TMainForm.BIHexModeClick(Sender: TObject);
begin
  With BIHexMode do begin
    State:=Not State;
    HexMode:=State;
  End;
  DRIC.HexMode:=HexMode;
  DRSR.HexMode:=HexMode;
  DRAC.HexMode:=HexMode;
  DRMQ.HexMode:=HexMode;
  DRIR1.HexMode:=HexMode;
  DRIR2.HexMode:=HexMode;
  DRIR3.HexMode:=HexMode;
  DRIR4.HexMode:=HexMode;
  DRIR5.HexMode:=HexMode;
  DRIR6.HexMode:=HexMode;
  DRIR7.HexMode:=HexMode;
  DRCLR.HexMode:=HexMode;
  DRCCR.HexMode:=HexMode;
  DRCAR.HexMode:=HexMode;
  DRCWR.HexMode:=HexMode;
  DRCDR.HexMode:=HexMode;
End;

// Enable/Disable display of channel indicators
procedure TMainForm.RBChanClick(Sender: TObject);
begin
  DispChan:=Channels[(Sender As TComponent).Tag];
  UpdateDisplays;
end;

procedure TMainForm.CBSSAskClick(Sender: TObject);
begin
  // Set alert if we wanted to know when the CPU reads the SWRreg
  SSAlert:=CBSSAsk.Checked;
end;

// Process any change of the console switch register
procedure TMainForm.CBSSClick(Sender: TObject);
begin
  With (Sender As TButnInd) do begin
    State:=Not STate;
    RegSS[Tag]:=State;
  End;
end;

// Flash Sense switch when tested by SWT,
// Long flash on first test, Short flash next time (if enabled for this)
procedure TMainForm.SenseFlash(SN: Byte);
Var CB: TButnInd;
    DT: Integer;
    CN,CF: TColor;
Begin
  If (Not SenseFlags[SN]) Or (SSAlert And (InstCount-SwtfCount>500)) then begin
    Case SN Of
      1: CB:=CBSS1; 2: CB:=CBSS2; 3: CB:=CBSS3;
      4: CB:=CBSS4; 5: CB:=CBSS5; 6: CB:=CBSS6; Else CB:=NIL;
    End;
    If SenseFlags[SN] then DT:=25 else DT:=0500;
    With CB do begin
      CN:=ONColor; CF:=OFColor;
      ONColor:=CLRed; OFColor:=CLRed;
      Paint; Application.ProcessMessages; Sleep(DT);
      ONColor:=CN; OFColor:=CF;
      Paint; Application.ProcessMessages; Sleep(005);
    End;
    SenseFlags[SN]:=True;
  End;
  SwtfCount:=InstCount;
End;

// Refresh all console displays and indicators
Procedure TMainForm.UpdateDisplays;
Var FC: Single;
    AC: Int64;
Begin
//ShowTextValues:=True;
  DRIC.Value:=InstCtr;         // CPU registers
  DRSR.Value:=StorReg;
  AC:=RegAC; If SgnAC then AC:=AC Or $2000000000;
  DRAC.Value:=AC;
  DRMQ.Value:=RegMQ;
  DRSI.Value:=RegSI;
  BITTMode.State:=TrapMode;     // CPU Indicators
  BIOVFInd.State:=OVFInd;
  BIDVCInd.State:=DVCInd;
  BIIOCInd.State:=IOCInd;
  BICPUChk.State:=CPUCheck;
  DRIR1.Value:=XegXR[1];        // Index registers
  DRIR2.Value:=XegXR[2];
  DRIR3.Value:=XegXR[3];
  DRIR4.Value:=XegXR[4];
  DRIR5.Value:=XegXR[5];
  DRIR6.Value:=XegXR[6];
  DRIR7.Value:=XegXR[7];
  BIChnChk.State:=ChnCheck;
  BIChnStop.Visible:=ChnlActive;
  If DispChan<>NIL then         // Selected channel's registers
    With DispChan do begin
      DRCLR.Value:=LocnReg;
      DRCCR.Value:=ComdReg;
      DRCAR.Value:=AddrReg;
      DRCWR.Value:=WordReg;
      DRCDR.Value:=DataReg;
      BIChnAct.State:=Active;
      BIChnTrp.State:=TrapCtrlInd;
      BIChnTCK.State:=LCHInd;
      BIChnTCK.State:=TCKInd;
      BIChnEOF.State:=EOFInd;
    End;
  BISL1.State:=RegSL[1];        // Sense lights
  BISL2.State:=RegSL[2];
  BISL3.State:=RegSL[3];
  BISL4.State:=RegSL[4];
  Inc(TickCtr);
  If (TickCtr Mod 20)=0 then begin
    FC:=InstCount;
    Try
      LAInstCount.Caption:=Format('%18.0n',[FC]);
    Except End;
    If (TickCtr Mod 40)=0 then begin
      CoreRefresh;              // Core display update
    End;
  End;
//ShowTextValues:=False;
End;

//******************************************************************************
// Event handlers for the major control buttons
//******************************************************************************
// System Clear button
procedure TMainForm.BIClearClick(Sender: TObject);
begin
  BIResetClick(Self);             // Reset system
  MTagMode:=True;                 // Not done by reset 
  CoreClear;                      // Clear Core Storage
  PrinterForm.Clear;
  UpdateDisplays;
  ShowMsg('Clear & Reset complete');
end;

// System Reset button
procedure TMainForm.BIResetClick(Sender: TObject);
begin
  CPUReset;                       // Reset CPU
  ResetChannels;                  // Reset Channels & devices
  FillChar(SenseFlags,SizeOf(SenseFlags),0);
  UpdateDisplays;
  ShowMsg('Reset complete');
  BILoadCard.Show;                // Re-enable load buttons
  BILoadTape.Show;
end;

// Program Load button
procedure TMainForm.BILoadClick(Sender: TObject);
Var DS: String;
    DA: TAddr;
    LD: TDevice;
Begin
  BILoadCard.Hide;                // Disable further loads
  BILoadTape.Hide;
  // Reset 704 mode flags
  SNulMode:=False;
  STrpMode:=False;
  CTrpMode:=False;
  FTrpMode:=True;
  // Set load unit address
  If TControl(Sender).Tag=1 then DS:='01321'     // Card
                            else DS:='01221';    // Tape
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,'Load from device '+DS);
  BIRun.State:=True;              // Show running
  With Channels[1] do begin
    // Reset channel
    ResetChannel;
    // Simulated RDS instruction (Read Select)
    If (TIGEN In TraceRecdFlags) then
      Trace(TIGEN,'Implicit Read Select of load unit');
    DA:=OctStrToInt(DS);
    LD:=GetDevice(DA,True);
    SelectDevice(LD,False);       // Read Select the device
    If CurChanNum<>1 then Exit;   // Did it work? If not, Exit
    // Simulated RCHx instruction (Reset and Load Channel x)
    If (TIGEN In TraceRecdFlags) then
      Trace(TIGEN,'Implicit load of 3 words to location 0');
    SetCore(0,$8000C0000);        // IOCP for 3 word to location 0
    LoadChannelCommand(0);        // Load it and start channel
    LocnReg:=0;                   // Reset next CCW address
  End;
  // Reads 3 words, then CcIOCT transfers to CCW loaded into locn 0
  Try
    Repeat
      ProcessChannels;
    Until (Not ChnlActive) Or ChannelStop; // Wait for completion
  Except
    On E: Exception do begin
      ShowMsg(E.Message);       // Show error status
      Exit;
    End;
  End;
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,'Load Complete. Starting CPU...');
  // Start execution from location 00001
  CorTrcAdr:=-1;                   // Reset last core change by channels
  SetIC(00001,TCNone);             // Set program start location
  BIStartClick(Self);              // Pres start button
End;

// CPU Start button
procedure TMainForm.BIStartClick(Sender: TObject);
begin
  If (CPUCheck Or ChnCheck) then Exit;
  If CPURunning then begin        // Running?
    StopPend:=True;               // Yes, Tell system to stop
    Exit;
  End;
  // Not running, Start system
  StopPend:=False;                // Clear Stop flag
  StopMesg:='';                   // Clear stop reason message
  DispIPtr:=$8000;                // Force trace for first/only instruction
  // If HTR caused last stop, then begin from address it specified
  If HTRAddr<>$8000 then begin
    SetIC(HTRAddr,TCNone);        // Start from address set by last HTR
    HTRAddr:=$8000;               // Clear for next start
  End;
  // Begin run state. Turn on run indicators. Disable other controls
  BIStart.State:=True; BIRun.State:=True;
  BIReset.Hide; BIClear.Hide; BIPwrOff.Hide;
  Try
    Try
      ShowMsg('Operating...');    // Show status
      Process;                    // Excecute Instructions/Channel Commands
      ShowMsg(StopMesg);          // Done, Show why it stopped
    Except
      On E: Exception do begin    // Screwed up?
        ShowMsg(E.Message);       // Show error status
      End;
    End;
  Finally
    // Run state ended. Turn off run indicators, Re-enable other controls
    BIRun.State:=False; BIStart.State:=False;
    BIReset.Show; BIClear.Show; BIPwrOff.Show;
    UpdateDisplays;               // Ensure display is up to date
    ShowLastTrace;                // Ensure trace shows last entry
//    With OCLThread do
//      If ScriptState=SSCPUWAIT then  // Script waiting for this?
//        Restart;                     // Yes, Start scripter again}
  End;
End;

procedure TMainForm.BIChnStopClick(Sender: TObject);
begin
  If BIChnAct.State then
    ChannelStop:=True;
end;

procedure TMainForm.BIAboutClick(Sender: TObject);
begin
  SplashForm.Show;
end;

procedure TMainForm.BIConfigClick(Sender: TObject);
begin
  ConfigForm.ShowModal;
end;

// Update TrueFactor to control display refresh rate
procedure TMainForm.UDTrueFactClick(Sender: TObject; Button: TUDBtnType);
Var FP,FI: Integer;
begin
  FP:=UDTrueFact.Position;   // Get new value
  FI:=1000;                  // Form new increment for
  If FP<=5000 then FI:=500;  // value range
  If FP<=1000 then FI:=100;
  If FP<=100 then FI:=10;
  If FP<=10 then FI:=1;
  UDTrueFact.Increment:=FI;  // Set new increment for next click
  TrueFact:=FP;              // Save new value
end;

//*********************************************************
// Control events
//*********************************************************
procedure TMainForm.SetSenseSwitch(SN: Byte; SV: Boolean);
Var BI: TButnInd;
Begin
  Case SN Of
    1: BI:=CBSS1; 2: BI:=CBSS2; 3: BI:=CBSS3;
    4: BI:=CBSS4; 5: BI:=CBSS5; 6: BI:=CBSS6;
    Else BI:=NIL;
  End;
  BI.State:=SV;
  RegSS[SN]:=SV;
End;

// Events generated within the emulator are generally handled within the
// unit in which they arise. If further actions are required, control
// messages are sent to this central handler to complete processing
// of the event.
Procedure TMainForm.WMEMCMsg(Var Message: TMessage);
Var PM: PEMCMsg;
    SS: String;
Begin
  Try
    PM:=PEMCMsg(Message.LParam);
    With PM^ do begin
      If StrParm=NIL then SS:='' else SS:=StrParm^;
      Case FunCode of
        CCSCRIPT:        BIRunScript.State:=Parm1<>0;
        CCTRACE:         CBTrace.Checked:=Parm1<>0;
        CCSAVTRACE:      TraceForm.SaveTrace(SS);
        CCPLOTCLEAR:     Begin
                           PlotForm.PlotClear;
                           CBPlot.Checked:=True;
                         End;
        CCPLOTONOFF:     CBPlot.Checked:=Parm2=0;
        CCGENCHNS:       SetupChnDev(Parm1);
        CCENDCONFIG:     TapesForm.DisplayDrives;
        CCPRESS:         Case Parm1 Of
                           CTRESET:  BIReset.Click;
                           CTCLEAR:  BIClear.Click;
                           CTSTART:  BIStart.Click;
                           CTLODCRD: BILoadCard.Click;
                           CTLODTAP: BILoadTape.Click;
                           Else     ShowError('Messaging','Unexpected "Press" code: '+IntToStr(Parm1));
                         End;
        CCTAPEDEF:       TapesForm.DefineDrive(Parm1);
        CCTAPEDEL:       TapesForm.DeleteDrive(Parm1);
        CCTAPVADR:       TapeViewForm.SelectDrive(TDevTAP(GetDevice(Parm1,True)));
        CCTAPVMOD:       TapeViewForm.SetDispMode(Boolean(Parm1));
        CCTAPVLIN:       TapeViewForm.SetLineNumb(Parm1);
        CCTAPVSAV:       TapeViewForm.SaveToFile(SS);
        CCSETSENSW:      SetSenseSwitch(Parm1,Parm2<>0);
        CCSCRIPTRESUME:  OCLThread.Resume;
        CCDISPSHOW:      ScriptForm.ShowHide(Parm1=1);
        Else             ShowError('Messaging','Unexpected Control Message. Code='+IntToStr(FunCode));
      End;
    End;
    Application.ProcessMessages;
  Except
    On E: Exception do begin
      ShowError('Control Execution',E.Message);
    End;
  End;
End;

// I/O control messages (from scripter) are handled here.
// The indicated device DeviceControl method is invoked to perform the function
procedure TMainForm.WMIOCMsg(var Message: TMessage);
Var PM: PIOControl;
    DV: TDevice;
begin
  Try
    PM:=PIOControl(Message.LParam);
    With PM^ do begin
      DV:=GetDevice(DevAddr,True);
      DV.DeviceControl(PM);
    End;
    Application.ProcessMessages;
  Except
    On E: Exception do begin
      ShowError('I/O operation',E.Message);
    End;
  End;
End;

End.
