//******************************************************************************
//  IBM 7094 Emulator - Tape Drive display form
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit defines the TDevTAP object which descends from TDevice.
//  TDevTAP provides the functions for the IBM 729 tape drive
//  There may be several instances of TDevTAP.
//  Each instance of TDevTAP has an associated form window to display its
//  status. These forms are instances of the TFormTape class, whose definitions
//  and methods are also contained here.
//  Instances of these windows are all contained in a parent display window
//  "TTapesForm" which is defined in B709DTAP.
//------------------------------------------------------------------------------
//  Significant sections of the code in this unit were debugged and enhanced
//  by James Fehlinger. Thanks Jim.
//--------------------------------------------------------------------------
Unit B709TapF;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics,
     Controls, Forms, Dialogs, ComCtrls, StdCtrls, ExtCtrls,
     B709Strm, // Customized TMemoryStream class
     B709Cmps, // Display Components
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Misc, // Miscellaneous utility functions
     B709Trce, // Log/Trace functions
     B709DTAP, // Tape Drives container form
     B709Chan; // I/O Channel and Device functions

Const TAPEMARK=$8F;

Type TDevTAP=Class(TDevice)
       Constructor Create(DT: TDevType; CH: TChannel; DA: TAddr);      Override;
     Private
       FirstWord: Boolean;
       BegRecFlag: Boolean;
       Procedure ConvertASCtoBIN;
       function  TestTapeBeg: Boolean;
       procedure BackupOneRecd(RM: Boolean);
     Protected
       Procedure AttachFile(FN: ShortString; WM: Boolean);             Override;
       Procedure RewindFile;                                           Override;
       Function  DeviceSelect(WR: Boolean): Boolean;                   Override;
       Function  ReadWord: TWord;                                      Override;
       Function  TestEOR: Boolean;                                     Override;
       Procedure WriteWord(WW: TWord);                                 Override;
       Procedure WriteEndOfRecord;                                     Override;
       Procedure WriteEndOfFile;                                       Override;
       Procedure MoveToEndOfRecord;                                    Override;
     Public
       Procedure Operate(FC: TOprFunc);                                Override;
     End;

Type
  TFormTape = class(TForm)
    PADrive: TPanel;
    Label4: TLabel;
    Label7: TLabel;
    LACUU: TLabel;
    LAFileName: TLabel;
    LAPosn: TLabel;
    LASize: TLabel;
    BIWP: TButnInd;
    BIRD: TButnInd;
    BIWR: TButnInd;
    BIBin: TButnInd;
    BIRewind: TButnInd;
    BIOpen: TButnInd;
    TBPosn: TTrackBar;
    BIRun: TButnInd;
    procedure BIWPClick(Sender: TObject);
    procedure BIRewindClick(Sender: TObject);
    procedure BIOpenClick(Sender: TObject);
    procedure TBPosnChange(Sender: TObject);
    procedure BIRunClick(Sender: TObject);
    procedure LACUUClick(Sender: TObject);
  private
    procedure ShowFileDetails(Sender: TObject);
    Procedure RefreshDisplay(Sender: TObject);
  public
    TAPDev:   TDevTAP;
    Procedure Initialize;
  End;

Var FormTape: TFormTape;               // Form instance

Implementation
 
Uses B709TAPV; // Tape data viewer

{$R *.DFM}

//******************************************************************************
//  TDevTAP Methods
//------------------------------------------------------------------------------
Constructor TDevTAP.Create(DT: TDevType; CH: TChannel; DA: TAddr);
begin
  Inherited Create(DT,CH,DA);
  BinAddr:=DA+$10;                     // Form address for Binary mode
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' 729 Tape Drive defined');
end;

// Associate a PC data file with the tape drive
procedure TDevTAP.AttachFile(FN: ShortString; WM: Boolean);
Var FE: String;
begin
  Inherited;
  // If not *.BIN or *.BCD, assume ASCII format and translate
  // file data in memory to BIN file format
  FE:=ExtractFileExt(UpperCase(FN));
  If (FE<>'') And (FE<>'.BIN') And (FE<>'.BCD') then
    ConvertASCtoBIN;
End;

// Select this tape drive as the active device on its channel
Function  TDevTAP.DeviceSelect(WR: Boolean): Boolean;
begin
  Result:=Inherited DeviceSelect(WR);
End;

// Reset device data buffer pointer to start of data
procedure TDevTAP.RewindFile;
begin
  Inherited;
// Do this here (tapes only) or in TDevice (all device types)??
  With Channel do begin             // Reset channel indicators
    BOTInd:=True; EOTInd:=False;
  End;
  FileNum:=0;                             // Reset stuff
  RecdNum:=0;
  WordNum:=0;
  FirstWord:=True;
  IOStream.Seek(0,soFromBeginning);       // Go to start of memory image
//;SetDataPosition(0);                     // Go to start of memory image
End;

// See if next byte to be read has Gap Mark. Set EndOfRecd if so.
Function TDevTAP.TestEOR: Boolean;
Var DP: Longint;
    BB: Byte;
Begin
  DP:=DataPosn;                        // Get current stream position
  If DP>DataSize then begin       // End of data?
    Result:=True;
  End else Begin
//  If (DP=0) then Exit;
    BB:=IOStream.Peek(DP);             // Get stream byte
    If (BB And $80)>0 then Result:=True     // Record gap?
                      else Result:=False;
  End;
End;

// Skip to end of current tape record
Procedure TDevTAP.MoveToEndOfRecord;
Var IP: Longint;
begin
  IP:=DataPosn;
  If (Not EndOfRec) then begin
    Repeat
      ReadWord;
    Until EndOfRec;
  End;
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' Skip to EOR from  '+IntToStr(IP)+' to '+IntToStr(DataPosn));
End;

// Read a word from the device data buffer
Function TDevTAP.ReadWord: TWord;
Var BC: Byte;
Begin
  Result:=0;
  If FirstWord then begin
    GetNextByte;                            // Prime CurrChar
    FirstWord:=False;
  End else Begin
    Inc(WordNum);
  End;
  For BC:=1 to 6 do begin                   // 6 bytes (chars) to a word
    Result:=(Result Shl 6) Or CurrChar;     // Use current byte
    GetNextByte;                        // Get next one ready
    If CurrByte=TAPEMARK then begin
      GetNextByte;                          // Flush it out of the data stream!
      If (TIDEV In TraceRecdFlags) then
        Trace(TIDEV,'Dev   '+DevAdrString+' Tape Mark at '+IntToStr(DataPosn));
      EndOfRec:=True;                       // Set flags
      PendOfFil:=True;
      Result:=Result Shl (6-BC);
      Break;                                // End word here
    End;
  End;
  // Only "see" a record mark at the beginning of a word.
  // If there's one anywhere else, we have a corrupt tape,
  // and we're fried anyway.
  If (CurrByte And $80)>0 then begin            // Record gap?
    If (TIDEV In TraceRecdFlags) then
      Trace(TIDEV,'Dev   '+DevAdrString+' Record Gap at '+IntToStr(DataPosn));
    EndOfRec:=True;                       // Set EOR flag
  End;
  // Update associated display occasionally
  If (WordNum Mod 100)=0 then
    UpdateStatus;
End;

// Write a word to the device data buffer
Procedure TDevTAP.WriteWord(WW: TWord);
Var WB: Array[1..6] Of Byte;
    BC: Byte;
begin
  For BC:=6 Downto 1 do begin          // Work backwards
    WB[BC]:=(WW And $3F);              // Get rightmost 6-bit Char, put
                                       // in rightmost (higher-index) position
                                       // in buffer.
    WW:=WW Shr 6;                      // Move next char to rightmost 6 bits
  End;
  If FirstWord Or BegRecFlag then Begin
    WB[1]:=WB[1] Or $80;               // If new record, Set EOR bit in first char
    BegRecFlag:=False;            // (rightmost, in this buffer) byte
    FirstWord:=False;
  End;
  PutBytes(WB,6);                      // This write starts from the WB[1] side of the buffer.
  Inc(WordNum);
  // Update associated display occasionally
  If (WordNum Mod 100)=0 then
    UpdateStatus;
end;

// Write an EOR mark to data buffer
Procedure TDevTAP.WriteEndOfRecord;
begin
  // This will signal next word written to have EOR bit turned on in first byte.
  BegRecFlag:=True;
end;

// Write an EOF mark to data buffer
Procedure TDevTAP.WriteEndOfFile;
Var BB: Byte;
begin
  BB:=TAPEMARK; PutBytes(BB,1);              // Write file mark
end;

Function TDevTAP.TestTapeBeg: Boolean;
Begin
  Result:=DataPosn<6;
  If Result then
    RewindFile;
End;

Procedure TDevTAP.BackupOneRecd(RM: Boolean);
Var BF: Boolean;
    BB: Byte;
Begin
  BF:=TestTapeBeg;
  If Not BF then Begin
   Repeat
     IOStream.Seek(-6,SOFromCurrent);     // Back up to previous word
     Dec(WordNum);                        // Adjust word number
     BF:=TestTapeBeg;
     If BF then Break;
     IOStream.Read(BB,1);                 // Read byte
     IOStream.Seek(-1,SOFromCurrent);     // Put it back
     If (BB And $80) > 0 then Break;      // Record mark?
     IOStream.Seek(-1,SOFromCurrent);     // Back up to byte **before** first byte of current word
     IOStream.Read(BB,1);                 // Read it
     If BB=TAPEMARK then                  // File mark?
       If RM then IOStream.Seek(-1,SOFromCurrent)   // If so, back up again to just past end of preceding word.
             else Break;
   Until False;
  End;
End;

// Perform indicated drive operation
procedure TDevTAP.Operate(FC: TOprFunc);
Var BF: Boolean;
    IP: Longint;
    BB: Byte;
    TS: String;
Begin
  TS:='Dev   '+DevAdrString+' ';
  IP:=DataPosn;                   // Get current file posn
  Case FC Of
    FCBSR: Begin
             // First, skip back over any Tape mark
             BF:=TestTapeBeg;
             If Not BF then Begin
               IOStream.Seek(-2,SOFromCurrent);       // Back up to byte **before** first byte of current word
               IOStream.Read(BB,1);                   // Read it
               If BB=TAPEMARK then                    // File mark?
                 IOStream.Seek(-1,SOFromCurrent);     // If so, back up again to just past end of preceding word.
             End;
             // Back up to beginning of current record.
             BackupOneRecd(True);
             // Back up one more record.
             BackupOneRecd(True);
             // Fix rec num and prime
             If Not BF then Begin
               Dec(RecdNum);                           // Adjust record number
               If Not WRMode then
                 GetNextByte;                          // Re-Prime
             End;
             TS:=TS+'BackSpace RECD from '+IntToStr(IP)+' to '+IntToStr(DataPosn);
           End;
    FCBSF: Begin
             BF:=TestTapeBeg;
             If Not BF then Begin
               If WRMode then Begin
                 IOStream.Seek(-1,SOFromCurrent);       // Back up to byte **before** first byte of current word
               End else Begin
                 IOStream.Seek(-2,SOFromCurrent);       // Back up to byte **before** first byte of current word
               End;
               IOStream.Read(BB,1);                     // Read it
               If BB=TAPEMARK then                      // File mark?
                 IOStream.Seek(-1,SOFromCurrent);       // If so, back up again to just past end of preceding word.
             End;
             // First, back up to beginning of current file.
             BackupOneRecd(False);
             // Back up one more file.
             BackupOneRecd(False);
             // Fix rec num and prime
             If Not BF then Begin
               Dec(FileNum);                            // Bump file number down
               RecdNum:=0;                              // Reset record number
               If Not WRMode then
                 GetNextByte;                           // Re-Prime
             End;
             TS:=TS+'BackSpace FILE from '+IntToStr(IP)+' to '+IntToStr(DataPosn);
           End;
(*
    FCBSR: Begin
             // Start at byte prior to last one read
             DP:=DataPosn-0;
             // Move backwards to the gap at the start of this record
             Repeat
               BB:=IOStream.Peek(DP);
               If BB=TAPEMARK then begin
                 Dec(DP); Break;
               End;
               If (BB And $80)>0 then
                 Break; // Gap? If so, at start of recd
               Dec(DP);					    // No, Keep going
             Until DP=0;				    // Stop if we hit start of tape
             // Backup over the gap to get to previous record
             Dec(DP);
             // Move backwards to the gap at the start of the previous record
             Repeat
               If (IOStream.Peek(DP) And $80)>0 then Break; // Gap? If so, at start of recd
               Dec(DP);					    // No, Keep going
             Until DP=0;				    // Stop if we hit start of tape
             // Backup one char for priming
             If DP>0 then
               SetDataPosition(DP-1);
             Dec(RecdNum);             // Bump record number down
             If Not WRMode then
               GetNextByte;                       // Re-Prime
             TS:=TS+'BackSpace RECD from '+IntToStr(IP)+' to '+IntToStr(DataPosn);
           End;
    FCBSF: Begin
             ShowMessage('A BSF!');
{            If Not (WRMode Or CheckStartOfTape) then Begin
               SetDataPosition(DataPosn-2);           // Back up to byte **before** first byte of current word
               GetTapeChar;                           // And Re-Read
             End;
             // First, back up to beginning of current file.
             If Not CheckStartOfTape then             // Already in first word?
               Repeat
                 SetDataPosition(DataPosn-6);         // No, back up to previous word
                 Dec(WordNum);                        // Adjust word number
                 If CheckStartOfTape then Break;      // Hit first word yet?
                 If AtTapeMark then Begin             // Is next byte a tape mark?
                   SetDataPosition(DataPosn-1);       // Back up to previous byte
                   BB:=GetNextByte;                   // Read byte
                   If BB=TAPEMARK then Break;              // File mark?
                 End;
               Until False;
             // Back up one more file.
             If Not CheckStartOfTape then Begin       // Already in first word?
               Repeat
                 SetDataPosition(DataPosn-6);         // Back up to previous word
                 Dec(WordNum);                        // Adjust word number
                 If CheckStartOfTape then Break;      // Hit first word yet?
                 If AtTapeMark then Begin             // Is next byte a tape mark?
                   SetDataPosition(DataPosn-1);       // Back up to previous byte
                   BB:=GetNextByte;                   // Read byte
                   If BB=TAPEMARK then Break;              // File mark?
                 End;
               Until False;
               Dec(FileNum);                          // Bump file number down
               RecdNum:=0;                            // Reset record number
               If Not WRMode then
                 GetTapeChar;
             End;}
             TS:=TS+'BackSpace FILE from '+IntToStr(IP)+' to '+IntToStr(DataPosn);
           End;
*)
    FCWEF: Begin
             TS:=TS+'Write EOF at '+IntToStr(IP);
             WriteEndOfFile;
           End;
    FCREW: Begin
             RewindFile;
             TS:=TS+'Rewinding...';
           End;
    FCSDN: Begin                                 // Set Density
             TS:=TS+'Set Density';
           End;
    Else   Error('Invalid function for tape device');
  End;
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,TS);
end;

// Read ASCII lines, Convert to BIN file format
Procedure TDevTAP.ConvertASCtoBIN;
Var MS: TB709Stream;
    SL: TStringList;
    SS: String;
    BS: Array[1..80] Of Byte;
    SI: Integer;
    CI,CB: Byte;
Begin
  MS:=TB709Stream.Create;              // Create temp image for conversion
  SL:=TStringList.Create;
  With SL do begin
    LoadFromFile(FFileNam);            // Load from PC file
    For SI:=0 to Count-1 do begin
      SS:=Strings[SI];                 // Read a line
      FillChar(BS,80,$30);             // Fill output rec with zeroes
      For CI:=1 to Length(SS) do begin // Process each char in line
        CB:=Ord(SS[CI]);
        If CB>=$60 then                // Make UpperCase
          CB:=CB And $5F;
        CB:=TabASCToBCD[CB];           // Convert to BCD
        BS[CI]:=CB;
      End;
      MS.Write(BS,72);                 // Write converted record
    End;
  End;
  IOStream.LoadFromStream(MS);         // Replace ASCII data with BIN data
  MS.Free;
  SL.Free;
  RewindFile;                          // Setup to use it
End;

//******************************************************************************
//  Form Functions
//------------------------------------------------------------------------------
Procedure TFormTape.Initialize;
begin
  With TAPDev do begin
    OnFileChange:=ShowFileDetails;
    OnStatusChange:=RefreshDisplay;
//  OnEndOfRecord:=
    LACUU.Caption:=IntToOct(DevAddr,5);
    LAFileName.Caption:=FileNam;
    LAPosn.Caption:=IntToStr(0);
    LASize.Caption:=IntToStr(0);
  End;
End;

procedure TFormTape.BIWPClick(Sender: TObject);
begin
  With BIWP do begin
    State:=Not State;
    TAPDev.WRProt:=State;
  End;
End;

procedure TFormTape.ShowFileDetails(Sender: TObject);
begin
  With TAPDev do begin
    LAFileName.Caption:=FileNam;
    LASize.Caption:=IntToStr(DataSize);
    TBPosn.SelEnd:=DataSize Div 1024;
  End;
End;

Procedure TFormTape.RefreshDisplay(Sender: TObject);
Var FP: Longint;
Begin
  With TAPDev do begin
    BIWP.State:=WRProt;
    BIBin.State:=BinMode;
    BIRD.State:=Selected And (Not WRMode);
    BIWR.State:=Selected And WRMode;
    FP:=DataPosn;
    LAPosn.Caption:=IntToStr(FP);
    TBPosn.Position:=FP Div 1024;
    If WRMode then
      LASize.Caption:=IntToStr(DataSize);
  End;
End;

procedure TFormTape.TBPosnChange(Sender: TObject);
begin
  TBPosn.Position:=TAPDev.DataPosn Div 1024;
end;

procedure TFormTape.BIOpenClick(Sender: TObject);
Var FN: String;
begin
  With TAPDev do
    If ConfigForm.OpenFileDialog(FGTAP,FN,False) then
      IOCMsg(DevAddr,FCOPEN,0,FN);
end;

procedure TFormTape.BIRewindClick(Sender: TObject);
begin
  With TAPDev do
    IOCMsg(DevAddr,FCREWIND,0,'');
end;

procedure TFormTape.BIRunClick(Sender: TObject);
begin
  With TAPDev do begin
    IOCMsg(DevAddr,FCCLOSE,0,'');
    WRMode:=True;
  End;
End;

// File name label click - Open tape viewer
procedure TFormTape.LACUUClick(Sender: TObject);
begin
  With TapeViewForm do begin
    SelectDrive(TapDev);
    SetDispMode(False);
  End;
end;

End.

