//******************************************************************************
//  IBM 7094 Emulator - Log/Trace form
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit provides the Log/Trace display form and associated functions
//------------------------------------------------------------------------------
Unit B709Trce;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics,
     Controls, Forms, Dialogs, StdCtrls, ExtCtrls,
     B709Defs; // General definitions

// Types of possible trace entries. When a trace entry is created by a call to
// Trace, the first parameter muse be one of these codes. The trace string passed
// in the second parameter is stored into the raw trace listed with this code.
// Filter checkbox tags also have the ordinal value of these codes.
Type TTraceID=(
       TIXXX,      // 0 Not used, bases ordinal value to 1
       TIGEN,      // 1 General/Miscellaneous
       TIINS,      // 2 Instruction about to be executed
       TIMEM,      // 3 Core location value altered
       TIREG,      // 4 Register value altered
       TICCW,      // 5 Channel command about to be executed
       TIIOT,      // 6 I/O word transfer
       TIDEV,      // 7 I/O device event
       TIZZZ,      // 8 Testing
       TISCR);     // 9 Scripting events

Type TTraceIDSet=Set Of TTraceID;

Const TIALL=[TIGEN..TISCR];       // All Trace IDs

Const TIChars='GIMRCTDZS';         // Must match the above

Type
  TTraceForm = class(TForm)
    Panel1: TPanel;
    LBTraceDisp: TListBox;
    PBClear: TButton;
    LABrColor: TLabel;
    LAFOColor: TLabel;
    ColorDialog: TColorDialog;
    LALineCount: TLabel;
    PBSave: TButton;
    Label1: TLabel;
    PBClose: TButton;
    SBPos: TScrollBar;
    PBDisplay: TButton;
    PBLoad: TButton;
    GBDispFilters: TGroupBox;
    CBDREG: TCheckBox;
    CBDCCW: TCheckBox;
    CBDIOT: TCheckBox;
    CBDMEM: TCheckBox;
    CBDGEN: TCheckBox;
    CBDDEV: TCheckBox;
    CBDSCR: TCheckBox;
    GBRecdFilters: TGroupBox;
    CBTINS: TCheckBox;
    CBTREG: TCheckBox;
    CBTCCW: TCheckBox;
    CBTIOT: TCheckBox;
    CBTMEM: TCheckBox;
    CBTGEN: TCheckBox;
    CBTDEV: TCheckBox;
    CBTSCR: TCheckBox;
    Label2: TLabel;
    EBSkip: TEdit;
    Label3: TLabel;
    EBStop: TEdit;
    Label4: TLabel;
    EBKey: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure PBClearClick(Sender: TObject);
    procedure LBTraceDispDrawItem(Control: TWinControl; Index: Integer; Rect: TRect; State: TOwnerDrawState);
    procedure FormDestroy(Sender: TObject);
    procedure LABrColorClick(Sender: TObject);
    procedure LAFOColorClick(Sender: TObject);
    procedure LBTraceDispClick(Sender: TObject);
    procedure CBRecdFilterClick(Sender: TObject);
    procedure CBDispFilterClick(Sender: TObject);
    procedure PBCloseClick(Sender: TObject);
    procedure PBSaveClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure SBPosChange(Sender: TObject);
    procedure PBDisplayClick(Sender: TObject);
    procedure PBLoadClick(Sender: TObject);
    procedure EBKeyChange(Sender: TObject);
    procedure EBStopStopPress(Sender: TObject; var Key: Char);
  private
    TraceList:   TStringList;          // Trace recording list
    FilterList:  TStringList;          // Entries that passed disp filter
    FilterKey:   String;
    DispPageSize: Integer;             // # of lines in trace display
    Modified:     Boolean;
    Procedure SetColor(FB: Boolean);
    Procedure FilterAdd(TI: TTraceID; TS: String);
    procedure LoadFilterList;
    function  SaveFirstCheck: Boolean;
    procedure LoadTrace(FN: String);
  public
    Procedure Initialize;
    Procedure SaveTrace(FN: String);
  end;

Var TraceForm:    TTraceForm;

Type ColorRecd=Record
       CLBrush,CLFont: TColor;
     End;

Var TraceRecdFlags: TTraceIDSet;   // Trace record filter flags
    TraceDispFlags: Array[TTraceID] Of Boolean;   // Trace display filter flags
    TraceColors:    Array[TTraceID] of ColorRecd;
    TraceScrs:      Boolean;

Procedure Trace(TI: TTraceID; TS: String);
Procedure XTrace(TS: String);
Procedure ClearTrace;
Procedure ShowLastTrace;

Implementation

Uses B709CPU,  // For SetStopPending
     B709CnFg; // Open/Save dialogs

{$R *.DFM}

Const TraceFileName='B7094.TRC';

Var TraceCount: Integer;
    SkipCount:  Integer;
    StopCount:  Integer;

//******************************************************************************
//  Non-Form methods
//------------------------------------------------------------------------------
Procedure ClearTrace;
begin
  With TraceForm do begin
    TraceList.Clear;
    PBDisplayClick(NIL);
    Modified:=False;
    LALineCount.Caption:=IntToStr(TraceList.Count);   // Show line count
  End;
  TraceCount:=0;
End;

Procedure ShowLastTrace;
Begin
  With TraceForm.LBTraceDisp do
    If Visible then
      TopIndex:=Items.Count-1;
End;

Procedure Trace(TI: TTraceID; TS: String);
Begin
//If Not Tracing then Exit;            // Forget it?
//If Not (TI In [TIIOT,TIDEV,TICCW]) then Exit;
//  If (TI<>TIZZZ) And (Not Tracing) then Exit;            // Forget it?
//  TI:=TIDEV;
  Inc(TraceCount);
  If TraceCount<SkipCount then Exit;
  With TraceForm do begin
    If TraceList.Count>StopCount then Exit;
    TraceList.AddObject(TS,TObject(TI));    // Add entry to unfiltered list
//  FilterAdd(TI,TS);                       // Also add to filtered list
    LALineCount.Caption:=IntToStr(TraceList.Count);   // Show line count
    Modified:=True;
    Application.ProcessMessages;
//  If TraceList.Count<DispPageSize then SBPosChange(NIL);
  End;
End;

// Unmaskable Trace
Procedure XTrace(TS: String);
Begin
  With TraceForm do begin
    TraceList.AddObject(TS,TObject(TIGEN));
    LALineCount.Caption:=IntToStr(TraceList.Count);
    Application.ProcessMessages;
  End;
End;

//******************************************************************************
//  Form methods
//------------------------------------------------------------------------------
procedure TTraceForm.FormCreate(Sender: TObject);
Begin
  FormHandles[FITrace]:=Handle;
  Caption:=Heading+' - Log/Trace display';
  TraceList:=TStringList.Create;
  FilterList:=TStringList.Create;
  LBTraceDisp.DoubleBuffered:=True;
  Modified:=False;
  SkipCount:=000000; EBSkip.Text:=IntToStr(SkipCount);
  StopCount:=500000; EBStop.Text:=IntToStr(StopCount);
End;

procedure TTraceForm.FormDestroy(Sender: TObject);
begin
  SaveFirstCheck;                                     // Save current data?
  TraceList.Free;
  FilterList.Free;
end;

procedure TTraceForm.Initialize;
begin
end;

procedure TTraceForm.PBCloseClick(Sender: TObject);
begin
  Hide;
end;

Procedure TTraceForm.PBSaveClick(Sender: TObject);
Begin
  SaveTrace('');
End;

procedure TTraceForm.PBLoadClick(Sender: TObject);
Var FN: String;
begin
  SaveFirstCheck;                                     // Save current data?
  If ConfigForm.OpenFileDialog(FGTRC,FN,False) then   // Get new file name
    LoadTrace(FN);                                    // Load it
end;

procedure TTraceForm.PBClearClick(Sender: TObject);
begin
  SaveFirstCheck;                                     // Save current data?
  ClearTrace;
end;

Function TTraceForm.SaveFirstCheck: Boolean;
begin
  // Check if current file should be saved first
  If Modified then begin
    Result:=MessageDlg('Save current trace data?',MTWarning,[MBYES,MBNO],0)=MRYES;
    If Result then
       PBSaveClick(Self);
  End else
    Result:=False;
  Modified:=False;
End;

procedure TTraceForm.PBDisplayClick(Sender: TObject);
Var CI: Integer;
    CC: TControl;
begin
  Screen.Cursor:=CRHourGlass;
  // Load display filters
  With GBDispFilters do
    For CI:=0 to ControlCount-1 do begin
      CC:=Controls[CI];
      If CC Is TCheckBox then
        TraceDispFlags[TTraceID(CC.Tag)]:=TCheckBox(CC).Checked;
    End;
  //PBDisplay.Enabled:=False;
  LBTraceDisp.Clear;
  FilterKey:=Trim(EBKey.Text);
  // Reload Filtered list from Raw list using new filter settings
  LoadFilterList;
  // Load listbox from filtered list
  SBPos.Position:=1;
  SBPosChange(NIL);
  Screen.Cursor:=CRDefault;
end;

procedure TTraceForm.CBRecdFilterClick(Sender: TObject);
begin
  // Set filter flag for trace code corresping to clicked checkbox
  With Sender As TCheckBox do
    If Checked then TraceRecdFlags:=TraceRecdFlags+[TTraceID(Tag)]
               else TraceRecdFlags:=TraceRecdFlags-[TTraceID(Tag)];
//PBDisplay.Enabled:=True;
End;

procedure TTraceForm.CBDispFilterClick(Sender: TObject);
begin
  // Set filter flag for trace code corresping to clicked checkbox
  With Sender As TCheckBox do
    TraceDispFlags[TTraceID(Tag)]:=Checked;
//PBDisplay.Enabled:=True;
End;

procedure TTraceForm.EBKeyChange(Sender: TObject);
begin
//PBDisplay.Enabled:=True;
end;

Procedure TTraceForm.SaveTrace(FN: String);
Var TL: TStringList;
    LI: Integer;
    TS: STring;
begin
  Screen.Cursor:=CRHourGlass;
  TL:=TStringList.Create;
  With TraceList do
    For LI:=0 to Count-1 do begin
      TS:=TIChars[Byte(Objects[LI])]+Strings[LI];
      TL.Add(TS);
    End;
  If FN='' then FN:=TraceFileName;
  TL.SaveToFile(FN);
  TL.Free;
  Modified:=False;
  Screen.Cursor:=CRDefault;
End;

Procedure TTraceForm.LoadTrace(FN: String);
Var TL: TStringList;
    LI: Integer;
    TS: STring;
    TI: TTraceID;
begin
  TL:=TStringList.Create;
  TL.LoadFromFile(FN);
  ClearTrace;
  With TL do
    For LI:=0 to Count-1 do begin
      TS:=Strings[LI];
      If TS='' then Continue;
      TI:=TTraceID(Pos(TS[1],TIChars));
      System.Delete(TS,1,1);
      TraceList.AddObject(TS,TObject(TI));    // Add entry to unfiltered list
    End;
  TL.Free;
  LALineCount.Caption:=IntToStr(TraceList.Count);   // Show line count
End;

// Add entry to Filtered list if it passes current filter settings
Procedure TTraceForm.FilterAdd(TI: TTraceID; TS: String);
Var FC: Integer;
begin
  If TraceDispFlags[TI] then begin             // Trace code enabled?
    If (FilterKey<>'') And (Pos(FilterKey,TS)=0) then Exit;
    FilterList.AddObject(TS,TObject(TI));   // Yes, add entry to filtered list
    FC:=FilterList.Count;
    If FC>DispPageSize then
      FC:=FC-DispPageSize;
    SBPos.Max:=FC;
  End;
End;

// Load Filtered list from Raw list using current filter settings
procedure TTraceForm.LoadFilterList;
Var RI: Integer;
begin
  FilterList.Clear;
  With TraceList do begin
    For RI:=0 to Count-1 do
      FilterAdd(TTraceID(Objects[RI]),Strings[RI]);
  End;
End;

procedure TTraceForm.LBTraceDispDrawItem(Control: TWinControl; Index: Integer; Rect: TRect; State: TOwnerDrawState);
Var SS: ShortString;
    TI: TTraceID;
begin
  With Control As TListBox do begin
    SS:=Items[Index];
    TI:=TTraceID(Items.Objects[Index]);
    With Canvas do begin
      With TraceColors[TI] do begin
        Brush.Color:=CLBrush; Font.Color:=CLFont;
      End;
      FillRect(Rect);
      TextOut(Rect.Left,Rect.Top,SS);
    End;
  End;
End;

Procedure TTraceForm.SBPosChange(Sender: TObject);
Var SP,LI,LC,EI: Integer;
    SS: String;
    TI: TObject;
begin
  SP:=SBPos.Position-1;
  LC:=FilterList.Count-1;
  With LBTraceDisp do begin
    Clear;
    EI:=SP+DispPageSize-1;
    If EI>LC then EI:=LC;
    For LI:=SP to EI do begin
      With FilterList do begin
        SS:=Strings[LI];
        TI:=Objects[LI];
      End;
      Items.AddObject(SS,TI);
    End;
  End;
end;

procedure TTraceForm.FormResize(Sender: TObject);
begin
  With LBTraceDisp do
    DispPageSize:=ClientHeight Div ItemHeight;
  SBPos.PageSize:=DispPageSize;
  SBPos.LargeChange:=DispPageSize;
  LBTraceDisp.Repaint;
End;

procedure TTraceForm.SetColor(FB: Boolean);
Var CI: TTraceID;
Begin
  With LBTraceDisp do begin
    If ItemIndex<0 then Exit;
    CI:=TTraceID(Items.Objects[ItemIndex]);
  End;
  With ColorDialog do begin
    With TraceColors[CI] do
      If FB then Color:=CLFont
            else Color:=CLBrush;
    If Execute then begin
      With TraceColors[CI] do
        If FB then CLFont:=Color
              else CLBrush:=Color;
      LBTraceDisp.Repaint;
    End;
  End;
End;

procedure TTraceForm.LAFOColorClick(Sender: TObject);
Begin
  SetColor(True);
end;

procedure TTraceForm.LABrColorClick(Sender: TObject);
Begin
  SetColor(False);
End;

procedure TTraceForm.LBTraceDispClick(Sender: TObject);
Var EF: Boolean;
begin
  EF:=LBTraceDisp.ItemIndex>=0;
  LABRColor.Enabled:=EF;
  LAFOColor.Enabled:=EF;
end;

procedure TTraceForm.EBStopStopPress(Sender: TObject; var Key: Char);
begin
  If Not (Key In [#8,'0'..'9']) then Key:=#0;
  SkipCount:=StrToInt(EBSkip.Text);
  StopCount:=StrToInt(EBStop.Text);
end;

Initialization
  TraceCount:=0;
  TraceRecdFlags:=[];
End.



