//******************************************************************************
//  IBM 7094 Emulator - Channel definitions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit defines the TChannel and TDevice objects.
//  TChannel provides the base functions of the IBM 7067 data channel
//  TDevice provides the base functions for all I/O devices
//  Specific TDevXXX objects descend from TDevice
//------------------------------------------------------------------------------
//  Significant sections of the code in this unit were debugged and enhanced
//  by James Fehlinger. Thanks Jim.
//--------------------------------------------------------------------------
Unit B709Chan;

Interface

Uses SysUtils,Classes,Contnrs,Forms,
     B709Strm, // Customized TMemoryStream class
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Misc, // Miscellaneous utility functions
     B709Trce, // Log/Trace functions
     B709Core; // Core Storage functions

// Channel command opcodes
Const CcIOCD=0; // IOCD
      CcTCH =1; // TCH
      CcIORP=2; // IORP
      CcIORT=3; // IORT
      CcIOCP=4; // IOCP
      CcIOCT=5; // IOCT
      CcIOSP=6; // IOSP
      CcIOST=7; // IOST

// Standard device addresses
Const DaRDR=209;        // Card Reader - octal 321
      DaPUN=225;        // Card Punch - octal 341
      DaLST=241;        // Line Printer - octal 361
      DaTAP=128;        // Tape Drive base address (BCD Mode)- octal 200

// Device Function codes
Type  TOprFunc=(FCXXX,FCBSR,FCBSF,FCWEF,FCREW,FCRUN,FCSDN);

Type TDevice=Class;               // Forward refernce

     // TChannel Object - One instance created for each installed channel
     TChannel=Class
       Constructor Create(CN: Byte);
     Private
       NonXMit:   Boolean;      // Non-Trasfer flag
       PrevLReg:  TAddr;        // Previous location register value
       FEOFInd:   Boolean;      // End of File indicator
       Procedure ReadTransfer;
       Procedure WriteTransfer;
       Procedure ComdTransfer;
       procedure SetEOFInd(const Value: Boolean);
     Public
       ChanNum:  Byte;          // Channel Number 1-8
       ComdReg:  TComd;         // Channel registers
       WordReg:  TAddr;
       AddrReg:  TAddr;
       LocnReg:  TAddr;
       DataReg:  TWord;
       WordCnt:  TAddr;         // Initial WordReg value when command is loaded
       Active:   Boolean;       // Channel is currently active
       CurDevice: TDevice;      // Currently selected device
       // Data Channel Trapping flags
       TrapCtrlInd: Boolean;    // General channel Trap Control Indicator
       LCHTrapEnab: Boolean;    // LCHx (end of command) Trap is enabled
       TCKTrapEnab: Boolean;    // Tape Check Trap is enabled
       EOFTrapEnab: Boolean;    // End of File Trap is enabled
       // Indicators
       LCHInd:   Boolean;       // No LCHc instruction is waiting
       TCKInd:   Boolean;       // Tape check indicator
       BOTInd:   Boolean;       // Begin of Tape indicator
       EOTInd:   Boolean;       // End of Tape indicator
       Function  AddDevice(DT: TDevType; UN: Byte): TDevice;
       Procedure SelectDevice(DV: TDevice; WR: Boolean);
       Procedure LoadChannelCommand(CA: TAddr);
       Procedure StopChannel;
       Procedure ResetChannel;
       Procedure OperateRD;
       Procedure OperateWR;
       Procedure TestEndOfTape;
       Property  EOFInd: Boolean  Read FEOFInd Write SetEOFInd;
     End;

     // TDevice Object - Never instantiated, specific devices decend from this
     TDevice=Class
       Constructor Create(DT: TDevType; CH: TChannel; DA: TAddr);      Virtual;
       Destructor  Destroy;                                            Override;
     Private
       Procedure SetEndOfRec(Const ER: Boolean);
       Procedure SetEndOfFil(Const EF: Boolean);
       function  GetFileNam: String;
       procedure SetFilNam(Const NV: String);
       procedure SetWRMode(Const NV: Boolean);
       procedure SetWRProt(Const NV: Boolean);
       procedure SetBinMode(Const NV: Boolean);
       procedure SaveFile(SS: String);
     Protected
       IOStream:  TB709Stream;              // Device data buffer
       Modified:  Boolean;                  // Device data has been modified
       FEndOfRec: Boolean;                  // End of Record Indicator
       FEndOfFil: Boolean;                  // End of File Indicator
       FBinMode:  Boolean;                  // Operating in Binary mode
       FWRProt:   Boolean;                  // Data is Write Protected
       FWRMode:   Boolean;                  // Write mode
       FFileNam:  String;                   // Name of attached PC file
       FOnFileChange:   TNotifyEvent;       // Callback when file changes
       FOnStatusChange: TNotifyEvent;       // Callback to update status display
       FOnEndOfRecord:  TNotifyEvent;       // Callback when record completed
       Procedure GetNextByte;
//     Function  GetDeviceChar: Byte;
       Procedure PutBytes(Var BF; BC: Integer);
       Procedure ConvertASCBCD;
       Procedure ConvertBCDASC;
       Procedure AttachFile(FN: ShortString; WM: Boolean);             Virtual;
       Procedure DetachFile;                                           Virtual;
       Procedure RewindFile;                                           Virtual;
       Procedure ReloadFile;                                           Virtual;
       Procedure ResetDevice;                                          Virtual;
       Function  DeviceSelect(WR: Boolean): Boolean;                   Virtual;
       Procedure DeviceStop;                                           Virtual;
       Procedure ProcessEndOfRecord;                                   Virtual;
       Procedure WriteEndOfRecord;                                     Virtual;
       Procedure WriteEndOfFile;                                       Virtual;
       Procedure MoveToEndOfRecord;                                    Virtual;
       Procedure UpdateStatus;                                         Virtual;
       Function  ReadWord: TWord;                                      Virtual;
       Function  TestEOR: Boolean;                                     Virtual;
       Procedure WriteWord(WW: TWord);                                 Virtual;
       Procedure AppendRecord(RS: String);                             Virtual;
       Procedure AppendFile(FN: String);                               Virtual;
       Procedure AppendEOF;                                            Virtual;
     Public
       DevType:  TDevType;         // Device class
       Channel:  TChannel;         // Owning channel
       DevAddr:  TAddr;            // Device address
       BinAddr:  TAddr;            // Device address in Binary mode
       LastFile: String;           // Name of last PC file opened
       Selected: Boolean;          // Device is Selected/Active
       CurrByte: Byte;             // Last byte read from IOStream, unmasked
       CurrChar: Byte;             // Above value masked to 6 bits
       WordNum:  Integer;          // Word number within record
       RecdNum:  Integer;          // Current record number
       FileNum:  Integer;          // Current Tape file number
       PendOfFil: Boolean;
       Function  DataPosn: Longint;   // Current size of data buffer
       Function  DataSize: Longint;   // Current position in data buffer
       Procedure SetDataPosition(FP: Integer);
       Function  PeekByte(DP: Integer): Byte;
       Function  DevAdrString: String;
       Procedure DeviceControl(PM: PIOControl);
       Procedure Operate(FC: TOprFunc);                                Virtual;
       Property  EndOfRec: Boolean  Read FEndOfRec  Write SetEndOfRec;
       Property  EndOfFil: Boolean  Read FEndOfFil  Write SetEndOfFil;
       Property  WRMode:   Boolean  Read FWRMode    Write SetWRMode;
       Property  WRProt:   Boolean  Read FWRProt    Write SetWRProt;
       Property  BinMode:  Boolean  Read FBinMode   Write SetBinMode;
       Property  FileNam:  String   Read GetFileNam Write SetFilNam;
       Property  OnFileChange:   TNotifyEvent Read FOnFileChange   Write FOnFileChange;
       Property  OnStatusChange: TNotifyEvent Read FOnStatusChange Write FOnStatusChange;
       Property  OnEndOfRecord:  TNotifyEvent Read FOnEndOfRecord  Write FOnEndOfRecord;
     End;

Var Channels:    Array[1..8] Of TChannel;   // Array of channel objects
    CurChanNum:  Byte;                      // Currently selected channel (0 if none)

Var ChnTrcFlg:   Boolean;                   // Trace channel activity flag
    ChannelStop: Boolean;                   // Flag to stop channels running

Var ChnCheck:    Boolean;                   // Machine Check in Channels

Procedure ResetChannels;
Procedure DropChannels;
Function  GetDevice(DA: TAddr; SE: Boolean): TDevice;
Function  GetNextDevice(Var DI: Integer; DT: TDevType): TDevice;
Function  DefineDevice(DA: Word; DT: TDevType): TDevice;
Procedure DeleteDevice(DA: Word);
Procedure DeleteTypeDevices(DT: TDevType);

Implementation

Uses B709Main, // For caption
     B709CPU,  // For ChanWait
     B709DRDR, // To create TDevRDR
     B709DLST, // To create TDevLST
     B709TAPF; // To create TDevTAP

// Channel Command record
Type  TCommand=Record
        OpCode:   TComd;
        Mnemonic: String[4];
        Desc:     String[40];
      End;

Const MaxComd=8;

// Channel Command table
Var Commands: Array[1..MaxComd] of TCommand;
    LastComd: Integer;

Var DevicesList: TObjectList;   // List of defined devices

Var NextIsPT: Boolean;          // Next channel command is IORP,IORT,IOSP or IOST

//******************************************************************************
//  Non-Method Procedures
//------------------------------------------------------------------------------
// Return command mnemonic string
Function ComdMnemonic(OC: TComd): ShortString;
Var II: Integer;
Begin
  For II:=1 to LastComd do
    With Commands[II] do
      If OC=OpCode then begin
        Result:=Mnemonic+' '+Desc;
        Exit;
      End;
  Result:='???? Unknown Command Code';
End;

//******************************************************************************
//  TChannel Methods
//------------------------------------------------------------------------------
constructor TChannel.Create(CN: Byte);
begin
  ChanNum:=CN;
end;

// Create a device of the specified type, Add to this channel
Function TChannel.AddDevice(DT: TDevType; UN: Byte): TDevice;
Var DA: TAddr;
Begin
  Result:=NIL;
  DA:=ChanNum Shl DaChShft;  // Setup Channel part of device address
  Case DT of
    DT716: Result:=TDevLST.Create(DT,Self,DA Or DaLST);
    DT711: Result:=TDevRDR.Create(DT,Self,DA Or DaRDR);
    DT721: ; //Result:=TDevPUN.Create(DT,Self,DA Or DaPUN,True,FN);
    DT729: Result:=TDevTAP.Create(DT,Self,(DA Or DaTAP)+UN);
    Else   Error('Unknown device type '+IntToStr(Ord(DT)));
  End;
  If Result<>NIL then
    DevicesList.Add(Result);              // Add to global device list
End;

// Reset channel and all attached devices
procedure TChannel.ResetChannel;
Var DI: Integer;
begin
  ComdReg:=0;
  WordReg:=0;
  AddrReg:=0;
  LocnReg:=0;
  DataReg:=0;
  NonXMit:=False;
  Active:=False;
  TrapCtrlInd:=False;
  LCHTrapEnab:=False;
  TCKTrapEnab:=False;
  EOFTrapEnab:=False;
  With DevicesList do                  // Go through device list
    For DI:=0 to Count-1 do
      With TDevice(Items[DI]) do       // For each device:-
        If Channel=Self then           // If this is its channel,
          ResetDevice;                 // reset the device
End;

// Select the specified device, in the specified R/W mode
procedure TChannel.SelectDevice(DV: TDevice; WR: Boolean);
Var SS: String;
begin
  CurDevice:=DV;                     // Set current device
  // Select device. If it works, Set this as the currently selected channel
  With CurDevice do begin
    Selected:=DeviceSelect(WR);
    If Selected then CurChanNum:=ChanNum
                else CurChanNum:=0;
    UpdateStatus;                 // Notify form of new status
    If (TIDEV In TraceRecdFlags) then begin
      If WR then SS:='(WR)' else SS:='(RD)';
      If Selected then SS:='Device Selected '+SS+' '
                  else SS:='Device Select '+SS+' Failed';
      Trace(TIDEV,'Dev   '+DevAdrString+' '+SS);
    End;
  End;
  LCHInd:=False;                  // Clear command trap indicator
end;

// Stop activity on this channel
procedure TChannel.StopChannel;
begin
  CurDevice.DeviceStop;           // Stop currently active device
  Active:=False;                  // Mark channel inactive
  If ChanWait then begin          // Is an LCHA instruction waiting to execute?
    LoadChannelCommand(ChWCAddr); // Yes, Get address of command to run
    ChanWait:=False;              // and the CPU can resume now
  End;
End;

// Transfer - Load next Channel Command from address in waiting LCHA instruction
Procedure TChannel.ComdTransfer;
Begin
  If ChanWait then begin          // Is an LCHA instruction waiting to execute?
    LoadChannelCommand(ChWCAddr); // Yes, Get address of command to run
    ChanWait:=False;              // Let CPU resume now
  End else begin
    StopChannel;                  // No, Disconnect
    LCHInd:=True;                 // Set trap indicator
  End;
End;

// Load Channel Command from specified address, Start channel running
// Can be called by:-
//   "Load Card" and "Load Tape" buttons - With address=0
//   RCHc and LCHc Instructions          - With instruction address field
//   "Transfer in Channel" command       - With command address field
//   Channel I/O "Proceed" commands      - With next location address
//   Channel I/O "Transfer" commands     - With address from waiting LCHc
Procedure TChannel.LoadChannelCommand(CA: TAddr);
Var CW: TWord;
    NC: TComd;
    TS: String;
begin
  // Get specified CCW from core
  CW:=GetCore(CA);
  // Load channel registers from the CCW
  ComdReg:= (CW And COpCMask) Shr COpCShft;
  WordReg:= (CW And COpWMask) Shr COpWShft;
  AddrReg:= (CW And AddrMask);
  NonXMit:=((CW And COpNMask) Shr COpNShft)>0;
  // Indirect CCW Data Address?
  If ((CW And COpFMask)<>0) And                       // Indirect Flag?
     ( Not ((ComdReg In [CcIOCP,CcIOSP]) And          // and not IOCP/IOSP with
           (WordReg>0))) then                         // a zero WordCount?
    AddrReg:=GetCore(AddrReg) And AddrMask;           // Yes, Get indirect addr.
  WordCnt:=WordReg;                                   // Note inital word count
  // Log trace details for command about to be executed
  If (TICCW In TraceRecdFlags) then begin
    TS:=IntToOct(CA,5)+' '+ComdMnemonic(ComdReg);
    If ComdReg=CcTCH then begin
      Trace(TICCW,TS+' to '+IntToOct(AddrReg,5));
    End else Begin
      Trace(TICCW,TS+' CUU='+CurDevice.DevAdrString);
{      If (CW And COpFMask)<>0 then Begin               // Indirect Flag?
        TS:='      CCW='+IntToOct(CW,12)+' '+'*Address='+IntToOct(AddrReg,5)+
            ' WordCount='+IntToStr(WordReg);
      End else Begin}
        TS:='      CCW='+IntToOct(CW,12)+' '+'Address='+IntToOct(AddrReg,5)+
            ' WordCount='+IntToStr(WordReg);
//    End;
      If NonXMit then TS:=TS+' No Transmit';
      Trace(TICCW,TS);
    End;
  End;
//CurDevice.FudgeMode:=WordReg>30;
  PrevLReg:=CA;                   // Note address of this CCW
  LocnReg:=CA+1;                  // Point to next
  // Read ahead for the next command, Note if it's a Proceed Or Transfer
  CW:=GetCore(LocnReg);
  NC:=(CW And COpCMask) Shr COpCShft;
  NextIsPT:=(NC In [CcIORP,CcIORT,CcIOSP,CcIOST]);
  // Start channel operating
//If WordReg>0 then               // Anything to do?
    Active:=True;                 // Yes, Mark channel active
End;

// Set Channel EOF indicator if past EOF on tape
procedure TChannel.TestEndOfTape;
Begin
  With CurDevice do
    If PendOfFil then Begin
      PendOfFil:=False;
      EndOfFil:=True;
      Channel.EOFInd:=True;
    End;
End;

// Channel execution procedures for Read and Write modes
// Calls to these methods are interleaved with execution of CPU instructions
// until the channel disconnects.
procedure TChannel.OperateRD;
Begin
  If ChannelStop then                          // This is to get out of a
    Error ('Channel Stop by Operator');        // channel command loop
  With CurDevice do begin
    // Execute one cycle of the current channel command
    Case ComdReg Of                            // Using current command:-
      // Transfer in Channel
      CcTCH:  If EndOfFil then Begin
                StopChannel;
              End else Begin
                LoadChannelCommand(AddrReg);     // Load next command
              End;
      // I/O Under Count Control and Disconnect
      CcIOCD: Begin         // Read or Write?
                TestEndOfTape;                   // Set Channel EOFInd if past EOF on tape
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If WordReg=0 then begin        // Count exhausted?
                    MoveToEndOfRecord;           // Yes, If not at EOR, get there
                    TestEndOfTape;
                    StopChannel;                 // And Disconnect
                  End else Begin
                    ReadTransfer;                // Read a word
                  End;
                End;
              End;
      // I/O Under Count Control and Proceed
      CcIOCP: Begin         // Read or Write?
                TestEndOfTape;                   // Set Channel EOFInd if past EOF on tape
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If WordReg=0 then begin        // Count exhausted?
                    If EndOfRec And NextIsPT then  // Yes, Was it End of record?
                      StopChannel                   // Yes, Disconnect
                    Else
                      LoadChannelCommand(LocnReg);  // No, Load next command
                  End else Begin
                   ReadTransfer;                  // Read a word
                  End;
                End;
              End;
      // I/O Under Count Control and Transfer
      CcIOCT: Begin         // Read or Write?
                TestEndOfTape;                   // Set Channel EOFInd if past EOF on tape
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If WordReg=0 then Begin        // Count exhausted?
                    If EndOfRec And NextIsPT then  // Yes, Was it End of record?
                      StopChannel                   // Yes, Disconnect
                    Else
                      ComdTransfer                 // Yes, Get next command (LCHA)
                  End else Begin
                   ReadTransfer;                  // Read a word
                  End;
                End;
              End;
      // I/O Until Signal, then Proceed
      CcIOSP: Begin         // Read or Write?
                TestEndOfTape;                   // Set Channel EOFInd if past EOF on tape
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If (WordReg=0) Or EndOfRec then
                    LoadChannelCommand(LocnReg)  // and Load next command
                  Else
                    ReadTransfer;                  // Read a word
                End;
              End;
      // I/O of a Record and Transfer
      CcIORT: Begin         // Read or Write?
                If Not EndOfRec then             // Set Channel EOFInd if past EOF on tape
                  TestEndOfTape;
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If WordReg=0 then begin        // Count exhausted?
                    MoveToEndOfRecord;           // Yes, If not at EOR, get there
                    ComdTransfer;                // And get next command (LCHA)
                  End else Begin
                    If EndOfRec then begin       // Or End of record?
                      EndOfRec:=False;
                      ComdTransfer;
                    End else Begin
                      ReadTransfer;              // No, Read a word
                    End;
                  End;
                End;
              End;
      // I/O of a Record and Proceed
      CcIORP: Begin         // Read or Write?
                If Not EndOfRec then             // Set Channel EOFInd if past EOF on tape
                  TestEndOfTape;
                If EndOfFil then Begin
                  StopChannel;
                End else Begin
                  If WordReg=0 then begin        // Count exhausted?
                    MoveToEndOfRecord;           // Yes, If not at EOR, get there
                    LoadChannelCommand(LocnReg); // And Load next command
                  End else Begin
                    If EndOfRec then begin       // Or End of record?
                      EndOfRec:=False;
                      LoadChannelCommand(LocnReg); // And Load next command
                    End else Begin
                      ReadTransfer;              // No, Read a word
                    End;
                  End;
                End;
              End;
      Else    Error('Unhandled Channel Command: '+ComdMnemonic(ComdReg)+
                                     ' at '+IntToOct(PrevLReg,5));
    End;
  End;
End;

procedure TChannel.OperateWR;
Begin
  If ChannelStop then                          // This is to get out of a
    Error ('Channel Stop by Operator');        // channel command loop
  With CurDevice do begin
    // Execute one cycle of the current channel command
    Case ComdReg Of                            // Using current command:-
      // Transfer in Channel
      CcTCH:  If EndOfFil then Begin
                StopChannel;
              End else Begin
                LoadChannelCommand(AddrReg);     // Load next command
              End;
      // I/O Under Count Control and Disconnect
      CcIOCD: Begin
                If WordReg=0 then begin        // Count exhausted?
                  StopChannel;                 // And Disconnect
                End else Begin
                  WriteTransfer;               // Write a word
                End;
              End;
      // I/O Under Count Control and Proceed
      CcIOCP: Begin
                If WordReg=0 then begin        // Count exhausted?
                  LoadChannelCommand(LocnReg); // And Load next command
                End else Begin
                  WriteTransfer;                 // Write a word
                End;
              End;
      // I/O Under Count Control and Transfer
      CcIOCT: Begin
                If WordReg=0 then begin        // Count exhausted?
                  ComdTransfer;                // Yes, Get next command (LCHA)
                End else Begin
                  WriteTransfer;               // Write a word
                End;
              End;
      // I/O Until Signal, then Proceed
      CcIOSP: Begin
                If WordReg=0 then Begin        // Count exhausted?
                  LoadChannelCommand(LocnReg)  // Start next command
                End else Begin
                  WriteTransfer;               // Write a word
                End;
              End;
      // I/O of a Record and Transfer
      CcIORT: Begin
                If WordReg=0 then begin        // Count exhausted?
                  WriteEndOfRecord;            // Yes, Write end of record
                  ComdTransfer;
                End else Begin
                  WriteTransfer;               // Write a word
                End;
              End;
      // I/O of a Record and Proceed
      CcIORP: Begin
                If WordReg=0 then begin        // Count exhausted?
                  WriteEndOfRecord;            // Yes, Write end of record
                  LoadChannelCommand(LocnReg); // And Load next command
                End else Begin
                  WriteTransfer;                 // Write a word
                End;
              End;
      Else    Error('Unhandled Channel Command: '+ComdMnemonic(ComdReg)+
                                     ' at '+IntToOct(PrevLReg,5));
    End;
  End;
End;

// Transfer one word from I/O device to core storage
procedure TChannel.ReadTransfer;
Var IP: Longint;
    SS: String;
begin
  If WordReg>0 then begin                   // Count exhausted yet?
    // Read word from device, Build trace string
    With CurDevice do begin
      IP:=DataPosn; //If IP<0 then IP:=0;
      DataReg:=ReadWord;
      If (TIIOT In TraceRecdFlags) then begin
        SS:=' .. ';
        If TestEOR then SS[2]:='R';
        If EndOfFil then SS[3]:='F';
        Trace(TIIOT,'Dev   '+DevAdrString+SS+'Read '+Format('%8.8d: ',[IP])+
                    IntToOct(DataReg,12)+' to '+IntToOct(AddrReg,5));
//                    ' Fil#'+IntToStr(FileNum)+' Rec#'+IntToStr(RecdNum)+
//                    ' Word '+IntToStr(WordNum)+' WR='+IntToStr(WordReg));
      End;
    End;
    If Not NonXMit then begin               // Transmit enabled?
      SetCore(AddrReg,DataReg);             // Yes, Store to core
      CorTrcAdr:=-1;                        // Inhibit trace - Done by GetWord
    End;
    // Bump address and count
    Inc(AddrReg); Dec(WordReg);
    If WordReg=0 then
      CurDevice.UpdateStatus;               // Do end of block processing
  End;
End;

// Transfer one word to I/O device from core storage
procedure TChannel.WriteTransfer;
begin
  If WordReg>0 then begin                   // Count exhausted yet?
    DataReg:=GetCore(AddrReg);              // No, Fetch next word
    // Write word to the device, Build trace string
    With CurDevice do begin
      If (TIIOT In TraceRecdFlags) then
        Trace(TIIOT,'Dev   '+DevAdrString+' Write '+Format('%8.8d: ',[DataPosn])+
                    IntToOct(AddrReg,5)+'='+IntToOct(DataReg,12)+
                    ' WR='+IntToStr(WordReg)+' '+WordDump(DataReg));
      WriteWord(DataReg);
    End;
    // Bump address and count
    Inc(AddrReg); Dec(WordReg);
    If WordReg=0 then                       // Exhausted now?
      CurDevice.UpdateStatus;               // Yes, Do end of block processing
  End;
End;

//******************************************************************************
//  TDevice Methods
//------------------------------------------------------------------------------
constructor TDevice.Create(DT: TDevType; CH: TChannel; DA: TAddr);
begin
  DevType:=DT;                         // Load device details
  Channel:=CH;
  DevAddr:=DA;
  IOStream:=TB709Stream.Create;        // Create data buffer
end;

Destructor TDevice.Destroy;
begin
  SaveFile('');
  IOStream.Free;                       // Release data buffer
end;

Function TDevice.DataPosn: Longint;
begin
  Result:=IOStream.Position;
end;

Function TDevice.DataSize: Longint;
begin
  Result:=IOStream.Size;
end;

Function  TDevice.PeekByte(DP: Integer): Byte;
begin
  Result:=IOStream.Peek(DP);
end;

// IOControl message handler. Performs device control function as per message
// Invoked by main form when IOControl message is received
Procedure TDevice.DeviceControl(PM: PIOControl);
Var SS: String;
Begin
  With PM^ do begin
    // Set null string, if string parameter is NIL
    If StrParm=NIL then SS:='' else SS:=StrParm^;
    // Select required device function for message action code
    Case Action Of
      FCOPEN:   AttachFile(SS,False);
      FCCLOSE:  DetachFile;
      FCREWIND: RewindFile;
      FCRELOAD: ReloadFile;
      FCSEEK:   SetDataPosition(IntParm);
      FCWRPROT: WRProt:=IntParm=0;
      FCLODREC: AppendRecord(SS);
      FCLODFIL: AppendFile(SS);
      FCLODEOF: AppendEOF;
      FCSAVE:   SaveFile(SS);
      Else      IOCheck('Unknown device control action code: '+IntToStr(Action),0);
    End;
  End;
  UpdateStatus;
End;

// Associate a PC file with the device and load file data
// Invoked by DeviceControl method or Reload method
procedure TDevice.AttachFile(FN: ShortString; WM: Boolean);
Const BC=10030;
Var BB: Array[1..BC] Of Byte;
begin
  FN:=UpperCase(FN);
  // If no file name, Assume it's a scratch tape, set write mode
  If FN='' then begin
    FN:='SCRATCH'+DevAdrString+'.BIN';
    WM:=True;
    // Temp fix - Scratch tapes must have some data (This can be eliminated)
    FillChar(BB,BC,#$30);
    IOStream.Write(BB,BC);
  End else Begin
    // If Read mode, Check file exists, Load to data buffer
    If Not WM then begin
      If Not FileExists(FN) then
        Error('No such input file:-'+#$D+FN);
      IOStream.LoadFromFile(FN);
    End;
  End;
  FileNam:=FN;                    // Note PC file name
  LastFile:=FN;                   // Remember for reload also
  WRMode:=WM;                     // Set Write mode
//  WRProt:=Not WM;                 // If Read mode, Protect it
  RewindFile;                     // Reset position and flags etc.
  // Trace it and notify associated form of new file loaded
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' File "'+FFileNam+'" attached');
  If Assigned(FOnFileChange) then FOnFileChange(Self);
End;

// Remove device's association with a PC file afer saving any modified data
// Invoked by DeviceControl method or Reload method
procedure TDevice.DetachFile;
begin
  SaveFile('');
  IOStream.Clear;               // Empty memory image
  // Trace and notify associated form of no file
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' File "'+FFileNam+'" detached');
  FileNam:='';
  If Assigned(FOnFileChange) then FOnFileChange(Self);
End;

// Select device for subsequent channel operations
// Invoked via Channel's SelectDevice method by Load button or RDS/WRS instructions
Function TDevice.DeviceSelect(WR: Boolean): Boolean;
Var SS: String;
begin
  // Check media loaded
  If FileNam='' then begin
    If WR then SS:='Write' else SS:='Read';
    Error(SS+' Select for device with no file attached: '+DevAdrString);
  End;
  // Check R/W mode is compatible with protect status
  Result:=Not (FWRProt And WR);
  If Not Result then begin
    Error('Write Select on Write Protected device: '+DevAdrString);
  End;
  If EndOfFil then Begin
    Inc(FileNum);                           // Count file
    RecdNum:=0;                             // Reset record count
    EndOfFil:=False;                        // Reset EOF flag
  End;
  If EndOfRec then Begin
    Inc(RecdNum);                           // Count record
    EndOfRec:=False;                        // Reset EOR flag
  End;
  FWRMode:=WR;               // Set new R/W mode
End;

// De-select device (stop it)
procedure TDevice.DeviceStop;
begin
  Selected:=False;
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' Stopped');
  UpdateStatus;                  // Notify display form
end;

// Re-assosiacte last PC file used with device
procedure TDevice.ReLoadFile;
begin
  If LastFile<>'' then
    AttachFile(LastFile,WRMode);
End;

// Save modified file data
procedure TDevice.SaveFile(SS: String);
begin
  If Not WRMode then Exit;        // Exit if not in write mode,
  If Not Modified then Exit;      // or not modified
  If SS='' then SS:=FFileNam;     // No name given? Use current name
  IOStream.SaveToFile(SS);
  Modified:=False;
End;

procedure TDevice.ResetDevice;
begin
  UpdateStatus;                  // Notify display form
End;

// Reset device data buffer pointer to start of PC data file,
// Prime next byte for next read call
procedure TDevice.RewindFile;
begin
  If Not WRMode then begin
    IOStream.Seek(0,soFromBeginning);       // Go to start of memory image
    GetNextByte;                            // Prime CurrChar
//; SetDataPosition(0);                     // Go to start of memory image
  End;
  FileNum:=0;                               // Reset stuff
  RecdNum:=0;
  WordNum:=0;
  UpdateStatus;                             // Notify display form
  EndOfFil:=False;
  PendOfFil:=False;
End;

// Position device data buffer pointer to specified offset
procedure TDevice.SetDataPosition(FP: Integer);
begin
  If FP<0 then FP:=0;
//If FP=-1 then FP:=0;       // Allow -1 to indicate start
//If FP<0 then               // More negative is error
//  Error('Invalid arg to SetDataPosition: '+IntToSTr(FP));
  IOStream.Seek(FP,SOFromBeginning);        // Seek to new posn
//  UpdateStatus;                             // Notify display form
  GetNextByte;                              // Prime CurrChar
  UpdateStatus;                             // Notify display form
End;

// Update various status values, invoke any assigned callback to display status
procedure TDevice.UpdateStatus;
begin
//EndOfFil:=DataPosn>=DataSize;
//If (Not WRMode) And EndOfFil then
//  Channel.EOFInd:=True;                   // EOF if past EOF on read
  // Notify associated form of new status
  If Assigned(FOnStatusChange) then FOnStatusChange(Self);
End;

// Read next byte from device's data buffer, update various status values
Procedure TDevice.GetNextByte;
Var BC: Longint;
begin
  Try
    If DataPosn>=DataSize then Begin
      EndOfFil:=True;
      EndOfRec:=True;
      CurrByte:=0;                                 // Set zero
    End else Begin
      BC:=IOStream.Read(CurrByte,1);               // Get next byte,
      If BC=0 then begin
        EndOfFil:=True;                            // No more, Set EOF, etc.
        EndOfRec:=True;
        CurrByte:=0;                               // Set zero
      End;
    End;
  Except
    On E: Exception do begin
      Error('I/O Error: '+E.Message);
    End;
  End;
  CurrChar:=CurrByte And $3F;
End;
{
// Read next byte from device's data buffer, update various status values
Function TDevice.GetDeviceChar: Byte;
Var BC: Longint;
begin
  Try
    BC:=IOStream.Read(CurrByte,1);               // Get next byte,
    If BC=0 then begin
      EndOfFil:=True;                            // No more, Set EOF, etc.
      EndOfRec:=True;
      CurrByte:=0;                               // Return zero
      Result:=0;
      Error('Read past end of data');
      Exit;
    End;
  Except
    On E: Exception do begin
      Error('I/O Error: '+E.Message);
    End;
  End;
  Result:=CurrByte And $3F;            // And return it masked to 6 bits
End;
}
// Write block of bytes to device's data buffer
Procedure TDevice.PutBytes(Var BF; BC: Integer);
Begin
  IOStream.WriteBuffer(BF,BC);         // Write bytes to buffer
  Modified:=True;
End;

// Invoke any callback assigned for end of record condition
procedure TDevice.ProcessEndOfRecord;
begin
  // Update status info
  UpdateStatus;
  // Notify associated form that record has been processed
  If Assigned(OnEndOfRecord) then OnEndOfRecord(Self);
End;

// Form device address string
Function TDevice.DevAdrString: String;
Begin
  If BinMode then Result:=IntToOct(DevAddr+$10,5)
             else Result:=IntToOct(DevAddr,5);
End;

// Return PC file name, without path information
function TDevice.GetFileNam: String;
begin
  Result:=ExtractFileName(FFileNam);
end;

// Set name of associated PC data file
procedure TDevice.SetFilNam(Const NV: String);
begin
  FFileNam:=NV; UpdateStatus;
end;

// Set device's R/W mode
procedure TDevice.SetWRMode(Const NV: Boolean);
begin
  FWRMode:=NV;  UpdateStatus;
end;

// Set device's Write protect status
procedure TDevice.SetWRProt(Const NV: Boolean);
begin
  FWRProt:=NV;  UpdateStatus;
end;

// Set device's operating mode BCD or binary
procedure TDevice.SetBinMode(Const NV: Boolean);
begin
  FBinMode:=NV; UpdateStatus;
end;

// Set end of record status
procedure TDevice.SetEndOfRec(Const ER: Boolean);
begin
  FEndOfRec:=ER;
end;

// Set end of file status
procedure TDevice.SetEndOfFil(Const EF: Boolean);
begin
  FEndOfFil:=EF;
end;

// Called by descendant device instance when reading a word,
// Sets end of file indicator in associated channel
Function TDevice.ReadWord: TWord;
begin
  Result:=0;                                     // Always overridden
end;

// The follwoing are all dummy virtual methods, overriden in descendants
procedure TDevice.Operate(FC: TOprFunc);
begin
end;

Procedure TDevice.WriteWord(WW: TWord);
begin
end;

procedure TDevice.MoveToEndOfRecord;
begin
End;

procedure TDevice.WriteEndOfRecord;
begin
end;

procedure TDevice.WriteEndOfFile;
begin
end;

procedure TDevice.ConvertASCBCD;
begin
end;

procedure TDevice.ConvertBCDASC;
begin
end;

procedure TDevice.AppendRecord(RS: String);
begin
end;

procedure TDevice.AppendFile(FN: String);
begin
end;

procedure TDevice.AppendEOF;
begin
end;

//******************************************************************************
//  Channel and Device Management procedures
//------------------------------------------------------------------------------
// Reset each channel and all its attached devices
Procedure ResetChannels;
Var CI: Byte;
    CH: TChannel;
begin
  ChnCheck:=False;                     // Reset check flag
  ChannelStop:=False;                  // And stop flag
  For CI:=1 to LastChan do begin
    CH:=Channels[CI];
    If CH<>NIL then                    // Reset each channel
      CH.ResetChannel;                 // (Also resets all its devices)
  End;
  CurChanNum:=0;                          // No channel selected
End;

// Free all channels
Procedure DropChannels;
Var CI: Byte;
    CH: TChannel;
begin
  For CI:=1 to 8 do begin
    CH:=Channels[CI];
    If CH<>NIL then
      CH.Free;
  End;
End;

// See if device exists
Function DeviceExists(DA: TAddr): Boolean;
Var DI: Integer;
    DV: TDevice;
Begin
  Result:=True;
  With DevicesList do
    For DI:=0 to Count-1 do begin
      DV:=TDevice(Items[DI]);
      If DA=DV.DevAddr then Exit;          // Standard address specified?
      If DA=DV.DevAddr+$10 then Exit;      // Binary mode address specified?
    End;
  Result:=False;
End;

// Find device specified
Function GetDevice(DA: TAddr; SE: Boolean): TDevice;
Var DI: Integer;
Begin
  With DevicesList do
    For DI:=0 to Count-1 do begin
      Result:=TDevice(Items[DI]);
      With Result do begin
        If DA=Result.DevAddr then begin     // Standard address specified?
          BinMode:=False; Exit              // Set selected mode and exit found
        End;
        If DA=Result.BinAddr then begin     // Binary mode address specified?
          BinMode:=True; Exit;              // Set selected mode and exit found
        End;
      End;
    End;
  If SE then
    Error('No device at defined Address='+IntToOct(DA,5));
  Result:=NIL;
End;

// Return next device of specified type from list of devices
Function GetNextDevice(Var DI: Integer; DT: TDevType): TDevice;
Begin
  With DevicesList do
    Repeat
      If DI>=Count then begin
        Result:=NIL; Exit;
      End else Begin
        Result:=TDevice(Items[DI]);
        Inc(DI);
        If (DT=DTXXX) Or (Result.DevType=DT) then Exit;
      End;
    Until False;
End;

// Define a new device, add to list of devices
Function  DefineDevice(DA: Word; DT: TDevType): TDevice;
Var CN,UN: Byte;
Begin
  CN:=(DA And DaChMask) Shr DaChShft;
  If CN>LastChan then Error('No channel for device define: '+IntToOct(DA,5));
  UN:=DA And DaUnMask;
  Result:=Channels[CN].AddDevice(DT,UN);
End;

// Remove device at specified address from list of devices
Procedure DeleteDevice(DA: Word);
Var DV: TDevice;
Begin
  DV:=GetDevice(DA,True);
  DevicesList.Remove(DV);        // Remove form global list
End;

// Remove all devices of specified type from list of devices
Procedure DeleteTypeDevices(DT: TDevType);
Var DI: Integer;
    DV: TDevice;
    DF: Boolean;
Begin
  With DevicesList do
    Repeat
      DF:=False;
      For DI:=0 to Count-1 do begin
        DV:=TDevice(Items[DI]);
        If DV.DevType=DT then begin
          Remove(DV);        // Remove form global list
          DF:=True;
          Break;
        End;
      End;
    Until Not DF;
End;

//------------------------------------------------------------------------------
// Load channel command table
//------------------------------------------------------------------------------
// Store command details to channel command table
Procedure SetupComd(OC: TComd; MN,ID: ShortString);
Begin
  Inc(LastComd);
  With Commands[LastComd] do begin
    OpCode:=OC; Mnemonic:=MN; Desc:=ID;
  End;
End;

// Load details for each channel command
Procedure SetupCommands;
Begin
  LastComd:=0;
  SetUpComd(CcIOCD,'IOCD','I/O Under Count Control and Disconnect'); // 0
  SetUpComd(CcTCH, 'TCH', 'Transfer in Channel');                    // 1
  SetUpComd(CcIORP,'IORP','I/O of a Record and Proceed');            // 2
  SetUpComd(CcIORT,'IORT','I/O of a Record and Transfer');           // 3
  SetUpComd(CcIOCP,'IOCP','I/O Under Count Control and Proceed');    // 4
  SetUpComd(CcIOCT,'IOCT','I/O Under Count Control and Transfer');   // 5
  SetUpComd(CcIOSP,'IOSP','I/O Until Signal, then Proceed');         // 6
  SetUpComd(CcIOST,'IOST','I/O Until Signal, then Transfer');        // 7
End;
{
Procedure DumpDevs;
Var DI: Integer;
    DV: TDevice;
Begin
  With DevicesList do
    For DI:=0 to Count-1 do begin
      DV:=TDevice(Items[DI]);
      Trace(DevTypeStrings[DV.DevType]+' '+DV.DevAdrString);
    End;
End;
}
procedure TChannel.SetEOFInd(const Value: Boolean);
Var SS: String;
begin
  If Value<>FEOFInd then begin
    FEOFInd:=Value;
    If FEOFInd then SS:='SET'
               else SS:='RESET';
    If (TICCW In TraceRecdFlags) then
      Trace(TICCW,'Channel '+Chr($40+ChanNum)+' EOF Indicator '+SS);
  End;
end;

Function TDevice.TestEOR: Boolean;
begin
  Result:=False; // This gets Overriden
end;

Initialization
  SetupCommands;   // Load channel command table
  DevicesList:=TObjectList.Create;
Finalization
//DumpDevs;
  DevicesList.Free;
End.

