//******************************************************************************
//  IBM 7094 Emulator - Log/Trace form
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit provides Core Storage Display/Alter  functions
//------------------------------------------------------------------------------
Unit B709Core;

Interface

Uses Windows, Messages, SysUtils, Classes, Graphics,
     Controls, Forms, Dialogs,StdCtrls, ExtCtrls,
     B709Defs, // General definitions
     B709Cmps, // Display Components
     B709Misc, // Miscellaneous utility functions
     B709Plot, // Core Usage Plot display
     B709Stop, // Process Stops Panel
     B709Trce; // Log/Trace functions

Const Heading='IBM 7302 Core Storage';
      MaxCore=32768;

Type
  TCoreForm = class(TForm)
    PACtrl: TPanel;
    Label1: TLabel;
    EBAddr: TEdit;
    PBBeg: TButton;
    PBEnd: TButton;
    BIHexMode: TButnInd;
    PACore: TPanel;
    Timer: TTimer;
    Procedure Paint;                                                   Override;
    procedure FormCreate(Sender: TObject);
    procedure BIHexModeClick(Sender: TObject);
    procedure EBAddrKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure EBAddrKeyPress(Sender: TObject; var Key: Char);
    procedure PBBegClick(Sender: TObject);
    procedure PBEndClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
  private
    HexMode:  Boolean;
    BaseAddr,
    EndAddr:  Integer;
    TxtWid,
    TxtHgt:   Integer;
    LineCnt:  Integer;
    DrwBitMap: TBitMap;
    DrwCanvas: TCanvas;
    DrwRect:   TRect;
    function  Word2Str(IV: TWord): ShortString;
    Procedure CoreUpDate(MA: TAddr);
    procedure SetBaseAddr(BA: Integer);
  public
    Procedure Initialize;
  end;

Procedure CoreInit(CS: Word);
Procedure CoreClear;
Function  GetCore(AD: TAddr): TWord;
Procedure SetCore(AD: TAddr; WV: TWord);
Procedure CoreRefresh;

Var CoreForm:  TCoreForm;
    CorTrcAdr: Integer;                // Trace core storage change address
    CoreSize:  TAddr;
    CoreWrap:  TAddr;

Implementation

Uses B709Main, // For main form position
     B709CPU;  // For InstCtr

{$R *.DFM}

Var Core:     Array[0..MaxCore] Of TWord;
    CoreFlag: Boolean;

// Reset and Clear core storage
Procedure CoreInit(CS: Word);
begin
  CS:=(CS Div 4096)*4096;
  CoreSize:=CS;
  CoreWrap:=CS-1;
  FillChar(Core,SizeOf(Core),#0);
  If (TIGEN In TraceRecdFlags) then
    Trace(TIGEN,'Core Size: '+IntToStr(CoreSize)+' words');
  CoreClear;
end;

Procedure CoreClear;
begin
  FillChar(Core,SizeOf(Core),#0);
  CoreForm.CoreUpDate($8000);                     // Update display
  PlotForm.PlotClear;
end;

// Return core storage word from specified address
function GetCore(AD: TAddr): TWord;
begin
  AD:=AD And CoreWrap;
  If PlotDisp then PlotForm.PlotRead(AD);
  Result:=Core[AD] And Wd36Mask;
end;

// Write core storage word to specified address
procedure SetCore(AD: TAddr; WV: TWord);
Var SS: String;
begin
  AD:=AD And CoreWrap;                      // Ensure address is valid
  If PlotDisp then PlotForm.PlotWrite(AD);
  // Check for Store to Address Stop
  If AD=AlterStopAddr then begin
    SS:='Store to Address '+IntToOct(AD,5)+' ('+IntToOct(WV,12)+')';
    If AlterStopStop then                   // CPU stop reqd on Address stop?
      SetStopPending(SS);                   // Yes, tell CPU to stop
    If AlterStopTrace then begin            // Trace reqd on Address stop?
      MainForm.SetTraceOn;                  // Yes, enable tracing
      Trace(TIMEM,SS);                      // Write initial entry
    End;
  End;
  // Write data to core, Set address for tracing, Update core display
  Core[AD]:=(WV And Wd36Mask);
  CorTrcAdr:=AD;
  CoreForm.CoreUpDate(AD);
end;

procedure TCoreForm.FormCreate(Sender: TObject);
begin
  FormHandles[FICore]:=Handle;
  Caption:=Heading;
  BaseAddr:=0;
  HexMode:=False;
  CoreFlag:=True;
End;

procedure TCoreForm.FormDestroy(Sender: TObject);
begin
  DrwBitMap.Free;
end;

Procedure TCoreForm.Initialize;
begin
//Left:=0;
//Top:=MainForm.Top+MainForm.Height;
//Height:=Screen.Height-Top-2-20;
end;

procedure TCoreForm.PBBegClick(Sender: TObject);
begin
  SetBaseAddr(0);
end;

procedure TCoreForm.PBEndClick(Sender: TObject);
begin
  SetBaseAddr(CoreSize-(LineCnt*4));
end;

procedure TCoreForm.BIHexModeClick(Sender: TObject);
begin
  With BIHexMode do begin
    State:=Not State;
    HexMode:=State;
  End;
  If HexMode then EBAddr.MaxLength:=4
             else EBAddr.MaxLength:=5;
  SetBaseAddr(BaseAddr);
end;

procedure TCoreForm.EBAddrKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  Case Key of
    VK_Prior:  SetBaseAddr(BaseAddr-(LineCnt*4));
    VK_Next:   SetBaseAddr(BaseAddr+(LineCnt*4));
    Else       Exit;
  End;
End;

Procedure TCoreForm.EBAddrKeyPress(Sender: TObject; var Key: Char);
Var BA: Integer;
Begin
  Case Key of
    #8,
    '0'..'9': Begin End;
    'a'..'f',
    'A'..'F': If Not HexMode then Key:=#0;
    #$D:      Begin
                If HexMode then BA:=StrToInt('$'+Trim(EBAddr.Text))
                           else BA:=OctStrToInt(Trim(EBAddr.Text));
                SetBaseAddr(BA);
              End;
    Else      Key:=#0;
  End;
  Inherited;
end;

procedure TCoreForm.FormResize(Sender: TObject);
begin
  With TCanvasPanel(PACore).Canvas.TextExtent('X') do begin
    TxtWid:=CX; TxtHgt:=CY;
  End;
  LineCnt:=ClientHeight Div TxtHgt;
  ClientHeight:=LineCnt*TxtHgt;
  If DrwBitMap=NIL then           // Create off-screen bitmap
    DrwBitMap:=TBitMap.Create;
  With DrwBitMap do begin
    Width:=PACore.Width;          // Set to display area size,
    Height:=PACore.ClientHeight;  // reduced by scrollbar width
    DrwCanvas:=Canvas;            // Make it available for drawing on
  End;
  With DrwCanvas do begin
    DrwRect:=ClientRect;          // Make size available
    Font.Assign(PACore.Font);     // Set to same font as panel
  End;
  Invalidate;
end;

procedure TCoreForm.Paint;
Var LA: TAddr;
    LC,YP: Integer;
begin
  Inherited;
  // Work with offscreen bitmap's canvas
  With DrwCanvas do begin
    // Clear it
    Brush.Color:=CLWhite;
    FillRect(DrwRect);
    LA:=BaseAddr And AddrMask;
    YP:=0;
    For LC:=1 to LineCnt do begin
      Brush.Color:=ClWhite; TextOut(000,YP,Num2Str(LA,5,HexMode));
      Brush.Color:=ClAqua;  TextOut(045,YP,Word2Str(Core[LA+0]));
      Brush.Color:=ClWhite; TextOut(160,YP,Word2Str(Core[LA+1]));
      Brush.Color:=ClAqua;  TextOut(275,YP,Word2Str(Core[LA+2]));
      Brush.Color:=ClWhite; TextOut(390,YP,Word2Str(Core[LA+3]));
//    Brush.Color:=ClAqua;
      Inc(LA,4); Inc(YP,TxtHgt);
    End;
    EndAddr:=LA-4;
  End;
  TCanvasPanel(PACore).Canvas.Draw(0,0,DrwBitMap);
End;

procedure TCoreForm.TimerTimer(Sender: TObject);
begin
  Invalidate;
end;

Procedure TCoreForm.SetBaseAddr(BA: Integer);
Var PS: Integer;
begin
  PS:=LineCnt*4;
  BaseAddr:=BA And $7FF8;
  If BaseAddr+PS>CoreSize then BaseAddr:=CoreSize-PS+8;
  If HexMode then EBAddr.Text:=IntToHex(BaseAddr,4)
             else EBAddr.Text:=IntToOct(BaseAddr,5);
  Invalidate;
End;

Function TCoreForm.Word2Str(IV: TWord): ShortString;
Var DC,CV,ND,RC: Byte;
Begin
  Result:='';
  If HexMode then begin
    ND:=9; RC:=4;
    For DC:=1 to ND do begin
      CV:=IV And $F;
      If CV>9 then Inc(CV,7);
      Result:=Chr(CV+$30)+Result;
      IV:=IV Shr 4;
      If (DC Mod RC)=0 then Result:=' '+Result;
    End
  End else Begin
    ND:=12; RC:=3;
    For DC:=1 to ND do begin
      Result:=Chr((IV And 7)+$30)+Result;
      IV:=IV Shr 3;
      If (DC Mod RC)=0 then Result:=' '+Result;
    End;
  End;
  Result:=Trim(Result);
End;

// Set Display refresh flag if specified address is visible (or if forced)
procedure TCoreForm.CoreUpDate(MA: TAddr);
begin
  // If updated address is visible
  If (MA=$8000) Or                               // Force update?
     ((MA>=BaseAddr) And (MA<=EndAddr)) then     // Address in display range?
    CoreFlag:=True;                              // Set for refresh
end;

procedure CoreRefresh;
begin
  If CoreFlag then begin                         // Refresh required?
    CoreForm.Invalidate;                         // Yes, Force repaint
    CoreFlag:=False;                             // and reset refresh flag
  End;
end;

End.
