//******************************************************************************
//  IBM 7094 Emulator - Card Reader device
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit defines the TDevRDR object which descends from TDevice
//  TDevRDR provides the functions for the IBM 711 card reader
//  There is only one instance of TDevRDR, "RDRDev", which has an
//  associated form window "ReaderForm" to display its data.
//  This unit also contains the definition and methods of ReaderForm.
//------------------------------------------------------------------------------
Unit B709DRDR;

Interface

Uses SysUtils,Windows,Forms,Messages, StdCtrls,
     Controls, Classes, ExtCtrls, Dialogs,
     B709Cmps, // Display Components
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Misc, // Miscellaneous utility functions
     B709Trce, // Log/Trace functions
     B709Chan; // I/O Channel and Device functions

Type TDevRDR=Class(TDevice)
       Constructor Create(DT: TDevType; CH: TChannel; DA: TAddr);      Override;
       Destructor  Destroy;                                            Override;
     Private
       CardImage: Array[1..80] of Word;
       Procedure ConvertASCIIFile;
       Function  ReadColumn: Word;
       Procedure ReadNextCard;
     Protected
//;     EndOfCard:  Boolean;
       procedure AttachFile(FN: ShortString; WM: Boolean);             Override;
       Procedure MoveToEndOfRecord;                                    Override;
//;       Function  TestEOR: Boolean;                                     Override;
       Function  ReadWord: TWord;                                      Override;
       Procedure AppendRecord(SS: String);                             Override;
       Procedure AppendFile(FN: String);                               Override;         
       Procedure AppendEOF;                                            Override;
       Function  DeviceSelect(WR: Boolean): Boolean;                   Override;
     Public
     End;

Const Heading='IBM 711 Card Reader';

Type TDispMode=(DMBCD,DMOCT,DMHEX);

Type
  TReaderForm = class(TForm)
    Panel1: TPanel;
    Label1: TLabel;
    Deck: TUnitRecordDisplay;
    RBHex: TRadioButton;
    RBOct: TRadioButton;
    RBBCD: TRadioButton;
    CBSize: TCheckBox;
    LACardNum: TLabel;
    Label2: TLabel;
    LACardCount: TLabel;
    CBCBM: TCheckBox;
    BIReload: TButnInd;
    BIOpen: TButnInd;
    BIClear: TButnInd;
    procedure FormResize(Sender: TObject);
    procedure CBSizeClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure DeckPositionChange(SE: TObject; RN: Word);
    procedure RBBOHClick(Sender: TObject);
    procedure BIOpenClick(Sender: TObject);
    procedure BIReloadClick(Sender: TObject);
    procedure BIClearClick(Sender: TObject);
  private
    CardsList: TStringList;       // Card data display strings
    BotCard:   Integer;           // Number of first displayed card
    DispMode:  TDispMode;         // Current display format
    procedure LoadCardsList;
    procedure ShowFileDetails(Sender: TObject);
    Procedure RefreshDisplay(Sender: TObject);
    Procedure CardReadIn(Sender: TObject);
  public
    Procedure Initialize;
    Procedure Clear;
  end;

Var RDRDev:     TDevRDR;               // Device instance
    ReaderForm: TReaderForm;           // Form instance

Implementation

Uses B709Main; // For form positioning

{$R *.DFM}

// Column Binary card record found in *.CBN files
// 80 columns of two 6 bit groups:-
// First byte of each pair:  Rows 9-4 in bits 5-0
// Second byte of each pair: Rows 3-0,11,12 in bits 5-0
// Or is that:-
// First byte of each pair:  Rows 4-9 in bits 5-0
// Second byte of each pair: Rows 12,11,3-0 in bits 5-0
// 12 11 0 1 2 3 4 5 6 7 8 9     - Card columns
//               2 1 8 4 2 1     - Bit positions in first byte
//  2  1 8 4 2 1                 - Bit positions in second byte
Type TCBNRec=Array[1..160] Of Byte;

// EOF Card is:-
// 7,8    punch in cols 1 & 2
// 12,1,4 punch in cols 3 & 4
Const EOFCard: Array[1..8] Of Byte=(
               $06,$00,         // 7,8 punch in col 1
               $06,$00,         // 7,8 punch in col 2
               $20,$24,         // 12,1,4 punch in col 3
               $20,$24);        // 12,1,4 punch in col 4

// Translate an ASCII string into a Column Binary card record
Function ASCStrToCBNRec(SS: String): TCBNRec;
Var CI,BC,CB: Byte;
    CW: Word;
Begin
  SS:=Pad(SS,80);
  BC:=1;                                    // Reset output column ptr
  For CI:=1 to 80 do begin
    CB:=Ord(SS[CI]);                        // Get next input char value
    If CB>=$60 then CB:=CB And $5F;         // Make it UpperCase
    CB:=TabASCToBCD[CB];                    // Convert to BCD
    CW:=TabBCDtoCRD[CB];                    // Convert to CRD format
    Result[BC]:=(CW Shr 6) And $3F;         // Split 12 CRD bits into
    Result[BC+1]:=CW And $3F;               // two 6 bit BCD chars
    Inc(BC,2);                              // Point to next output column
  End;
End;

//******************************************************************************
//  TDevRDR Methods
//------------------------------------------------------------------------------
constructor TDevRDR.Create(DT: TDevType; CH: TChannel; DA: TAddr);
begin
  Inherited Create(DT,CH,DA);
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' 711 Card Reader defined');
end;

Destructor TDevRDR.Destroy;
begin
  Inherited;
end;

// Associate device with PC file, Load file data to buffer
procedure TDevRDR.AttachFile(FN: ShortString; WM: Boolean);
Begin
  // Load device data buffer from file
  Inherited;
  // If not *.CBN, assume ASCII format and translate
  // device data buffer to CBN file format
  If ExtractFileExt(FileNam)<>'.CBN' then begin
    ConvertASCIIFile;
    // Notify associated form of altered file data
    If Assigned(FOnFileChange) then FOnFileChange(Self);
  End;
End;

// Select device for I/O, setting R/W mode
Function  TDevRDR.DeviceSelect(WR: Boolean): Boolean;
begin
  // Check not trying to write
  If WR then Error('Write to Card Reader!?');
  // Select device, Set R/W mode, Etc.
  Result:=Inherited DeviceSelect(WR);
//;  EndOfCard:=False;           // No I/O done yet, so EOR is false
End;

// Read word from device
Function TDevRDR.ReadWord: TWord;
Var BI,BB,BC: Byte;
    WB: TWord;
    WW: TWord;
Begin
  Result:=0;
  If WordNum=0 then Begin                     // Nead a new card?
    ReadNextCard;
    If EndOfFil then Begin                    // Nothing to read?
      Channel.EOFInd:=True;
      Exit;
    End;
  End;
//;  Result:=Inherited ReadWord;
//;  If EndOfFil then Exit;                    // Nothing to read?
//;  If WordNum=0 then ReadNextCard;           // Nead a new card?
  If Odd(WordNum) then BB:=36 else BB:=0;   // Select left or right part of card
  BI:=WordNum Div 2;                        // Set required bit number
  WW:=0;
  For BC:=1 to 36 do begin                  // Bit counter
    WB:=((CardImage[BB+BC] Shr BI) And 1);  // Get bit
    WW:=(WW Shl 1) Or WB;                   // Shift into result
  End;
  Result:=WW And Wd36Mask;                  // Mask to full word
  Inc(WordNum);                             // Count words read from card
  If WordNum=24 then begin                  // Done all?
    EndOfRec:=True;                         // Yes, Set end of record
//;    EndOfCard:=True;                         // Yes, Set end of record
    WordNum:=0;
//  UpdateStatus;                           // Set EOF status now
  End;
End;

// Read CBN format card image to 711 card reader format in CardImage
Procedure TDevRDR.ReadNextCard;
Var BI: Byte;
    CW: Word;
    BF: Boolean;
Begin
  EndOfRec:=False;                          // Start of a new record
  BF:=True;                                 // Reset blank card flag
  // Read 160 bytes, drop record bit ($80 in first byte) and parity bits ($40)
  // Map 160*6 bits chars into 80*12 bit words (a full 80 column card image)
  For BI:=1 to 80 do begin
    CW:=ReadColumn;                         // Get column word
    If CW<>0 then                           // Note if card is blank
      BF:=False;
    CardImage[BI]:=CW;                      // Store column to card
  End;
  Inc(RecdNum);                             // Count records
  // Update associated display
  ProcessEndOfRecord;
  If BF then                                // Set EOF if card is blank
    EndOfFil:=True;                         // But this can't be right?!
//  If CompareMem(@CardImage,@EOFCard,8) then // Set EOF if card is EOF punched
//    EndOfFil:=True;
  If (TIDEV In TraceRecdFlags) then
    Trace(TIDEV,'Dev   '+DevAdrString+' Read Card');
End;

// Read next pair of CBN bytes, Form into 12-bit column data
Function TDevRDR.ReadColumn: Word;
Var W1,W2: Word;
Begin
  W1:=CurrChar; GetNextByte;   // Get rows 9-4
  W2:=CurrChar; GetNextByte;   // Get rows 3-12
  Result:=(W1 Shl 6) Or W2;            // Combine into 12 bits
End;

// Convert loaded file stream from ASCII text to CBN file format
Procedure TDevRDR.ConvertASCIIFile;
Var MS: TMemoryStream;
    RS: String;
    RC: TCBNRec;
    BC: Byte;
Begin
  MS:=TMemoryStream.Create;                 // Create temp image for conversion
  // Read ASCII lines, Convert to CBN file format
  While Not EndOfFil do begin
    // Read text line from device data buffer
    RS:=StringOfChar(' ',80); BC:=0;
    Repeat
      If CurrByte=$D then GetNextByte;      // Ignore CR's
      If (CurrByte<>$A) And                 // Use LF, Or line too long,
        (BC<80) then begin                  // as end of line
        Inc(BC);                            // Bump char ptr
        RS[BC]:=Chr(CurrByte);              // Store char to line string
        GetNextByte;                        // Get the next
      End else Begin
        GetNextByte;                        // EOL, Eat the LF char
        RC:=ASCStrToCBNRec(RS);             // Translate line to CBN record
        MS.Write(RC,160);                   // Write it to work stream
        Break;                              // Start new card
      End;
    Until EndOfFil;
  End;
  IOStream.LoadFromStream(MS);    // Replace ASCII data with CBN data
  MS.Free;
  RewindFile;                     // Setup to use it
End;

procedure TDevRDR.MoveToEndOfRecord;
begin
  WordNum:=0;                     // Force ReadCard on next ReadWord
end;

// Add EOF card to data buffer
procedure TDevRDR.AppendEOF;
Var FP: Longint;
    RC: TCBNRec;
Begin
  FillChar(RC,SizeOf(RC),#0);     // Clear record
  Move(EOFCard,RC,8);             // Insert EOF punches
  // Add to data buffer
  FP:=DataPosn;                   // Note current data position
  SetDataPosition(DataSize);      // Move to end of device data stream
  PutBytes(RC,160);               // Append new card data to it
  SetDataPosition(FP);            // Restore original position
  EndOfFil:=False;
  // Notify associated form of altered file data
  If Assigned(FOnFileChange) then FOnFileChange(Self);
End;

// Create card image from ASCII string, Add to data buffer
procedure TDevRDR.AppendRecord(SS: String);
Var FP: Longint;
    RC: TCBNRec;
begin
  If FileNam='' then              // Set file name if blank (scripted load)
    FileNam:='SCRATCH'+DevAdrString+'.CBN';
  RC:=ASCStrToCBNRec(SS);         // Translate text to CBN record
  FP:=DataPosn;                   // Note current data position
  SetDataPosition(DataSize);      // Move to end of device data stream
  PutBytes(RC,160);               // Append new card data to it
  SetDataPosition(FP);            // Restore original position
  EndOfFil:=False;
  // Notify associated form of altered file data
  If Assigned(FOnFileChange) then FOnFileChange(Self);
End;

// Load card images from ASCII file, Add to data buffer
procedure TDevRDR.AppendFile(FN: String);
Var SL: TStringList;
    LI: Integer;
Begin
  SL:=TStringList.Create;
  With SL do begin
    LoadFromFile(FN);
    For LI:=0 to Count-1 do
      AppendRecord(Strings[LI]);
    Free;
  End;
  EndOfFil:=False;
End;

//******************************************************************************
//  Form Functions
//------------------------------------------------------------------------------
procedure TReaderForm.FormCreate(Sender: TObject);
begin
  FormHandles[FIReader]:=Handle;
  Caption:=Heading;
  DispMode:=DMBCD;
  LACardNum.Caption:='';
  LACardCount.Caption:='';
  CardsList:=TStringList.Create;
  Deck.Records:=CardsList;
  BotCard:=1;
end;

procedure TReaderForm.FormDestroy(Sender: TObject);
begin
  CardsList.Free;
end;

procedure TReaderForm.Initialize;
Begin
  RDRDev.OnFileChange:=ShowFileDetails;
//RDRDev.OnStatusChange:=RefreshDisplay;
  RDRDev.OnEndOfRecord:=CardReadIn;
end;

procedure TReaderForm.Clear;
begin
  RDRDev.DetachFile;
end;

//******************************************************************************
//  Device events
//------------------------------------------------------------------------------
procedure TReaderForm.ShowFileDetails(Sender: TObject);
begin
  Caption:=Heading+' - '+RDRDev.FileNam;
  LoadCardsList;
  RefreshDisplay(Self);
End;

procedure TReaderForm.RefreshDisplay(Sender: TObject);
begin
  Deck.SetDisplayPosition(1,BotCard);
  Deck.Invalidate;
  LACardCount.Caption:=IntToStr(CardsList.Count);
End;

Procedure TReaderForm.CardReadIn(Sender: TObject);
Begin
  With CardsList do
    If Count>0 then
      Delete(0);                         // Remove card from list
  RefreshDisplay(Self);
End;

//******************************************************************************
//  System events
//------------------------------------------------------------------------------
Procedure TReaderForm.FormResize(Sender: TObject);
begin
  Deck.SetSize;
end;

procedure TReaderForm.DeckPositionChange(SE: TObject; RN: Word);
begin
  BotCard:=RN;
  LACardNum.Caption:=IntToStr(RN);
end;

//******************************************************************************
//  User control events
//------------------------------------------------------------------------------
procedure TReaderForm.BIOpenClick(Sender: TObject);
Var FN: String;
begin
  If ConfigForm.OpenFileDialog(FGCRD,FN,False) then
    IOCMsg(RDRDev.DevAddr,FCOPEN,0,FN);
End;

procedure TReaderForm.BIReloadClick(Sender: TObject);
begin
  IOCMsg(RDRDev.DevAddr,FCRELOAD,0,'');
  RBBOHClick(Self);               // Reload display
end;

procedure TReaderForm.BIClearClick(Sender: TObject);
begin
  IOCMsg(RDRDev.DevAddr,FCCLOSE,0,'');
End;

procedure TReaderForm.RBBOHClick(Sender: TObject);
Var FP: Longint;
begin
  DispMode:=TDispMode(TWinControl(Sender).Tag);  // Set new display format
//  FP:=RDRDev.DataPosn-1;                         // Note current file position
//  IOCMsg(RDRDev.DevAddr,FCSEEK,FP,'');           // Reposition and Reset EOF
  LoadCardsList;                                 // Load display from there
//  IOCMsg(RDRDev.DevAddr,FCSEEK,FP,'');           // Restore original position
End;

Procedure TReaderForm.CBSizeClick(Sender: TObject);
begin
  With Deck do begin
    If CBSize.Checked then Font.Size:=8
                      else Font.Size:=4;
    SetSize;
  End;
End;

// Load list of card display strings from device data buffer
Procedure TReaderForm.LoadCardsList;
Var SS: String;
    CC: Byte;
    CW: Word;
    FP: Integer;
begin
  // Load stringlist from device file for display
  CardsList.Clear;
  With RDRDev do begin
//    FP:=DataPosn;
//    SetDataPosition(FP);
    RewindFile;
    While DataPosn<DataSize do begin
      SS:=''; EndOfRec:=False;
      For CC:=1 to 80 do begin
        CW:=ReadColumn;
        Case DispMode Of
          DMBCD: SS:=SS+TabBCDtoASC[TabCRDtoBCD[CW]+1];
          DMOCT: SS:=SS+IntToOct(CW,4);
          DMHEX: SS:=SS+IntToHex(CW,3);
        End;
        If (CurrByte And $80)<>0 then Break;
      End;
      CardsList.Add(SS);
    End;
    RewindFile;
//    SetDataPosition(FP);
  End;
  RefreshDisplay(Self);
End;
{//;
function TDevRDR.TestEOR: Boolean;
begin
  Result:=EndOfCard;
end;
}
End.
