//******************************************************************************
//  IBM 7094 Emulator - Basic Instruction set functions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains the execution functions for most of the:-
//  Basic Instruction set
//------------------------------------------------------------------------------
//  Instruction Implementation notes:-
//  ENK  Panel switches are treated as always 0
//  SPTn/SPRn/SPUn Never skip
//------------------------------------------------------------------------------
//  Instruction functions are collected together in groups, and specific
//  instructions selected via Case statements.
//  This was handy during development, but eventually each instruction
//  probably should have a separate procedure.
//------------------------------------------------------------------------------
Unit B709IBIS;

Interface

Uses SysUtils,Dialogs,Controls,
     B709Defs, // General definitions
     B709Cnfg, // Configuration Information
     B709Misc, // Miscellaneous utility functions
     B709Inst, // CPU Instruction definitions
     B709Core, // Core Storage functions
     B709Chan, // I/O Channel functions
     B709Trce; // Log/Trace functions

Procedure IXShiftR(IR: TInstruction);
Procedure IXLodSto(IR: TInstruction);
Procedure IXContrl(IR: TInstruction);
Procedure IXTtmXfr(IR: TInstruction);
Procedure IXSensLS(IR: TInstruction);
Procedure IXTestsO(IR: TInstruction);
Procedure IXCompar(IR: TInstruction);
Procedure IXIdxReg(IR: TInstruction);
Procedure IXLogico(IR: TInstruction);
Procedure IXSnsInd(IR: TInstruction);
Procedure IXConvrt(IR: TInstruction);
Procedure IXChTrap(IR: TInstruction);
Procedure IXSysCom(IR: TInstruction);

Implementation

Uses B709Main, // For SenseToggle
     B709CPU;  // Access to CPU Registers

//******************************************************************************
//  Shift/Rotate
//------------------------------------------------------------------------------
Procedure IXShiftR(IR: TInstruction);
Var CF: Boolean;
    SV,SC: Integer;
    SF: Boolean;
    AC,MQ: TWord;
begin
  SV:=AddrFld And $FF;
  Case IR.ExRef of
    01: Begin                                                        // ALS
          If SV=0 then Exit;
          AC:=RegAC;
          For SC:=1 to SV do begin
            AC:=AC Shl 1;
            If (AC And PBitMask)>0 then begin
              OVFInd:=True; INDTrcFlg:=True;     // Set OVF indicator
            End;
          End;
          SetAC(AC And Wd37Mask);
        End;
    02: Begin                                                        // ARS
          If SV=0 then Exit;
          AC:=RegAC;
          For SC:=1 to SV do
            AC:=AC Shr 1;
          SetAC(AC And Wd37Mask);
        End;
    03: Begin                                                        // LLS
          SF:=(RegMQ And PBitMask)>0;       // Save MQ sign
          SetMQ(RegMQ And Wd35Mask);        // but drop from register
          AC:=RegAC; MQ:=RegMQ;             // Get working copies
          For SC:=1 to SV do begin
            MQ:=MQ Shl 1;                   // Shift registers
            AC:=AC Shl 1;
            If (MQ And PBitMask)<>0 then    // Move MQ MSB
              AC:=AC Or 1;                  // to AC LSB
            If (AC And PBitMask)<>0 then begin   // Set OVF if P is set
              OVFInd:=True; INDTrcFlg:=True;
            End;
          End;
          MQ:=MQ And Wd35Mask;              // Clean up
          AC:=AC And Wd37Mask;
          If SF then                        // Restore MQ sign
            MQ:=MQ Or PBitMask;
          SetSB(SF);                        // Make AC sign the same
          SetAC(AC); SetMQ(MQ);             // Save
        End;
    04: Begin                                                        // LRS
          SetMQ(RegMQ And Wd35Mask);        // Discard MQ sign
          AC:=RegAC; MQ:=RegMQ;             // Get working copies
          For SC:=1 to SV do begin
            CF:=(AC And 1)>0;               // Save LSB out of AC
            AC:=AC Shr 1;                   // Shift registers
            MQ:=MQ Shr 1;
            If CF then
              MQ:=MQ Or Bit1Mask;           // Set MQ bit 1
          End;
          MQ:=MQ And Wd35Mask;              // Clean up
          AC:=AC And Wd37Mask;
          If SgnAC then
            MQ:=MQ Or PBitMask;             // Set MQ sign same as AC
          SetAC(AC); SetMQ(MQ);             // Save
        End;
    05: Begin                                                        // LGL
          If SV=0 then Exit;
          AC:=RegAC; MQ:=RegMQ;             // Get working copies
          For SC:=1 to SV do begin
            CF:=(MQ And PBitMask)>0;        // Save MQ-S
            AC:=AC Shl 1;                   // Shift registers left
            MQ:=MQ Shl 1;
            If CF then AC:=AC Or 1;         // Set MQ-S to AC-1
            If (AC And PBitMask)>0 then     // Check for OVF
              OVFInd:=True;
          End;
          SetMQ(MQ And Wd36Mask);           // Clean up & save
          SetAC(AC And Wd37Mask);
        End;
    06: Begin                                                        // LGR
          If SV=0 then Exit;
          AC:=RegAC; MQ:=RegMQ;             // Get working copies
          For SC:=1 to SV do begin
            CF:=(AC And 1)>0;               // Save AC-1
            AC:=AC Shr 1;                   // Shift registers right
            MQ:=MQ Shr 1;
            If CF then
              MQ:=MQ Or PBitMask;           // Set MQ-S to AC-1
          End;
          SetMQ(MQ And Wd36Mask);           // Clean up save
          SetAC(AC And Wd37Mask);
        End;
    07: Begin                                                        // RQL
          If SV=0 then Exit;
          MQ:=RegMQ;                        // Get working copies
          For SC:=1 to SV do begin
            CF:=(MQ And PBitMask)>0;        // Save MSB
            MQ:=MQ Shl 1;
            If CF then                      // Move MSB
              MQ:=MQ Or 1;                  // to LSB
          End;
          SetMQ(MQ);                        // Save
        End;
  End;
End;

//******************************************************************************
//  Load/Store
//------------------------------------------------------------------------------
Procedure IXLodSto(IR: TInstruction);
Var MQ,YD: TWord;
begin
  Case IR.ExRef of
    01: SetMQ(GetCore(AddrFld));                                     // LDQ
    02: SetCore(AddrFld,RegMQ);                                      // STQ
    03: Begin                                                        // SLQ
          YD:=GetCore(AddrFld) And RtHaMask;
          YD:=YD Or (RegMQ And LfHaMask);
          SetCore(AddrFld,YD);
        End;
    04: Begin                                                        // STO
          YD:=RegAC And Wd35Mask;
          If SgnAC then YD:=YD Or PBitMask;
          SetCore(AddrFld,YD);
        End;
    05: SetCore(AddrFld,RegAC);                                      // SLW
    06: Begin                                                        // STP
          YD:=GetCore(AddrFld) And NotPrfxMask;
          YD:=YD Or (RegAC And PrfxMask);
          SetCore(AddrFld,YD);
        End;
    07: Begin                                                        // STD
          YD:=(GetCore(AddrFld) And NotDecrMask);
          YD:=YD Or (RegAC And DecrMask);
          SetCore(AddrFld,YD);
        End;
    08: Begin                                                        // STT
          YD:=GetCore(AddrFld) And NotTagsMask;
          SetCore(AddrFld,YD Or (RegAC And TagsMask));
        End;
    09: Begin                                                        // STA
          YD:=(GetCore(AddrFld) And NotAddrMask);
          YD:=YD Or (RegAC And AddrMask);
          SetCore(AddrFld,YD);
        End;
    10: Begin                                                        // STL
          YD:=GetCore(AddrFld) And NotAddrMask;
          SetCore(AddrFld,YD Or InstCtr);
        End;
    11: Begin                                                        // STR
          YD:=GetCore(0) And NotAddrMask;
          SetCore(0,YD Or InstCtr);
          SetIC(2,TCTrap);
        End;
    12: SetCore(AddrFld,0);                                          // STZ
    13: Begin                                                        // XCA
          YD:=RegMQ;
          MQ:=RegAC And Wd35Mask;
          If SgnAC then MQ:=MQ Or PBitMask;
          SetAC(YD And Wd35Mask);
          SetMQ(MQ);
          SetSB((YD And PBitMask)<>0);
        End;
    14: Begin                                                        // XCL
          YD:=RegMQ;
          SetMQ(RegAC And Wd36Mask);
          SetAS(YD And Wd36Mask,SA0);
        End;
    15: SetMQ(0);                                                    // ENK
    Else What(IR);
  End;
End;

//******************************************************************************
//  Control instructions
//------------------------------------------------------------------------------
Procedure IXContrl(IR: TInstruction);
begin
  Case IR.ExRef of
    01: Begin End;                                                   // NOP
    02: SetStopPending('Press Start');                               // HPR
    03: XECute(AddrFld,True);                                        // XEC
    04: TrapMode:=True;                                              // ETM
    05: TrapMode:=False;                                             // LTM
    06: SetIC(AddrFld,TCTran);                                       // TTR
    Else What(IR);
  End;
End;

//******************************************************************************
//  Control Transfer - Instructions affected by Transfer Trap Mode
//------------------------------------------------------------------------------
Type TCompareResult=(CRLT,CREQ,CRGT);

Function Compare(SA,SB: Boolean; MA,MB: TWord): TCompareResult;
Var VA,VB: Int64;
    SS: String;
Begin
  VA:=MA; If SA then VA:=-VA;
  VB:=MB; If SB then VB:=-VB;
  Result:=CREQ;
  Repeat
    If VA<VB then begin
      Result:=CRLT; Break;
    End;
    If VA>VB then begin
      Result:=CRGT; Break;
    End;
    If MA<>0 then begin
      Result:=CREQ; Break;
    End;
    If SA=SB then begin
      Result:=CREQ; Break;
    End;
    If SA then Result:=CRLT
          else Result:=CRGT;
  Until True;
  // Tracing
  If (TIREG In TraceRecdFlags) then begin
    If SA then MA:=MA Or PBitMask;
    If SB then MB:=MB Or PBitMask;
    SS:='      '+IntToOct(MA,12)+' ';
    Case Result Of
      CRLT: SS:=SS+'<';
      CREQ: SS:=SS+'=';
      CRGT: SS:=SS+'>';
    End;
    SS:=SS+' '+IntToOct(MB,12);
    Trace(TIREG,SS);
  End;
End;

Procedure IXTtmXfr(IR: TInstruction);
Var MQ,MA: TWord;
    TA: Int64;
    XV: TAddr;
    SQ: Boolean;
begin
  If TrapMode then begin
    TA:=GetCore(0);                    // Trapmode: set IC into Locn 0
    TA:=TA And NotAddrMask;            // Whther transfer occurs or not
    SetCore(0,TA Or (InstCtr-1));
  End;
  TA:=0;
  Case IR.ExRef of
    01: Begin                                                        // HTR
          SetStopPending('Press Start');
          SetIC(InstCtr-1,TCNone);
          HTRAddr:=AddrFld;
          Exit;
        End;
    02: Begin                                                        // TRA
          TA:=AddrFld;
          If (TA=PrevICtr) And (Not TrapMode) then
            Error('TRA loop at '+IntToOct(TA,5));
        End;
    03: If (RegAC And Wd37Mask)=0 then TA:=AddrFld;                  // TZE
    04: If (RegAC And Wd37Mask)<>0 then TA:=AddrFld;                 // TNZ
    05: If Not SgnAC then TA:=AddrFld;                               // TPL
    06: If SgnAC then TA:=AddrFld;                                   // TMI
    07: If OVFInd then begin                                         // TOV
          TA:=AddrFld;
          OVFInd:=False; INDTrcFlg:=True;
        End;
    08: If OVFInd then begin                                         // TNO
          OVFInd:=False; INDTrcFlg:=True;
        End else
          TA:=AddrFld;
    09: If (RegMQ And PBitMask)=0 then TA:=AddrFld;                  // TQP
    10: If (Not FTrpMode) And MQOInd then begin                      // TQO
          TA:=AddrFld;
          MQOInd:=False; INDTrcFlg:=True;
        End;
    11: Begin                                                        // TLQ
          SQ:=(RegMQ And PBitMask)>0;                 // Get sign
          MQ:=RegMQ And Wd35Mask;                     // Get magnitudes
          MA:=RegAC And Wd37Mask;
          If Compare(SQ,SgnAC,MQ,MA)=CRLT then        // Compare
            TA:=AddrFld;
        End;
    12: Begin                                                        // TSX
          If XECAddr<>$8000 then XV:=XECAddr    // If under XEC, get XEC address
                            else XV:=InstCtr;   // otherwise, get TSX address
          SetXR(TagsFld,-(XV-1));
          TA:=AddrFld;
        End;
    13: Begin                                                        // TXI
          XV:=GetXR(TagsFld);
          SetXR(TagsFld,XV+DecrFld);
          TA:=AddrFld;
        End;
    14: Begin                                                        // TXH
          XV:=GetXR(TagsFld);
          If XV>DecrFld then TA:=AddrFld;
        End;
    15: Begin                                                        // TXL
          XV:=GetXR(TagsFld);
          If XV<=DecrFld then TA:=AddrFld;
        End;
    16: Begin                                                        // TIX
          XV:=GetXR(TagsFld);
          If XV>DecrFld then begin
            SetXR(TagsFld,XV-DecrFld);
            TA:=AddrFld;
          End;
        End;
    17: Begin                                                        // TNX
          XV:=GetXR(TagsFld);
          If XV>DecrFld then SetXR(TagsFld,XV-DecrFld)
                        else TA:=AddrFld;
        End;
    18: If (RegSI And RegAC)=(RegAC And Wd36Mask) then TA:=AddrFld;                 // TIO
    19: If (RegSI And RegAC)=0 then TA:=AddrFld;                     // TIF
//  30: Begin                                                        // ESNT
//        SNulMode:=True;
//        TA:=AddrFld;
//      End;
//  31: Begin                                                        // LSNM
//        SNulMode:=False;
//      End;
    Else What(IR);
  End;
  // See if trap is required
  If TA<>0 then                        // Did instruction transfer?
    If TrapMode then SetIC(1,TCTrap)   // Yes, If Trapmode, Goto 1
                else SetIC(TA,TCTran); // No, sequential execution
End;

//******************************************************************************
//  Sense Lights & Switches
//------------------------------------------------------------------------------
Procedure IXSensLS(IR: TInstruction);
Var SN: Byte;
    SI: Integer;
Begin
  SI:=0;                               // Reset skip increment
  Case IR.ExRef of
    01: Begin                                                        // SLF
          RegSL[1]:=False;
          RegSL[2]:=False;
          RegSL[3]:=False;
          RegSL[4]:=False;
        End;
    11..
    14: Begin                                                        // SLNn 1-4
          SN:=IR.ExRef-10;             // Base to 1
          RegSL[SN]:=True;
        End;
    21..
    26: Begin                                                        // SWTn 1-6
          SN:=IR.ExRef-20;
          MainForm.SenseFlash(SN);
          If RegSS[SN] then
            SI:=1;
        End;
    27: Begin End;                                                   // 9M71B
    31..
    34: Begin                                                        // SLTn 1-4
          SN:=IR.ExRef-30;             // Base to 1
          If RegSL[SN] then begin
            RegSL[SN]:=False;
            SI:=1;
          End;
        End;
    35: Begin End;                                                   // 9M71B
    // For channels, Not switches
    40..
    48: Begin                                                        // SPTn 1-8
        End;
    50..
    58: Begin                                                        // SPRn 1-8
        End;
    Else What(IR);
  End;
  SetIC(InstCtr+SI,TCTest);            // Update IC with increment
End;

//******************************************************************************
//  Tests (other than Sense Lights/Switches and Input/Output)
//------------------------------------------------------------------------------
Procedure IXTestsO(IR: TInstruction);
Var YD: TWord;
    SI: Integer;
Begin
  SI:=0;                               // Reset skip increment
  YD:=GetCore(AddrFld);
  Case IR.ExRef of
    01: If (RegAC And PBitMask)<>0 then SI:=1;                       // PBT
    02: If (RegAC And LowBMask)<>0 then SI:=1;                       // LBT
    03: If DVCInd then DVCInd:=False                                 // DCT
                  else SI:=1;
    04: If (YD And Wd35Mask)=0  then SI:=1;                          // ZET
    05: If (YD And Wd35Mask)<>0 then SI:=1;                          // NZT
    06: If (RegSI And YD)=YD then SI:=1;                             // ONT
    07: If (RegSI And YD)=0 then SI:=1;                              // OFT
   Else What(IR);
  End;
  SetIC(InstCtr+SI,TCTest);            // Update IC with increment
End;

//******************************************************************************
//  Compare
//------------------------------------------------------------------------------
Procedure IXCompar(IR: TInstruction);
Var YD,YA,MA,MD: TWord;
    SD: Boolean;
    SI: Integer;
Begin
  SI:=0;                               // Reset skip increment
  YD:=GetCore(AddrFld);
  Case IR.ExRef of
    01: Begin                                                        // CAS
          SD:=(YD And PBitMask)<>0;         // Get sign
          MA:=RegAC And Wd37Mask;           // Get magnitudes
          MD:=YD And Wd35Mask;
          Case Compare(SgnAC,SD,MA,MD) Of   // Compare
            CRLT: SI:=2;
            CREQ: SI:=1;
          End;
        End;
    02: Begin                                                        // LAS
          YA:=(RegAC And Wd37Mask);
          If YA<=YD then begin
            SI:=1;
            If YA<>YD then
              SI:=2;
          End;
          OVFInd:=(RegAC And PBitMask)>0;
        End;
    Else What(IR);
  End;
  SetIC(InstCtr+SI,TCTest);         // Update IC with increment
End;

//******************************************************************************
//  Index Registers
//------------------------------------------------------------------------------
Procedure IXIdxReg(IR: TInstruction);
Var AC,YD,XD: TWord;
Begin
  Case IR.ExRef of
    01: MTagMode:=True;                                              // EMTM
    02: MTagMode:=False;                                             // LMTM
    03: Begin                                                        // PCA
          AC:=(-GetXR(TagsFld));       // Separate steps required
          SetAS(AC And AddrMask,SA0);  // to keep Int64 result
        End;
    04: Begin                                                        // PCD
          AC:=(-GetXR(TagsFld));      // Separate steps required
          AC:=AC And AddrMask;
          SetAS(AC Shl DecrShft,SA0); // to keep Int64 result
        End;
    05: Begin                                                        // SCA
          YD:=GetCore(AddrFld) And NotAddrMask;
          XD:=(-GetXR(TagsFld)) And AddrMask;
          SetCore(AddrFld,YD Or XD);
        End;
    06: Begin                                                        // SCD
          YD:=GetCore(AddrFld) And NotDecrMask;
          XD:=(-GetXR(TagsFld)) And AddrMask;
          XD:=XD Shl DecrShft;
          SetCore(AddrFld,YD Or XD);
        End;
    07: If TagsFld<>0 then begin                                     // LXA
          YD:=GetCore(AddrFld) And AddrMask;
          SetXR(TagsFld,YD);
        End;
    08: If TagsFld<>0 then begin                                     // LAC
          YD:=(GetCore(AddrFld) And AddrMask);
          SetXR(TagsFld,-YD);
        End;
    09: If TagsFld<>0 then begin                                     // LXD
          YD:=(GetCore(AddrFld) And DecrMask) Shr DecrShft;
          SetXR(TagsFld,YD);
        End;
    10: If TagsFld<>0 then begin                                     // LDC
          YD:=(GetCore(AddrFld) And DecrMask) Shr DecrShft;
          YD:=(-YD) And AddrMask;
          SetXR(TagsFld,YD);
        End;
    11: If TagsFld<>0 then begin                                     // AXT
          SetXR(TagsFld,AddrFld);
        End;
    12: If TagsFld<>0 then begin                                     // AXC
          SetXR(TagsFld,-AddrFld);
        End;
    13: If TagsFld<>0 then begin                                     // PAX
          SetXR(TagsFld,(RegAC And AddrMask));
        End;
    14: If TagsFld<>0 then begin                                     // PAC
          YD:=(RegAC And AddrMask);
          SetXR(TagsFld,-YD);
        End;
    15: SetAS(GetXR(TagsFld),SA0);                                   // PXA
    16: Begin                                                        // PXD
          AC:=GetXR(TagsFld);
          SetAS(AC Shl DecrShft,SA0);
        End;
    17: If TagsFld<>0 then begin                                     // PDX
          SetXR(TagsFld,RegAC Shr DecrShft);
        End;
    18: If TagsFld<>0 then begin                                     // PDC
          YD:=(RegAC And DecrMask) Shr DecrShft;
          SetXR(TagsFld,-YD);
        End;
    19: Begin                                                        // SXA
          YD:=(GetCore(AddrFld) And NotAddrMask);
          XD:=GetXR(TagsFld);
          SetCore(AddrFld,YD Or XD);
        End;
    20: Begin                                                        // SXD
          YD:=(GetCore(AddrFld) And NotDecrMask);
          XD:=GetXR(TagsFld);
          XD:=XD Shl DecrShft;
          SetCore(AddrFld,YD Or XD);
        End;
    Else What(IR);
  End;
End;

//******************************************************************************
//  Logic operations
//------------------------------------------------------------------------------
Procedure IXLogico(IR: TInstruction);
Var AC,YD: TWord;
Begin
  YD:=GetCore(AddrFld);
  Case IR.ExRef of
    01: SetAC((RegAC Or YD) And Wd37Mask);                           // ORA
    02: Begin                                                        // ORS
          YD:=YD Or RegAC;
          YD:=YD And Wd36Mask;
          SetCore(AddrFld,YD);
        End;
    03: Begin                                                        // ANA
          AC:=RegAC And YD;
          SetAS(AC And Wd36Mask,SA0);
        End;
    04: Begin                                                        // ANS
          YD:=(RegAC And YD) And Wd36Mask;
          SetCore(AddrFld,YD);
        End;
    05: SetAS((RegAC Xor YD) And Wd36Mask,SA0);                      // ERA
    06: SetAC(Not RegAC);                                            // COM
    07: SetAC(0);                                                    // CLM
    08: SetSB(Not SgnAC);                                            // CHS
    09: SetSB(False);                                                // SSP
    10: SetSB(True);                                                 // SSM
   Else What(IR);
  End;
End;

//******************************************************************************
//  Sense Indicator register
//------------------------------------------------------------------------------
Procedure IXSnsInd(IR: TInstruction);
Var LM,RM: TWord;
    SI: Integer;
Begin
  SI:=0;                               // Reset skip increment
  RM:=(StorReg And RtHaMask);          // Get R field as L & R masks
  LM:=RM Shl HalfShft;
  Case IR.ExRef of
    01: SetSI(RegAC);                                                // PAI
    02: Begin                                                        // PIA
          SetAC(RegSI And Wd36Mask);
          SetSB(False);
        End;
    03: SetSI(GetCore(AddrFld));                                     // LDI
    04: SetCore(AddrFld,RegSI);                                      // STI
    05: SetSI(RegSI Or RegAC);                                       // OAI
    06: SetSI(RegSI Or GetCore(AddrFld));                            // OSI
    07: SetSI(RegSI Or LM);                                          // SIL
    08: SetSI(RegSI Or RM);                                          // SIR
    09: SetSI(RegSI And (Not RegAC));                                // RIA
    10: SetSI(RegSI And (Not GetCore(AddrFld)));                     // RIS
    11: SetSI(RegSI And (Not LM));                                   // RIL
    12: SetSI(RegSI And (Not RM));                                   // RIR
    13: SetSI(RegSI Xor RegAC);                                      // IIA
    14: SetSI(RegSI Xor GetCore(AddrFld));                           // IIS
    15: SetSI(RegSI Xor LM);                                         // IIL
    16: SetSI(RegSI Xor RM);                                         // IIR
    17: If (RegSI And LM)=LM then SI:=1;                             // LNT
    18: If (RegSI And RM)=RM then SI:=1;                             // RNT
    19: If (RegSI And LM)=0 then SI:=1;                              // LFT
    20: If (RegSI And RM)=0 then SI:=1;                              // RFT
    Else What(IR);
  End;
  SetIC(InstCtr+SI,TCTest);         // Update IC with increment
End;

//******************************************************************************
//  Convert instructions
//------------------------------------------------------------------------------
Procedure IXConvrt(IR: TInstruction);
Var AC,MQ,SR,AQ: TWord;
    XV: TWord;
    SC: Byte;
    TF: Boolean;
Begin
  SR:=AddrFld;                                   // Load Y address
  SC:=DecrFld And $FF;                           // Get count
  TF:=(TagsFld And 1)<>0;                        // Get Tag value
  Case IR.ExRef of
    01: Begin                                                        // CVR
          AC:=RegAC;
          AQ:=RegAC And QBitMask;
          AQ:=AQ Shr 6;
          Repeat
            If SC=0 then begin
              If TF then SetXR(1,(SR And AddrMask));
              Break;
            End;
            XV:=AC And $3F;                       //  Get lookup char
            SR:=GetCore((SR And AddrMask)+XV);    // Look it up
            XV:=SR And CharMask;                  // Get the result char
            AC:=AC Shr 6;                         // Shift AC 6 bits
            AC:=AC Or XV;                         // Insert result char
            AC:=AC Or AQ;                         // Insert result char
            AQ:=0;
            Dec(SC);
          Until False;
          SetAC(AC);
        End;
    02: Begin                                                        // CRQ
          MQ:=RegMQ;            // Get working copy
          Repeat
            If SC=0 then begin
              If TF then SetXR(1,(SR And AddrMask));
              Break;
            End;
            XV:=(MQ And CharMask) Shr 30;      // Get lookup char
            SR:=GetCore((SR And AddrMask)+XV);    // Look it up
            XV:=(SR And CharMask) Shr 30;         // Get the result char
            MQ:=(MQ Shl 6) And Wd36Mask;    // Shift MQ 6 bits
            MQ:=MQ Or XV;                   // Insert result char
            Dec(SC);
          Until False;
          SetMQ(MQ);                               // Save
        End;
    03: Begin                                                        // CAQ
          AC:=RegAC; MQ:=RegMQ;            // Get working copies
          Repeat
            If SC=0 then begin
              If TF then SetXR(1,(SR And AddrMask));
              Break;
            End;
            XV:=(MQ And CharMask) Shr 30;      // Get lookup char
            MQ:=(MQ Shl 6) And Wd36Mask;    // Rotate MQ 6 bits
            MQ:=MQ Or XV;
            SR:=GetCore((SR And AddrMask)+XV);    // Look it up
            AC:=AC+SR;                      // Add to AC (Logical)
            Dec(SC);                              // Next Char
          Until False;
          SetAC(AC And Wd37Mask);              // Clean AC. Leave sign alone
          SetMQ(MQ);                               // Save
        End;
    Else What(IR);
  End;
End;

//******************************************************************************
//  Channel Trap
//------------------------------------------------------------------------------
Procedure IXChTrap(IR: TInstruction);
Var TD,TM,CM: TWord;
    CI: Byte;
    SL,SC: String;
begin
  Case IR.ExRef of
    01: Begin                                                        // ENB
          TD:=GetCore(AddrFld) And TrapMask;// Get Traps Map
          TM:=$000000001;                   // Channel Trap bit for Ch A
          CM:=$000040000;                   // Tape Check Trap bit for Ch A
          SL:=''; SC:='';                   // Clear trace lists
          For CI:=1 to LastChan do begin
            With Channels[CI] do begin      // For each channel
              TrapCtrlInd:=True;            // Enable trapping generally
              LCHTrapEnab:=(TD And TM)>0;   // Enable Command End trap?
              TCKTrapEnab:=(TD And CM)>0;   // Enable Tape Check trap?
              EOFTrapEnab:=LCHTrapEnab;     // EOF same as LCH
              // Build channel trap lists for trace
              If (TICCW In TraceRecdFlags) then begin
                If LCHTrapEnab then SL:=SL+Chr($40+CI);
                If TCKTrapEnab then SC:=SC+Chr($40+CI);
              End;
            End;
            TM:=TM Shl 1;                   // Move bits to next channel
            CM:=CM Shl 1;
          End;
          If (TICCW In TraceRecdFlags) then
            Trace(TICCW,'Channel Traps set to: CMD/EOF='+Sl+' Tape Check='+SC);
        End;
    Else What(IR);
  End;
  TrapInhib:=True;      // Inhibit channel traps for next instruction
End;

//******************************************************************************
//  System Compatibliity (704 & 709 modes)
//------------------------------------------------------------------------------
Procedure IXSysCom(IR: TInstruction);
Begin
  Case IR.ExRef of
//  01: Begin End;                                                   // ESTM
//  02: Begin End;                                                   // ECTM
    03: FTrpMode:=True;                                              // EFTM
    04: FTrpMode:=False;                                             // LFTM
    Else What(IR);
  End;
End;

End.



