//**************************************************************************
//  IBM 7094 Emulator - Floating Point Instruction functions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//--------------------------------------------------------------------------
//  This unit contains the execution functions for:-
//  Floating Point Arithmetic instructions (Single and Double Precision)
//--------------------------------------------------------------------------
//  The code in this unit was developed by Mike Hore. (but hacked about by me)
//  Thanks Mike, I wasn't looking forward do doing this part!
//--------------------------------------------------------------------------

Unit B709IFPA;

Interface

Uses SysUtils,Dialogs,Controls,
     B709Defs, // General definitions
     B709Misc, // Miscellaneous utility functions
     B709Inst, // CPU Instruction definitions
     B709Core, // Core Storage functions
     B709Chan, // I/O Channel functions
     B709Trce; // Log/Trace functions

Procedure IXFloatS(IR: TInstruction);
Procedure IXFloatD(IR: TInstruction);

Implementation

Uses B709CPU;  // Access to CPU Registers

// Floating point fields in a 36 bit word
// S00000000011111111112222222222333333
// S12345678901234567890123456789012345
// X................................... Sign
// .XXXXXXXX........................... Characteristic
// .........XXXXXXXXXXXXXXXXXXXXXXXXXXX Fraction
// ............XXXXXX.................. Spill code stored in Locn 0
// 842184218421842184218421842184218421

// Fields in FracRV working variable
// 0000000000111111111122222222223333333333444444444455555555556666
// 0123456789012345678901234567890123456789012345678901234567890123
// 6666555555555544444444443333333333222222222211111111110000000000
// 3210987654321098765432109876543210987654321098765432109876543210
// .........XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
//         Valid part
// .........X......................................................
//         Overfow 
// ..........XXXXXXXXXXXXXXXXXXXXXXXXXXX...........................
//         AC part
// .....................................XXXXXXXXXXXXXXXXXXXXXXXXXXX
//         MQ part
// 8421842184218421842184218421842184218421842184218421842184218421

// Masks to extract fields from a Floating Point word
Const SignMask: TWord=$800000000; // Floating Point Sign Mask
      CharMask: TWord=$7F8000000; // Floating Point Characteristic Mask
      FracMask: TWord=$007FFFFFF; // Floating Point Fraction Mask

Const RVOVFBit: TWord=$0040000000000000; // Result Overfow bit
      RVMSBit:  TWord=$0020000000000000; // Result most significant bit
      RVMSMask: TWord=$003FFFFFF8000000; // Result Most significant bits
      RVLSMask: TWord=$0000000007FFFFFF; // Result Least significant bits

// Bit shift counts to move Floating Point fields to/from LSB
Const CharShft=27;      // Bit offset to Characteristic field in a Word

// Intermediate/Working variables

Var // Floating point components of AC register
    SignAC: Boolean;       // Sign
    CharAC: Integer;       // Characteristic
    FracAC: Int64;         // Fraction
    PrevCharAC: Integer;   // Saves CharAC value at start of instruction

    // Floating point components of MQ register
    SignMQ: Boolean;       // Sign
    CharMQ: Integer;       // Characteristic
    FracMQ: Int64;         // Fraction

    // Floating point components of a word from storage
    SignSR: Boolean;       // Sign
    CharSR: Integer;       // Characteristic
    FracSR: Int64;         // Fraction

    // Floating point components of Sense Indicator register (SI)
    // Used in Double Precision operations for the high word from storage.
    SignSI: Boolean;       // Sign
    CharSI: Integer;       // Characteristic
    FracSI: Int64;         // Fraction

    FracRV: Int64;        // Fraction component of result

    FPTrap:    Boolean;   // Set during execution if a trap comes up
    FPTrapFlags: TWord;   // Flag bits for FP trap
    Dividing: Boolean;    // True if a division is current

//**************************************************************************
//  Common functions
//--------------------------------------------------------------------------
// Execute Floating Point Trap/Spill.
Procedure FloatingPointTrap;
Var TD: TWord;
Begin
//TD:=GetCore(0) And NotAddrMask;            // Get locn 0 contents
  TD:=0;
  TD:=TD Or InstCtr;                         // Insert IC+1 to Addr field
  TD:=TD Or (FPTrapFlags Shl DecrShft);      // Insert Code to Decr field
  SetCore(0,TD);                             // Save spill data to locn 0
  SetIC(00008,TCFPSp);                       // Jump to FP trap address
End;

// Set a floating point trap condition to be taken at the end
// of the current instruction.
Procedure SetFPTrap ( flags : TWord );
Begin
  If FTrpMode then begin
    FPTrapFlags:=FPTrapFlags Or flags;
    If Dividing then
      FPTrapFlags:=FPTrapFlags Or 8;  // Set div error bit in flags
    FPTrap:=True;
  End else Begin
    // Not in trap mode - just set AC or MQ overflow
    If (Flags And 2)<>0 then OVFInd:=True
                        else MQOInd:=True;
  End
End;

// Check if Floating Point Trap is required
//  The test on the AC is tricky.  It includes the P and Q bits.
//  It seems we have to test for a carry setting the P bit when
//  both P and Q were originally clear.
Procedure SpillCheck;
Begin
  If CharAC=0 then begin
    CharMQ:=0;
  End else Begin
    // AC char gone negative?
    If CharAC<0 then begin
      SetFPTrap($2);                    // Yes, AC underflow
    End else Begin
      If ((PrevCharAC And $300)=0) And  // Both P and Q were 0?
         ((CharAC And $100)<>0) then    // and P is now 1?
        SetFPTrap($6);                  // Yes, AC ovfl
    End;
    If CharMQ>255 then                  // MQ char too high?
      SetFPTrap($5);                    // Yes, MQ ovfl
    If CharMQ<0 then                    // MQ char gone negative?
      SetFPTrap($1);                    // Yes, MQ underflow
  End;
  CharAC:=CharAC And $3FF;              // Can set P and Q bits
  CharMQ:=CharMQ And $FF;
End;

Procedure DoNormalize;
Var SC: Integer;
Begin
  If FracRV = 0 then begin    // result fraction zero?
    CharAC:=0; FracAC:=0;     // Note sign is unchanged
  End else Begin
    SC:=0;
    While ((FracRV And RVMSBit)=0) And (SC<60) do begin      // Sanity check
      FracRV:=FracRV Shl 1;
      Inc(SC);
    End;
    CharAC:=CharAC-SC;
    CharMQ:=CharAC-27;
  End
End;

// Utility procedure for DP add, to do a double word
//  exchange between AC-MQ and SI-SR.
Procedure  DblExchange;
Var WV: Integer;   // Work variable
    WB: Boolean;   // Work boolean
Begin
  WB:=SignAC; SignAC:=SignSI; SignSI:=WB;
  WV:=CharAC; CharAC:=CharSI;
  CharSI:=(WV And $FF);          // Masking out any P/Q
  WV:=FracAC; FracAC:=FracSI; FracSI:=WV;
  WB:=SignMQ; SignMQ:=SignSR; SignSR:=WB;
  WV:=CharMQ; CharMQ:=CharSR; CharSR:=WV;
  WV:=FracMQ; FracMQ:=FracSR; FracSR:=WV;
End;

// Add SR to AC, Result to AC-MQ.
Procedure DoFAdd ( normalize, double : boolean );
Var WV: Integer;        // Work variable
    WB: Boolean;        // Work boolean
    W64: Int64;         // Work Int64
    SC: Integer;        // Shift count
    signsSame: Boolean;
    // The following two vars are needed to get the final SI
    //  right in the double precision case.
    exchange: Boolean;  // True if we do an exchange
    bigShift: Boolean;  // True if smaller operand shifted out of existence
Begin
  FracRV:=0;
  exchange:=false;  bigShift:=false;
  If not double then begin
    If charSR<charAC then begin                    // Interchange AC and SR
      WB:=SignAC; SignAC:=SignSR; SignSR:=WB;
      WV:=CharAC; CharAC:=CharSR;
      CharSR:=(WV And $FF);          // Masking out any P/Q
      WV:=FracAC; FracAC:=FracSR; FracSR:=WV;
      // A nonzero P bit throws a spanner in the works:
      If (RegAC And $0800000000)<>0  then  SignSR:=True;
    End;
    SignMQ:=SignAC;
    SC:=CharSR-CharAC;
  End else Begin
    // double precision case
    If charSI < charAC then begin
      DblExchange;
      exchange:=True;
    End;
    signMQ:=signAC;
    SC:=charSI - charAC;
  End;
  FracRV:=FracAC Shl CharShft;
  If double then
    FracRV:=FracRV Or fracMQ;
// Now we shift the fraction with the smaller exponent right.
// Note however, a nonzero P or Q can lead to SC being negative!
// In this case the hardware interprets it as a big positive
// number and shifts the fraction right out of existence, so
// here we just clear it.
  If (SC<0) Or (SC>63) then begin
    FracRV:=0;
    bigShift:=True
  End else Begin
    FracRV:=FracRV Shr SC;
  End;
// Now we set up for the actual add or subtract:
  If not double  then begin
    CharAC:=CharSR;
    W64:=FracSR Shl CharShft;
    signsSame:=(signAC=signSR);
  End else Begin
    charAC:=CharSI;
    W64:=(fracSI Shl CharShft) Or fracSR;
    signsSame:=(signAC=signSI);
  End;
// Now we do it:
  If signsSame then begin
    // Signs same - True add
    FracRV:=FracRV + W64;
    If (FracRV And RVOVFBit)<>0 then begin        // Got an overflow bit - shift right by 1
      FracRV:=FracRV Shr 1;
      Inc(CharAC);
      SpillCheck;
    End;
  End else Begin
    // Signs differ - complement add
    FracRV:=FracRV-W64;
    If FracRV<0 then begin
      // recomplement
      // In the DP case the SI gets used in a rather nonobvious
      // way -- the following code reproduces the cases exposed
      // by the diagnostic but I don't have enough info to be
      // sure it's completely right.  It probably doesn't matter
      // anyway.
      If bigShift And signSI  then
        If exchange then begin
          signSI:=false; charSI:=0; fracSI:=0;
        End else Begin
          fracSI:=fracAC
        End;
      FracRV:=- FracRV;
      SignAC:=SignSR;  SignMQ:=SignSR
    End;
  End;
  If normalize then DoNormalize
               else CharMQ:=CharAC - 27;
  // if unnorm, this happens no matter what
  SpillCheck;
// Return result fraction back to AC and MQ
  FracAC:=(FracRV And RVMSMask) Shr CharShft;
  FracMQ:=(FracRV And RVLSMask);
End;

// Round the AC by adding one if the high bit of the MQ fraction is 1.
Procedure DoFRound;
begin
  if (FracMQ and $04000000)<>0 then begin   // top bit of MQ frac set?
    FracAC:=FracAC + 1;                     // increment AC fraction
    if (FracAC and $08000000)<>0 then begin // carried?
      FracAC:=FracAC Shr 1;                 // yes, shift AC frac right
      CharAC:=CharAC+1;                     // increment characteristic
      SpillCheck;
    end
  end
end;

{   Common routine for DP mult and div, to shift the result fraction
    one bit left if the top bit is a zero.  We could surely use it
    for single precision as well, but it's better to leave working
    code alone.

    This is not a full normalization -- in the mult case only one
    top bit can be zero if the operands were normalized, and in
    the DP divide case only one bit can be zero anyway.  In the
    SP divide case this can't happen, so this routine isn't called.

    This was surely a single sequence in the hardware, especially
    since even if we're Dividing, a spill doesn't set the divide
    error bit.

    One diagnostic test for DFDP shows that there's a spill test
    before the normalization shift, and if there's a spill the
    trap is taken without doing the shift.  The multiplication
    tests in the diagnostic don't expose this condition, but it
    probably holds and we can always fix it later if it doesn't. }
Procedure  OneBitNormalize;
Begin
  Dividing:=false;        // An error here doesn't set the div error bit
  SpillCheck;
  If not FPtrap then begin  // If trap condition we don't do anything else
    If (FracRV And RVMSBit)=0 then begin
      fracRV:=fracRV Shl 1;
      Dec(charAC); Dec(charMQ);
    End;
{   Next if the AC fraction is zero, we also clear the AC
    characteristic, yielding a normal zero.
    The manual reads as if this step comes before the
    normalization shift, but it doesn't, at least not on the
    7094.  Maybe it did on the 704? }
    If (FracRV And RVMSMask)=0 then   // i.e. result AC frac
      CharAC:=0;
    If CharAC=0 then CharMQ:=0
                else CharMQ:=CharAC-27;    // Set MQ char appropriately
    SpillCheck;
  End;
End;

// Mult AC by SR, result to AC-MQ.
Procedure DoFMult ( normalize, double : boolean );
begin
{  To multiply, we basically multiply the fractions and add the
  the characteristics, subtracting the 128 bias so it isn't
  added twice.  If the operands are normalized, the most we
  need to shift the result left is one, and the spec ASSUMES
  this.  This means that if the operands are unnormalized,
  then even if this is a FMP, the result probably won't be
  normalized.  Strange but True. }
  If not double then begin
    CharAC:=0;    // Clear initial result (but SignAC left alone)
    PrevCharAC:=0;  // Even if P was set, we can overflow!
    FracAC:=0;
    SignAC:=(SignMQ<>SignSR );        // set result sign
  end;
  If (CharSR=0) And (FracSR=0) then begin
    // Multiplicand is normal zero.  Bail out early,
    FracMQ:=0;    // with low order result also zero
    CharMQ:=0;
  End else Begin
    FracRV:=FracMQ * FracSR;          // multiply fractions
    CharAC:=CharMQ + CharSR - 128;    // add chars and adjust bias
    If normalize then begin
      // If top bit of result fraction is 0, we shift left by 1.
      //  We just do this once, i.e. NOT a full normalization!
      If (FracRV And RVMSBit)=0 then begin
        FracRV:=FracRV Shl 1;
        CharAC:=CharAC-1
      End;
  {   Next if the AC fraction is zero, we also clear the AC
      characteristic, yielding a normal zero.
      The manual reads as if this step comes before the
      normalization shift, but it doesn't, at least not on the
      7094.  Maybe it did on the 704?}
      If (FracRV And RVMSMask)=0 then   // i.e. result AC frac
        CharAC:=0;
      If CharAC=0 then CharMQ:=0
                  else CharMQ:=CharAC-27;      // Set MQ char appropriately
    End else        // normalize
      CharMQ:=CharAC-27;    // Unnormalized op -- just
                            // set MQ char without a check
    SignMQ:=SignAC;
    SpillCheck;
// Return result fraction back to AC and MQ
    FracAC:=(FracRV And RVMSMask) Shr CharShft;
    FracMQ:=(FracRV And RVLSMask);
  End
End;

{  Double mult is sufficiently different to single that it's
    better to have a separate routine.
    We multiply the AC-MQ (A and B) by the doubleword from storage
    which will be in SI-SR (C and D).  An interesting variety of
    things happen to the SI, and though this doesn't affect any
    real results we try to observe it to keep the diagnostic
    happy. }
Procedure DoDFMult ( normalize: Boolean );
Var AtimesD:    Int64;
    BtimesC:    Int64;
Begin
  signAC:=(signAC <> signSI);        // set result sign
  signMQ:=signAC;
{ There are a couple of early end checks.
  1. If either operand is a normal zero.
  2. If the A and C fractions are both zero, since the lower parts
      can't contribute any significant bits.  This isn't really True
      mathematically if the original operands are very unnormalized,
      but it's what the implementation does.  The 7094 didn't have
      enough registers to do any better, and anyway most operands were
      probably normalized.}
  If ((charAC=0) And (fracAC=0) Or
      (charSI=0) And (fracSI=0) Or          // Either operand is a normal zero
      (fracAC=0) And (fracSI=0)) then begin // A and C fractions both zero
    // Early end - clear AC and MQ except for signs
    charAC:=0; fracAC:=0;
    charMQ:=0; fracMQ:=0;
  End else Begin
    charAC:=charAC + charSI - 128;    // add chars and adjust bias
    charMQ:=charAC - 27;                // set charMQ as well
    signMQ:=signAC;                    // and signMQ
    // Third early-end test: A and B fractions both zero, or ditto C and D:
    If ((fracAC=0) And (fracMQ=0) Or
        (fracSI=0) And (fracSR=0)) then begin
      // Early end, condition 3:
      If normalize then begin        // chars left alone, but fracSI cleared!
        fracAC:=0;  fracMQ:=0;
        fracSI:=0;
      End else Begin        // chars get cleared, and SI strange...
        fracAC:=0;  fracMQ:=0;
        charAC:=0;  charMQ:=0;
        If fracSI=0 then  charSI:=0;
      End
    End else Begin       // We actually do the multiplication!
      fracRV :=fracAC * fracSI;                    // hi order term
      AtimesD:=(fracAC * fracSR) Shr charShft;    // lo order
      BtimesC:=(fracMQ * fracSI) Shr charShft;    // lo order
      fracSI:=BtimesC;            // This is what happens!
      fracRV:=fracRV + AtimesD + BtimesC;        // add the terms
      If normalize then OneBitNormalize
                   else SpillCheck;
// Return result fraction back to AC and MQ
      FracAC:=(FracRV And RVMSMask) Shr CharShft;
      FracMQ:=(FracRV And RVLSMask);
    End;
  End;
End;

// Divide AC by SR, quotient to MQ, remainder to AC
Procedure DoFDiv ( double : boolean );
// To divide, we basically divide the fractions and subtract the
// characteristics, adding back the 128 bias.
Begin
  Dividing:=True;
  CharMQ:=0;  FracMQ:=0;
  SignMQ:=(SignAC<>SignSR);
  If FracAC>=(2*FracSR) then begin // Out of range (incl. zero divisor)?
    DVCInd:=True;                  // Set divide check, abort operation
  End else Begin
    If FracAC = 0 then begin      // Dividend zero?
      // Yes, set result zero.
      SignAC:=false;
      CharAC:=0;
      FracAC:=0
    End else Begin                    // We actually do the division!
      FracRV:=FracAC Shl CharShft;
      If FracAC >= FracSR then begin
        FracRV:=FracRV Shr 1;
        CharAC:=CharAC+1;
        SpillCheck;
      End;
      CharMQ:=CharAC - CharSR + 128;    // Set quotient char
      FracMQ:=FracRV Div FracSR;        // Quotient
      FracAC:=FracRV Mod FracSR;        // Remainder
      CharAC:=CharAC-27;
      SpillCheck;
    End
  End
End;

{    The dreaded double-precision divide.  The algorithm is the Taylor
    expansion to 2 terms (2 being enough, as further terms don't
    contrubute to the 54-bit result fraction):

    (x+y)^-1 = x^-1 - y*x^-2 +...

    We need to rewrite this to use the A B C D convention.  To keep
    track of what's high and what's low, we'll use an apostrophe
    to indicate a multiplication by 2^-27.

    So we get
    (A+B') * (C+D')^-1 = (A+B') * C^-1 - (A+B') * D'* C^-2 +...

    Now A+B' can be left as is, since the basic hardware division
    handles this.  So we start with one division to get the first
    term on the right above - (A+B')/C.  This is called Q1:

    Q1 = (A+B')/C

    It has only 27 bits, so to get the low-order 27 bits we need to
    divide the remainder R by C.
    Thus the first term above becomes Q1 + (R/C)'.

    The second term can now be written
    - (Q1*D')/C, where we can ignore the lower 27 bits of Q1.
    This is scaled by 2^-27, because of the D', and so we can
    write it - ((Q1*D)/C)'.  It's the same scale as (R/C)', so
    these can be directly added.  Thus the whole RHS above can be
    written
    Q1 + (R - Q1*D)'/C
    We can compute the second term with one more division, giving

    Q2' = (R - Q1*D)'/C

    Thus the final result is Q1 + Q2' as written in the manual.
    I hope this explanation was a bit clearer....}
Procedure DoDFDiv;
Var signQ1:    Boolean;
    charQ1:    Integer;
    dividend,
    fracQ1,
    Q2,
    rem:    Int64;
    signRem,
    signQ2:    Boolean;
Begin
  Dividing:=True;
  signQ1:=(signAC <> signSI);        // set result sign
// Legality check on first divide.  If it fails, the manual
//  specifies that SI must be cleared, presumably so the trap
//  routine can tell which divide failed.
  If fracAC>=(2*fracSI) then begin
    DVCInd:=True;             // Set divide check
    signSI:=false;
    charSI:=0; fracSI:=0;    // Clear SI
  End Else
  Begin
    dividend:=(fracAC Shl charShft) Or fracMQ;
    If fracAC >= fracSI then begin
      dividend:=dividend Shr 1;
      Dec(charAC);
      SpillCheck
    End;
    charQ1:=charAC - charSI + 128;    // Set quotient char
// First dividion, to get Q1 fraction:
    fracQ1:=dividend Div fracSI;        // quotient Q1
    rem   :=dividend Mod fracSI;        // remainder R
    signRem:=signAC;                    // R sign = dividend sign
// Get Q1*D to fracAC:
    fracAC:=(fracQ1 * fracSR) Shr charShft; // We only need the top half
    signAC:=(signQ1<>signSI);        // Set its sign
// Get R-Q1*D to fracAC:
    If signAC=signRem then fracAC:=rem-fracAC
                      else fracAC:=rem+fracAC;
    If fracAC < 0 then begin
        fracAC:=- fracAC;
        signAC:=not signAC;
    End;
    signMQ:=signAC;

// Check legality of second divide:
    If fracAC >= (2*fracSI) then begin
      DVCInd:=True;            // Set divide check
      charAC:=0; charMQ:=0; fracMQ:=0;
    // The manual specifies that any pending overflow/underflow
    // trap condition is cancelled:
      FPTrap:=false;  SetCore(0, 0);
    // And Q1 is left in SI:
       signSI:=signQ1; charSI:=charQ1; fracSI:=fracQ1;
    End else Begin  // All OK - do 2nd division to get Q2:
      Q2:=(fracAC Shl charShft) Div fracSI;    // Q2 = (R-Q1*D)/C
      signQ2:=(signAC<>signSI);
      // Get final result fraction to fracRV:
      fracRV:=fracQ1 Shl charShft;
      // Q2 needs to be added, but if signs differ, it will be a subtraction
      If signQ1=signQ2 then fracRV:=fracRV+Q2
                       else fracRV:=fracRV-Q2;
      // Result sign is the same as Q1:
      signAC:=signQ1;  signMQ:=signQ1;
      // Set result characteristics:
      charAC:=charQ1;
      charMQ:=charQ1 - 27;
      // Now we call OneBitNormalize.  Q1 should be normalized
      // already, but if the second term is subtractive, it can lead
      // to one high-order zero in the result fraction.
      OneBitNormalize;
      // Return result fraction back to AC and MQ
      FracAC:=(FracRV And RVMSMask) Shr CharShft;
      FracMQ:=(FracRV And RVLSMask);
      // And Q1 is left in SI, probably to help debugging:
      signSI:=signQ1; charSI:=charQ1; fracSI:=fracQ1;
    End;
  End;
End;

//**************************************************************************
//  Single-Precision Floating Point
//--------------------------------------------------------------------------
Procedure IXFloatS(IR: TInstruction);
Var SR, MQ, WV: TWord;
Begin
  // Load working variables from CPU registers
  SignAC:=SgnAC;                                   // normal AC sign bit
  CharAC:=(RegAC And $1FF8000000) Shr CharShft;    // Including P and Q!
  PrevCharAC:=CharAC;
  FracAC:=(RegAC And FracMask);
  SignMQ:=( RegMQ And SignMask )<>0;
  CharMQ:=( RegMQ And CharMask ) Shr CharShft;
  FracMQ:=( RegMQ And FracMask );
  SR:=GetCore ( AddrFld );
  SignSR:=(SR And SignMask)<>0;
  CharSR:=(SR And CharMask) Shr CharShft;
  FracSR:=(SR And FracMask);
// Clear FP trap flags etc
  FPTrap:=false;
  FPTrapFlags:=0;
  Dividing:=false;
// Process specific function
  Case IR.ExRef of
    01: Begin                                                  // FAD
          DoFAdd(True, false);
        End;
    02: Begin                                                  // FAM
          SignSR:=False;
          DoFAdd(True, false);
        End;
    03: Begin                                                  // UFA
          DoFAdd(false, false);
        End;
    04: Begin                                                  // FSB
          SignSR:=Not SignSR;
          DoFAdd(True, false);
        End;
    05: Begin                                                  // UAM
          SignSR:=False;
          DoFAdd(false, false);
        End;
    06: Begin                                                  // FSM
          SignSR:=True;
          DoFAdd(True, false);
        End;
    07: Begin                                                  // UFS
          SignSR:=Not SignSR;
          DoFAdd(false, false);
        End;
    08: Begin                                                  // USM
          SignSR:=True;
          DoFAdd(false, false);
        End;
    09: Begin                                                  // FRN
          DoFRound;
        End;
    10: Begin                                                  // FMP
          DoFMult(True, false);
        End;
    11: Begin                                                  // UFM
          DoFMult(false, false);
        End;
    13: Begin                                                  // FDP
          DoFDiv(false);
        End;
    Else What(IR);
  End;  // Case
// Load Results to CPU Registers
  // Set AC Reg from work vars
  WV:=CharAC And $3FF;                            // Force to 64 bits
  WV:=WV Shl CharShft;
  SetAC(WV Or FracAC);
  SetSB(SignAC);                                  // Set AC Sign
  // Form MQ value from work vars
  WV:=CharMQ And $FF;                             // Force to 64 bits
  WV:=WV Shl CharShft;
  MQ:=WV Or FracMQ;
  If SignMQ then MQ:=MQ Or SignMask;              // Insert MQ sign
  SetMQ(MQ);                                      // Set MQ Reg
  // Finally we take a FP trap if indicated
  If FPTrap  then  FloatingPointTrap;
End;

//**************************************************************************
//  Double-Precision Floating Point
//--------------------------------------------------------------------------
Procedure IXFloatD(IR: TInstruction);
Var SI, SR, AC, MQ,WV : TWord;        // Temp full register values
begin
  // Load working variables from CPU registers
  SignAC:=SgnAC;                                 // normal AC sign bit
  CharAC:=(RegAC And $1FF8000000) Shr CharShft;    // Including P and Q!
  PrevCharAC:=CharAC;
  FracAC:=( RegAC And FracMask );
  SignMQ:=( RegMQ And SignMask ) <> 0;
  CharMQ:=( RegMQ And CharMask ) Shr CharShft;
  FracMQ:=( RegMQ And FracMask );
  // Clear FP trap flags etc
  FPTrap:=false;
  FPTrapFlags:=0;
  Dividing:=false;
{ Load operand variables from the double-length storage
  word.  The address must be even or an odd-address
  trap is indicated.  If we're in trap mode, the trap
  is taken and the operation is aborted.  If we're
  not in trap mode, the operation proceeds but with
  the word in the odd address being used for both the
  high and low part of the double precision operand.

  The exceptions are:

  1. DLD - if we're in trap mode it isn't aborted, but
           loads the regs, then takes the trap if indicated
           at the end of the op.  If it's an odd address,
           both AC and MQ get the word at the odd address.

  2. DST - Never aborts, never traps.  AC goes to the
           given address, MQ to the address plus 1,
           exactly as for two single word stores.}
  If (AddrFld And 1)<>0 then
    SetFPTrap($10);
  If (IR.ExRef=12) then begin                   // DLD
    AC:=GetCore(AddrFld);
    SetAC(AC And Wd35Mask);      // (These 2 from CLA) Clear Q,P Load 1..35
    SetSB((AC And PBitMask)<>0); // Set bool from sign bit posn
    MQ:=GetCore(AddrFld Or 1);
    SetMQ(MQ);
    Exit;
  End;
  If IR.ExRef = 13 then begin                  // DST
    WV:=RegAC And Wd35Mask;                // (These 3 from STO)
    If SgnAC then WV:=WV Or PBitMask;
    SetCore(AddrFld,WV);
    SetCore(AddrFld+1,RegMQ);
    Exit;
  End;
  If FPTrap then Exit;
  // Proceed with the operation (Not DLD or DST, and not FPTrap)
  // Load double operand variables from storage word.
  // High part goes to SI, low part to SR.
  SI:=GetCore ( AddrFld ) ;
  SignSI:=(SI And SignMask)<>0;
  CharSI:=(SI And CharMask) Shr CharShft;
  FracSI:=(SI And FracMask);
  SR:=GetCore ( AddrFld Or 1 );
  SignSR:=(SR And SignMask)<>0;
  CharSR:=(SR And CharMask) Shr CharShft;
  FracSR:=(SR And FracMask);
  // Process specific function
  Case IR.ExRef of
    01: Begin                                                 // DUFA
          DoFAdd ( false, True );
        End;
    02: Begin                                                 // DFAD
          DoFAdd ( True, True );
        End;
    03: Begin                                                 // DUFS
          signSI:=not signSI;
          DoFAdd ( false, True );
        End;
    04: Begin                                                 // DFSB
          signSI:=not signSI;
          DoFAdd ( True, True );
        End;
    05: Begin                                                 // DUAM
          signSI:=false;
          DoFAdd ( false, True );
        End;
    06: Begin                                                 // DFAM
          signSI:=false;
          DoFAdd ( True, True );
        End;
    07: Begin                                                 // DUSM
          signSI:=True;
          DoFAdd ( false, True );
        End;
    08: Begin                                                 // DFSM
          signSI:=True;
          DoFAdd ( True, True );
        End;
    09: Begin                                                 // DUFM
          DoDFMult ( false );
        End;
    10: Begin                                                 // DFMP
          DoDFMult ( True );
        End;
    11: Begin                                                 // DFDP
          DoDFDiv;
        End;
    Else What(IR);
  End;
  // Load Results back to CPU Registers
  // Set AC Reg from work vars
  WV:=CharAC And $3FF;                          // Force to 64 bits
  WV:=WV Shl CharShft;
  SetAC(WV Or FracAC);
  SetSB(SignAC);                                  // Set AC Sign
  // Form MQ value from work vars
  WV:=CharMQ And $FF;                           // Force to 64 bits
  WV:=WV Shl CharShft;
  MQ:=WV Or FracMQ;
  If SignMQ then MQ:=MQ Or SignMask;              // Insert MQ sign
  SetMQ(MQ);                                      // Set MQ Reg
  // %%% DP ops leave a value in the SI reg:
  WV:=CharSI And $FF;                         // Force to 64 bits
  WV:=WV Shl CharShft;
  SI:=WV Or FracSI;
  If SignSI then SI:=SI Or SignMask;            // Insert SI sign
  SetSI(SI);                                      // Set SI Reg
  // Finally we take a FP trap if indicated
  If FPTrap then
    FloatingPointTrap;
End;

End.

