//******************************************************************************
//  IBM 7094 Emulator - Basic Instruction set functions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains the execution functions for:-
//  Fixed Point (Integer) Arithmetic instructions
//------------------------------------------------------------------------------
Unit B709IINA;

Interface

Uses SysUtils,Dialogs,Controls,
     B709Defs, // General definitions
     B709Misc, // Miscellaneous utility functions
     B709Inst, // CPU Instruction definitions
     B709Core, // Core Storage functions
     B709Chan, // I/O Channel functions
     B709Trce; // Log/Trace functions

Procedure IXAddSub(IR: TInstruction);
Procedure IXMulDiv(IR: TInstruction);            

Implementation

Uses B709CPU;  // Access to CPU Registers

//******************************************************************************
//  Add/Subtract
//------------------------------------------------------------------------------
Procedure AddToAC(YD: TWord);
Var AC,RR: TWord;
    YS: Boolean;
Begin
  YS:=(YD And PBitMask)<>0;            // Get operand sign
  YD:=(YD And Wd35Mask);               // Get operand magnitude
  AC:=RegAC And Wd37Mask;              // Get AC magnitude (sign is RegSB)
  If YS=SgnAC then begin
    // Same signs, Just add
    RR:=AC+YD;                         // Add YD to AC
  End else Begin
    // Different signs, Subtract the smallest. Result sign same as largest.
    If AC>YD then begin
      RR:=AC-YD                        // Subtract smallest
    End else Begin
      RR:=YD-AC;                       // Subtract smallest
      If RR<>0 then SetSB(YS);         // For 0, keep original AC sign
    End;
  End;
  If (RR And PBitMask)<>0 then begin   // Overflowed?
    OVFInd:=True; INDTrcFlg:=True;     // Set OVF indicator
  End;
  SetAC(RR And Wd37Mask);              // Clean up and save
End;

Procedure IXAddSub(IR: TInstruction);
Var YD,WD: TWord;
begin
  YD:=GetCore(AddrFld);
  Case IR.ExRef of
    01: Begin                                                        // CLA
          SetAC(YD And Wd35Mask);      // Clear Q,P Load 1..35
          SetSB((YD And PBitMask)<>0); // Set bool from sign bit posn
        End;
    02: SetAS(YD And Wd36Mask,SA0);                                  // CAL
    03: Begin                                                        // CLS
          SetAC(YD And Wd35Mask);      // Clear Q,P Load 1..35
          SetSB((YD And PBitMask)=0);  // Set bool from inverted sign bit posn
        End;
    04: AddToAC(YD);                                                 // ADD
    05: AddToAC(YD And (Not PBitMask));// Clear sign & add           // ADM
    06: AddToAC(YD Xor PBitMask);      // Reverse sign & add         // SUB
    07: AddToAC(YD Or PBitMask);       // Set sign & add             // SBM
    08: Begin                                                        // ACL
          WD:=(RegAC And Wd36Mask)+YD;
          If (WD And QBitMask)>0 then
            Inc(WD);
          WD:=WD And Wd36Mask;
          SetAC((RegAC And (Not Wd36Mask)) Or WD);
        End;
    Else What(IR);
  End;
End;

//******************************************************************************
//  Multiply/Divide
//------------------------------------------------------------------------------
// Couldn't get multiply to work by the book (PoP).
// Comments "//***//" indicate where a simpler but slower method has been
// substituted.
Procedure ShiftACMQRight2;
Var AC,MQ: TWord;
    CF: Boolean;
Begin
  // Shift
  CF:=(RegAC And 1)>0;                 // Save LSB out of AC
  AC:=RegAC Shr 1;                     // Shift registers
  MQ:=RegMQ Shr 1;
  If CF then
    MQ:=MQ Or Bit1Mask; // Move AC35 to MQ1
(* //***//
  // Shift again
  CF:=(RegAC And 1)>0;                 // Save LSB out of AC
  SetAC(RegAC Shr 1);                  // Shift registers
  SetMQ(RegMQ Shr 1);
  If CF then
    SetMQ(RegMQ Or Bit1Mask); // Move AC35 to MQ1
*)
  SetAC(AC);
  SetMQ(MQ);
End;

Procedure ShiftACMQLeft1;
Var AC: TWord;
Begin
  AC:=RegAC;
  AC:=AC Shl 1;                        // Shift registers left
  SetMQ(RegMQ Shl 1);
  If (RegMQ And PBitMask)<>0 then      // Move MQ1
    AC:=AC Or 1;                       // to AC35
  SetAC(AC);
End;

Procedure Multiply(SC: Integer; YD: TWord);
Var SP,SD: Boolean;
    MQ,WD: TWord;
Begin
  MQ:=RegMQ;
  SP:=(MQ And PBitMask)>0;             // Get multiplier sign
  SD:=(YD And PBitMask)>0;             // Get multiplicand sign
  WD:=YD And Wd35Mask;                 // Get multiplicand magnitude
  SetAC(0);
  MQ:=MQ And Wd35Mask;                 // Discard MQ sign
  If WD=0 then begin                   // Is multiplicand zero?
    SetSB(False); MQ:=0;
  End;
  Repeat
    Case (MQ And $1) Of //***//// Examine 2 LSB's of MQ
      0: Begin
           SetMQ(MQ); ShiftACMQRight2; MQ:=RegMQ;
         End;
      1: Begin
           AddToAC(WD);
           SetMQ(MQ); ShiftACMQRight2; MQ:=RegMQ;
         End;
{     2: Begin//***//
           WD:=WD Shl 1;
           AddToAC(WD);
           SetMQ(MQ); ShiftACMQRight2; MQ:=RegMQ;
         End;
      3: Begin
           AddToAC(WD Or PBitMask); // Subtract by set sign and add
           SetAC(RegAC Or QBitMask Or PBitMask;
           SetMQ(MQ); ShiftACMQRight2; MQ:=RegMQ;
         End;}
    End;
    Dec(SC,1);//***//
  Until SC<=0;
  If SP=SD then begin                       // Arguments have same signs?
    SetSB(False); SetMQ(MQ And Wd35Mask);   // Set result positive
  End else Begin
    SetSB(True); SetMQ(MQ Or PBitMask);     // Else result negative
  End;
End;

Procedure Divide(SC: Integer; YD: TWord);
Var SE,SD: Boolean;
    MQ,WD: TWord;
Begin
  SE:=SgnAC;                      // Save dividend sign
  SD:=(YD And PBitMask)>0;        // Get divisor sign
  WD:=YD And Wd35Mask;            // Get divisor magnitude
  If WD<=RegAC then begin         // Divisor too large?
    DVCInd:=True; Exit;           // If so, set Divide Check Indicator
  End;
  MQ:=RegMQ;
  Repeat
    SetSB(False);                 // Keep AC positive (magnitude only)
    SetMQ(MQ); ShiftACMQLeft1; MQ:=RegMQ;
    If WD<=RegAC then begin
      AddToAC(WD Or PBitMask);    // Subtract by set sign and add
      MQ:=MQ Or 1;
    End;
    Dec(SC,1);
  Until SC=0;
  SetSB(SE);                      // 704 - Sign of remainder always same as dividend
  If SD=SE then                   // Arguments have same signs?
    SetMQ(MQ And Wd35Mask)        // Set quotient positive
  Else
    SetMQ(MQ Or PBitMask);        // Else quotient negative
End;

Procedure IXMulDiv(IR: TInstruction);
Var SC: Integer;
    YD: TWord;
begin
  YD:=GetCore(AddrFld);
  Case IR.ExRef of
    01: Begin                                                        // MPY
          SC:=35;                                // Set fixed shift count
          Multiply(SC,YD);                       // Multiply
        End;
    02: Begin                                                        // MPR
          SC:=35;                                // Set fixed shift count
          Multiply(SC,YD);                       // Multiply
          If (RegMQ And Bit1Mask)<>0 then begin  // And Round
            Inc(RegAC);
            SetAC(RegAC And Wd37Mask);
          End;
        End;
    03: If (RegMQ And Bit1Mask)<>0 then begin                        // RND
          Inc(RegAC);
          SetAC(RegAC And Wd37Mask);
        End;
    04: Begin                                                        // VLM
          SC:=(StorReg And VLACMask) Shr HalfShft;  // Get shift count
          If SC<>0 then Multiply(SC,YD);            // Multiply if SC<>0
        End;
    05: Begin                                                        // DVH
          SC:=35;                                   // Set fixed shift count
          Divide(SC,YD);                            // Do Division
          If DVCInd then                            // Did it check?
            SetStopPending('Divide Check');         // Set stop pending,
        End;
    06: Begin                                                        // DVP
          SC:=35;                                   // Set fixed shift count
          Divide(SC,YD);
        End;
    07: Begin                                                        // VDH
          SC:=(StorReg And VLACMask) Shr HalfShft;  // Get shift count
          If SC<>0 then Divide(SC,YD);              // Divide if SC<>0
          If DVCInd then                            // Did it check?
            SetStopPending('Divide Check');         // Set stop pending,
        End;
    08: Begin                                                        // VDP
          SC:=(StorReg And VLACMask) Shr HalfShft;  // Get shift count
          If SC<>0 then Divide(SC,YD);              // Divide if SC<>0
        End;
    Else What(IR);
  End;
End;

End.
