//******************************************************************************
//  IBM 7094 Emulator - Input/Output Instruction set functions
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
//  This unit contains the execution functions for:-
//  Input/Output Instructions
//------------------------------------------------------------------------------
Unit B709IIOC;

Interface

Uses SysUtils,
     B709Defs, // General definitions
     B709Misc, // Miscellaneous utility functions
     B709Inst, // CPU Instruction definitions
     B709Core, // Core Storage functions
     B709Chan, // I/O Channel and Device functions
     B709Trce; // Log/Trace functions

Procedure IXChnXfr(IR: TInstruction);
Procedure IXTestIO(IR: TInstruction);
Procedure IXInpOut(IR: TInstruction);
Procedure IXIOTran(IR: TInstruction);

Implementation

Uses B709CPU; // Access to CPU Registers

// Return TChannel object for current instruction
// either from specified instruction record or instruction decoder address field
Function GetChannel(Const IR: TInstruction): TChannel;
Var CI: Byte;
Begin
  // Get channel number from ICR ChanID or current instruction address field
  CI:=IR.ChanID;                  // Number in ICR
  If CI=0 then begin              // No, Extract from decoder address field
    CI:=(AddrFld And DaChMask) Shr DaChShft;
  End;
  // Validate channel number
  If Not (CI In [1..8]) then
    Error('Invalid channel number: '+IntToStr(CI)+
          ' used at addr '+IntToOct(PrevICtr,5)+': '+IR.Mnemonic+' - '+IR.Desc);
  Result:=Channels[CI];           // Return TChannel object
  If Result=NIL then
    Error('I/O instruction for for uninstalled Channel: '+Chr(CI+$40));
End;

//******************************************************************************
//  Input-Output commands
//------------------------------------------------------------------------------
Procedure IXInpOut(IR: TInstruction);
Var DV: TDevice;
begin
  With GetChannel(IR) do begin                   // Get TChannel object
// Should these instructions spin if the channel is busy?
//  If Not Active then begin
//    FindDevice(AddrFld,DV,BM);
      While Active do begin
//      Operate;
        If CurDevice.WRMode then OperateWR       // Do write operations
                            else OperateRD;      // Or read operations
      End;
      DV:=GetDevice(AddrFld,True);
      Case IR.ExRef of
        01: SelectDevice(DV,False);                                 // RDS
        02: SelectDevice(DV,True);                                  // WRS
        03: DV.Operate(FCBSR);                                      // BSR
        04: DV.Operate(FCBSF);                                      // BSF
        05: DV.Operate(FCWEF);                                      // WEF
        06: DV.Operate(FCREW);                                      // REW
//      07: DV.Operate(FCXXX);                                      // RUN
        08: DV.Operate(FCSDN);                                      // SDN
        Else What(IR);
      End;
//  End else Begin
//  End;
  End;
End;

//******************************************************************************
//  Input-Output transmission
//------------------------------------------------------------------------------
Procedure IXIOTran(IR: TInstruction);
Var SC,SL,SA,SD: TWord;
Begin
  With GetChannel(IR) do                         // Get TChannel object
    Case IR.ExRef of
      01: Begin                                                      // RDCc
            StopChannel;                         // Stop activity
            ResetChannel;                        // Reset channel
          End;
      02: Begin                                                      // SCHc
//          If Active then begin
              SC:=ComdReg;                       // Byte to TWord
              SC:=SC Shl 33;
              SL:=(LocnReg And AddrMask) Shl DecrShft;
              SA:=AddrReg And AddrMask;
              SD:=SC Or SL Or SA;
//          End else Begin
//            SD:=0;
//          End;
            SetCore(AddrFld,SD);
          End;
      03: Begin                                                      // RCHc
            If CurChanNum<>ChanNum then     // Is this the selected channel?
              IOCInd:=True;                 // No, Set I/O Check indicator?
//          TrapCtrlInd:=False;             // Reset trap control indicator,
            ResetChannel;                        // Reset channel
            LoadChannelCommand(AddrFld);    // then load command
          End;
      04: If Active then begin              // Channel active?       // LCHc
            ChanWait:=True;                 // Yes, Go into channel wait
            ChWCAddr:=AddrFld;              // And save address for resume
          End else Begin
            LoadChannelCommand(AddrFld);    // No, Load & Start channel
          End;
      Else Error('Unimplemented I/O instruction: '+IR.Mnemonic);
    End;
End;

//******************************************************************************
//  Control Transfer - Instructions affected by Transfer Trap Mode
//------------------------------------------------------------------------------
Procedure IXChnXfr(IR: TInstruction);
Var TA: TAddr;
Begin
  If TrapMode then begin
    TA:=GetCore(0);                    // Trapmode: set IC into Locn 0
    TA:=TA And NotAddrMask;            // Whther transfer occurs or not
    SetCore(0,TA Or (InstCtr-1));
  End;
  TA:=0;
  With GetChannel(IR) do begin                   // Get TChannel object
    Case IR.ExRef of
      01:  If Active then TA:=AddrFld;                                // TC0c
      02:  If Not Active then TA:=AddrFld;                            // TCNc
      03:  If TCKInd then begin                                       // TRCc
             TCKInd:=False;                      // Clear indicator
             TA:=AddrFld;
           End;
      04:  If EOFInd then begin                                       // TEFc
             EOFInd:=False;                      // Clear indicator
             TA:=AddrFld;
           End;
      Else Error('Unimplemented I/O Program Transfer instruction: '+IR.Mnemonic);
    End;
  End;
  // See if trap is required
  If TA<>0 then                        // Did instruction transfer?
    If TrapMode then SetIC(1,TCTrap)   // Yes, If Trapmode, Goto 1
                else SetIC(TA,TCTran); // No, sequential execution
End;

//******************************************************************************
//  I/O Tests
//------------------------------------------------------------------------------
Procedure IXTestIO(IR: TInstruction);
Var SI: Integer;
Begin
  SI:=0;                                         // Reset skip increment
  Case IR.ExRef of
    01:  With GetChannel(IR) do             // Get TChannel object
           If BOTInd then BOTInd:=False     // Clear indicator       // BTT
                     else SI:=1;            // Set skip increment
    02:  With GetChannel(IR) do
           If EOTInd then EOTInd:=False                              // ETT
                     else SI:=1;
    03:  If IOCInd then IOCInd:=False                                // IOT
                   else SI:=1;
    Else What(IR);
  End;
  SetIC(InstCtr+SI,TCTest);         // Update IC with increment
End;

End.
