//******************************************************************************
//  IBM 7094 Emulator - Operator control language Interpreter
//  By Rob Storey 2001-2004 intabits@optushome.com.au
//------------------------------------------------------------------------------
// This unit implements the interpreter for the Operator Control Language
// scripting language used by the IBM 7094 emulator.
// It runs as a separate Windows processing thread, interpreting and executing
// script langauge commands until it suspends itself to wait for the main thread
// to execute the current command. The main thread resumes the script thread
// when processing for the command has completed.
//------------------------------------------------------------------------------
Unit B709OCLI;

Interface

Uses SysUtils,Classes,Dialogs,Forms,Controls,
     StdCtrls,Windows,Messages,Contnrs,Graphics,
     B709Defs, // General definitions
     B709Misc, // Miscellaneous utility functions
     B709Eror, // Error message display form
     B709Trce, // Log/Trace functions
     B709Core, // Core Storage functions
     B709Chan, // I/O Channel and Device functions
     B709OCLD; // Script message display form

// Scripter major states
Type TScriptState=(
       SSNONE,     // No script exists
       SSRUN,      // Script is running
       SSWAIT,     // Waiting for User action, Or CPU to stop
       SSSTOP);    // Stopped, About to terminate/Return

// Language token types
Type TTokTyp=(
     TKNONE,       // Nothing
     TKEOFIL,      // End of source file
     TKCOMT,
     // Operators/Punctuation
     TKPLUS,       // "+"
     TKEQ,         // "="
     TKLT,         // "<"
     TKGT,         // ">"
     TKNE,         // "<>"
     TKCOLON,      // ":"
     TKCOMMA,      // ","
     TKLPREN,      // "("
     TKRPREN,      // ")"
     TKLSQBK,      // "["
     TKRSQBK,      // "]"
     // Literal constants
     TKNUMB,       // Numeric value
     TKSTRL,       // Literal string in quotes
     // Misc
     TKTO,
     // Commands
     TKSHOW,
     TKHIDE,
     TKCONFIG,
     TKADD,
     TKDEL,
     TKCHANS,
     TKMOUNT,
     TKLOAD,
     TKSAVE,
     TKREWIND,
     TKRELOAD,
     TKSENSW,
     TKRESET,
     TKCLEAR,
     TKDISPLAY,TKPAUSE,
     TKEXIT,
     TKHALT,
     TKCHKICTR,
     TKIF,
     TKGOTO,
     TKCALL,
     TKRETURN,
     TKCORE,
     TKZAP,
     TKEXEC,
     TKSET,
     TKSLEEP,
     TKPRESS,
     TKTRACE,TKPLOT,
     TKADRSTOP,
     // Command options
     TKSIZE,TKFILE,
     TKLODCRD,TKLODTAP,TKSTART,
     // Built in Functions
     TKOPTION,
     // Built in Identifiers & Constants
     TKMAIN,TKREADER,TKPUNCH,TKPRINTR,TKTAPE,TKTAPVIEW,
     TKCARD,TKMODE,
     TKEOF,
     TKWRPROT,
     TKON,TKOFF,
     TKBIN,TKBCD,
     TKALL,
     TKPARM,
     TKCOUNT,TKLINE,
     TKSCRTCH,
     TKWINDOW,
     TKTOP,TKLEFT,TKWIDTH,TKHEIGHT,
     TKSPACE,
     // Colour Constants
     TKBLACK, TKWHITE,
     TKMAROON,TKGREEN,TKOLIVE, TKNAVY,TKPURPLE, TKTEAL,TKGRAY,  TKSILVER,
     TKRED,   TKLIME, TKYELLOW,TKBLUE,TKFUCHSIA,TKAQUA,TKLTGRAY,TKDKGRAY,
     TKINFOBK,
     // Built-in variables
     TKINSTCTR,    // Instruction Counter Register
     TKOPTNUM,     // Selected Option number
     TKBCOLOUR,    // Script window background colour
     TKTCOLOUR,    // Script window text colour
     TKHCOLOUR,    // Script window hyperlink colour
     // Intermediate
     TKWORD);      // Unclassified AlphaNumeric

// Keyword and special symbol constants (order must be as above)
Const KeyWords: Array[Low(TTokTyp)..High(TTokTyp)] of String[16]=(
      '','endoffile ','comment',
      '+','=','<','>','<>',':',',','(',')','[',']',
      'number','string',
      'TO',
      'SHOW','HIDE',
      'CONFIG','ADD','DEL',
      'CHANNELS',
      'MOUNT','LOAD','SAVE',
      'REWIND','RELOAD',
      'SENSW',
      'RESET','CLEAR',
      'DISPLAY','PAUSE',
      'EXIT',
      'HALT',
      'CHKICTR',
      'IF','GOTO','CALL','RETURN',
      'CORE','ZAP',
      'EXEC',
      'SET',
      'SLEEP',
      'PRESS',
      'TRACE','PLOT',
      'ADRSTOP',
      'SIZE','FILE',
      'LOADCARD','LOADTAPE','START',
      'OPTION',
      'MAIN','READER','PUNCH','PRINTER','TAPE','TAPEVIEW',
      'CARD','MODE',
      'EOF',
      'PROTECT',
      'ON','OFF',
      'BIN','BCD',
      'ALL',
      'PARM',
      'COUNT','LINE',
      'SCRATCH',
      'WINDOW',
      'TOP','LEFT','WIDTH','HEIGHT',
      'SPACE',
      'BLACK','WHITE',
      'MAROON','GREEN','OLIVE','NAVY','PURPLE','TEAL','GRAY','SILVER',
      'RED','LIME','YELLOW','BLUE','FUCHSIA','AQUA','LTGRAY','DKGRAY',
      'INFOBK',
      'INSTCTR',
      'OPTNUM','BCOLOUR','TCOLOUR','HCOLOUR',
      'word');

Const   CTRESET  =100+Ord(TKRESET);
        CTCLEAR  =100+Ord(TKCLEAR);
        CTSTART  =100+Ord(TKSTART);
        CTLODCRD =100+Ord(TKLODCRD);
        CTLODTAP =100+Ord(TKLODTAP);

// Parameter list passed to sub-scripts
Type TPList=Array[0..9] Of String;

Type
  TScripter=Class
    Constructor Create;
    Destructor Destroy; Override;
  Private
    ScrStream: TMemoryStream;     // Script source text in memory stream
    ScrPosn:   Integer;           // Current position in source stream
    CurChar:   Char;              // Current source character
    Token:     String;            // Current token extracted from script source
    TokTyp:    TTokTyp;           // Its type
    TokVal:    Integer;           // And value (if numeric)
    ExecOK:    Boolean;           // Execute flag - Commands can execute
    CfgFlag:   Boolean;           // Last command was config
    DispText:  String;            // Text string built by DISPLAY command
    Parms:     TPList;            // Passed parameter list
    ScriptState:  TScriptState;         // Current scripter run state
    // Basic procedures
    procedure LoadScript(SP: String);  // Load new script file
    procedure ProcessCommands;         // Read and Process script commands
    procedure NextChar;                // Read next source character
    procedure NextToken;               // Extract next source token
    // Misc utilty procedures
    Procedure SyntaxError(ES: String);
    Procedure SyntaxErrorEF(ES: String);
    Procedure SyntaxErrorCH(ES: String);
    Procedure SyntaxErrorTK(ES: String);
    Function  GetCondition: Boolean;
    Function  GetFileSpec: String;
    Function  GetStrExpr: String;
    Function  GetTapeRef: Integer;
    Function  GetNumber(V1,V2: Int64): Integer;
    Function  GetOnOff: Boolean;
    Procedure GotoLabel(LB: String);
    Procedure FlushDispText;
    // Command implementation procedures
    Procedure ExCore;
    Procedure ExDisplay;
    Procedure ExGoto;
    Procedure ExIf;
    Procedure ExReader;
    Procedure ExCHKICTR;
    Procedure ExCall;
    Procedure ExReturn;
    Procedure ExExec;
    Procedure ExExit;
    Procedure ExSave;
    Procedure ExColour(TT: TTokTyp);
    Procedure ExLoad(TT: TTokTyp);
    procedure ExRewind;
    Procedure ExSet;
    Procedure ExSleep;
    Procedure ExZap;
    Procedure ExConfig;
    procedure ExTrace;
    procedure ExPlot;
    procedure ExPress;
    procedure ExHalt;
    procedure ExWindow;
    procedure ExShowHide(TT: TTokTyp);
    Function  CheckCPUIsOK: Boolean;
    procedure AfterConfigCheck;
    procedure ScriptError(ES: String);
    procedure PreserveLineInfo;
    procedure RestoreLineInfo;
    procedure BuildErrorText(ES: String);
    function  GetWindowHandle: HWnd;
    procedure ExAdrStop;
    procedure ExTapView;
    procedure ExPause;
  Protected
  Public
    ScriptPath: String;           // Source script pathname
    LineNumber: Word;
    CmdLineNum: Word;
    PrevLine:   String;
    CurrLine:   String;
  End;

Type
  TOCLThread=Class(TThread)
    constructor Create(RS: Boolean);
    destructor Destroy; override;
  Private
    Scripter: TScripter;                    // Currently active scripter
    ScrStack: TStack;                       // Stack of running scripters
    procedure DoShowErrorText;
    procedure RunScripter;
  Protected
    Procedure Execute; Override;            // Main execution Procedure of thread
  Public
    Procedure Reset;                        // Terminate all scripts
    Procedure RunScript(SP: String; PL: TPList);   // Run specified script
    Procedure TerminateScript;              // Terminate the current script
    Procedure TerminateAllScripts;          // Terminate all scripts
    Procedure ReStart;                      // Restart current script
    Function  ScriptState: TScriptState;    // Returns current script status
    Function  ScriptName: String;           // Returns current script pathname
  End;

Var OCLThread: TOCLThread;

Implementation

Uses B709CPU,  // To monitor CPU status
     B709Stop, // To access stop address
//   B709DPUN, // To get punch address
     B709DRDR, // To get reader address
     B709DLST, // To get printer address
     B709DTAP; // To create tape drive devices

// Some character equates
Const TB=#$09;
      LF=#$0A;
      CR=#$0D;
      EF=#$1A;
      SP=#$20;
      QU=#$27;

Const WhiteSpace=[TB,CR,LF,SP];
      Alphas=['A'..'Z','a'..'z'];
      OctDigits=['0'..'7'];
      DecDigits=['0'..'9'];
      AlphaNums=Alphas+DecDigits;
      HexDigits=DecDigits+['A'..'F','a'..'f'];

Var   OctalMode: Boolean;

Procedure ScrTrace(SS: String);
Begin
  If TraceScrs then begin
//  SendMessage(TraceFormHandle,WM_Trace,0,Longint(@SS));
//    Trace(TISCR,SS);
  End;
End;

//******************************************************************************
// OCLScripter methods
//******************************************************************************
Constructor TScripter.Create;
Begin
  ScrStream:=TMemoryStream.Create;
End;

Destructor TScripter.Destroy;
Begin
  EMCMsg(CCDISPSHOW,0,0,'');
  ScrStream.Free;
  Inherited;
End;

//*********************************************************
// Exception generation & Error handling
//*********************************************************
// Line information save areas. Used to preserve error context when the
// script file position will be altered by command functions
Var SaveLineNumber: Integer;
    SaveCmdLineNum: Integer;
    SavePrevLine:   String;
    SaveCurrLine:   String;

// Save/Restore line information needed for error reporting
Procedure TScripter.PreserveLineInfo;
Begin
  SaveLineNumber:=LineNumber;
  SaveCmdLineNum:=CmdLineNum;
  SavePrevLine:=PrevLine;
  SaveCurrLine:=CurrLine;
End;

Procedure TScripter.RestoreLineInfo;
Begin
  LineNumber:=SaveLineNumber;
  CmdLineNum:=SaveCmdLineNum;
  PrevLine:=SavePrevLine;
  CurrLine:=SaveCurrLine;
End;

// Formatted error message string passed for display to VCL by Synchronize
Var ErrorText:      String;

// Procedure invoked by Synchronize to display errors using the main VCL thread
Procedure TOCLThread.DoShowErrorText;
Begin
  ShowError('Script',ErrorText);
End;

// Build formatted error message string for display
Procedure TScripter.BuildErrorText(ES: String);
Begin
  ErrorText:='Error in line '+IntToStr(CmdLineNum)+
             ' of '+ExtractFileName(ScriptPath)+':-'+#$D+#$D;
  Try
    While LineNumber=CmdLineNum do NextToken;
  Except End;
  ErrorText:=ErrorText+PrevLine;
  ErrorText:=ErrorText+#$D+#$D+ES;
End;

// Script error message and exception generators.
// Several forms allow for inclusion of various additional text items
Procedure TScripter.ScriptError(ES: String);
Begin
  Raise Exception.Create(ES);
End;

Procedure TScripter.SyntaxError(ES: String);
Begin
  Raise Exception.Create('Syntax Error: '+ES);
End;

Procedure TScripter.SyntaxErrorCH(ES: String);
Begin
  Raise Exception.Create(ES+' character: "'+CurChar+'"');
End;

Procedure TScripter.SyntaxErrorTK(ES: String);
Begin
  Raise Exception.Create(ES+': "'+Token+'"');
End;

Procedure TScripter.SyntaxErrorEF(ES: String);
Begin
  Raise Exception.Create(ES+' expected, found: "'+Token+'"');
End;

//*********************************************************
// Character and Token Parsing
//*********************************************************
Procedure TScripter.NextChar;
Begin
  // Get CurChr to next char from script source stream, Set EOF if end of stream
  With ScrStream do
    If Position=Size then
      CurChar:=EF
    Else
      Try
        ScrStream.ReadBuffer(CurChar,1);
      Except
        CurChar:=EF;
      End;
  // Build source line for reporting
  If CurChar>=' ' then
    CurrLine:=CurrLine+CurChar;
End;

Procedure TScripter.NextToken;
Var KI: TTokTyp;
Begin
  Token:=''; TokTyp:=TKNONE;
  // Skip whitespace
  While CurChar In WhiteSpace do begin
    // On line feeds, Count lines and roll current line text into previous
    If CurChar=LF then begin
      Inc(LineNumber);
      If Trim(CurrLine)<>'' then PrevLine:=CurrLine;
      CurrLine:='';
    End;
    NextChar;
  End;
  // Classify and Build tokens
  Case CurChar of
    ';','*',
    '/':      Begin
                TokTyp:=TKCOMT;
                Repeat
                  NextChar;
                Until CurChar In [CR,LF,EF];
              End;
    'A'..'Z',
    'a'..'z': Begin
                TokTyp:=TKWORD;
                Repeat
                  Token:=Token+CurChar;
                  NextChar;
                Until Not (CurChar In AlphaNums);
              End;
    '0'..'9': If Not OctalMode then begin
                TokTyp:=TKNUMB;
                Repeat
                  Token:=Token+CurChar;
                  NextChar;
                Until Not (CurChar In DecDigits);
                Try
                  TokVal:=StrToInt(Token);
                Except
                  SyntaxErrorCH('Invalid numeric');
                End;
              End else Begin
                Tokval:=0;
                If Not (CurChar In OctDigits) then
                  SyntaxErrorCH('Invalid octal digit');
                While CurChar In OctDigits do begin
                  Token:=Token+CurChar;
                  TokVal:=(TokVal*8)+Ord(CurChar)-$30;
                  NextChar;
                End;
                TokTyp:=TKNUMB;
              End;
    '$':      Begin
                TokTyp:=TKNUMB;
                Repeat
                  Token:=Token+CurChar;
                  NextChar;
                Until Not (CurChar In HexDigits);
                Try
                  TokVal:=StrToInt(Token);
                Except
                  SyntaxErrorCH('Invalid Hexadecimal');
                End;
              End;
    '&':      Begin
                TokTyp:=TKPARM;
                Token:=CurChar;
                NextChar;
                If Not (CurChar In DecDigits) then
                  SyntaxErrorCH('Invalid Parameter specifier &');
                Token:=Token+CurChar;
                TokVal:=Ord(CurChar)-$30;
                NextChar;
              End;
    QU:       Begin
                TokTyp:=TKSTRL;
                NextChar;
                While Not (CurChar In [QU,CR,LF]) do begin
                  Token:=Token+CurChar;
                  NextChar;
                End;
                If CurChar<>QU then
                  SyntaxError('Unterminated string: "'+Token+'"');
                NextChar;
              End;
    EF:       TokTyp:=TKEOFIL;
    '+':      Begin
                TokTyp:=TKPLUS; Token:=CurChar; NextChar;
              End;
    ':':      Begin
                TokTyp:=TKCOLON; Token:=CurChar; NextChar;
              End;
    ',':      Begin
                TokTyp:=TKCOMMA; Token:=CurChar; NextChar;
              End;
    '(':      Begin
                TokTyp:=TKLPREN; Token:=CurChar; NextChar;
              End;
    ')':      Begin
                TokTyp:=TKRPREN; Token:=CurChar; NextChar;
              End;
    '[':      Begin
                TokTyp:=TKLSQBK; Token:=CurChar; NextChar;
              End;
    ']':      Begin
                TokTyp:=TKRSQBK; Token:=CurChar; NextChar;
              End;
    '=':      Begin
                TokTyp:=TKEQ; Token:=CurChar; NextChar;
              End;
    '<':      Begin
                TokTyp:=TKLT;
                NextChar;
                If CurChar='>' then begin
                  TokTyp:=TKNE;
                  NextChar;
                End;
              End;
    Else      SyntaxErrorCH('Invalid character: "'+CurChar+'"'+
                            '($'+IntToHex(Ord(CurChar),2)+')');
  End;
  // Classify Keyword tokens
  If TokTyp=TKWORD then begin
    Token:=UpperCase(Token);
    For KI:=Low(TTokTyp) to High(TTokTyp) do
      If KeyWords[KI]=Token then begin
        TokTyp:=KI; Break;
      End;
  End;
  // Substitute parameter values
  If TokTyp=TKPARM then begin
    TokTyp:=TKSTRL;
    Token:=Parms[TokVal];
  End;
End;

//*********************************************************
// Common syntax construct parsers
//*********************************************************
// Get a boolean value
// [=] ON|OFF
Function TScripter.GetOnOff: Boolean;
Begin
  If TokTyp=TKEQ then NextToken;
  If Not (TokTyp In [TKON,TKOFF]) then
    SyntaxErrorEF('ON or OFF');
  Result:=TokTyp=TKON;
  NextToken;
End;

// Get a string expression: str [ + str ]...
// <string> ["+" <string>]
Function TScripter.GetStrExpr: String;
Begin
  Result:='';
  Repeat
    If Not (TokTyp In [TKSTRL,TKWORD]) then
      SyntaxErrorEF('String expression');
    Result:=Result+Token;
    NextToken;
    If TokTyp<>TKPLUS then Break;
    NextToken;
  Until False;
End;

// Get a numeric value, within specified range
// <number>
Function TScripter.GetNumber(V1,V2: Int64): Integer;
Begin
  If TokTyp<>TKNUMB then
    SyntaxErrorEF('Numeric value');
  If (TokVal<V1) And (V1<>0) then
    SyntaxErrorTK('Value too small. Min allowed='+IntToStr(V1));
  If (TokVal>V2) And (V2<>0) then
    SyntaxErrorTK('Value too large. Max allowed='+IntToStr(V2));
  Result:=TokVal;
  NextToken;
End;

// Get a Relational expression. Returns the expression's boolean value
// OPTNUM|<numeric value>|<variable name>  "="|"<>" <numeric value>
Function TScripter.GetCondition: Boolean;
Var V1,V2: Integer;
    RO: TTokTyp;
Begin
  Case TokTyp of
    TKINSTCTR: V1:=INSTCTR;
    TKOPTNUM:  V1:=ScriptForm.SelectIndex;
    TKSTRL:    Try
                 V1:=StrToInt(Token);
               Except
                 V1:=0;
                 SyntaxErrorTK('Invalid numeric value');
               End;
    Else       Begin
                 V1:=0;
                 SyntaxErrorEF('Variable identifier');
               End;
  End;
  NextToken;
  Case TokTyp of
    TKEQ,
    TKNE:     RO:=TokTyp;
    Else      Begin
                RO:=TKNONE;
                SyntaxErrorEF('"=" or "<>"');
              End;
  End;
  NextToken;
  Case TokTyp of
    TKNUMB:   V2:=TokVal;
    Else      Begin
                V2:=0;
                SyntaxErrorEF('Value');
              End;
  End;
  Case RO of
    TKEQ: Result:=V1=V2;
    TKNE: Result:=V1<>V2;
    Else  Result:=False;
  End;
  NextToken;
End;

// Get a tape reference, using either:-
// "CCUUU"=octal address or "CUU" channel letter+unit number
// TAPE [=] ccuuu|Cuu
Function TScripter.GetTapeRef: Integer;
Var UU: Byte;
Begin
  Result:=0;
  If TokTyp<>TKTAPE then SyntaxErrorEF('TAPE');
  NextToken; If TokTyp=TKEQ then NextToken;
  Case TokTyp Of
    TKNUMB: Result:=TokVal;
    TKWORD: Begin
              If Not (Token[1] In ['A'..'H']) then
                SyntaxError('Invalid tape channel: '+Token[1]);
              Result:=((Ord(Token[1])-$40)*512)+128;
              Try
                Delete(Token,1,1);
                UU:=StrToInt(Token);
              Except
                SyntaxError('Invalid tape unit number: '+Token);
                UU:=0;
              End;
              If Not (UU In [1..10]) then SyntaxError('Invalid tape unit number: '+Token);
              Result:=Result+UU;
            End;
    Else    SyntaxErrorEF('Tape address');
  End;
  NextToken;
End;

// Get a file specification
// SCRATCH|(FILE ["="] <string>)
Function TScripter.GetFileSpec: String;
Begin
  If TokTyp=TKSCRTCH then begin
    Result:=''; NextToken; Exit;
  End;
  If TokTyp<>TKFILE then SyntaxErrorEF('FILE');
  NextToken;
  If TokTyp=TKEQ then NextToken;
  If TokTyp<>TKSTRL then
    SyntaxErrorEF('Tape file name');
  Result:=GetStrExpr;
End;

// Get a window name, return its handle
// MAIN|CORE|READER|PUNCH|PRINTER|TAPE|TRACE
Function TScripter.GetWindowHandle: HWnd;
Var FI: TFormID;
Begin
  Case TokTyp of
    TKMAIN:    FI:=FIMain;
    TKCORE:    FI:=FICore;
    TKREADER:  FI:=FIReader;
    TKPUNCH:   FI:=FIPunch;
    TKPRINTR:  FI:=FIPrinter;
    TKTAPE:    FI:=FITapes;
    TKTAPVIEW: FI:=FITapeView;
    TKTRACE:   FI:=FITrace;
    Else       Begin
                 SyntaxErrorTK('Unknown window ID'); FI:=FIMain;
               End;
  End;
  Result:=FormHandles[FI];
  NextToken;
End;

//*********************************************************
// Common command execution functions
//*********************************************************
// Reposition source stream to specified label
Procedure TScripter.GotoLabel(LB: String);
Begin
  // Save line context for error reporting
  PreserveLineInfo;
  // Rewind script file and Re-prime lexical analyzer
  ScrStream.Seek(0,SOFromBeginning);
  NextChar; NextToken;
  // Search for specified label
  LB:=UpperCase(LB);
  Repeat
    While TokTYP<>TKCOLON do begin                    // Find the next colon
      If TokTyp=TKEOFIL then begin                    // Hit EOF?
        RestoreLineInfo;                              // Yes, Setup for error
        ScriptError('Label not found: "'+LB+'"');     // and report
      End;
      NextToken;
    End;
    NextToken;                                   // ":" found, what follows...
    If TokTyp<>TKWORD then                       // Must be a TKWORD
      SyntaxErrorEF('Label identifier');
  Until Token=LB;                                // Is it the label we want?
  NextToken;                                     // Yes, found
End;

// Load and run a new script file
Procedure TScripter.LoadScript(SP: String);
begin
  ScrTrace('Script Load '+SP);
  If ExtractFileExt(SP)='' then
    SP:=SP+'.EC7';
  ScriptPath:=SP;
  // Load script file and prime lexical analyzer
  Try
    ScrStream.LoadFromFile(SP);
  Except
    On E: Exception do
      Raise Exception.Create('Error opening file "'+SP+'"');
  End;
  LineNumber:=1;  PrevLine:=''; CurrLine:='';
  NextChar;
  NextToken;
  // Make it runable
  ScriptState:=SSRUN;
End;

//*********************************************************
// Script language command execution Procedures
//*********************************************************

// SHOW|HIDE [WINDOW "="] *(<window name> ","}
Procedure TScripter.ExShowHide(TT: TTokTyp);
Var WH: HWnd;
    SS: Integer;
Begin
  If TokTyp=TKWINDOW then begin
    NextToken;
    If TokTyp=TKEQ then NextToken;
  End;
  Repeat
    WH:=GetWindowHandle;
    If ExecOK And (WH<>0) then begin
      If TT=TKSHOW then SS:=SW_Show
                   else SS:=SW_Hide;
      ShowWindow(WH,SS);
    End;
    // More window ID's to be shown/hidden?
    If TokTyp=TKCOMMA then NextToken
                      else Break;
  Until False;
End;

// WINDOW <window name> *(TOP|LEFT|WIDTH|HEIGHT ["="] <value> ",")
Procedure TScripter.ExWindow;
Var WH: HWnd;
    FT: String;
    FC: TTokTyp;
    WR: TRect;
    NV: Integer;
Begin
  WH:=GetWindowHandle;
  Repeat
    OctalMode:=False;
    If Not (TokTyp In [TKTOP,TKLEFT,TKWIDTH,TKHEIGHT]) then
      SyntaxError('Invalid WINDOW function: '+FT);
    FC:=TokTyp;
    NextToken;
    If TokTyp=TKEQ then NextToken;
    NV:=GetNumber(0,1000);
    If ExecOK And (WH<>0) then begin
      // Get window screen extents
      GetWindowRect(WH,WR);
      // Convert Bottom/Right to Height/Width
      With WR do begin
        Right:=Right-Left;
        Bottom:=Bottom-Top;
      End;
      // Update the specified value
      Case FC Of
        TKTOP:    WR.Top:=NV;
        TKLEFT:   WR.Left:=NV;
        TKWIDTH:  WR.Right:=NV;
        TKHEIGHT: WR.Bottom:=NV;
      End;
      // Change the window's position/size
      With WR do
        MoveWindow(WH,Left,Top,Right,Bottom,True);
    End;
    // More attributes to change for this window?
    If TokTyp=TKCOMMA then NextToken
                      else Break;
  Until False;
End;

// <number>|<colourname>
Procedure TScripter.ExColour(TT: TTokTyp);
Var CV: Integer;
Begin
  If Not (TokTyp In [TKNUMB,TKBLACK..TKWHITE]) then
    SyntaxErrorEF('Colour constant or value');
  If TokTyp=TKNUMB then CV:=TokVal
                   else CV:=StringToColor('CL'+Token);
  NextToken;
  If ExecOK then begin
    ScrTrace('Colour');
    Case TT of
      TKTCOLOUR: ScriptForm.TColour:=CV;
      TKHCOLOUR: ScriptForm.HColour:=CV;
      Else       ScriptForm.Color:=CV;
    End;
  End;
End;

// (SET CHANNELS ["="] <channelcount>) |
// (ADD TAPE ["="] <deviceaddr>)        |
// (DEL (TAPE ["="] <deviceaddr>)|ALL)
Procedure TScripter.ExConfig;
Var FT,
    TT: TTokTyp;
    DA: Integer;
Begin
  FT:=TokTyp;
  If Not (FT In [TKSET,TKADD,TKDEL]) then
    SyntaxErrorEF('"ADD" "DEL" or "SET" Config function type');
  NextToken;
  TT:=TokTyp;
  Case TT Of
    TKCHANS: Begin
               If FT<>TKSET then
                 ScriptError('CONFIG CHANNELS function must be "SET"');
               NextToken; If TokTyp=TKEQ then NextToken;
               If TokTyp<>TKNUMB then SyntaxErrorEF('Channel count value');
               If TokVal>8       then SyntaxErrorTK('Invalid channel');
               DA:=TokVal;
               NextToken;
             End;
    TKTAPE:  Begin
               DA:=GetTapeRef;
             End;
    TKALL:   Begin
               If FT<>TKDEL then
                 ScriptError('"ALL" valid only for CONFIG DEL');
               DA:=0;
               NextToken;
             End;
    Else     Begin;
               SyntaxErrorTK('Invalid CONFIG operand');
               DA:=0;
             End;
  End;
  If ExecOK then begin
    ScrTrace('Config '+KeyWords[FT]);
    Case FT Of
      TKSET: EMCMsg(CCGENCHNS,DA,0,'');
      TKADD: EMCMsg(CCTAPEDEF,DA,0,'');
      TKDEL: EMCMsg(CCTAPEDEL,DA,0,'');
    End;
  End;
  CfgFlag:=True;
  Application.ProcessMessages;
End;

// ((LOAD|MOUNT) (TAPE <tape ref>)|READER <file spec>) |
// (RELOAD (TAPE <tape ref>)|READER)
Procedure TScripter.ExLoad(TT: TTokTyp);
Var DA: Word;
    FN: String;
Begin
  Case TokTyp Of
    TKREADER: Begin DA:=RDRDev.DevAddr; NextToken; End;
    TKTAPE:   DA:=GetTapeRef;
    Else      Begin
                SyntaxErrorEF('TAPE or READER'); DA:=0;
              End;
  End;
  If TT In [TKLOAD,TKMOUNT] then
    FN:=GetFileSpec;
  If ExecOK then begin
    Case TT Of
      TKMOUNT,
      TKLOAD:   IOCMsg(DA,FCOPEN,0,FN);
      TKRELOAD: IOCMsg(DA,FCRELOAD,0,FN);
    End;
  End;
End;

// REWIND (TAPE <tape ref>)|READER
Procedure TScripter.ExRewind;
Var DA: Word;
Begin
  DA:=0;
  Case TokTyp Of
    TKTAPE:   DA:=GetTapeRef;
    TKREADER: Begin
                DA:=RDRDev.DevAddr;
                NextToken;
              End;
    Else      SyntaxErrorEF('TAPE or READER');
  End;
  If ExecOK then
    IOCMsg(DA,FCREWIND,0,'');
End;

// READER CLEAR|(CARD <string>)|LOAD <file spec>
Procedure TScripter.ExReader;
Var FC: Word;
    SS: String;
Begin
  FC:=0; SS:='';
  Case TokTyp Of
    TKCLEAR: Begin
               NextToken;
               FC:=FCCLOSE;
             End;
    TKCARD:  Begin
               NextToken;
               If TokTyp=TKEOF then begin
                 FC:=FCLODEOF; NextToken;
               End else Begin
                 FC:=FCLODREC; SS:=GetStrExpr;
               End;
             End;
    TKLOAD:  Begin
               NextToken;
               SS:=GetFileSpec;
               FC:=FCLODFIL;
             End
    Else     SyntaxErrorEF('CLEAR, CARD or LOAD');
  End;
  If ExecOK then
    IOCMsg(RDRDev.DevAddr,FC,0,SS);
End;

// SAVE PUNCH|PRINTER|(TAPE <taperef>) [[TO] <file spec>]
Procedure TScripter.ExSave;
Var WT: TTokTyp;
    DA: Word;
    FN: String;
Begin
  WT:=TokTyp;
  Case WT Of
//  TKPUNCH:  Begin DA:=PUNDev.DevAddr; NextToken; End;
    TKPRINTR: Begin DA:=LSTDev.DevAddr; NextToken; End;
    TKTAPE:   DA:=GetTapeRef;
    Else      Begin
                SyntaxErrorEF('PUNCH, PRINTER or TAPE'); DA:=0;
              End;
  End;
  If TokTyp=TKTO then begin
    NextToken;
    FN:=GetFileSpec;
  End else Begin
    If TokTyp=TKFILE then FN:=GetFileSpec
                     else FN:='';
  End;
  If ExecOK then begin
    ScrTrace('Save');
    IOCMsg(DA,FCSAVE,0,FN);
  End;
End;

// PRESS RESET|CLEAR|START|LOADCARD|LOADTAPE
Procedure TScripter.ExPress;
Var FC: Word;
Begin
  Case TokTyp Of
    TKRESET:  FC:=CTRESET;
    TKCLEAR:  FC:=CTCLEAR;
    TKSTART:  FC:=CTSTART;
    TKLODCRD: FC:=CTLODCRD;
    TKLODTAP: FC:=CTLODTAP;
    Else      Begin
                SyntaxErrorTK('Invalid PRESS option:'); FC:=0;
              End;
  End;
  NextToken;
  If ExecOK then begin
    EMCMsg(CCPRESS,FC,0,''); // Tell CPU to Start/Load/Reset etc
  End;
End;

// GOTO <labelref>
Procedure TScripter.ExGoto;
Var SL: String;
Begin
  If TokTyp<>TKWORD then
    SyntaxErrorEF('Label identifier');
  SL:=Token;
  NextToken;
  If ExecOK then begin
    ScrTrace('Goto '+SL);
    GotoLabel(SL);
  End;
End;

// CALL <label ref> ["(" <string> ")"]
Procedure TScripter.ExCall;
Var SL,PS: String;
Begin
  If TokTyp<>TKWORD then
    SyntaxErrorEF('Label identifier');
  SL:=Token;
  NextToken;
  If TokTyp=TKLPREN then begin
    NextToken;
    If TokTyp<>TKSTRL then
      SyntaxError('CALL Parameter must be a string literal');
    PS:=GetStrExpr;
    If TokTyp<>TKRPREN then
      SyntaxErrorEF('")"');
    NextToken;
  End else Begin
    PS:='';
  End;
  If ExecOK then begin
    ScrTrace('Call '+SL+'('+PS+')');
    Parms[1]:=PS;
    ScrPosn:=ScrStream.Position;
    GotoLabel(SL);
  End;
End;

// RETURN
Procedure TScripter.ExReturn;
Begin
  If ExecOK then begin
    ScrTrace('Return');
    If ScrPosn=0 then
      ScriptError('"RETURN" with no previous CALL');
    ScrStream.Position:=ScrPosn;
    NextChar;
  End;
End;

// EXEC <script name> ["(" <string>|<number> [8([","] <string>|<number>)] ")"]
Procedure TScripter.ExExec;
Var SP: String;
    PL: TPList;
    PC: Integer;
Begin
  If Not (TokTyp In [TKSTRL,TKWORD]) then
    SyntaxErrorEF('Script file name');
  SP:=Token;
  NextToken;
  // Load parameter list
  If TokTyp=TKLPREN then begin
    NextToken;
    PC:=0; PL[0]:=SP;             // Insert exec name into PL
    While TokTyp<>TKRPREN do
      If PC=9 then
        SyntaxError('Too many parameters on Exec call');
      Case TokTyp Of
        TKSTRL,
        TKNUMB:  Begin
                   Inc(PC); PL[PC]:=Token; NextToken;
                 End;
        TKCOMMA: NextToken;
        Else     SyntaxError('Invalid EXEC parameter "'+Token+'"');
      End;
    If TokTyp<>TKRPREN then
      SyntaxErrorEF('")"');
    NextToken;
  End;
  // Run the new script
  If ExecOK then begin
    ScrTrace('Exec '+SP);
    OCLThread.RunScript(SP,PL);   // Activate the new one
  End;
End;

// EXIT
Procedure TScripter.ExExit;
Begin
  If ExecOK then begin
    ScrTrace('Exit');
    EMCMsg(CCDISPSHOW,0,0,'');
    ScriptState:=SSSTOP;          // End this script
  End;
End;

// HALT
Procedure TScripter.ExHalt;
Begin
  If ExecOK then begin
    ScrTrace('Halt');
    OCLThread.TerminateAllScripts;  // End this script and all others
    ScriptState:=SSSTOP;            // Make non runable to exit command loop
  End;
End;

// CHKICTR <address>
Procedure TScripter.ExCHKICTR;
Var SA: TAddr;
    MS: String;
Begin
  If TokTyp<>TKNUMB then
    SyntaxErrorEF('Address value');
  SA:=GetNumber(0,CoreSize);
  If TokTyp=TKSTRL then begin
    MS:=Trim(Token); NextToken;
  End else Begin
    MS:='';
  End;
  If ExecOK then begin
    ScrTrace('IARCheck');
    If SA<>INSTCTR then begin
      With ScriptForm do begin
        Color:=CLRed; TColour:=CLWhite;
        ClearDispItems;
        ContEnable:=True;
        If MS<>'' then AddDispItem(MS,DITextLabel)
                  else AddDispItem('Unexpected Error:-',DITextLabel);
        AddDispItem('',DIBlankLine);
        AddDispItem('The emulator has stopped at address '+IntToOct(INSTCTR,5)+#$D+
                    ', but a stop at address '+IntToOct(SA,5)+' was expected.',
                    DITextLabel);
        AddDispItem('',DIBlankLine);
      End;
      EMCMsg(CCDISPSHOW,1,0,'');
      ScriptState:=SSWAIT;  // Wait for user action
    End;
  End;
End;

// CORE (SIZE ["="] <address>) | ("[" <address> "]" "=" <number>)
Procedure TScripter.ExCore;
Var TK: TTokTyp;
    SA: TAddr;
    NV: TWord;
Begin
  TK:=TokTyp; SA:=0; NV:=0;
  Case TokTyp of
    TKSIZE:  Begin
                NextToken; If TokTyp=TKEQ then NextToken;
                SA:=GetNumber(4096,MaxCore);
             End;
    TKLSQBK: Begin
               NextToken;
               SA:=GetNumber(0,CoreSize);
               If TokTyp<>TKRSQBK then SyntaxErrorEF(']'); NextToken;
               If TokTyp<>TKEQ    then SyntaxErrorEF('='); NextToken;
               NV:=GetNumber(0,(1 Shl 36));
             End;
    Else     SyntaxErrorEF('SIZE or "["');
  End;
  If ExecOK then begin
    If TK=TKSIZE then EMCMsg(CCSETCORSIZ,SA,0,'')
//               else EMCMsg(CCSETCORVAL,SA,NV,'');
                 else SetCore(SA,NV);
  End;
End;

// ZAP READER|(TAPE <taperef>) "[" <number> "]" "=" <number>
Procedure TScripter.ExZap;
Var DA: Word;
    SA: LongWord;
    NV: Byte;
Begin
  Case TokTyp Of
    TKREADER: Begin DA:=RDRDev.DevAddr; NextToken; End;
    TKTAPE:   DA:=GetTapeRef;
    Else      Begin
                SyntaxErrorEF('TAPE or READER'); DA:=0;
              End;
  End;
  SA:=GetNumber(0,MaxInt);
  If TokTyp<>TKRSQBK then SyntaxErrorEF(']'); NextToken;
  If TokTyp<>TKEQ    then SyntaxErrorEF('='); NextToken;
  NV:=GetNumber(0,255);
  If ExecOK then begin
    ScrTrace('Zap');
    IOCMsg(DA,FCZAPA,SA,'');
    IOCMsg(DA,FCZAPV,NV,'');
  End;
End;

// "IF" command sets "ExecOK" to true or false, Others commands then test it
// IF <condition> <command>
Procedure TScripter.ExIf;
Begin
  ExecOK:=GetCondition;
  If ExecOK then ScrTrace('IF - Condition True')
            else ScrTrace('IF - Condition False');
End;

// DISPLAY *(<string>|SPACE|(OPTION "(" <string> ")"))
Procedure TScripter.ExDisplay;
Begin
  With ScriptForm do begin
    // Clear display form of items from previous displays
    ClearDispItems;
    // Process parameters to command, Creating display items as required
    ContEnable:=True;
    Repeat
      Case TokTyp Of
        TKSTRL:   Begin
                    DispText:=Token;
                    FlushDispText;
                  End;
        TKSPACE:  AddDispItem('',DIBlankLine);
  //    TKINSTCTR:DispText:=DispText+IntToStr(INSTCTR);
        TKOPTION: Begin
                    FlushDispText;
                    NextToken;
                    If TokTyp<>TKLPREN then SyntaxErrorEF('"("'); NextToken;
                    If TokTyp<>TKSTRL then SyntaxErrorEF('String literal');
                    AddDispItem(Token,DIClikLabel);
                    NextToken;
                    If TokTyp<>TKRPREN then SyntaxErrorEF('")"');
                    ContEnable:=False;
                  End;
        Else      Break;
      End;
      NextToken;                 // Get next item
    Until False;
    FlushDispText;               // Output any remaining text
  End;
  If ExecOK then begin           // Active?
    ScrTrace('Display Wait');
    EMCMsg(CCDISPSHOW,1,0,'');
    ScriptState:=SSWAIT;      // Wait for user action
  End;
End;

// Message the display form to create a textlabel for any text built by DISPLAY
Procedure TScripter.FlushDispText;
Begin
  If DispText<>'' then begin
    ScriptForm.AddDispItem(DispText,DITextLabel);
    DispText:='';
  End
End;

// PAUSE [<string>]
Procedure TScripter.ExPause;
Begin
  With ScriptForm do begin
    // Clear display form of items from previous displays
    ClearDispItems;
    AddDispItem('Script Paused.',DIClikLabel);
    // Process parameters to command, Creating display items as required
    ContEnable:=True;
    If TokTyp=TKSTRL then begin
      AddDispItem(Token,DITextLabel);
      NextToken;
    End;
    FlushDispText;               // Output any remaining text
  End;
  If ExecOK then begin           // Active?
    ScrTrace('Display Wait');
    EMCMsg(CCDISPSHOW,1,0,'');
    ScriptState:=SSWAIT;      // Wait for user action
  End;
End;

// SET (SENSESW <number>|ALL ON|OFF)|(TAPE <taperef> WRPROT ON|OFF)
Procedure TScripter.ExSet;
Var DA: Word;
    BV: Boolean;
Begin
  Case TokTyp Of
    TKSENSW: Begin
               NextToken;
               If TokTyp=TKALL then begin
                 NextToken;
                 BV:=GetOnOff;
                 If ExecOK then
                   For DA:=1 to 6 do
                      EMCMsg(CCSETSENSW,DA,Ord(BV),'');
               End else Begin
                 DA:=GetNumber(1,6);
                 BV:=GetOnOff;
                 If ExecOK then
                   EMCMsg(CCSETSENSW,DA,Ord(BV),'');
               End;
             End;
    TKTAPE:  Begin
               DA:=GetTapeRef;
               If TokTyp<>TKWRPROT then SyntaxErrorEF('WRPROT');
               NextToken; If TokTyp=TKEQ then NextToken;
               BV:=GetOnOff;
               If ExecOK then
                IOCMsg(DA,FCWRPROT,Ord(BV),'');
             End;
    Else     SyntaxErrorEF('SET command option');
  End;
End;

// SLEEP <number>
Procedure TScripter.ExSleep;
Var MS: Integer;
Begin
  If TokTyp=TKNUMB then begin
    MS:=TokVal; NextToken;
  End else
    MS:=1000;
  If ExecOK then begin
    ScrTrace('Sleep');
    Sleep(MS);
  End;
End;

// TRACE ON|OFF|(SAVE [[TO] <filespec>])
Procedure TScripter.ExTrace;
Var TK: TTokTyp;
    BV: Boolean;
    FN: String;
Begin
  TK:=TokTyp; BV:=False;
  If TokTyp=TKSAVE then begin
    NextToken;
    If TokTyp=TKTO then begin
      NextToken;
      FN:=GetFileSpec;
    End else Begin
      If TokTyp=TKFILE then FN:=GetFileSpec
                       else FN:='';
    End;
  End else Begin
    BV:=GetOnOff;
  End;
  If ExecOK then
    If TK=TKSAVE then EMCMsg(CCSAVTRACE,0,0,FN)
                 else EMCMsg(CCTRACE,Ord(BV),0,'');
End;

// PLOT *(ON|OFF|CLEAR [","])
Procedure TScripter.ExPlot;
Var BV: Boolean;
Begin
  Repeat
    Case TokTyp Of
      TKON,
      TKOFF:   Begin
                 BV:=GetOnOff;
                 If ExecOK then EMCMsg(CCPLOTONOFF,Ord(BV),0,'');
               End;
      TKCLEAR: Begin
                 NextToken;
                 If ExecOK then EMCMsg(CCPLOTCLEAR,0,0,'');
               End;
      Else     SyntaxErrorEF('ON, OFF or CLEAR');
    End;
    If TokTyp=TKCOMMA then NextToken
                      else Break;
  Until False;
End;

// ADSTOP ("=" <address> [COUNT "=" <number>])|OFF
Procedure TScripter.ExAdrStop;
Var AD,SC: Word;
Begin
  SC:=1;
  If TokTyp=TKOFF then begin
    NextToken;
    AD:=$8000;
  End else Begin
    If TokTyp=TKEQ then NextToken;
    AD:=GetNumber(0,CoreSize);
    If TokTyp=TKCOUNT then begin
      NextToken; If TokTyp=TKEQ then NextToken;
      SC:=GetNumber(1,MaxInt);
    End;
  End;
  If ExecOK then begin
    FetchStopAddr:=AD;
    FetchStopCount:=SC;
    FetchStopStop:=True;
    FetchStopTrace:=False;
  End;
End;

// TAPEVIEW (TAPE <taperef>)|([MODE ["="] BIN|BCD)|
//          (LINE ["="] <number>)|(SAVE [TO] <file spec>)
Procedure TScripter.ExTapView;
Var DA,LN: Integer;
    BM: Boolean;
    SS: String;
Begin
  Repeat
    Case TokTyp Of
      TKTAPE:  Begin
                 DA:=GetTapeRef;
                 If ExecOK then EMCMsg(CCTAPVADR,DA,0,'');
               End;
      TKMODE:  Begin
                 NextToken; If TokTyp=TKEQ then NextToken;
                 BM:=TokTyp=TKBCD; NextToken;
                 If ExecOK then EMCMsg(CCTAPVMOD,Ord(BM),0,'');
               End;
      TKLINE:  Begin
                 OctalMode:=False;
                 NextToken; If TokTyp=TKEQ then NextToken;
                 LN:=GetNumber(1,0);
                 If ExecOK then EMCMsg(CCTAPVLIN,LN,0,'');
               End;
      TKSAVE:  Begin
                 NextToken; If TokTyp=TKTO then NextToken;
                 SS:=GetFileSpec;
                 If ExecOK then EMCMsg(CCTAPVSAV,0,0,SS);
               End;
    End;
    If TokTyp=TKCOMMA then NextToken
                      else Break;
  Until False;
End;

// Check that CPU is operational, And whether script should continue
Function TScripter.CheckCPUIsOK: Boolean;
Begin
  // Check in CPU or Channels?
  Result:=Not (CPUCheck Or ChnCheck);
  If Not Result then begin
    // Yes, find out if we should continue
    Result:=MessageDlg('CPU/Channel Check exists. Continue running script?',
                       MTWarning,[MBYES,MBNO],0)=MRYES;
    If Not Result then begin
      ResetChannels;                  // Reset Channels & devices
      OCLThread.TerminateAllScripts;
    End;
  End;
End;

// To avoid screen flicker from creation of dynamic displays (eg. tape forms),
// Creation is deferred until a non-config command appears
Procedure TScripter.AfterConfigCheck;
Begin
  // If the configuration command flag has beem set, and the next command is a
  // not a config command, invoke creation of dynamic displays all in one hit
  If (TokTyp<>TKCONFIG) And       // Not a config command?
     CfgFlag then begin           // But config has been executed?
    EMCMsg(CCENDCONFIG,0,0,'');   // Yes, Create any oustanding displays
    CfgFlag:=False;               // Reset flag. Will be set by next config command
  End;
End;

//*********************************************************
// Scripter command processing loop
//*********************************************************
Procedure TScripter.ProcessCommands;
Var TT: TTokTyp;
    TK: String;
Begin
  ExecOK:=True;
  Repeat
    Application.ProcessMessages;       // Let other stuff hve a go
    If Not CheckCPUIsOK then Exit;     // Check that CPU is operational
    AfterConfigCheck;                  // Generate any dynamic displays pending
    CmdLineNum:=LineNumber;            // Note line number for error reporting
    TT:=TokTyp; TK:=Token;             // Save command token details,
    OctalMode:=True;                   // Default to octal numbers
    NextToken;                         // Get parameter token ready
    // Process command
    Case TT of
      TKCOMT:     Begin End;
      TKCOLON:    NextToken;
      TKSHOW,
      TKHIDE:     ExShowHide(TT);
      TKBCOLOUR,
      TKTCOLOUR,
      TKHCOLOUR:  ExColour(TT);
      TKCONFIG:   ExConfig;
      TKMOUNT,
      TKRELOAD,
      TKLOAD:     ExLoad(TT);
      TKREWIND:   ExRewind;
      TKREADER:   ExReader;
      TKSAVE:     ExSave;
      TKPRESS:    ExPress;
      TKSLEEP:    ExSleep;
      TKTRACE:    ExTrace;
      TKPLOT:     ExPlot;
      TKADRSTOP:  ExAdrStop;
      TKWINDOW:   ExWindow;
      TKTAPVIEW:  ExTapView;
      TKDISPLAY:  ExDisplay;
      TKPAUSE:    ExPause;
      TKGOTO:     ExGoto;
      TKCALL:     ExCall;
      TKRETURN:   ExReturn;
      TKEXEC:     ExExec;         // Activates new script
      TKEXIT:     ExExit;
      TKHALT:     ExHALT;
      TKCHKICTR:  ExCHKICTR;
      TKCORE:     ExCore;
      TKZAP:      ExZap;
      TKIF:       ExIf;           // IF sets up ExecOK, others test it
      TKSET:      ExSet;
      TKEOFIL:    ScriptState:=SSSTOP;
      Else        Begin
                    Token:=TK;
                    SyntaxErrorTK('Unknown script command');
                  End;
    End;
    // Enable execution for next command,
    // unless last command was IF, which may have disabled it
    If TT<>TKIF then ExecOK:=True;
  Until ScriptState<>SSRUN;
End;

//******************************************************************************
// OCLThread methods
//******************************************************************************
Constructor TOCLThread.Create(RS: Boolean);
Begin
  Inherited Create(RS);
  ScrStack:=TStack.Create;
//Priority:=TPIdle;
End;

Destructor TOCLThread.Destroy;
Begin
  Inherited;
  Reset;
  ScrStack.Free;
End;

// Destroy all scripter objects
Procedure TOCLThread.Reset;
begin
  While Scripter<>NIL do
    TerminateScript;
  If ScriptForm<>NIL then
    With ScriptForm do begin
      Color:=CLInfoBk;
      TColour:=CLBlack;
      HColour:=CLNavy;
    End;
End;

// Run a script
Procedure TOCLThread.RunScript(SP: String; PL: TPList);
begin
  If Scripter<>NIL then              // Was a script already running?
    Scripter.ScriptState:=SSWAIT;    // Yes, Disable it (to exit command loop)
  ScrStack.Push(Scripter);           // Push current scripter
  Scripter:=TScripter.Create;        // Create a new one
  Scripter.LoadScript(SP);           // Load the script
  Scripter.Parms:=PL;                // Load its PList
  Application.ProcessMessages;
End;

Procedure TOCLThread.TerminateScript;
begin
  If Scripter<>NIL then begin        // Is there a scripter?
    Scripter.Free;                   // Yes, Free it
    Scripter:=ScrStack.Pop;          // And pop the next one off the stack
  End;
  If Scripter<>NIL then              // Popped the last one (NIL)?
    Scripter.ScriptState:=SSRUN;     // No, Restore to runable
End;

Procedure TOCLThread.TerminateAllScripts;
begin
  While Scripter<>NIL do             // Pop off and free until no scripters left
    TerminateScript;
  EMCMsg(CCSCRIPT,0,0,'');           // Indicate on main form
End;

procedure TOCLThread.ReStart;
begin
  If Scripter<>NIL then begin
    Scripter.ScriptState:=SSRUN;
    Resume;
  End;
end;

Function TOCLThread.ScriptState: TScriptState;
begin
  If Scripter=NIL then Result:=SSNONE
                  else Result:=Scripter.ScriptState;
end;

function TOCLThread.ScriptName: String;
begin
  If Scripter=NIL then Result:=''
                  else Result:=Scripter.ScriptPath;
end;

// OCLThread Active Scripter loop
Procedure TOCLThread.RunScripter;
Begin
  If Scripter=NIL then Exit;           // Is there a scripter?
  EMCMsg(CCSCRIPT,1,0,'');             // Yes, Indicate active in main window
  With Scripter do begin
    While ScriptState=SSRUN do begin   // Is it runable?
      Try
        ProcessCommands;               // Yes, Process script commands
      Except
        On E: Exception do begin
          BuildErrorText(E.Message);         // Format error text
          Synchronize(DoShowErrorText);      // Display using VCL thread
          ScriptState:=SSSTOP;               // Set script to stop
          ShowWindow(FormHandles[FIMain],SW_Show);// Ensure main window is visible
        End;
      End;
      If Scripter=NIL then Break;      // Has scripter been terminated?              
      If ScriptState=SSSTOP then       // Has script stopped?
        TerminateScript;               // Yes, terminate (restores any caller)
    End;
  End;
  EMCMsg(CCSCRIPT,0,0,'');             // Indicate inactive in main window
End;

// OCLThread Main loop
Procedure TOCLThread.Execute;
Begin
  While Not Terminated do begin        // Run thread until terminated
    RunScripter;                       // Run scripter until it needs to wait
    If ScriptState=SSRUN then          // Did it bounce out due to EXEC/EXIT?
      Continue;                        // Yes, Continue with new scripter
    // Sleep
    Suspend;                           // Wait for CPU/User
    // Wakeup
    If Scripter<>NIL then              // Wakeup. Is there still a scripter?
      With Scripter do
        If ScriptState=SSWAIT then     // And is it waiting?
          ScriptState:=SSRUN;          // Yes, make it runable again
  End;
End;

Initialization
  OCLThread:=TOCLThread.Create(True);
Finalization
  TerminateThread(OCLThread.Handle,9);
  OCLThread.Free;
End.


